/*
 *  SR Run-Time Support.  Input-Output Routines for use under UNIX.
 *
 *  Warning:  These routines depend to some extent on the internals of the
 *  4.3BSD standard I/O library.  Portability to other systems is questionable.
 *  The code could be more portable if we didn't want to make SR files
 *  compatible with fprintf etc. for use by externals.
 */

#include <ctype.h>
#include <varargs.h>
#include "rts.h"

static int rdbool(), rdline(), rdtok(), inchar(), outchar(), check_error();
static void outstr(), flushout(), wready();




/*
 *  Input or output a character.
 */

#define INCH(f) (--(f)->_cnt>=0?(int)(*(unsigned char*)(f)->_ptr++):inchar(f))
#define OUCH(f,c) (--(f)->_cnt>=0?(*(f)->_ptr++=(c)):outchar(f,c))



/*
 *  Return rv if fp is the noop file.
 *  Abort if the null file, or if the pointer is null.
 */
#define CHECK_FILE(fp,rv)	{ \
    if (!(fp)) \
	sr_abort ("null pointer for file descriptor"); \
    if ((fp) == (FILE *) NOOP_FILE) \
	return rv; \
    if ((fp) == (FILE *) NULL_FILE) \
	sr_abort ("operation attempted on null file"); \
    if ((fp)->_flag & (_IOREAD|_IOWRT|_IORW) == 0) \
	sr_abort ("file not open"); \
}





/*
 *  Copy file name to null-terminated string and open the file.
 *  Return descriptor, or a descriptor for the null file if open fails.
 */
FILE *
sr_open (fname, len, mode)			/* RTS Primitive */
char *fname;
int len;
int mode;
{
    char name [MAX_PATH], *fn;
    FILE *fp;
    static char *flags [] = { "r", "w", "r+" };

    sr_check_stk();
    if (len >= MAX_PATH)
	sr_abort ("file name too large for open");
    for (fn = name ; len-- ; *fn++ = *fname++)
	;
    *fn = '\0';

    if ((fp = fopen (name, flags [mode])) == NULL)
	return ((FILE *) NULL_FILE);
    else
	return (fp);
}



/*
 *  Flush a file's buffers.  The file remains open.
 */
void
sr_flush (pfp)					/* RTS Primitive */
FILE **pfp;
{
    sr_check_stk();
    CHECK_FILE (*pfp, /*void*/);
    flushout(*pfp);
}



/*
 *  Close a file.  Invalidate file descriptor.
 *  For asynchronous terminals reset descriptor flags.
 */
void
sr_close (pfp)					/* RTS Primitive */
FILE **pfp;
{
    sr_check_stk();
    CHECK_FILE (*pfp, /*void*/);
    flushout(*pfp);
    if (fclose (*pfp) == EOF)
	check_error (*pfp);
    *pfp = (FILE *) NULL_FILE;
}



/*
 *  Read zero or more arguments according to given format.
 */
/*VARARGS*/
int
sr_read (va_alist)				/* RTS Primitive */
va_dcl
{
    va_list ap;
    FILE *fp;
    char c, f, buf[20], *fmt, *cp;
    int nread, *ip, len, i, n, ret;

    sr_check_stk();
    va_start (ap);
    fp = va_arg (ap, FILE *);
    CHECK_FILE (fp, EOF);

    fmt = va_arg (ap, char *);
    nread = ret = 0;
    while (f = *fmt++)  {
	switch (f) {
	    case 'b':	/* boolean argument */
		cp = va_arg (ap, char *);
		ret = rdbool (fp, cp);
		break;
	    case 'd':	/* integer argument */
		ip = va_arg (ap, int *);
		n = rdtok(fp,buf,sizeof(buf)-1);
		if (n < 0)
		    ret = EOF;
		else {
		    c = buf[n] = '\0';
		    n = sscanf (buf, "%d%c", &i, &c);
		    ret = (n != 1) || (c != '\0');
		    if (ret == 0)
			*ip = i;	/* store only if valid */
		}
		break;
	    case 'c':	/* char array argument */
		cp = va_arg (ap, char *);
		len = va_arg (ap, int);
		n = rdline (fp, cp, len);
		if (n < 0)
		    ret = EOF;
		else
		    while (n < len)
			cp[n++] = ' ';
		break;
	    case 's':	/* string argument */
		ip = (int *) va_arg (ap, char *);
		cp = (char *) (ip + 1);
		len = va_arg (ap, int);
		n = rdline (fp, cp, len);
		if (n < 0)
		    ret = EOF;
		else
		    *ip = n;
		break;
	    default:
		sr_abort("bad read format");
	}
	if (ret != 0)  {
	    if (nread > 0)
		return (nread);
	    else if (ret == EOF)
		return (EOF);
	    else
		return (0);
	}
	nread++;
    }
    va_end (ap);
    return (nread);
}



/*
 *  Read a Boolean literal into given address from the named file.
 *  The accepted values are "true" and "false".
 *  Return 1 if bad, 0 if okay, EOF for EOF.
 */
static int
rdbool (fp, pbool)
FILE *fp;
char *pbool;
{
    int n;
    char buf[5];

    n = rdtok(fp,buf,6);
    if (n == EOF)
	return (EOF);
    if (n==4 && buf[0]=='t' && buf[1]=='r' && buf[2]=='u' && buf[3]=='e') {
	*pbool = TRUE;
	return (0);
    }
    if (n==5&&buf[0]=='f'&&buf[1]=='a'&&buf[2]=='l'&&buf[3]=='s'&&buf[4]=='e') {
	*pbool = FALSE;
	return (0);
    }
    return (1);
}



/*
 *  Read a line to buffer s of size n; return number of chars stuffed, or EOF.
 */
static int
rdline (fp, s, n)
FILE *fp;
char *s;
int n;
{
    register int c, i;

    i = 0;
    for (;;) {
	c = INCH (fp);
	if (c == EOF)
	    return ((i != 0) ? i : EOF);
	if (c == '\n')
	    return (i);
	if (++i > n)  {
	    ungetc (c, fp);
	    return (n);
	}
	*s++ = c;
    }
}



/*
 *  Read a token to buffer s of size n; return number of chars stuffed, or EOF.
 */
static int
rdtok (fp, s, n)
FILE *fp;
char *s;
int n;
{
    int c, i;

    while (isspace (c = INCH (fp)))
	;
    if (c == EOF)
	return (EOF);
    i = 0;
    while ((c != EOF) && (!isspace (c)))  {
	if (i < n)
	    s[i++] = c;
	c = INCH (fp);
    }
    while (c != '\n' && isspace (c))
	c = INCH (fp);
    if (c != '\n')
	ungetc (c, fp);
    return (i);
}




/*
 *  Write zero or more arguments according to given format.
 */
/*VARARGS*/
void
sr_write (va_alist)				/* RTS Primitive */
va_dcl
{
    va_list ap;
    FILE *fp;
    char *fmt, *str, c, *p;
    char buf[20];
    int n;
    long l;

    sr_check_stk();
    va_start (ap);
    fp = va_arg (ap, FILE *);
    CHECK_FILE (fp, /*void*/);

    fmt = va_arg (ap, char *);
    while (c = *fmt++) {
	switch (c)  {
	    case 'b':	/* boolean argument */
		outstr (fp, va_arg (ap, int) ? "true" : "false", -1);
		break;
	    case 'd':	/* integer argument */
		n = va_arg (ap, int);
		sprintf (buf, "%d", n);
		outstr (fp, buf, -1);
		break;
	    case 'p':	/* integer argument */
		p = va_arg (ap, char *);
		if (!p)
		    strcpy(buf,"==null==");
		else {
		    l = (long) p;
		    sprintf (buf, "%08lX", l);
		}
		outstr (fp, buf, -1);
		break;
	    case 's':	/* string argument */
		str = va_arg (ap, char *);
		n = va_arg (ap, int);
		outstr (fp, str, n);
		break;
	    default:
		OUCH (fp, c);
	}
    }
    va_end (ap);
}



/*
 *  Output a string to a file.  If len < 0, output up to '\0'.
 */
static void
outstr (fp, s, len)
FILE *fp;
char *s;
int len;
{
    if (len < 0)
	while (*s)
	    OUCH (fp, *s++);
    else
	while (len--)
	    OUCH (fp, *s++);
}



/*
 *  Read up to "len" characters from the specified file.
 *  Set new string length in *slen if given.
 */
int
sr_get (fp, str, len, slen)			/* RTS Primitive */
FILE *fp;
char *str;
int len;
int *slen;
{
    int c;
    int count = 0;

    sr_check_stk();
    CHECK_FILE (fp, EOF);
    for ( ; len > 0 ; len--, count++) {
	if ((c = INCH (fp)) == EOF) {
	    if (count == 0)
		return (EOF);
	    break;
	}
	*str++ = c;
    }
    if (slen)
	*slen = count;
    return (count);
}



/*
 *  Move the file pointer.
 */
int
sr_seek (fp, seektype, offset)			/* RTS Primitive */
FILE *fp;
int seektype;
int offset;
{
    sr_check_stk();
    CHECK_FILE (fp, 0);
    if (fseek (fp, (long) offset, seektype) < 0)
	check_error (fp);
    return ((int) ftell (fp));
}



/*
 *  Determine the position of the file pointer.
 */
int
sr_where (fp)					/* RTS Primitive */
FILE *fp;
{
    long pos;

    sr_check_stk();
    CHECK_FILE (fp, 0);
    if ((pos = ftell (fp)) < 0L)
	check_error (fp);
    return ((int) pos);
}



/*
 *  Remove the named file after converting name to UNIX string.
 */
bool
sr_remove (fname, len)				/* RTS Primitive */
char *fname;
int len;
{
    char name [MAX_PATH], *fn;

    sr_check_stk();
    if (len >= MAX_PATH)
	sr_abort ("file name too large for remove");
    for (fn = name ; len-- ; *fn++ = *fname++);
    *fn = '\0';
    return ((unlink (name) < 0) ? FALSE : TRUE);
}



/*
 *  Initialize I/O pointers.  The "stdxxx" file macros from
 *  <sysio.h> aren't legal static initializers on all systems.
 */
void
sr_init_io ()
{
    sr_stdin = stdin;
    sr_stdout = stdout;
    sr_stderr = stderr;
}



/*
 *  Input a character without blocking other processes.
 *  (Called by INCH macro when the buffer is empty.)
 */
static int
inchar (fp)
FILE *fp;
{
    int c;

    fp->_cnt++;			/* reset count clobbered by INCH macro */
    if (fp == stdin)  {
	flushout(stdout);	/* flush out any prompts */
	flushout(stderr);
    }
    wready (fp, INPUT);		/* wait for input file ready */
    c = getc(fp);		/* now that we know it won't wait... */
    if (c < 0)
	return (check_error(fp));
    else
	return (c);
}



/*
 *  Output a character without blocking other processes.
 *  (Called by OUCH macro when the count is zero.)
 *
 *  For some buffering schemes, the count is always zero, so check the buffer
 *  contents;  if there's already something there, we checked before and
 *  nothing's been written since, so we need not check again.
 *
 *  NOT FOOLPROOF; depends on how stdio works.
 */
static int
outchar (fp, c)
FILE *fp;
char c;
{
    fp->_cnt++;			/* reset count clobbered by OUCH macro */
    if (fp->_ptr == fp->_base)	/* only need to select() once */
	wready (fp, OUTPUT);	/* wait for output file ready */
    if (putc(c,fp) == EOF)	/* now we know this is safe */
	check_error (fp);	/* handle error */
    return (c);
}



/*
 *  Flush an output file without blocking other processes.
 *
 *  NOT FOOLPROOF: has been seen to fail on Vax, work on Sun.
 */
static void
flushout (fp)
FILE *fp;
{
    if (fp->_ptr == fp->_base)	/* if buffer empty, return */
	return;
    wready (fp, OUTPUT);	/* wait until ready for output */
    if (fflush(fp) == EOF)	/* flush buffer */
	check_error (fp);	/* handle error */
}



/*
 *  Wait until a file is ready for input or output.
 */
static void
wready (fp, inout)
FILE *fp;
enum io_type inout;
{
    int fd;
    fd_set fdset;
    static fd_set zeroset;

    fd = fileno (fp);
    fdset = zeroset;
    FD_SET (fd, &fdset);
    sr_iowait (&fdset, &fdset, inout);
}




/*
 *  Check for I/O errors and abort if there were any.
 *  If not, clear possible EOF condition.  Return EOF if so, or 0.
 */
static int
check_error (fp)
FILE *fp;
{
    if (ferror(fp))  {
	perror("SR I/O");
	sr_abort("I/O error");
    }
    if (feof(fp))  {
	clearerr (fp);		/* clear EOF */
	return (EOF);
    }
    return (0);
}
