/*
 *  SR Run-Time Support.  Invocation and Proc Operation Termination Routines.
 */

#include "rts.h"

static sem op_spawn();



/*
 *  Invoke a proc or input operation by either call or send.
 */
invb
sr_invoke (ibp)					/* RTS Primitive */
invb ibp;
{
    oper op;
    sem wait;

    sr_check_stk();

    /*
     *  Handle a co send by making a copy of the block,
     *  which is given back to the GC,
     *  and converting the original block into a regular SEND.
     *  note: if ever throw in exceptions on invocations,
     *  this approach might need to be changed.
     */
    if (ibp->type == COSEND_IN) {
	sr_co_send (ibp);
	ibp->type = SEND_IN;
    }

    /*
     *	Check for null or noop capability.
     */
    if (ibp->opc.oper_index == 0) {
	if (ibp->opc.seqn == NOOP_SEQN) {
	    if (ibp->type == SEND_IN) {
	    	sr_free ((daddr) ibp);
		return (NULL);
	    }
	    if (ibp->type == COCALL_IN) {
		/* Simulate the normal actions for a co call. */
		sr_co_call (ibp);
		sr_co_call_done (ibp);
	    }
	    return (ibp);
	}
	sr_abort ("attempting to invoke null operation");
    }

    if (ibp->opc.vm != sr_my_vm) {
	pach ph;

	/* Send invocation request to remote machine.  */

	if (ibp->type == COCALL_IN) {
	    sr_co_call (ibp);
	    ibp->type = REM_COCALL_IN;
	}

	ph = (pach) ibp;
	ibp = (invb) sr_remote (ibp->opc.vm, REQ_INVOKE, ph, ph->size);
	sr_free ((daddr) ph);

	if (ibp->type == REM_COCALL_IN)
	    sr_co_call_done (ibp);

	return (ibp);
    }

    op = sr_optab + ibp->opc.oper_index;
    if (ibp->opc.seqn != op->seqn) {
	/* raise exception */
	sr_abort
	    ("attempting to invoke operation that no longer exists");
	return (NULL);
    }

    ibp->replied = FALSE;
    
    if (op->type == INPUT_OP)
	switch (ibp->type) {
	    case CALL_IN:
	    case REM_COCALL_IN:
		ibp->wait = sr_make_sem (0);
		sr_invk_iop (ibp, op->u.clap);
		P (ibp->wait);
		sr_kill_sem (ibp->wait);
		break;
		
	    case SEND_IN:
		sr_invk_iop (ibp, op->u.clap);
		break;
		
	    case COCALL_IN:
		sr_co_call (ibp);
		sr_invk_iop (ibp, op->u.clap);
		break;
	    
	    case COSEND_IN:
		sr_abort ("co send in invoke (in) -- shouldn't happen");
		
	    default:
		sr_abort ("GC error in invoke (in)");
	}

    else  /* op->type == PROC_OP or PROC_REP_OP */
	switch (ibp->type) {
	    case CALL_IN:
	    case REM_COCALL_IN:
		/* Do direct call if proc does not reply to its invoker. */
		if (op->type != PROC_REP_OP) {
		    enum in_type old_itype;
		    invb old_ibp;
		    
		    old_itype = sr_cur_proc->itype;
		    old_ibp = sr_cur_proc->ibp;
		    
		    sr_cur_proc->itype = CALL_IN;
		    sr_cur_proc->ibp = ibp;

		    (*op->u.code) (op->res->crb_addr,
			op->res->rv_base, ibp, RTS_OWN);

		    sr_cur_proc->itype = old_itype;
		    sr_cur_proc->ibp = old_ibp;
		}
		else {
		    wait = op_spawn (ibp, op, CALL_IN);
		    P (wait);
		    sr_kill_sem (wait);
		}
		break;

	    case SEND_IN:
		(void) op_spawn (ibp, op, SEND_IN);
		break;

	    case COCALL_IN:
		sr_co_call (ibp);
		(void) op_spawn (ibp, op, COCALL_IN);
		break;
	
	    case COSEND_IN:
		sr_abort ("co send in invoke (proc) -- shouldn't happen");

	    default:
		sr_abort ("GC error in invoke (proc)");
	}

    return (ibp);
}



/*
 *  Create a new process to service a proc operation
 *  invocation.  Return waiting semaphore for CALL_IN's.
 */
static sem
op_spawn (ibp, op, type)
invb ibp;
oper op;
enum in_type type;
{
    proc pr;
    sem wait;

    pr = sr_spawn (op->u.code, op->res,
	    op->res->crb_addr, op->res->rv_base, ibp, RTS_OWN);

    pr->ptype = PROC;
    pr->itype = type;
    if (type == CALL_IN)  pr->wait = wait = sr_make_sem (0);
    pr->ibp = ibp;

    sr_activate (pr);
    return (wait);
}



/*
 *  Send an early reply to the invoker of an operation.
 *  Copy invocation block so invoker and invokee do not share
 *  the same argument area.  Return pointer to new copy.
 *  Also does replies for initial/final code (indicated by null
 *  ibp).  Copy resource capability when replying in initial.
 */
invb
sr_reply (ibp)					/* RTS Primitive */
invb ibp;
{
    daddr src, dest;
    oper op;
    invb new_ibp;

    sr_check_stk();

    if (ibp == NULL) {
	/*
	 *  The reply is in initialization or finalization code.
	 *  Act like sr_finished_init() or sr_finished_final().
	 */
	if (sr_cur_proc->ptype == INITIAL) {
	    if (sr_cur_res->status & INIT_REPLY) {
		rts_error ("ignoring extra reply in initial");
		return (NULL);
	    }

	    if ((dest = (daddr) sr_cur_res->rcp) != NULL) {
		src = sr_cur_res->rv_base;
		while (sr_cur_res->rc_size--)
		    *dest++ = *src++;
	    }

	    sr_cur_res->status |= INIT_REPLY;
	    V (sr_cur_proc->wait);
	    return (NULL);
	}

	else if (sr_cur_proc->ptype == FINAL) {
	    sr_abort ("reply in final not implemented");

/* old code follows; didn't work because after reply, our caller would
 * proceed to free our memory, etc., while we continued.
 *	    if (sr_cur_res->status & FINAL_REPLY) {
 *		rts_error ("ignoring extra reply in final");
 *		return (NULL);
 *	    }
 *	    sr_cur_res->status |= FINAL_REPLY;
 *	    V (sr_cur_proc->wait);
 *	    return (NULL);
 */
	}
	else
	    sr_abort ("invalid reply (null ibp, not I/F)");
    }

    /*
     *	Create a new ibp and copy the old invocation to the new.
     *  Make it look like a send invocation now that we've replied.
     *  This will get it automatically freed later.
     */
    ibp->replied = TRUE;
    new_ibp = sr_dup_invb (ibp);
    new_ibp->type = SEND_IN;		/* make it look like a send */

    op = sr_optab + ibp->opc.oper_index;
    if (op->type == INPUT_OP)
	switch (ibp->type) {
	    case CALL_IN:
	    case REM_COCALL_IN:
		V (ibp->wait);
		return (new_ibp);
		
	    case SEND_IN:
		sr_abort ("reply to send invocation");
		/*NOTREACHED*/
		
	    case COCALL_IN:
		sr_co_call_done (ibp);
		return (new_ibp);
	}
    
    else if (op->type == PROC_OP || op->type == PROC_REP_OP)
	switch (ibp->type) {
	    case CALL_IN:
	    case REM_COCALL_IN:
		if (op->type == PROC_OP)
		    sr_abort ("reply in called proc");

		sr_cur_proc->ibp = new_ibp;
		V (sr_cur_proc->wait);
		return (new_ibp);

	    case SEND_IN:
		sr_abort ("reply to send invocation");
		/*NOTREACHED*/

	    case COCALL_IN:
		sr_co_call_done (ibp);
		return (new_ibp);
	}
    
    else
	sr_abort ("invalid operation type in reply");
    /*NOTREACHED*/
}



/*
 *	An input operation has finished.  Clean up.
 */
void
sr_finished_input (ibp)				/* RTS Primitive */
invb ibp;
{

    	sr_check_stk();

    	switch (ibp->type) {
	    case CALL_IN:
	    case REM_COCALL_IN:
	    	if (! ibp->replied) {
		    V (ibp->wait);
		    /* formerly did a context switch here, but no need to. */
	    	}
	    	return;
	    
	    case SEND_IN:
	    	sr_free ((daddr) ibp);
	    	return;
	    
	    case COCALL_IN:
	    	if (! ibp->replied)  sr_co_call_done (ibp);
	    	return;
	}
}



/*
 *  A proc operation has finished.  Release the invocation block.
 *  If the proc was called, notify the invoker.  Commit suicide.
 */
void
sr_finished_proc ()				/* RTS Primitive */
{
    oper op;

    sr_check_stk();

    switch (sr_cur_proc->itype) {
	case CALL_IN:
	case REM_COCALL_IN:
	    op = sr_optab + sr_cur_proc->ibp->opc.oper_index;
	    if (op->type != PROC_REP_OP)  return;
	    if (! sr_cur_proc->ibp->replied)  V (sr_cur_proc->wait);
	    break;

	case SEND_IN:
	    sr_free ((daddr) sr_cur_proc->ibp);
	    break;

	case COCALL_IN:
	    if (! sr_cur_proc->ibp->replied)
		sr_co_call_done (sr_cur_proc->ibp);
	    break;
    }

    sr_kill (sr_cur_proc, TRUE);
}



/*
 *  Reject an invocation because the operation was killed
 *  before the invocation was accepted.
 */
void
sr_rej_inv (ibp)
invb ibp;
{
    oper op;

    op = sr_optab + ibp->opc.oper_index;
    if (op->type == PROC_OP || op->type == PROC_REP_OP)
	sr_abort ("rejecting a proc op");

    switch (ibp->type) {
	case CALL_IN:
	case REM_COCALL_IN:
	    /* indicate rejection in status field */
	    V (ibp->wait);
	    return;
	    
	case SEND_IN:
	    sr_free ((daddr) ibp);
	    return;
	    
	case COCALL_IN:
	    /* indicate rejection in status field */
	    sr_co_call_done (ibp);
	    return;
    }
}


/*
 *  Duplicate an invocation block and return the address of the copy.
 */
invb
sr_dup_invb (ibp)
invb ibp;
{
    invb new;
    int n;

    n = INVOCATION_HEADER_SIZE + ibp->arg_size;
    new = (invb) sr_own_alloc (n, sr_cur_res);
    memcpy ((daddr) new, (daddr) ibp, n);
    return new;
}
