/*
 * File: rmemmgt.c
 *  Contents: allocation routines, block description arrays, dump routines,
 *  garbage collection, sweep
 */

#include "../h/rt.h"
#include "gc.h"

#ifdef IconAlloc
/*
 *  If IconAlloc is defined the system allocation routines are not overloaded.
 *  The names are changed so that Icon's allocation routines are independently
 *  used.  This works as long as no other system calls cause the break value
 *  to change.
 */
#define malloc mem_alloc
#define free mem_free
#define realloc mem_realloc
#define calloc mem_calloc
#endif					/* IconAlloc */

#ifdef RunStats
#include "rxinstr.h"
#endif					/* RunStats */

#if !MSDOS			/* Microsoft C chokes on this */
char *memcpy();			/* used to copy memory in chunks */
#endif					/* MSDOS */

/*
 * The following code is operating-system dependent. Definition of brk().
 */

#if PORT
#endif					/* PORT */

#if AMIGA || ATARI_ST
/* nothing needed because of FixedRegions */
#endif					/* AMIGA || ATARI_ST */

#if MSDOS
/* taken care of elsewhere */
#endif					/* MSDOS */

#if MACINTOSH || UNIX || VMS
extern char *brk();
#endif					/* MACINTOSH || UNIX || VMS */

#if VM || MVS
#endif					/* MV || MVS */

/*
 * End of operating-system specific code.
 */

extern char *sbrk();

word coll_stat = 0;
word coll_str = 0;
word coll_blk = 0;
word coll_tot = 0;

struct descrip **quallist;	/* string qualifier list */
struct descrip **qualfree;	/* qualifier list free pointer */
struct descrip **equallist;	/* end of qualifier list */

int qualfail;			/* flag: quailifer list overflow */


/*
 * Note: function calls beginning with "MM" are just empty macros
 * unless MEMMON is defined.
 */

/*
 * Allocated block size table (sizes given in bytes).  A size of -1 is used
 *  for types that have no blocks; a size of 0 indicates that the
 *  second word of the block contains the size; a value greater than
 *  0 is used for types with constant sized blocks.
 */

int bsizes[] = {
    -1, 			/* 0, not used */
    -1, 			/* 1, not used */
    0,				/* T_Bignum (2), bignum */
     sizeof(struct b_real),	/* T_Real (3), real number */
     sizeof(struct b_cset),	/* T_Cset (4), cset */
     sizeof(struct b_file),	/* T_File (5), file block */
     0, 			/* T_Proc (6), procedure block */
     sizeof(struct b_list),	/* T_List (7), list header block */
     sizeof(struct b_table),	/* T_Table (8), table header block */
     0, 			/* T_Record (9), record block */
     sizeof(struct b_telem),	/* T_Telem (10), table element block */
     0, 			/* T_Lelem (11), list element block */
     sizeof(struct b_tvsubs),	/* T_Tvsubs (12), substring trapped variable */
    -1, 			/* T_Tvkywd (13), keyword trapped variable */
     sizeof(struct b_tvtbl),	/* T_Tvtbl (14), table element trapped variable */
     sizeof(struct b_set),	/* T_Set  (15), set header block */
     sizeof(struct b_selem),	/* T_Selem  (16), set element block */
     0, 			/* T_Refresh (17), refresh block */
    -1, 			/* T_Coexpr (18), co-expression block */
     0,				/* T_External (19) external block */
    };

/*
 * Table of offsets (in bytes) to first descriptor in blocks.  -1 is for
 *  types not allocated, 0 for blocks with no descriptors.
 */
int firstd[] = {
    -1, 			/* 0, not used */
    -1, 			/* 1, not used */
     0,				/* T_Bignum (2), bignum */
     0, 			/* T_Real (3), real number */
     0, 			/* T_Cset (4), cset */
     3*WordSize,		/* T_File (5), file block */
     7*WordSize,		/* T_Proc (6), procedure block */
     2*WordSize,		/* T_List (7), list header block */
     2*WordSize,		/* T_Table (8), table header block */
     2*WordSize,		/* T_Record (9), record block */
     2*WordSize,		/* T_Telem (10), table element block */
     5*WordSize,		/* T_Lelem (11), list element block */
     3*WordSize,		/* T_Tvsubs (12), substring trapped variable */
    -1, 			/* T_Tvkywd (13), keyword trapped variable */
     2*WordSize,		/* T_Tvtbl (14), table element trapped variable */
     2*WordSize,		/* T_Set  (15), set header block */
     2*WordSize,		/* T_Selem  (16), set element block */
     (4+Wsizeof(struct pf_marker))*WordSize,
				/* T_Refresh (17), refresh block */
    -1, 			/* T_Coexpr (18), co-expression block */
     0,				/* T_External (19), external block */
    };

/*
 * Table of block names used by debugging functions.
 */
char *blkname[] = {
   "illegal object",			/* T_Null (0), not block */
   "illegal object",			/* T_Integer (1), not block */
   "bignum",				/* T_Bignum (2) */
   "real number",			/* T_Real (3) */
   "cset",				/* T_Cset (4) */
   "file",				/* T_File (5) */
   "procedure",				/* T_Proc (6) */
   "list",				/* T_List (7) */
   "table",				/* T_Table (8) */
   "record",				/* T_Record (9) */
   "table element",			/* T_Telem (10) */
   "list element",			/* T_Lelem (11) */
   "substring trapped variable",	/* T_Tvsubs (12) */
   "keyword trapped variable",		/* T_Tvkywd (13) */
   "table element trapped variable",	/* T_Tvtbl (14) */
   "set",				/* T_Set (15) */
   "set elememt",			/* T_Selem (16) */
   "refresh block",			/* T_Refresh (17) */
   "co-expression",			/* T_Coexpr (18) */
   "external block",			/* T_External (19) */
   };

#ifdef FixedRegions
#include "rmemfix.c"
#else					/* FixedRegions */
#include "rmemexp.c"
#endif					/* FixedRegions */

/*
 * alcblk - returns pointer to nbytes of free storage in block region.
 */

static union block *alcblk(nbytes,tcode)
uword nbytes;
int tcode;
   {
   register uword fspace, *sloc;

   Inc(al_n_total);
   IncSum(al_bc_btotal,nbytes);
   /*
    * See if there is enough room in the block region.
    */
   fspace = (uword)blkend - (uword)blkfree;
   if (fspace < nbytes)
      syserr("block allocation botch");

   /*
    * If monitoring, show the allocation.
    */
   MMAlc(nbytes,tcode);

   /*
    * Decrement the free space in the block region by the number of bytes
    *  allocated and return the address of the first byte of the allocated
    *  block.
    */
   sloc = (uword *)blkfree;
   blkneed -= nbytes;
   blkfree = blkfree + nbytes;
   BlkType(sloc) = tcode;
   return (union block *)(sloc);
   }

/*
 * alcreal - allocate a real value in the block region.
 */

struct b_real *alcreal(val)
double val;
   {
   register struct b_real *blk;

   Inc(al_n_real);
   blk = (struct b_real *) alcblk((uword)sizeof(struct b_real), T_Real);
#ifdef Double
/* access real values one word at a time */
   { int *rp, *rq;
     rp = (word *) &(blk->realval);
     rq = (word *) &val;
     *rp++ = *rq++;
     *rp   = *rq;
   }
#else					/* Double */
   blk->realval = val;
#endif					/* Double */
   return blk;
   }

/*
 * alcbignum - allocate an n-digit bignum in the block region
 */

struct b_bignum *alcbignum (n)
{
    struct b_bignum *blk;
    uword size = sizeof (struct b_bignum) - sizeof (DIGIT)
	+ (n * sizeof (DIGIT));
    size = (size + 3) & -4;
    if (blkreq (size) == Error)
	runerr (0, 0);
    blk = (struct b_bignum *) alcblk (size, T_Bignum);
    blk->blksize = size;
    blk->left = blk->sign = 0;
    blk->right = n - 1;

    return blk;}

/*
 * alccset - allocate a cset in the block region.
 */

struct b_cset *alccset()
   {
   register struct b_cset *blk;
   register int i;

   Inc(al_n_cset);
   blk = (struct b_cset *)alcblk((uword)sizeof(struct b_cset), T_Cset);
   blk->size = -1;		/* flag size as not yet computed */

   /*
    * Zero the bit array.
    */
   for (i = 0; i < CsetSize; i++)
     blk->bits[i] = 0;
   return blk;
   }

/*
 * alcfile - allocate a file block in the block region.
 */

struct b_file *alcfile(fd, status, name)
FILE *fd;
int status;
struct descrip *name;
   {
   register struct b_file *blk;

   Inc(al_n_file);
   blk = (struct b_file *) alcblk((uword)sizeof(struct b_file), T_File);
   blk->fd = fd;
   blk->status = status;
   blk->fname = *name;
   return blk;
   }

/*
 * alcrecd - allocate record with nflds fields in the block region.
 */

struct b_record *alcrecd(nflds, recptr)
int nflds;
struct descrip *recptr;
   {
   register struct b_record *blk;
   register int i, size;

   Inc(al_n_recd);
   size = Vsizeof(struct b_record) + nflds*sizeof(struct descrip);
   blk = (struct b_record *) alcblk((uword)size, T_Record);
   blk->blksize = size;
   blk->recdesc.dword = D_Proc;
   BlkLoc(blk->recdesc) = (union block *)recptr;
   /*
    * Assign &null to each field in the record.
    */
   for (i = 0; i < nflds; i++)
       blk->fields[i] = nulldesc;
   return blk;
   }

/*
 * alcextrnl - allocate an external block.
 */

struct b_external *alcextrnl(n)
int n;
   {
   register struct b_external *blk;

   blk = (struct b_external *)alcblk((uword)(n * sizeof(word)), T_External);
   blk->blksize = (n + 3) * sizeof(word);
   blk->descoff = 0;
   /* probably ought to clear the rest of the block */
   return blk;
   }

/*
 * alclist - allocate a list header block in the block region.
 */

struct b_list *alclist(size)
uword size;
   {
   register struct b_list *blk;

   Inc(al_n_list);
   blk = (struct b_list *) alcblk((uword)sizeof(struct b_list), T_List);
   blk->size = size;
   blk->listhead = nulldesc;
   blk->listtail = nulldesc;
   return blk;
   }

/*
 * alclstb - allocate a list element block in the block region.
 */

struct b_lelem *alclstb(nslots, first, nused)
uword nslots, first, nused;
   {
   register struct b_lelem *blk;
   register word i, size;

   Inc(al_n_lstb);
   size = Vsizeof(struct b_lelem)+nslots*sizeof(struct descrip);
   blk = (struct b_lelem *)alcblk((uword)size, T_Lelem);
   blk->blksize = size;
   blk->nslots = nslots;
   blk->first = first;
   blk->nused = nused;
   blk->listprev = nulldesc;
   blk->listnext = nulldesc;
   /*
    * Set all elements to &null.
    */
   for (i = 0; i < nslots; i++)
      blk->lslots[i] = nulldesc;
   return blk;
   }

/*
 * alctable - allocate a table header block in the block region.
 */

struct b_table *alctable(def)
struct descrip *def;
   {
   register int i;
   register struct b_table *blk;

   Inc(al_n_table);
   blk = (struct b_table *) alcblk((uword)sizeof(struct b_table), T_Table);
   blk->size = 0;
   blk->defvalue = *def;
   /*
    * Zero out the buckets.
    */
   for (i = 0; i < TSlots; i++)
      blk->buckets[i] = nulldesc;
   return blk;
   }

/*
 *  alctelem - allocate a table element block in the block region.
 */

struct b_telem *alctelem()
   {
   register struct b_telem *blk;

   Inc(al_n_telem);
   blk = (struct b_telem *) alcblk((uword)sizeof(struct b_telem), T_Telem);
   blk->hashnum = 0;
   blk->clink = nulldesc;
   blk->tref = nulldesc;
   blk->tval = nulldesc;
   return blk;
   }

/*
 * alcset - allocate a set header block.
 */

struct b_set *alcset()
   {
   register int i;
   register struct b_set *blk;

   blk = (struct b_set *) alcblk((uword)sizeof(struct b_set), T_Set);
   blk->size = 0;
   /*
    *  Zero out the buckets.
    */
   for (i = 0; i < SSlots; i++)
      blk->sbucks[i] = nulldesc;
      return blk;
     }

/*
 *   alcselem - allocate a set element block.
 */

struct b_selem *alcselem(mbr,hn)
word hn;
struct descrip *mbr;

   { register struct b_selem *blk;

     blk = (struct b_selem *) alcblk((uword)sizeof(struct b_selem), T_Selem);
     blk->clink = nulldesc;
     blk->setmem = *mbr;
     blk->hashnum = hn;
     return blk;
     }

/*
 * alcsubs - allocate a substring trapped variable in the block region.
 */

struct b_tvsubs *alcsubs(len, pos, var)
word len, pos;
struct descrip *var;
   {
   register struct b_tvsubs *blk;

   Inc(al_n_subs);
   blk = (struct b_tvsubs *) alcblk((uword)sizeof(struct b_tvsubs), T_Tvsubs);
   blk->sslen = len;
   blk->sspos = pos;
   blk->ssvar = *var;
   return blk;
   }

/*
 * alctvtbl - allocate a table element trapped variable block in the block
 *  region.
 */

struct b_tvtbl *alctvtbl(tbl, ref, hashnum)
register struct descrip *tbl, *ref;
word hashnum;
   {
   register struct b_tvtbl *blk;

   Inc(al_n_tvtbl);
   blk = (struct b_tvtbl *) alcblk((uword)sizeof(struct b_tvtbl), T_Tvtbl);
   blk->hashnum = hashnum;
   blk->clink = *tbl;
   blk->tref = *ref;
   blk->tval = nulldesc;
   return blk;
   }

/*
 * alcstr - allocate a string in the string space.
 */

/* >alcstr */
char *alcstr(s, slen)
register char *s;
register word slen;
   {
   register char *d;
   register uword fspace;
   char *ofree;

   Inc(al_n_str);
   IncSum(al_bc_stotal,slen);
   MMStr(slen);
   /*
    * See if there is enough room in the string space.
    */
   fspace = (uword)strend - (uword)strfree;
   if (fspace < slen)
      syserr("string allocation botch");
   strneed -= slen;

   /*
    * Copy the string into the string space, saving a pointer to its
    *  beginning.  Note that s may be null, in which case the space
    *  is still to be allocated but nothing is to be copied into it.
    */
   ofree = d = strfree;
   if (s) {
      while (slen-- > 0)
         *d++ = *s++;
      }

   else
      d += slen;
   strfree = d;
   return ofree;
   }
/* <alcstr */

/*
 * alccoexp - allocate a co-expression stack block.
 */

struct b_coexpr *alccoexp()
   {
   struct b_coexpr *ep;
   pointer malloc();
   static int serial = 1;

   Inc(al_n_estk);

   ep = (struct b_coexpr *)malloc((unsigned int)stksize);

#ifdef FixedRegions
   /*
    * If malloc failed, attempt to free some co-expression blocks and retry.
    */
   if (ep == NULL) {
      collect(Static);
      ep = (struct b_coexpr *)malloc((unsigned int)stksize);
      }
#endif					/* FixedRegions */


   if (ep == NULL) {
      k_errornumber = -305;
      k_errortext = "";
      k_errorvalue = nulldesc;
      return NULL;
      }
   ep->title = T_Coexpr;
   ep->es_actstk = NULL;
   ep->size = 0;
   ep->id = ++serial;
   ep->nextstk = stklist;
   stklist = ep;
   MMPaint((char *)ep, stksize, T_Coexpr);
   return ep;
   }

/*
 * alcactiv - allocate a co-expression activation block.
 */

struct astkblk *alcactiv()
   {
   struct astkblk *abp;

   abp = (struct astkblk *)malloc(sizeof(struct astkblk));

#ifdef FixedRegions
   /*
    * If malloc failed, attempt to free some co-expression blocks and retry.
    */
   if (abp == NULL) {
      collect(Static);
      abp = (struct astkblk *)malloc(sizeof(struct astkblk));
      }
#endif					/* FixedRegions */

   if (abp == NULL) {
      k_errornumber = -305;
      k_errortext = "";
      k_errorvalue = nulldesc;
      return NULL;
      }
   abp->nactivators = 0;
   abp->astk_nxt = NULL;
   return abp;
   }

/*
 * alcrefresh - allocate a co-expression refresh block.
 */

struct b_refresh *alcrefresh(entryx, na, nl)
word *entryx;
int na, nl;
   {
   int size;
   struct b_refresh *blk;

   Inc(al_n_eblk);
   size = Vsizeof(struct b_refresh) + (na + nl) * sizeof(struct descrip);
   blk = (struct b_refresh *) alcblk((uword)size, T_Refresh);
   blk->blksize = size;
   blk->ep = entryx;
   blk->numlocals = nl;
   return blk;
   }

/*
 * blkreq - insure that at least bytes of space are left in the block region.
 *  The amount of space needed is transmitted to the collector via
 *  the global variable blkneed.
 */

blkreq(bytes)
uword bytes;
   {
   blkneed = bytes;
   if (bytes > (uword)blkend - (uword)blkfree) {
      coll_blk++;
      collect(Blocks);
      if (bytes > (uword)blkend - (uword)blkfree)
         RetError(-307, nulldesc);
      }
   return Success;
   }

/*
 * strreq - insure that at least n of space are left in the string
 *  space.  The amount of space needed is transmitted to the collector
 *  via the global variable strneed.
 */

strreq(n)
uword n;
   {
   strneed = n; 		/* save in case of collection */
   if (n > (uword)strend - (uword)strfree) {
      coll_str++;
      collect(Strings);
      if (n > (uword)strend - (uword)strfree) {

#ifdef FixedRegions
         if (qualfail)
            RetError(-304, nulldesc);
#endif					/* FixedRegions */

         RetError(-306, nulldesc);
         }
      }
   return Success;
   }

/*
 * cofree - collect co-expression blocks.  This is done after
 *  the marking phase of garbage collection and the stacks that are
 *  reachable have pointers to data blocks, rather than T_Coexpr,
 *  in their type field.
 */

cofree()
   {
   register struct b_coexpr **ep, *xep;
   extern word mstksize; 	/* main stack size */
   register struct astkblk *abp, *xabp;

   /*
    * Reset the type for &main.
    */
   BlkLoc(k_main)->coexpr.title = T_Coexpr;

   /*
    * The co-expression blocks are linked together through their
    *  nextstk fields, with stklist pointing to the head of the list.
    *  The list is traversed and each stack that was not marked
    *  is freed.
    */
   ep = &stklist;
   while (*ep != NULL) {
      if (BlkType(*ep) == T_Coexpr) {
         xep = *ep;
         *ep = (*ep)->nextstk;
	 /*
	  * Free the astkblks.  There should always be one and it seems that
	  *  it's not possible to have more than one, but nonetheless, the
	  *  code provides for more than one.
	  */
 	 for (abp = xep->es_actstk; abp; ) {
 	    xabp = abp;
 	    abp = abp->astk_nxt;
 	    free((pointer)xabp);
 	    }
         free((pointer)xep);
         }
      else {
         BlkType(*ep) = T_Coexpr;
         MMPaint((char *)(*ep), stksize, -T_Coexpr);
         ep = &(*ep)->nextstk;
         }
      }
   MMPaint((char *)stack,mstksize,-T_Coexpr);  /* Also paint main stack */
   }

/*
 * collect - do a garbage collection.
 *  the static region is needed.
 */

collect(region)
int region;
   {
   register struct descrip *dp;
   struct b_coexpr *cp;

#ifdef RunStats
   struct tms tmbuf;

   times(&tmbuf);
   gc_t_start = tmbuf.tms_utime + tmbuf.tms_stime;
#endif					/* RunStats */
   MMBGC();


   coll_tot++;

   /*
    * Garbage collection cannot be done until initializaion is complete.
    */
   if (sp == NULL)
      return;

   /*
    * Sync the values (used by sweep) in the coexpr block for &current
    *  with the current values.
    */
   cp = (struct b_coexpr *)BlkLoc(k_current);
   cp->es_pfp = pfp;
   cp->es_gfp = gfp;
   cp->es_efp = efp;
   cp->es_sp = sp;

   /*
    * Reset qualifier list.
    */
#ifndef FixedRegions
   quallist = (struct descrip **)blkfree;
#endif					/* FixedRegions */
   qualfree = quallist;
   qualfail = 0;

   /*
    * Mark the stacks for &main and the current co-expression.
    */
   markblock(&k_main);
   markblock(&k_current);
   /*
    * Mark &subject and the cached s2 and s3 strings for map.
    */
   postqual(&k_subject);
   if (Qual(maps2))			/*  caution:  the cached arguments of */
      postqual(&maps2); 		/*  map may not be strings. */
   else if (Pointer(maps2))
      markblock(&maps2);
   if (Qual(maps3))
      postqual(&maps3);
   else if (Pointer(maps3))
      markblock(&maps3);
   /*
    * Mark the tended descriptors and the global and static variables.
    */
   for (dp = &tended[1]; (uword)dp <= (uword)&tended[ntended]; dp++)
      if (Qual(*dp))
         postqual(dp);
      else if (Pointer(*dp))
         markblock(dp);
   for (dp = globals; (uword)dp < (uword)eglobals; dp++)
      if (Qual(*dp))
         postqual(dp);
      else if (Pointer(*dp))
         markblock(dp);
   for (dp = statics; (uword)dp < (uword)estatics; dp++)
      if (Qual(*dp))
         postqual(dp);
      else if (Pointer(*dp))
         markblock(dp);

   reclaim(region);

#ifdef RunStats
   times(&tmbuf);
   gc_t_last =
      1000*(((tmbuf.tms_utime + tmbuf.tms_stime)-gc_t_start)/(double)Hz);
   IncSum(gc_t_total,gc_t_last);
#endif					/* RunStats */
   MMEGC();

#ifndef FixedRegions
   if (qualfail && (region == Strings || statneed) &&
       (uword)quallist - (uword)blkfree > Sqlinc)
      /*
       * The string region could not be collected, but it looks like it
       *  needs to be. Collecting the block region gave more room for
       *  the qualifer list, so try again.
       */
       collect(region);
#endif				/* FixedRegions */
   return;
   }

/*
 * markblock - mark each accessible block in the block region and build
 *  back-list of *  descriptors pointing to that block. (Phase I of garbage
 *  collection.)
 */

/* >markblock */
markblock(dp)
struct descrip *dp;
   {
   register struct descrip *dp1;
   register char *endblock, *block;
   static word type, fdesc, off;

   /*
    * Get the block to which dp points.
    */

   block = (char *)BlkLoc(*dp);
   if ((uword)block >= (uword)blkbase && (uword)block < (uword)blkfree) {
      if (Var(*dp) && !Tvar(*dp)) {

         /*
          * The descriptor is a variable; point block to the head of the
          *  block containing the descriptor to which dp points.
          */
         off = Offset(*dp);
         if (off == 0)
            return;
         else
/*	    block = (char *)((word *)block - off); */
            block = (char *)(((uword)block) - off * sizeof(word));
            }

         type = BlkType(block);
         if ((uword)type <= MaxType)  {

            /*
             * The type is valid, which indicates that this block has not
             *	been marked.  Point endblock to the byte past the end
             *	of the block.
             */
            endblock = block + BlkSize(block);
            MMMark(block,type);
            }

         /*
          * Add dp to the back chain for the block and point the
          *  block (via the type field) to dp.
          */
         BlkLoc(*dp) = (union block *)type;
         BlkType(block) = (uword)dp;
         if (((unsigned int)type <=  MaxType) && ((fdesc = firstd[type]) > 0))

            /*
             * The block has not been marked, and it does contain
             *	descriptors. Mark each descriptor.
             */
            for (dp1 = (struct descrip *)(block + fdesc);
               (char *) dp1 < endblock; dp1++) {
               if (Qual(*dp1))
                  postqual(dp1);
               else if (Pointer(*dp1))
                  markblock(dp1);
            }
         }
      else if (dp->dword == D_Coexpr &&
         (unsigned int)BlkType(block) <= MaxType) {
 	 struct b_coexpr *cp;
 	 struct astkblk *abp;
 	 int i;
 	 struct descrip adesc;

         /*
          * dp points to a co-expression block that has not been
          *  marked.  Point the block to dp.  Sweep the interpreter
          *  stack in the block.  Then mark the block for the
          *  activating co-expression and the refresh block.
          */
         BlkType(block) = (uword)dp;
         sweep((struct b_coexpr *)block);
#ifndef NoCoexpr
 	 /*
 	  * Mark the activators of this co-expression.   The activators are
 	  *  stored as a list of addresses, but markblock requires the address
 	  *  of a descriptor.  To accommodate markblock, the dummy descriptor
 	  *  adesc is filled in with each activator address in turn and then
 	  *  marked.  Since co-expressions and the descriptors that reference
 	  *  them don't participate in the back-chaining scheme, it's ok to
 	  *  reuse the descriptor in this manner.
 	  */
 	 cp = (struct b_coexpr *)block;
 	 adesc.dword = D_Coexpr;
 	 for (abp = cp->es_actstk; abp != NULL; abp = abp->astk_nxt) {
 	    for (i = 1; i <= abp->nactivators; i++) {
 	       BlkLoc(adesc) = (union block *)abp->arec[i-1].activator;
 	       markblock(&adesc);
 	       }
 	    }
         markblock(&((struct b_coexpr *)block)->freshblk);
#endif					/* NoCoexpr */
         }
   }
/* <markblock */

/*
 * adjust - adjust pointers into the block region, beginning with block oblk
 *  and *  basing the "new" block region at nblk.  (Phase II of garbage
 *  collection.)
 */

/* >adjust */
adjust(source,dest)
char *source, *dest;
   {
   register struct descrip *nxtptr, *tptr;

   /*
    * Loop through to the end of allocated block region, moving source
    *  to each block in turn and using the size of a block to find the
    *  next block.
    */
   while ((uword)source < (uword)blkfree) {
      if ((uword)(nxtptr = (struct descrip *)BlkType(source)) > MaxType) {

         /*
          * The type field of source is a back pointer.  Traverse the
          *  chain of back pointers, changing each block location from
          *  source to dest.
          */
         while ((uword)nxtptr > MaxType) {
            tptr = nxtptr;
            nxtptr = (struct descrip *)BlkLoc(*nxtptr);
            if (Var(*tptr) && !Tvar(*tptr))
               BlkLoc(*tptr) = (union block *)((word *)dest + Offset(*tptr));
            else
               BlkLoc(*tptr) = (union block *)dest;
            }
         BlkType(source) = (uword)nxtptr | F_Mark;
         dest += BlkSize(source);
         }
      source += BlkSize(source);
      }
   }
/* <adjust */

/*
 * compact - compact good blocks in the block region. (Phase III of garbage
 *  collection.)
 */

/* >compact */
compact(source)
char *source;
   {
   register char *dest;
   register word size;

   /*
    * Start dest at source.
    */
   dest = source;

   /*
    * Loop through to end of allocated block space, moving source
    *  to each block in turn, using the size of a block to find the next
    *  block.  If a block has been marked, it is copied to the
    *  location pointed to by dest and dest is pointed past the end
    *  of the block, which is the location to place the next saved
    *  block.  Marks are removed from the saved blocks.
    */
   while ((uword)source < (uword)blkfree) {
      size = BlkSize(source);
      if (BlkType(source) & F_Mark) {
         BlkType(source) &= ~F_Mark;
         if (source != dest)
            mvc((uword)size,source,dest);
         dest += size;
         }
      source += size;
      }

   /*
    * dest is the location of the next free block.  Now that compaction
    *  is complete, point blkfree to that location.
    */
   blkfree = dest;
   }
/* <compact */

/*
 * postqual - mark a string qualifier.	Strings outside the string space
 *  are ignored.
 */

/* >postqual */
postqual(dp)
struct descrip *dp;
   {
   char *newend;

   if ((uword)StrLoc(*dp) >= (uword)strbase &&
     (uword)StrLoc(*dp) < (uword)strend) {
      /*
       * The string is in the string space.  Add it to the string qualifier
       *  list, but before adding it, expand the string qualifier list if
       *  necessary.
       */
      if ((uword)qualfree >= (uword)equallist) {
#ifdef FixedRegions
         qualfail = 1;
         return;
#else					/* FixedRegions */
         newend = (char *)equallist + Sqlinc;
         /*
          * Make sure region has not changed and that it can be expanded.
          */
         if (currend != sbrk((word)0) || (int) brk(newend) == -1) {
            qualfail = 1;
            return;
            }
         equallist = (struct descrip **)newend;
         currend = sbrk((word)0);
#ifdef QuallistExp
         fprintf(stderr,"size of quallist = %d\n",
            (uword)equallist - (uword)quallist);
         fflush(stderr);
#endif					/* QuallistExp */
#endif					/* FixedRegions */
         }
      *qualfree++ = dp;
      }
   }
/* <postqual */

/*
 * scollect - collect the string space.  quallist is a list of pointers to
 *  descriptors for all the reachable strings in the string space.  For
 *  ease of description, it is referred to as if it were composed of
 *  descriptors rather than pointers to them.
 */

/* >scollect */
scollect(extra)
word extra;
   {
   register char *source, *dest;
   register struct descrip **qptr;
   char *cend;
   extern int qlcmp();

   if ((uword)qualfree <= (uword)quallist) {
      /*
       * There are no accessible strings.  Thus, there are none to
       *  collect and the whole string space is free.
       */
      strfree = strbase;
      return;
      }
   /*
    * Sort the pointers on quallist in ascending order of string
    *  locations.
    */
   qsort((char *)quallist,
      (int)(((uword)qualfree - (uword)quallist)/sizeof(struct descrip *)),
      sizeof(struct descrip *), qlcmp);
   /*
    * The string qualifiers are now ordered by starting location.
    */
   dest = strbase;
   source = cend = StrLoc(**quallist);

   /*
    * Loop through qualifiers for accessible strings.
    */
   for (qptr = quallist; (uword)qptr < (uword)qualfree; qptr++) {
      if ((uword)StrLoc(**qptr) > (uword)cend) {

         /*
          * qptr points to a qualifier for a string in the next clump.
          *  The last clump is moved, and source and cend are set for
          *  the next clump.
          */
         MMSMark(source,cend - source);
         while ((uword)source < (uword)cend)
            *dest++ = *source++;
         source = cend = StrLoc(**qptr);
         }
      if ((uword)(StrLoc(**qptr) + StrLen(**qptr)) > (uword)cend)
         /*
          * qptr is a qualifier for a string in this clump; extend
          *  the clump.
          */
         cend = StrLoc(**qptr) + StrLen(**qptr);
      /*
       * Relocate the string qualifier.
       */
      StrLoc(**qptr) = StrLoc(**qptr) + (uword)dest - (uword)source +
         (uword)extra;
      }

   /*
    * Move the last clump.
    */
   MMSMark(source,cend - source);
   while ((uword)source < (uword)cend)
      *dest++ = *source++;
   strfree = dest;
   }
/* <scollect */

/*
 * qlcmp - compare the location fields of two string qualifiers for qsort.
 */

qlcmp(q1,q2)
struct descrip **q1, **q2;
   {
#if IntBits == 16
   long l;
   l = (long)((uword)StrLoc(**q1) - (uword)StrLoc(**q2));
   if (l < 0)
      return -1;
   else if (l > 0)
      return 1;
   else
      return 0;
#else					/* IntBits = 16 */
   return (int)((uword)StrLoc(**q1) - (uword)StrLoc(**q2));
#endif					/* IntBits == 16 */

   }

#ifdef OldMvc
/*
 * mvc - move n bytes from src to dst.
 */

mvc(n, s, d)
uword n;
register char *s, *d;
   {
   register int words;
   register int *srcw, *dstw;
   int bytes;

   if (n == 0)
      return;
   words = n / sizeof(int);
   bytes = n % sizeof(int);

   srcw = (int *)s;
   dstw = (int *)d;

   if ((uword)d < (uword)s) {
      /*
       * The move is from higher memory to lower memory.  (It so happens
       *  that leftover bytes are not moved.)
       */
      while (--words >= 0)
         *(dstw)++ = *(srcw)++;
      while (--bytes >= 0)
         *d++ = *s++;
      }
   else if ((uword)d > (uword)s) {
      /*
       * The move is from lower memory to higher memory.
       */
      s += n;
      d += n;
      while (--bytes >= 0)
         *--d = *--s;
      srcw = (int *)s;
      dstw = (int *)d;
      while (--words >= 0)
         *--dstw = *--srcw;
      }
   }
#else					/* OldMvc */
/*
 * mvc - move n bytes from src to dest
 *
 *      The algorithm is to copy the data (using memcpy) in the largest
 * chunks possible, which is the size of area of the source data not in
 * the destination area (ie non-overlapped area).  (Chunks are expected to
 * be fairly large.)
 */

mvc(n, src, dest)
uword n;
register char *src, *dest;
   {
   register char *srcend, *destend;        /* end of data areas */
   int copy_size;                  /* of size copy_size */
   int left_over;         /* size of last chunk < copy_size */

   if (n == 0)
     return;

   srcend  = src + n;    /* point at byte after src data */
   destend = dest + n;   /* point at byte after dest area */

   if ((destend <= src) || (srcend <= dest))  /* not overlapping */
      memcpy(dest,src,(int) n);

   else {                     /* overlapping data areas */
         if (dest < src)
           /*
            * The move is from higher memory to lower memory.
            */
            {
             copy_size = src - dest;

             /* now loop round copying copy_size chunks of data */

             do {
                 memcpy(dest,src,copy_size);
                 dest = src;
                 src += copy_size;
                }
             while ((srcend - src) > copy_size);

             left_over = srcend - src ;

             /* copy final fragment of data - if there is one */

             if (left_over > 0) memcpy(dest,src,left_over);
            }

         else if (dest > src)
           /*
            * The move is from lower memory to higher memory.
            */
            {
             copy_size = destend - srcend;

             /* now loop round copying copy_size chunks of data */

             do {
                 destend = srcend;
                 srcend  -= copy_size;
                 memcpy(destend,srcend,copy_size);
                }
             while ((srcend - src) > copy_size);

             left_over = srcend - src ;

             /* copy intial fragment of data - if there is one */

             if (left_over > 0) memcpy(dest,src,left_over);
            }

        } /* end of overlapping data area code */

  /*
   *  Note that src == dest implies no action
   */
   }
#endif					/* OldMvc */

/*
 * sweep - sweep the stack, marking all descriptors there.  Method
 *  is to start at a known point, specifically, the frame that the
 *  fp points to, and then trace back along the stack looking for
 *  descriptors and local variables, marking them when they are found.
 *  The sp starts at the first frame, and then is moved down through
 *  the stack.	Procedure, generator, and expression frames are
 *  recognized when the sp is a certain distance from the fp, gfp,
 *  and efp respectively.
 *
 * Sweeping problems can be manifested in a variety of ways due to
 *  the "if it can't be identified it's a descriptor" methodology.
 */
sweep(ce)
struct b_coexpr *ce;
   {
   register word *s_sp;
   register struct pf_marker *fp;
   register struct gf_marker *s_gfp;
   register struct ef_marker *s_efp;
   word nargs, type, gsize;

   fp = ce->es_pfp;
   s_gfp = ce->es_gfp;
   if (s_gfp != 0) {
      type = s_gfp->gf_gentype;
      if (type == G_Psusp)
         gsize = Wsizeof(*s_gfp);
      else
         gsize = Wsizeof(struct gf_smallmarker);
      }
   s_efp = ce->es_efp;
   s_sp =  ce->es_sp;
   nargs = 0;				/* Nargs counter is 0 initially. */

   while ((fp != 0 || nargs)) { 	/* Keep going until current fp is
					    0 and no arguments are left. */
      if (s_sp == (word *)fp + Vwsizeof(*pfp) - 1) {
					/* sp has reached the upper
					    boundary of a procedure frame,
					    process the frame. */
         s_efp = fp->pf_efp;		/* Get saved efp out of frame */
         s_gfp = fp->pf_gfp;		/* Get save gfp */
         if (s_gfp != 0) {
            type = s_gfp->gf_gentype;
            if (type == G_Psusp)
               gsize = Wsizeof(*s_gfp);
            else
               gsize = Wsizeof(struct gf_smallmarker);
            }
         s_sp = (word *)fp - 1; 	/* First argument descriptor is
					    first word above proc frame */
         nargs = fp->pf_nargs;
         fp = fp->pf_pfp;
         }
      else if (s_gfp != NULL && s_sp == (word *)s_gfp + gsize - 1) {
					/* The sp has reached the lower end
					    of a generator frame, process
					    the frame.*/
         if (type == G_Psusp)
            fp = s_gfp->gf_pfp;
         s_sp = (word *)s_gfp - 1;
         s_efp = s_gfp->gf_efp;
         s_gfp = s_gfp->gf_gfp;
            if (s_gfp != 0) {
            type = s_gfp->gf_gentype;
            if (type == G_Psusp)
               gsize = Wsizeof(*s_gfp);
            else
               gsize = Wsizeof(struct gf_smallmarker);
            }
         nargs = 1;
         }
      else if (s_sp == (word *)s_efp + Wsizeof(*s_efp) - 1) {
					    /* The sp has reached the upper
						end of an expression frame,
						process the frame. */
         s_gfp = s_efp->ef_gfp; 	/* Restore gfp, */
         if (s_gfp != 0) {
            type = s_gfp->gf_gentype;
            if (type == G_Psusp)
               gsize = Wsizeof(*s_gfp);
            else
               gsize = Wsizeof(struct gf_smallmarker);
            }
         s_efp = s_efp->ef_efp; 	/*  and efp from frame. */
         s_sp -= Wsizeof(*s_efp);	/* Move past expression frame marker. */
         }
      else {				/* Assume the sp is pointing at a
					    descriptor. */
         if (Qual(*((struct descrip *)(&s_sp[-1]))))
            postqual((struct descrip *)&s_sp[-1]);
         else if (Pointer(*((struct descrip *)(&s_sp[-1]))))
            markblock((struct descrip *)&s_sp[-1]);
         s_sp -= 2;			/* Move past descriptor. */
         if (nargs)			/* Decrement argument count if in an*/
            nargs--;			/*  argument list. */
         }
      }
   }

#ifdef MemMon
show_alloc()		/* show allocated block region on memory monitor */
   {
   HEADER *p;
   char *a;
   int h, n;

   for (p = (HEADER *) statbase; (uword)p < (uword)(HEADER *) statfree;
      p += p->s.bsize) {
         a = (char *)(p + 1);
         n = (p->s.bsize - 1) * sizeof(HEADER);
         h = *(int *)a;
         if (h == T_Coexpr || a == (char *) stack)
            MMPaint(a, n, T_Coexpr);
         else if (h == FREEMAGIC)
            MMPaint(a, n, 0);
         else
            MMPaint(a, n, 1);
         }
   a = (char *) p;
   if (a < strbase)
      MMPaint(a, strbase - a, 0);
   }
#endif					/* MemMon */

/*
 * descr - dump a descriptor.  Used only for debugging.
 */

descr(dp)
struct descrip *dp;
   {
   int i;

   fprintf(stderr,"%08lx: ",(long)dp);
   if (Qual(*dp))
      fprintf(stderr,"%15s","qualifier");
   else if (Var(*dp) && !Tvar(*dp))
      fprintf(stderr,"%15s","variable");
   else {
      i =  Type(*dp);
      switch (i) {
         case T_Null:
            fprintf(stderr,"%15s","null");
            break;
         case T_Integer:
            fprintf(stderr,"%15s","integer");
            break;
         default:
            fprintf(stderr,"%15s",blkname[i]);
         }
      }
   fprintf(stderr," %08lx %08lx\n",(long)dp->dword,(long)dp->vword.integr);
   }

/*
 * blkdump - dump the allocated block region.  Used only for debugging.
 */

blkdump()
   {
   register char *blk;
   register word type, size, fdesc;
   register struct descrip *ndesc;

   fprintf(stderr,
      "\nDump of allocated block region.  base:%08lx free:%08lx max:%08lx\n",
         (long)blkbase,(long)blkfree,(long)blkend);
   fprintf(stderr,"  loc     type              size  contents\n");

   for (blk = blkbase; (uword)blk < (uword)blkfree; blk += BlkSize(blk)) {
      type = BlkType(blk);
      size = BlkSize(blk);
      fprintf(stderr," %08lx   %15s   %4ld\n",(long)blk,blkname[type],
         (long)size);
      if ((fdesc = firstd[type]) > 0)
         for (ndesc = (struct descrip *) (blk + fdesc);
               (uword)ndesc < (uword)(struct descrip *) (blk + size);ndesc++) {
            fprintf(stderr,"                                 ");
            descr(ndesc);
            }
      fprintf(stderr,"\n");
      }
   fprintf(stderr,"end of block region.\n");
   }
