/*
 * File: rmisc.c
 *  Contents: deref, [gcvt], hash, [memcpy], [memset], outimage, [qsort],
 *  qtos, trace, tvkeys, pushact, popact, topact, dumpact, xdisp,
 *  findline, findfile
 */

#ifdef IconAlloc
#define free mem_free
#endif					/* IconAlloc */

#include "../h/rt.h"
#include <ctype.h>

/*
 * deref - dereference a descriptor.
 */

deref(dp)
struct descrip *dp;
   {
   register word i;
   register union block *bp;
   struct descrip v, tbl, tref;
   extern char *alcstr();

   if (!Tvar(*dp))
       /*
       * An ordinary variable is being dereferenced; just replace
       *  *dp with the descriptor *dp is pointing to.
       */
      *dp = *VarLoc(*dp);
   else switch (Type(*dp)) {

         case T_Tvsubs:
            Inc(ev_n_tsderef);
            /*
             * A substring trapped variable is being dereferenced.
             *  Point bp to the trapped variable block and v to
             *  the string.
             */
            bp = TvarLoc(*dp);
            v = bp->tvsubs.ssvar;
            if (DeRef(v) == Error)
               return Error;
            if (!Qual(v))
               RetError(103, v);
            if (bp->tvsubs.sspos + bp->tvsubs.sslen - 1 > StrLen(v))
               RetError(-205, nulldesc);
            /*
             * Make a descriptor for the substring by getting the
             *  length and pointing into the string.
             */
            StrLen(*dp) = bp->tvsubs.sslen;
            StrLoc(*dp) = StrLoc(v) + bp->tvsubs.sspos - 1;
            break;

         case T_Tvtbl:
            Inc(ev_n_ttderef);
            if (BlkLoc(*dp)->tvtbl.title == T_Telem) {
               /*
                * The tvtbl has been converted to a telem and is
                *  in the table.  Replace the descriptor pointed to
                *  by dp with the value of the element.
                */
                *dp = BlkLoc(*dp)->telem.tval;
                break;
                }

            /*
             *  Point tbl to the table header block, tref to the
             *  subscripting value, and bp to the appropriate 
             *  chain.  Point dp to a descriptor for the default
             *  value in case the value referenced by the subscript
             *  is not in the table.
             */
            tbl = BlkLoc(*dp)->tvtbl.clink;
            tref = BlkLoc(*dp)->tvtbl.tref;
            i = BlkLoc(*dp)->tvtbl.hashnum;
            *dp = BlkLoc(tbl)->table.defvalue;
            bp = BlkLoc(BlkLoc(tbl)->table.buckets[SlotNum(i,TSlots)]);

            /*
             * Traverse the element chain looking for the subscript
             *  value.  If found, replace the descriptor pointed to
             *  by dp with the value of the element.
             */
            while (bp != NULL && bp->telem.hashnum <= i) {
               if ((bp->telem.hashnum == i) &&
                  (equiv(&bp->telem.tref, &tref))) {
                     *dp = bp->telem.tval;
                     break;
                     }
               bp = BlkLoc(bp->telem.clink);
               }
            break;

         case T_Tvkywd:
            bp = TvarLoc(*dp);
            *dp = bp->tvkywd.kyval;
            break;

         default:
            syserr("deref: illegal trapped variable");
         }
#ifdef Debug
   if (!Qual(*d) && Var(*d))
      syserr("deref: didn't get dereferenced");
#endif					/* Debug */
   return Success;
   }

#ifdef IconGcvt
/*
 * gcvt - Convert number to a string in buf.  If possible, ndigit
 *  significant digits are produced, otherwise a form with an exponent is used.
 *
 *  The name is actually #defined as "icon_gcvt" in config.h.
 */
char *gcvt(number, ndigit, buf)
double number;
char *buf;
   {
   int sign, decpt;
   char *ecvt();
   register char *p1, *p2;
   register i;

   p1 = ecvt(number, ndigit, &decpt, &sign);
   p2 = buf;
   if (sign)
      *p2++ = '-';
   for (i=ndigit-1; i>0 && p1[i]=='0'; i--)
      ndigit--;
   if (decpt >= 0 && decpt-ndigit > 4
      || decpt < 0 && decpt < -3) { /* use E-style */
         decpt--;
         *p2++ = *p1++;
         *p2++ = '.';
         for (i=1; i<ndigit; i++)
            *p2++ = *p1++;
         *p2++ = 'e';
         if (decpt<0) {
            decpt = -decpt;
            *p2++ = '-';
            }
         if (decpt/10 > 0)
            *p2++ = decpt/10 + '0';
         *p2++ = decpt%10 + '0';
      } else {
         if (decpt<=0) {
         /* if (*p1!='0') */
         *p2++ = '0';
         *p2++ = '.';
         while (decpt<0) {
            decpt++;
            *p2++ = '0';
            }
         }
         for (i=1; i<=ndigit; i++) {
            *p2++ = *p1++;
            if (i==decpt)
               *p2++ = '.';
            }
      if (ndigit<decpt) {
         while (ndigit++<decpt)
            *p2++ = '0';
         *p2++ = '.';
         }
   }
   if (p2[-1]=='.')
      *p2++ = '0';
   *p2 = '\0';
   return(buf);
   }
#endif					/* IconGcvt */

/*
 * hash - compute hash value of arbitrary object for table and set accessing.
 */

word hash(dp)
struct descrip *dp;
   {
   word i;
   double r;
   register word j;
   register char *s;

   if (Qual(*dp)) {

      /*
       * Compute the hash value for the string by summing the value
       *  of all the characters (to a maximum of 10) plus the length.
       */
      i = 0;
      s = StrLoc(*dp);
      j = StrLen(*dp);
      for (j = (j <= 10) ? j : 10 ; j > 0; j--)
         i += *s++ & 0377;
      i += StrLen(*dp) & 0377;
      }
   else {
      switch (Type(*dp)) {
         /*
          * The hash value for numeric types is the bit-string
          *  representation of the value.
          */

         case T_Integer:
            i = IntVal(*dp);
            break;

         case T_Real:
            GetReal(dp,r);
            i = r;
            break;

	 case T_Bignum: {
	    struct b_bignum *b = &BlkLoc(*dp)->bignumblk;
	    i = ((b->right - b->left) << 16)
		^ (b->digits[b->left] << 8)
		^ b->digits[b->right];}
	    break;

         case T_Cset:
            /*
             * Compute the hash value for a cset by performing the
             *  exclusive-or of the words in the bit array.
             */
            i = 0;
            for (j = 0; j < CsetSize; j++)
               i ^= BlkLoc(*dp)->cset.bits[j];
            break;

         default:
            /*
             * For other types, use the type code as the hash
             *  value.
             */
            i = Type(*dp);
            break;
         }
      }

   return i;
   }

#ifndef SysMem
char *memcpy(to, from, n)
   register char *to, *from;
   register n;
   {
   register char *p = to;

   while (--n >= 0)
      *to++ = *from++;

   return p;
   }

char *memset(to, con, n)
   register char *to;
   register con, n;
   {
   register char *p = to;

   while (--n >= 0)
      *to++ = con;

   return p;
   }
#endif					/* SysMem */

#define StringLimit	16		/* limit on length of imaged string */
#define ListLimit	 6		/* limit on list items in image */

/*
 * outimage - print image of *dp on file f.  If restrict is nonzero,
 *  fields of records will not be imaged.
 */

outimage(f, dp, restrict)
FILE *f;
struct descrip *dp;
int restrict;
   {
   register word i, j;
   register char *s;
   register union block *bp, *vp;
   char *type;
   FILE *fd;
   struct descrip q;
   extern char *blkname[];
   double rresult;

outimg:

   if (Qual(*dp)) {
      /*
       * *dp is a string qualifier.  Print StringLimit characters of it
       *  using printimage and denote the presence of additional characters
       *  by terminating the string with "...".
       */
      i = StrLen(*dp);
      s = StrLoc(*dp);
      j = Min(i, StringLimit);
      putc('"', f);
      while (j-- > 0)
         printimage(f, *s++, '"');
      if (i > StringLimit)
         fprintf(f, "...");
      putc('"', f);
      return;
      }

   if (Var(*dp) && !Tvar(*dp)) {
      /*
       * *d is a variable.  Print "variable =", dereference it, and 
       *  call outimage to handle the value.
       */
      fprintf(f, "(variable = ");
      dp = VarLoc(*dp);
      outimage(f, dp, restrict);
      putc(')', f);
      return;
      }

   switch (Type(*dp)) {

      case T_Null:
         if (restrict == 0)
            fprintf(f, "&null");
         return;

      case T_Integer:
         fprintf(f, "%ld", (long)IntVal(*dp));
         return;

      case T_Bignum:
	 bigprint (f, BlkLoc(*dp));
	 return;

      case T_Real:
         {
         char s[30];
         struct descrip rd;

         GetReal(dp,rresult);
         rtos(rresult, &rd, s);
         fprintf(f, "%s", StrLoc(rd));
         return;
         }

      case T_Cset:
         /*
          * Check for distinguished csets by looking at the address of
          *  of the object to image.  If one is found, print its name.
          */
         if ((uword)BlkLoc(*dp) == (uword)&k_ascii) {
            fprintf(f, "&ascii");
            return;
            }
         else if ((uword)BlkLoc(*dp) == (uword)&k_cset) {
            fprintf(f, "&cset");
            return;
            }
         else if ((uword)BlkLoc(*dp) == (uword)&k_digits) {
            fprintf(f, "&digits");
            return;
            }
         else if ((uword)BlkLoc(*dp) == (uword)&k_lcase) {
            fprintf(f, "&lcase");
            return;
            }
         else if ((uword)BlkLoc(*dp) == (uword)&k_ucase) {
            fprintf(f, "&ucase");
            return;
            }
         /*
          * Use printimage to print each character in the cset.  Follow
          *  with "..." if the cset contains more than StringLimit
          *  characters.
          */
         putc('\'', f);
         j = StringLimit;
         for (i = 0; i < 256; i++) {
            if (Testb(i, BlkLoc(*dp)->cset.bits)) {
               if (j-- <= 0) {
                  fprintf(f, "...");
                  break;
                  }
               printimage(f, (int)i, '\'');
               }
            }
         putc('\'', f);
         return;

      case T_File:
         /*
          * Check for distinguished files by looking at the address of
          *  of the object to image.  If one is found, print its name.
          */
         if ((fd = BlkLoc(*dp)->file.fd) == stdin)
            fprintf(f, "&input");
         else if (fd == stdout)
            fprintf(f, "&output");
         else if (fd == stderr)
            fprintf(f, "&errout");
         else {
            /*
             * The file isn't a special one, just print "file(name)".
             */
            i = StrLen(BlkLoc(*dp)->file.fname);
            s = StrLoc(BlkLoc(*dp)->file.fname);
            fprintf(f, "file(");
            while (i-- > 0)
               printimage(f, *s++, '\0');
            putc(')', f);
            }
         return;

      case T_Proc:
         /*
          * Produce one of:
          *  "procedure name"
          *  "function name"
          *  "record constructor name"
          *
          * Note that the number of dynamic locals is used to determine
          *  what type of "procedure" is at hand.
          */
         i = StrLen(BlkLoc(*dp)->proc.pname);
         s = StrLoc(BlkLoc(*dp)->proc.pname);
         switch ((int)BlkLoc(*dp)->proc.ndynam) {
            default:  type = "procedure"; break;
            case -1:  type = "function"; break;
            case -2:  type = "record constructor"; break;
            }
         fprintf(f, "%s ", type);
         while (i-- > 0)
            printimage(f, *s++, '\0');
         return;

      case T_List:
         /*
          * listimage does the work for lists.
          */
         listimage(f, (struct b_list *)BlkLoc(*dp), restrict);
         return;

      case T_Table:
         /*
          * Print "table(n)" where n is the size of the table.
          */
         fprintf(f, "table(%ld)", (long)BlkLoc(*dp)->table.size);
         return;
      case T_Set:
	/*
         * print "set(n)" where n is the cardinality of the set
         */
	fprintf(f,"set(%ld)",(long)BlkLoc(*dp)->set.size);
	return;

      case T_Record:
         /*
          * If restrict is nonzero, print "record(n)" where n is the
          *  number of fields in the record.  If restrict is zero, print
          *  the image of each field instead of the number of fields.
          */
         bp = BlkLoc(*dp);
         i = StrLen(BlkLoc(bp->record.recdesc)->proc.recname);
         s = StrLoc(BlkLoc(bp->record.recdesc)->proc.recname);
         fprintf(f, "record ");
         while (i-- > 0)
            printimage(f, *s++, '\0');
         j = BlkLoc(bp->record.recdesc)->proc.nfields;
         if (j <= 0)
            fprintf(f, "()");
         else if (restrict > 0)
            fprintf(f, "(%ld)", (long)j);
         else {
            putc('(', f);
            i = 0;
            for (;;) {
               outimage(f, &bp->record.fields[i], restrict+1);
               if (++i >= j)
                  break;
               putc(',', f);
               }
            putc(')', f);
            }
         return;

      case T_Tvsubs:
         /*
          * Produce "v[i+:j] = value" where v is the image of the variable
          *  containing the substring, i is starting position of the substring
          *  j is the length, and value is the string v[i+:j].	If the length
          *  (j) is one, just produce "v[i] = value".
          */
         bp = BlkLoc(*dp);
         dp = VarLoc(bp->tvsubs.ssvar);
         if (dp == (struct descrip *)&tvky_sub)
            fprintf(f, "&subject");
         else outimage(f, dp, restrict);
         if (bp->tvsubs.sslen == 1)
            fprintf(f, "[%ld]", (long)bp->tvsubs.sspos);
         else
            fprintf(f, "[%ld+:%ld]", (long)bp->tvsubs.sspos,
               (long)bp->tvsubs.sslen);
         if (dp == (struct descrip *)&tvky_sub) {
            vp = BlkLoc(bp->tvsubs.ssvar);
            if (bp->tvsubs.sspos + bp->tvsubs.sslen - 1 >
                  StrLen(vp->tvkywd.kyval))
               return;
            StrLen(q) = bp->tvsubs.sslen;
            StrLoc(q) = StrLoc(vp->tvkywd.kyval) + bp->tvsubs.sspos-1;
            fprintf(f, " = ");
            dp = &q;
            goto outimg;
            }
         else if (Qual(*dp)) {
            if (bp->tvsubs.sspos + bp->tvsubs.sslen - 1 >
                  StrLen(*VarLoc(bp->tvsubs.ssvar)))
               return;
            StrLen(q) = bp->tvsubs.sslen;
            StrLoc(q) = StrLoc(*VarLoc(bp->tvsubs.ssvar)) + bp->tvsubs.sspos-1;
            fprintf(f, " = ");
            dp = &q;
            goto outimg;
            }
         return;

      case T_Tvtbl:
         bp = BlkLoc(*dp);
         /*
          * It is possible that the descriptor that thinks it is pointing
          *  to a tabel-element trapped variable may actually be pointing
          *  at a table element block which had been converted from a
          *  trapped variable. Check for this first and if it is a table
          *  element block, produce the outimage of its value.
          */
         if (bp->tvtbl.title == T_Telem) {
            outimage(f, &bp->tvtbl.tval, restrict);
            return;
            }
         /*
          * It really was a TVTBL - Produce "t[s]" where t is the image of
          *  the table containing the element and s is the image of the
          *  subscript.
          */
         else {
            outimage(f, &bp->tvtbl.clink, restrict);
            putc('[', f);
            outimage(f, &bp->tvtbl.tref, restrict);
            putc(']', f);
            return;
            }

      case T_Tvkywd:
         bp = BlkLoc(*dp);
         i = StrLen(bp->tvkywd.kyname);
         s = StrLoc(bp->tvkywd.kyname);
         while (i-- > 0)
            putc(*s++, f);
         fprintf(f, " = ");
         outimage(f, &bp->tvkywd.kyval, restrict);
         return;

      case T_Coexpr:
#ifdef NoCoexpr
         fprintf(f, "coexpression #1 (0)");
#else					/* NoCoexpr */
         fprintf(f, "co-expression #%ld",
            (long)((struct b_coexpr *)BlkLoc(*dp))->id);
#endif					/* NoCoexpr */
         return;

      case T_External:
         fprintf(f, "external(%d)",((struct b_external *)BlkLoc(*dp))->blksize);
         return;

      default:
         if (Type(*dp) <= MaxType)
            fprintf(f, "%s", blkname[Type(*dp)]);
         else
            syserr("outimage: unknown type");
      }
   }

/*
 * printimage - print character c on file f using escape conventions
 *  if c is unprintable, '\', or equal to q.
 */

static printimage(f, c, q)
FILE *f;
int c, q;
   {
   if (c >= ' ' && c < '\177') {
      /*
       * c is printable, but special case ", ', and \.
       */
      switch (c) {
         case '"':
            if (c != q) goto def;
            fprintf(f, "\\\"");
            return;
         case '\'':
            if (c != q) goto def;
            fprintf(f, "\\'");
            return;
         case '\\':
            fprintf(f, "\\\\");
            return;
         default:
         def:
            putc(c, f);
            return;
         }
      }

   /*
    * c is some sort of unprintable character.	If it one of the common
    *  ones, produce a special representation for it, otherwise, produce
    *  its octal value.
    */
   switch (c) {
      case '\b':                        /* backspace */
         fprintf(f, "\\b");
         return;
      case '\177':                        /* delete */
         fprintf(f, "\\d");
         return;
      case '\33':                        /* escape */
         fprintf(f, "\\e");
         return;
      case '\f':                        /* form feed */
         fprintf(f, "\\f");
         return;
      case '\n':                        /* new line */
         fprintf(f, "\\n");
         return;
/*
 * The following code is operating-system dependent. Handle \r if available.
 */

#if PORT
      case '\r':                        /* return */
         fprintf(f, "\\r");
         return;
#endif					/* PORT */

#if AMIGA
#endif					/* AMIGA */

#if MACINTOSH
#if MPW
   /* not available */
#endif					/* MPW */
#endif					/* MACINTOSH */

#if ATARI_ST || MSDOS || UNIX || VMS
      case '\r':                        /* return */
         fprintf(f, "\\r");
         return;
#endif					/* ATARI_ST || MSDOS || UNIX || VMS */

#if VM || MVS
#endif					/* VM || MVS */

/*
 * End of operating-system specific code.
 */
      case '\t':                        /* horizontal tab */
         fprintf(f, "\\t");
         return;
      case '\13':                        /* vertical tab */
         fprintf(f, "\\v");
         return;
      default:				      /* octal constant */
         fprintf(f, "\\%03o", c&0377);
         return;
      }
   }

/*
 * listimage - print an image of a list.
 */

static listimage(f, lp, restrict)
FILE *f;
struct b_list *lp;
int restrict;
   {
   register word i, j;
   register struct b_lelem *bp;
   word size, count;

   bp = (struct b_lelem *) BlkLoc(lp->listhead);
   size = lp->size;

   if (restrict > 0 && size > 0) {
      /*
       * Just give indication of size if the list isn't empty.
       */
      fprintf(f, "list(%ld)", (long)size);
      return;
      }

   /*
    * Print [e1,...,en] on f.  If more than ListLimit elements are in the
    *  list, produce the first ListLimit/2 elements, an ellipsis, and the
    *  last ListLimit elements.
    */
   putc('[', f);
   count = 1;
   i = 0;
   if (size > 0) {
      for (;;) {
         if (++i > bp->nused) {
            i = 1;
            bp = (struct b_lelem *) BlkLoc(bp->listnext);
            }
         if (count <= ListLimit/2 || count > size - ListLimit/2) {
            j = bp->first + i - 1;
            if (j >= bp->nslots)
               j -= bp->nslots;
            outimage(f, &bp->lslots[j], restrict+1);
            if (count >= size)
               break;
            putc(',', f);
            }
         else if (count == ListLimit/2 + 1)
            fprintf(f, "...,");
         count++;
         }
      }
   putc(']', f);
   }

#ifdef IconQsort
/* qsort(base,nel,width,compar) - quicksort routine
 *
 * A Unix-compatible public domain quicksort.
 * Based on Bentley, CACM 28,7 (July, 1985), p. 675.
 */
     
qsort (base, nel, w, compar)
char *base;
int nel, w;
int (*compar)();
{
   int i, lastlow;
    
   if (nel < 2)
      return;
   qswap (base, base + w * (rand() % nel), w);
   lastlow = 0;
   for (i = 1; i < nel; i++)
      if ((*compar) (base + w * i, base) < 0)
         qswap (base + w * i, base + w * (++lastlow), w);
   qswap (base, base + w * lastlow, w);
   qsort (base, lastlow, w, compar);
   qsort (base + w * (lastlow+1), nel-lastlow-1, w, compar);
}
    
static qswap (a, b, w)        /* swap *a and *b of width w for qsort*/
char *a, *b;
int w;
{
   register t;
    
   while (w--)  {
      t = *a;
      *a++ = *b;
      *b++ = t;
   }
}
#endif					/* IconQsort */

/*
 * qtos - convert a qualified string named by *dp to a C-style string.
 *  Put the C-style string in sbuf if it will fit, otherwise put it
 *  in the string region.
 */

qtos(dp, sbuf)
struct descrip *dp;
char *sbuf;
   {
   register word slen;
   register char *c;
   extern char *alcstr();

   c = StrLoc(*dp);
   slen = StrLen(*dp)++;
   if (slen >= MaxCvtLen) {
      if (strreq(slen + 1) == Error) 
         return Error;
      if (c + slen != strfree)
         StrLoc(*dp) = alcstr(c, slen);
      alcstr("",(word)1);
      }
   else {
      StrLoc(*dp) = sbuf;
      for ( ; slen > 0; slen--)
         *sbuf++ = *c++;
      *sbuf = '\0';
      }
   return Success;
   }

int findline();
char *findfile();

/*
 * ctrace - procedure *bp is being called with nargs arguments, the first
 *  of which is at arg; produce a trace message.
 */
ctrace(bp, nargs, arg)
struct b_proc *bp;
int nargs;
struct descrip *arg;
   {

   if (k_trace > 0)
      k_trace--;
   showline(findfile(ipc.opnd), findline(ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   putc('(', stderr);
   while (nargs--) {
      outimage(stderr, arg++, 0);
      if (nargs)
         putc(',', stderr);
      }
   putc(')', stderr);
   putc('\n', stderr);
   fflush(stderr);
   }

/*
 * rtrace - procedure *bp is returning *rval; produce a trace message.
 */

rtrace(bp, rval)
register struct b_proc *bp;
struct descrip *rval;
   {
   inst t_ipc;

   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the return instruction.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr, " returned ");
   outimage(stderr, rval, 0);
   putc('\n', stderr);
   fflush(stderr);
   }

/*
 * ftrace - procedure *bp is failing; produce a trace message.
 */

ftrace(bp)
register struct b_proc *bp;
   {
   inst t_ipc;

   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the fail instruction.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr, " failed");
   putc('\n', stderr);
   fflush(stderr);
   }

/*
 * strace - procedure *bp is suspending *rval; produce a trace message.
 */

strace(bp, rval)
register struct b_proc *bp;
struct descrip *rval;
   {
   inst t_ipc;

   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the suspend instruction.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr, " suspended ");
   outimage(stderr, rval, 0);
   putc('\n', stderr);
   fflush(stderr);
   }

/*
 * atrace - procedure *bp is being resumed; produce a trace message.
 */

atrace(bp)
register struct b_proc *bp;
   {
   inst t_ipc;

   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the instruction causing resumption.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr, " resumed");
   putc('\n', stderr);
   fflush(stderr);
   }

#ifndef NoCoexpr
/*
 * coacttrace -- co-expression is being activated; produce a trace message.
 */
coacttrace(ccp, ncp)
struct b_coexpr *ccp;
struct b_coexpr *ncp;
   {
   struct b_proc *bp;
   inst t_ipc;

   bp = (struct b_proc *)BlkLoc(*argp);
   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the activation instruction.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr,"; #%ld : ", (long)ccp->id);
   outimage(stderr, (struct descrip *)(sp - 3), 0);
   fprintf(stderr," @ #%ld\n", (long)ncp->id);
   fflush(stderr);
   }

/*
 * corettrace -- return from co-expression; produce a trace message.
 */
corettrace(ccp, ncp)
struct b_coexpr *ccp;
struct b_coexpr *ncp;
   {
   struct b_proc *bp;
   inst t_ipc;

   bp = (struct b_proc *)BlkLoc(*argp);
   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the coret instruction.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr,"; #%ld returned ", (long)ccp->id);
   outimage(stderr, (struct descrip *)(&ncp->es_sp[-3]), 0);
   fprintf(stderr," to #%ld\n", (long)ncp->id);
   fflush(stderr);
   }

/*
 * cofailtrace -- failure return from co-expression; produce a trace message.
 */
cofailtrace(ccp, ncp)
struct b_coexpr *ccp;
struct b_coexpr *ncp;
   {
   struct b_proc *bp;
   inst t_ipc;

   bp = (struct b_proc *)BlkLoc(*argp);
   if (k_trace > 0)
      k_trace--;
   /*
    * Compute the ipc of the cofail instruction.
    */
   t_ipc.op = ipc.op - 1;
   showline(findfile(t_ipc.opnd), findline(t_ipc.opnd));
   showlevel(k_level);
   putstr(stderr, &(bp->pname));
   fprintf(stderr,"; #%ld failed to #%ld\n", (long)ccp->id, (long)ncp->id);
   fflush(stderr);
   }

#endif					/* NoCoexpr */
/*
 * showline - print file and line number information.
 */
static showline(f, l)
char *f;
int l;
   {
   if (l > 0)
      fprintf(stderr, "%.10s: %d\t", f, l);
   else
      fprintf(stderr, "\t\t");
   }

/*
 * showlevel - print "| " n times.
 */
static showlevel(n)
register int n;
   {
   while (n-- > 0) {
      putc('|', stderr);
      putc(' ', stderr);
      }
   }

/*
 * putpos - assign value to &pos
 */

putpos(dp)
struct descrip *dp;
   {
   register word l1;
   long l2;
   switch (cvint(dp, &l2)) {

      case T_Integer:
         l1 = cvpos(l2, StrLen(k_subject));
         if (l1 == CvtFail)
            return Failure;
         k_pos = l1;
         return Success;

      default:
         RetError(101, *dp);
      }
   }

/*
 * putran - assign value to &random
 */

putran(dp)
struct descrip *dp;
   {
   long l1;
   switch (cvint(dp, &l1)) {

      case T_Integer:
         k_random = l1;
         return Success;

      default:
         RetError(101, *dp);
      }
   }

/*
 * putsub - assign value to &subject
 */

putsub(dp)
struct descrip *dp;
   {
   char sbuf[MaxCvtLen];
   extern char *alcstr();

   switch (cvstr(dp, sbuf)) {

      case Cvt:
         if (strreq(StrLen(*dp)) == Error)
            return Error;
         StrLoc(*dp) = alcstr(StrLoc(*dp), StrLen(*dp));
         /* no break */

      case NoCvt:
         k_subject = *dp;
         k_pos = 1;
         return Success;

     default:
        RetError(103, *dp);

      }
   }

/*
 * puttrc - assign value to &trace
 */

puttrc(dp)
struct descrip *dp;
   {
   long l1;
   switch (cvint(dp, &l1)) {

      case T_Integer:
         k_trace = (int)l1;
         return Success;

      default:
         RetError(101, *dp);
      }
   }

/*
 * puterr - assign value to &error
 */

puterr(dp)
struct descrip *dp;
   {
   long l1;
   switch (cvint(dp, &l1)) {

      case T_Integer:
         k_error = (int)l1;
         return Success;

      default:
         RetError(101, *dp);
      }
   }

#ifndef NoCoexpr
/*
 * pushact - push actvtr on the activator stack of ce
 */
int pushact(ce, actvtr)
struct b_coexpr *ce, *actvtr;
{
   struct astkblk *abp = ce->es_actstk, *nabp;
   struct actrec *arp;
   extern struct astkblk *alcactiv();

   /*
    * If the last activator is the same as this one, just increment
    *  its count.
    */
   if (abp->nactivators > 0) {
      arp = &abp->arec[abp->nactivators - 1];
      if (arp->activator == actvtr) {
         arp->acount++;
         return Success;
         }
      }
   /*
    * This activator is different from the last one.  Push this activator
    *  on the stack, possibly adding another block.
    */
   if (abp->nactivators + 1 > ActStkBlkEnts) {
      nabp = alcactiv();
      if (nabp == NULL)
         return Error;
      nabp->astk_nxt = abp;
      abp = nabp;
      }
   abp->nactivators++;
   arp = &abp->arec[abp->nactivators - 1];
   arp->acount = 1;
   arp->activator = actvtr;
   ce->es_actstk = abp;
   return Success;
}

/*
 * popact - pop the most recent activator from the activator stack of ce
 *  and return it.
 */
struct b_coexpr *popact(ce)
struct b_coexpr *ce;
{
   struct astkblk *abp = ce->es_actstk, *oabp;
   struct actrec *arp;
   struct b_coexpr *actvtr;

   /*
    * If the current stack block is empty, pop it.
    */
   if (abp->nactivators == 0) {
      oabp = abp;
      abp = abp->astk_nxt;
      free((char *)oabp);
      }

   if (abp == NULL || abp->nactivators == 0)
      syserr("empty activator stack\n");

   /*
    * Find the activation record for the most recent co-expression.
    *  Decrement the activation count and if it is zero, pop that
    *  activation record and decrement the count of activators.
    */
   arp = &abp->arec[abp->nactivators - 1];
   actvtr = arp->activator;
   if (--arp->acount == 0)
      abp->nactivators--;

   ce->es_actstk = abp;
   return actvtr;
}

/*
 * topact - return the most recent activator of ce.
 */
struct b_coexpr *topact(ce)
struct b_coexpr *ce;
{
   struct astkblk *abp = ce->es_actstk;
   
   if (abp->nactivators == 0)
      abp = abp->astk_nxt;
   return abp->arec[abp->nactivators-1].activator;
}

/*
 * dumpact - dump an activator stack
 */
dumpact(ce)
struct b_coexpr *ce;
{
   struct astkblk *abp = ce->es_actstk;
   struct actrec *arp;
   int i;

   if (abp)
      fprintf(stderr, "Ce %ld ", (long)ce->id);
   while (abp) {
      fprintf(stderr, "--- Activation stack block (%x) --- nact = %d\n",
         abp, abp->nactivators);
      for (i = abp->nactivators; i >= 1; i--) {
         arp = &abp->arec[i-1];
         /*for (j = 1; j <= arp->acount; j++)*/
         fprintf(stderr, "#%ld(%d)\n", (long)(arp->activator->id), arp->acount);
         }
      abp = abp->astk_nxt;
      }
}
#endif					/* NoCoexpr */

/*
 * Service routine to display variables in given number of
 *  procedure calls to file f.
 */

xdisp(count,f)
   int count;
   FILE *f;
   {
   struct pf_marker *fp;
   register struct descrip *dp;
   register struct descrip *np;
   register int n;
   struct b_proc *bp;
   extern struct descrip *globals, *eglobals;
   extern struct descrip *gnames;
   extern struct descrip *statics;

   fp = pfp;		/* start fp at most recent procedure frame */
   dp = argp;
   while (count--) {		/* go back through 'count' frames */

      bp = (struct b_proc *) BlkLoc(*dp); /* get address of procedure block */

      /*
       * Print procedure name.
       */
      putstr(f, &(bp->pname));
      fprintf(f, " local identifiers:\n");

      /*
       * Print arguments.
       */
      np = bp->lnames;
      for (n = abs(bp->nparam); n > 0; n--) {
         fprintf(f, "   ");
         putstr(f, np);
         fprintf(f, " = ");
         outimage(f, ++dp, 0);
         putc('\n', f);
         np++;
         }

      /*
       * Print locals.
       */
      dp = &fp->pf_locals[0];
      for (n = bp->ndynam; n > 0; n--) {
         fprintf(f, "   ");
         putstr(f, np);
         fprintf(f, " = ");
         outimage(f, dp++, 0);
         putc('\n', f);
         np++;
         }

      /*
       * Print statics.
       */
      dp = &statics[bp->fstatic];
      for (n = bp->nstatic; n > 0; n--) {
         fprintf(f, "   ");
         putstr(f, np);
         fprintf(f, " = ");
         outimage(f, dp++, 0);
         putc('\n', f);
         np++;
         }

      dp = fp->pf_argp;
      fp = fp->pf_pfp;
      }

   /*
    * Print globals.
    */
   fprintf(f, "global identifiers:\n");
   dp = globals;
   np = gnames;
   while (dp < eglobals) {
      fprintf(f, "   ");
      putstr(f, np);
      fprintf(f, " = ");
      outimage(f, dp++, 0);
      putc('\n', f);
      np++;
      }
   fflush(f);
   }

/*
 * findline - find the source line number associated with the ipc
 */
int findline(ipc)
word *ipc;
{
   uword ipc_offset;
   uword size;
   struct ipc_line *base;
   extern struct ipc_line *ilines, *elines;
   extern word *records;

   if ((uword)ipc < (uword)code || (uword)ipc >= (uword)records)
      return 0;
   ipc_offset = (uword)ipc - (uword)code;
   base = ilines;
   size = ((uword)elines - (uword)ilines) / sizeof(struct ipc_line);
   while (size > 1) {
      if (ipc_offset >= base[size / 2].ipc) {
         base = &base[size / 2];
         size -= size / 2;
         }
      else
         size = size / 2;
      }
   return base->line;
}

/*
 * findfile - find source file name associated with the ipc
 */
char *findfile(ipc)
word *ipc;
{
   uword ipc_offset;
   struct ipc_fname *p;
   extern struct ipc_fname *filenms, *efilenms;
   extern word *records;
   extern char *strcons;

   if ((uword)ipc < (uword)code || (uword)ipc >= (uword)records)
      return "?";
   ipc_offset = (uword)ipc - (uword)code;
   for (p = efilenms - 1; p >= filenms; p--)
      if (ipc_offset >= p->ipc)
         return strcons + p->fname;
   syserr("bad ipc/file name table");
}
