/*
 * File: rconv.c
 *  Contents: cvcset, cvint, cvnum, cvpos, cvreal, cvstr, mkint,
 *    mkreal, mksubs, strprc
 */

#include <math.h>
#include "../h/rt.h"

extern double bigtoreal ();

#ifndef NoStrInvoke
/*
 * Structure for mapping string names of procedures to block addresses.
 */
struct pstrnm {
   char *pstrep;
   struct b_proc *pblock;
   };

extern struct b_proc
#define FncDef(p) Cat(B,p),
#include "../h/fdefs.h"
   Bnoproc;	/* Hack to avoid ,; in expansion */
#undef FncDef

extern struct b_proc
   Basgn,
   Bbang,
   Bcat,
   Bcompl,
   Bdiff,
   Bdivide,
   Beqv,
   Binter,
   Blconcat,
   Blexeq,
   Blexge,
   Blexgt,
   Blexle,
   Blexlt,
   Blexne,
   Bminus,
   Bmod,
   Bmult,
   Bneg,
   Bneqv,
   Bnonnull,
   Bnull,
   Bnumber,
   Bnumeq,
   Bnumge,
   Bnumgt,
   Bnumle,
   Bnumlt,
   Bnumne,
   Bplus,
   Bpower,
   Brandom,
   Brasgn,
   Brefresh,
   Brswap,
   Bsect,
   Bsize,
   Bsubsc,
   Bswap,
   Btabmat,
   Btoby,
   Bunions,
   Bvalue;

struct pstrnm pntab[] = {
#define FncDef(p) Lit(p), Cat(&B,p),
#include "../h/fdefs.h"
#undef FncDef
	":=",            &Basgn,
	"!",             &Bbang,
	"||",            &Bcat,
	"~",             &Bcompl,
	"--",            &Bdiff,
	"/",             &Bdivide,
	"===",           &Beqv,
	"**",            &Binter,
	"|||",           &Blconcat,
	"==",            &Blexeq,
	">>=",           &Blexge,
	">>",            &Blexgt,
	"<<=",           &Blexle,
	"<<",            &Blexlt,
	"~==",           &Blexne,
	"-",             &Bminus,
	"%",             &Bmod,
	"*",             &Bmult,
	"-",             &Bneg,
	"~===",          &Bneqv,
	"\\",            &Bnonnull,
	"/",             &Bnull,
	"+",             &Bnumber,
	"=",             &Bnumeq,
	">=",            &Bnumge,
	">",             &Bnumgt,
	"<=",            &Bnumle,
	"<",             &Bnumlt,
	"~=",            &Bnumne,
	"+",             &Bplus,
	"^",             &Bpower,
	"?",             &Brandom,
	"<-",            &Brasgn,
	"^",             &Brefresh,
	"<->",           &Brswap,
	":",             &Bsect,
	"*",             &Bsize,
	"[]",            &Bsubsc,
	":=:",           &Bswap,
	"=",             &Btabmat,
	"...",           &Btoby,
	"++",            &Bunions,
	".",             &Bvalue,
	0,		 0
	};
#endif					/* NoStrInvoke */

#include <ctype.h>

#ifndef EBCDIC
#define tonum(c)	(isdigit(c) ? (c)-'0' : 10+(((c)|(040))-'a'))
#else					/* EBCDIC */
/*
 *  The following definition needs fixing to handle letters in radix
 *   literals.	Right now it treats them as zeroes.
 */
#define tonum(c)	(isdigit(c) ? (c) - '0' : '0')
#endif					/* EBCDIC */
/*
 * cvcset(dp, cs, csbuf) - convert dp to a cset and
 *  make cs point to it, using csbuf as a buffer if necessary.
 */

cvcset(dp, cs, csbuf)
register struct descrip *dp;
int **cs, *csbuf;
   {
   register char *s;
   register word l;
   char sbuf[MaxCvtLen];

   Inc(cv_n_cset);

   if (!Qual(*dp) && (dp)->dword == D_Cset) {
      Inc(cv_n_rcset);
      *cs = BlkLoc(*dp)->cset.bits;
      return T_Cset;
      }

   if (cvstr(dp, sbuf) == CvtFail)
      return CvtFail;

   for (l = 0; l < CsetSize; l++)
      csbuf[l] = 0;

   s = StrLoc(*dp);
   l = StrLen(*dp);
   while (l--) {
      Setb(*s, csbuf);
      s++;
      }
   *cs = csbuf;
   return T_Cset;
   }

/*
 * cvint - convert the value represented by dp into an integer and write
 *  the value into the location referenced by i.  cvint returns the type or
 *  CvtFail depending on the outcome of the conversion.
 */

cvint(dp, i)
register struct descrip *dp;
long *i;
   {
   union numeric result;

#ifdef RunStats
   Inc(cv_n_int);
   if (!Qual(*dp) && (dp)->dword == D_Integer)
      cv_n_rint++;
#endif					/* RunStats */

   /*
    * Use cvnum to attempt the conversion into "result".
    */
   switch (cvnum(dp, &result)) {

      case T_Integer:
	 *i = result.integer;
	 return T_Integer;

      case T_Bignum:
	  /*
	   *  Bignum, not in the range of an integer.  Fail as we do
	   *  for large reals.
	   */  
	  return CvtFail;

      case T_Real:
	 /*
	  * The value converted into a real number.  If it's not in the
	  *  range of an integer, return a 0, otherwise convert the
	  *  real value into an integer.  As before, distinguish between
	  *  integers and long integers if necessary.
	  */
	 if (result.real > MaxLong || result.real < MinLong)
	    return CvtFail;
	 *i = (long)result.real;
	 return T_Integer;

      default:
	 return CvtFail;
      }
   }

/*
 * cvnum - convert the value represented by d into a numeric quantity and
 *  place the value into *result. The value returned is the type or CvtFail.
 */

cvnum(dp,result)
register struct descrip *dp;
union numeric *result;
   {
   static char sbuf[MaxCvtLen];
   struct descrip cstring;
   register int t;

   cstring = *dp;  /* placed outside "if" to avoid Lattice 3.21 code gen bug */
   Inc(cv_n_num);
   if (Qual(*dp)) {
      qtos(&cstring, sbuf);
      if ((t = ston(StrLoc(cstring), result)) == T_Bignum)
	  mkbignum (result->bptr, dp); 
      return t;
      }

   switch (Type(*dp)) {

      case T_Integer:
	 Inc(cv_n_rnum);
	 result->integer = (long)IntVal(*dp);
	 return T_Integer;

      case T_Bignum: 
	 result->bptr = &BlkLoc(*dp)->bignumblk;
	 return T_Bignum;

      case T_Real:
	 Inc(cv_n_rnum);
	 GetReal(dp,result->real);
	 return T_Real;

      default:
	 /*
	  * Try to convert the value to a string and
	  *  then try to convert the string to an integer.
	  */
	 if (cvstr(dp, sbuf) == CvtFail)
	    return CvtFail;
	 if ((t = ston(StrLoc(*dp), result)) == T_Bignum)
	     mkbignum (result->bptr, dp); 
	 return t;
      }
   }

/*
 * ston - convert a string to a numeric quantity if possible.
 */
static ston(s, result)
register char *s;
union numeric *result;
   {
   register int c;
   int realflag = 0;	/* indicates a real number */
   char msign = '+';    /* sign of mantissa */
   char esign = '+';    /* sign of exponent */
   double mantissa = 0; /* scaled mantissa with no fractional part */
   int scale = 0;	/* number of decimal places to shift mantissa */
   int digits = 0;	/* total number of digits seen */
   int sdigits = 0;	/* number of significant digits seen */
   int exponent = 0;	/* exponent part of real number */
   double fiveto;	/* holds 5^scale */
   double power;	/* holds successive squares of 5 to compute fiveto */
   char *ssave = s;

/*
 * The following code is operating-system dependent. Get errno.
 */

#if PORT
   extern int errno;
#endif					/* PORT */

#if ATARI_ST
   int errno;
#endif					/* ATARI_ST */

#if AMIGA || MACINTOSH || UNIX
   extern int errno;
#endif					/* AMIGA || MACINTOSH || UNIX  */

#if MSDOS
   /* Already defined in math.h */
#endif					/* MSDOS */

#if VMS
#include <errno.h>
#endif					/* VMS */

#if VM || MVS
#endif					/* VM || MVS */
/*
 * End of operating-system specific code.
 */

   c = *s++;

   /*
    * Skip leading white space.
    */
   while (isspace(c))
      c = *s++;

   /*
    * Check for sign.
    */
   if (c == '+' || c == '-') {
      msign = c;
      c = *s++;
      }

   /*
    * Get integer part of mantissa.
    */
   while (isdigit(c)) {
      digits++;
      if (mantissa < Big) {
	 mantissa = mantissa * 10 + (c - '0');
	 if (mantissa > 0.0)
	    sdigits++;
	 }
      else
	 scale++;
      c = *s++;
      }

   /*
    * Check for based integer.
    */
   if (c == 'r' || c == 'R')
       return bigradix (msign, (int) mantissa, s, result);

   /*
    * Get fractional part of mantissa.
    */
   if (c == '.') {
      realflag++;
      c = *s++;
      while (isdigit(c)) {
	 digits++;
	 if (mantissa < Big) {
	    mantissa = mantissa * 10 + (c - '0');
	    scale--;
	    if (mantissa > 0.0)
	       sdigits++;
	    }
	 c = *s++;
	 }
      }

   /*
    * Check that at least one digit has been seen so far.
    */
   if (digits == 0)
      return CvtFail;

   /*
    * Get exponent part.
    */
   if (c == 'e' || c == 'E') {
      realflag++;
      c = *s++;
      if (c == '+' || c == '-') {
	 esign = c;
	 c = *s++;
	 }
      if (!isdigit(c))
	 return CvtFail;
      while (isdigit(c)) {
	 exponent = exponent * 10 + (c - '0');
	 c = *s++;
	 }
      scale += (esign == '+')? exponent : -exponent;
      }

   /*
    * Skip trailing white space.
    */
   while (isspace(c))
      c = *s++;

   /*
    * Check that entire string has been consumed.
    */
   if (c != '\0')
      return CvtFail;

   /*
    * Test for integer.
    */
   if (!realflag && mantissa >= MinLong && mantissa <= MaxLong) {
      result->integer = (msign == '+')? mantissa : -mantissa;
	 return T_Integer;
      }

   /*
    * Test for bignum
    */

   if (!realflag)
       return bigradix (msign, 10, ssave, result);

   /*
    * Rough tests for overflow and underflow.
    */
   if (sdigits + scale > LogHuge)
      return CvtFail;

   if (sdigits + scale < -LogHuge) {
      result->real = 0.0;
      return T_Real;
      }

   /*
    * Put the number together by multiplying the mantissa by 5^scale and
    *  then using ldexp() to multiply by 2^scale.
    */

   exponent = (scale > 0)? scale : -scale;
   fiveto = 1.0;
   power = 5.0;
   for (;;) {
      if (exponent & 01)
	 fiveto *= power;
      exponent >>= 1;
      if (exponent == 0)
	 break;
      power *= power;
      }
   if (scale > 0)
      mantissa *= fiveto;
   else
      mantissa /= fiveto;

   errno = 0;
   mantissa = ldexp(mantissa, scale);
   if (errno > 0 && mantissa > 0)
      /*
       * ldexp caused overflow.
       */
      return CvtFail;

   result->real = (msign == '+')? mantissa : -mantissa;
   return T_Real;
   }

/*
 * radix - convert string s in radix r into an integer in *result.  sign
 *  will be either '+' or '-'.
 */
static radix(sign, r, s, result)
char sign;
register int r;
register char *s;
union numeric *result;
   {
   register int c;
   long num;

   if (r < 2 || r > 36)
      return CvtFail;
   c = *s++;
   num = 0L;
   while (isalnum(c)) {
      c = tonum(c);
      if (c >= r)
	 return CvtFail;
      num = num * r + c;
      c = *s++;
      }

   while (isspace(c))
      c = *s++;

   if (c != '\0')
      return CvtFail;

   result->integer = (sign == '+')? num : -num;

   return T_Integer;
   }


/*
 * cvpos - convert position to strictly positive position
 *  given length.
 */

word cvpos(pos, len)
long pos;
register word len;
   {
   register word p;

   /*
    * Make sure the position is in the range of an int. (?)
    */
   if ((long)(p = pos) != pos)
      return CvtFail;
   /*
    * Make sure the position is within range.
    */
   if (p < -len || p > len + 1)
      return CvtFail;
   /*
    * If the position is greater than zero, just return it.  Otherwise,
    *  convert the zero/negative position.
    */
   if (pos > 0)
      return p;
   return (len + p + 1);
   }

/*
 * cvreal - convert to real and put the result into *r.
 */

cvreal(dp, r)
register struct descrip *dp;
double *r;
   {
   union numeric result;

#ifdef RunStats
   Inc(cv_n_real);
   if (!Qual(*dp) && (dp)->dword == D_Real)
      cv_n_rreal++;
#endif					/* RunStats */
   /*
    * Use cvnum to classify the value.	Cast integers into reals and
    *  fail if the value is non-numeric.
    */
   switch (cvnum(dp, &result)) {

      case T_Integer:
	 *r = result.integer;
	 return T_Real;

     case T_Bignum:
	 *r = bigtoreal (result.bptr);
	 return T_Real;

      case T_Real:
	 *r = result.real;
	 return T_Real;

      default:
	 return CvtFail;
      }
   }

/*
 * cvstr(dp,s) - convert dp (in place) into a string, using s as buffer
 *  if necessary.  cvstr returns CvtFail if the conversion fails, Cvt if dp
 *  wasn't a string but was converted into one, and NoCvt if dp was already
 *  a string.  When a string conversion takes place, sbuf gets the
 *  resulting string.
 */

/* >cvstr */
cvstr(dp, sbuf)
register struct descrip *dp;
char *sbuf;
   {
   double rres;

   Inc(cv_n_str);
   if (Qual(*dp)) {
      Inc(cv_n_rstr);
      return NoCvt;			/* It is already a string */
      }

   switch (Type(*dp)) {
      /*
       * For types that can be converted into strings, call the
       *  appropriate conversion routine and return its result.
       *  Note that the conversion routines change the descriptor
       *  pointed to by dp.
       */
      case T_Integer:
	 return itos((long)IntVal(*dp), dp, sbuf);

      case T_Bignum:
	 return bigtos (dp, dp, sbuf);

      case T_Real:
	 GetReal(dp,rres);
	 return rtos(rres, dp, sbuf);

      case T_Cset:
	 return cstos(BlkLoc(*dp)->cset.bits, dp, sbuf);

      default:
	 /*
	  * The value cannot be converted to a string.
	  */
	 return CvtFail;
      }
   }
/* <cvstr */

/*
 * itos - convert the integer num into a string using s as a buffer and
 *  making q a descriptor for the resulting string.
 */
static itos(num, dp, s)
long num;
struct descrip *dp;
char *s;
   {
   register char *p;
   long ival;

   p = s + MaxCvtLen - 1;
   ival = num;

   *p = '\0';
   if (num >= 0L)
      do {
	 *--p = ival % 10L + '0';
	 ival /= 10L;
	 } while (ival != 0L);
   else {
      do {
	 *--p = '0' - (ival % 10L);
	 ival /= 10L;
	 } while (ival != 0L);
      *--p = '-';
      }

   StrLen(*dp) = s + MaxCvtLen - 1 - p;
   StrLoc(*dp) = p;
   return Cvt;
   }

/*
 * rtos - convert the real number n into a string using s as a buffer and
 *  making a descriptor for the resulting string.
 */
rtos(n, dp, s)
double n;
struct descrip *dp;
char *s;
   {

   s++; 			/* leave room for leading zero */
/*
 * The following code is operating-system dependent. Convert real
 *  number to string.
 *
 * If IconGcvt is defined, icon_gcvt() is actually called, due to a #define
 *  in config.h.
 */

#if PORT
   gcvt(n, Precision, s);
#endif					/* PORT */

#if AMIGA
   gcvt(n, Precision, s);
#endif					/* AMIGA */

#if ATARI_ST
   gcvt(n, Precision, s);
#endif					/* ATARI_ST */

#if MACINTOSH
   sprintf(s,"%20g",n);
#endif					/* MACINTOSH */

#if MSDOS
   gcvt(n, Precision, s);
#endif					/* MSDOS */

#if UNIX
   gcvt(n, Precision, s);
#endif					/* UNIX */

#if VM || MVS
#endif					/* VM || MVS */

#if VMS
   gcvt(n, Precision, s);
#endif					/* VMS */

/*
 * End of operating-system specific code.
 */
   if (s[0] == '.') {
      s--;
      s[0] = '0';
      }
   else if (!index(s, '.'))
      strcat(s, ".0");
   if (s[strlen(s) - 1] == '.')
      strcat(s, "0");
   StrLen(*dp) = strlen(s);
   StrLoc(*dp) = s;
   return Cvt;
   }

/*
 * cstos - convert the cset bit array pointed at by cs into a string using
 *  s as a buffer and making a descriptor for the resulting string.
 */

static cstos(cs, dp, s)
int *cs;
struct descrip *dp;
char *s;
   {
   register unsigned int w;
   register int j, i;
   register char *p;

   p = s;
   for (i = 0; i < CsetSize; i++) {
      if (cs[i])
	 for (j=i*IntBits, w=cs[i]; w; j++, w >>= 1)
	    if (w & 01)
	       *p++ = (char)j;
      }
   *p = '\0';

   StrLen(*dp) = p - s;
   StrLoc(*dp) = s;
   return Cvt;
   }

/*
 * mkreal(r, dp) - make a real number descriptor and associated block
 *  for r and place it in *dp.
 */

int mkreal(r, dp)
double r;
register struct descrip *dp;
   {
   extern struct b_real *alcreal();

   if (blkreq((uword)sizeof(struct b_real)) == Error)
      return Error;
   dp->dword = D_Real;
   BlkLoc(*dp) = (union block *) alcreal(r);
   return Success;
   }

mkbignum (bp, dp)
register struct b_bignum *bp;
register struct descrip *dp;
{
    dp->dword = D_Bignum;
    BlkLoc(*dp) = (union block *) bp;
    return Success;
}

/*
 * mksubs - form a substring.  var is a descriptor for the string from
 *  which the substring is to be formed.  var may be a variable.  val
 *  is a dereferenced version of var.  The descriptor for the resulting
 *  substring is placed in *result.  The substring starts at position
 *  i and extends for j characters.
 */

mksubs(var, val, i, j, result)
register struct descrip *var, *val, *result;
word i, j;
   {
   extern struct b_tvsubs *alcsubs();

   if (Qual(*var) || !Var(*var)) {
      /*
       * var isn't a variable, just form a descriptor that points into
       *  the string named by val.
       */
      StrLen(*result) = j;
      StrLoc(*result) = StrLoc(*val) + i - 1;
      return;
      }

   if ((var)->dword == D_Tvsubs) {
      /*
       * If var is a substring trapped variable,
       *  adjust the position and make var the substrung string.
       */
	 i += BlkLoc(*var)->tvsubs.sspos - 1;
	 var = &BlkLoc(*var)->tvsubs.ssvar;
	 }

   /*
    * Make a substring trapped variable by passing the buck to alcsubs.
    */
   result->dword = D_Tvsubs;
   BlkLoc(*result) = (union block *) alcsubs(j, i, var);
   return;
   }

/*
 * strprc - Convert the qualified string named by *dp into a procedure
 *  descriptor if possible.  n is the number of arguments that the desired
 *  procedure has.  n is only used when the name of the procedure is
 *  non-alphabetic (hence, an operator).
 *
 */
strprc(dp, n)
struct descrip *dp;
word n;
   {

#ifdef NoStrInvoke
   return CvtFail;
#else					/* NoStrInvoke */

   extern struct descrip *gnames, *globals, *eglobals;
   struct descrip *np, *gp;
   struct pstrnm *p;
   char *s;
   word ns, l;

   /*
    * Look in global name list first.
    */
   np = gnames; gp = globals;
   while (gp < eglobals) {
   if (!lexcmp(np++,dp))
      if (BlkLoc(*gp)->proc.title == T_Proc) {
	 StrLen(*dp) = D_Proc; /* really type field */
	 BlkLoc(*dp) = BlkLoc(*gp);
	 return T_Proc;
	 }
   gp++;
   }
/*
 * The name is not a global, see if it is a function or an operator.
 */
   s = StrLoc(*dp);
   l = StrLen(*dp);
   for (p = pntab; p->pstrep; p++)
      /*
       * Compare the desired name with each standard procedure/operator
       *  name.
       */
      if (!slcmp(s,l,p->pstrep)) {
	 if (isalpha(*s)) {
	    /*
		* The names are the same and s starts with an alphabetic,
		*  so it's the one being looked for; return it.
		*/
	     StrLen(*dp) = D_Proc;
	     BlkLoc(*dp) = (union block *) p->pblock;
	     return T_Proc;
	     }
	  if ((ns = p->pblock->nstatic) < 0)
	     ns = -ns;
	  else
	     ns = abs(p->pblock->nparam);
	  if (n == ns) {
	     StrLen(*dp) = D_Proc; /* really type field */
	     BlkLoc(*dp) = (union block *) p->pblock;
	     return T_Proc;
	     }
	 }
   return CvtFail;
#endif					/* NoStrInvoke */
   }

/*
 * slcmp - lexically compare l1 bytes of s1 with null-terminated s2.
 */

slcmp(s1, l1, s2)
word l1;
char *s1,*s2;
   {
   register word minlen;
   word l2;

   l2 = strlen(s2);

   minlen = (l1 <= l2) ? l1 : l2;

   while (minlen--)
      if (*s1++ != *s2++)
	 return (int)((*--s1 & 0377) - (*--s2 & 0377));

   return (int)(l1 - l2);
   }
