#include "../h/rt.h"
#include <math.h>
#include <ctype.h>

/* The number of bits in each base-B digit; the type DIGIT in rt.h
   must be large enough to hold this many bits.
   It must be at least 2 and at most WordSize - 1. */

#define NB 16

/* Type for doing arithmetic on (2 * NB)-bit nonnegative numbers.
   Normally unsigned but may be signed (with NB reduced appropriately)
   if unsigned arithmetic is slow.  */

typedef unsigned long WORD;

/* Type the same size as WORD, must be signed. */

typedef long SWORD;

/* Type the same size as WORD, must be unsigned. */

typedef unsigned long UWORD;

/*
 *  Conventions:
 *
 *  Bignums entering this module and leaving it are too large to
 *  be represented with T_Integer.  So, externally, a given value
 *  is always T_Integer or always T_Bignum.
 *
 *  Routines outside this module operate on bignums by calling
 *  a routine like
 *
 *      bigadd (da, db, dx)
 *
 *  where da, db, and dx are pointers to tended descriptors.
 *  For the common case where one operand is a T_Integer, there
 *  are routines like
 *
 *      bigaddi (da, i, dx).
 *
 *  Where no appropriate routine exists, an integer can be
 *  converted to bignum form by using itobigl:
 *
 *      bigdiv (itobigl (a), db, dx)
 *
 *  itobigl converts its arg to bignum form and returns a descriptor
 *  to it.  This descriptor points to static storage and must not
 *  be put into any tended slot -- it is valid only as an argument
 *  to a big() routine.  The bigxxxi routines also can convert
 *  an integer to bignum form; they use itobigr.  itobigl is called
 *  only from outside this module, itobigr only from inside it.
 *
 *  The routines that actually do the work take (length, address)
 *  pairs specifying unsigned base-B digit strings.  The sign handling
 *  is done in the bigxxx routines.
 */

#define bitsizeof(t) (8 * sizeof (t))

/* The bignum radix, B */

#define B ((SWORD) 1 << NB)

/* Bits in a word (T_Integer) */

#define NW (bitsizeof (word))

/* Bignum digits in a word */

#define WORDLEN (NW / NB + (NW % NB != 0))

/* lo (WORD d) : the low digit of a WORD
   hi (WORD d) : the rest, d is unsigned
   signed_hi (WORD d) : the rest, d is signed
   dbl (DIGIT a, DIGIT b) : the two-digit WORD [a,b] */

#define lo(d) ((d) & (B - 1))
#define hi(d)  ((UWORD) (d) >> NB) 
#define dbl(a,b) (((WORD) (a) << NB) + (b))

#if ((-1) >> 1) < 0
#define signed_hi(d) ((SWORD) (d) >> NB)
#else
#define signbit ((WORD) 1 << (bitsizeof (WORD) - NB - 1))
#define signed_hi(d)  ((SWORD) ((((UWORD) (d) >> NB) ^ signbit) - signbit))
#endif

/* Icon block pointed to by descriptor, defined in rt.h */

typedef struct b_bignum BIGNUM; 

/* DBIG (struct descrip *dp) : the BIGNUM pointed to by dp */

#define DBIG(dp) (&BlkLoc(*dp)->bignumblk)

/* DLEN (struct descrip *dp) : number of significant digits */

#define DLEN(dp) (LEN(DBIG(dp)))

/* LEN (BIGNUM *b) : number of significant digits */

#define LEN(b) ((b)->right - (b)->left + 1)

/* Digits are stored in big-endian order */

#define msd left		/* most significant digit */
#define lsd right		/* least significant */

/* DIG (BIGNUM *b, int i): pointer to i'th most significant digit */

#define DIG(b,i) (&(b)->digits[(b)->msd+(i)])

/* D (DIGIT *d, int i): the i'th most significant digit */ 

#define D(d,i) ((d)[i])	

/* ceil, ln: ceil may be 1 too high in case ln is inaccurate */

#define ceil(x) ((word) ((x) + 1.01))
#define ln(n) (log ((double) n))

/* tempalloc (int n, type ty) : scratch space for n things of type ty */

#define tempalloc(n,ty) ((ty *) alloca ((n) * sizeof (ty)))

/* copied from rconv.c */

#ifndef EBCDIC
#define tonum(c) \
    (isdigit (c) ? (c) - '0' : (((c) | (040)) - 'a' + 10))
#else 
#define tonum(c) \
    index ("0123456789abcdefghijklmnopqrstuvwsyz", tolower (c))
#endif

/* copied from oref.c */

#define RandVal (RanScale*(k_random=(RandA*k_random+RandC)&MaxLong))

static DIGIT add1 ();
static word sub1 ();
static void mul1 ();
static void div1 ();
static word cmp1 ();
static DIGIT addi1 ();
static void subi1 ();
static DIGIT muli1 ();
static DIGIT divi1 ();
static DIGIT shifti1 ();
static word cmpi1 ();
static word compl1 ();

static word bigret ();
struct descrip *itobigl ();
static struct descrip *itobigr ();
static void itobig ();

extern BIGNUM *alcbignum ();
extern char *alcstr ();

#ifdef __GNUC__
#define alloca __builtin_alloca
#else
extern char *alloca ();
#endif

/*
 * retnum -- put value into a union numeric and return the appropriate type
 * retdesc -- put value into a descriptor and return void
 */

#define retdesc(x,dp) ((dp)->dword = bigret (x, &(dp)->vword))
#define retnum(x,np) ((bigret (x, np)) & TypeMask)

static word bigret (x, np)
    BIGNUM *x;
    union numeric *np;
{   int xlen, cmp;

    static DIGIT maxword[WORDLEN] = {1 << ((NW-1) % NB)};

    /* suppress leading zero digits */

    while (x->msd != x->lsd && *DIG(x,0) == 0) {
	x->msd++;}

    /* put it into a word if it fits, otherwise return the bignum */

    xlen = LEN(x);

    if (xlen < WORDLEN
	|| (xlen == WORDLEN
	    && ((cmp = cmp1 (DIG(x,0), maxword, WORDLEN)) < 0
		|| (cmp == 0 && x->sign)))) {
	word val = - (word) *DIG(x,0);
	int i;

	for (i = x->msd; ++i <= x->lsd; ) {
	    val = (val << NB) - x->digits[i];}
	if (!x->sign) val = -val;
	np->integer = val;
	return D_Integer;}
    else {
	np->bptr = x;
	return D_Bignum;}}

/*
 *  i -> big
 *
 *  the result is a temp only usable as an arg to a bigxxx routine
 */

static struct descrip dleft, dright;
static char bleft [sizeof (BIGNUM) + sizeof (DIGIT) * WORDLEN];
static char bright [sizeof (BIGNUM) + sizeof (DIGIT) * WORDLEN];

struct descrip *olditobigl();
struct descrip *itobigl (i)
    word i;
{   itobig (i, (BIGNUM *) bleft, &dleft);
    return &dleft;}

static struct descrip *itobigr (i)
    word i;
{   itobig (i, (BIGNUM *) bright, &dright);
    return &dright;}

static void itobig (i, x, dx)
    word i;
    BIGNUM *x;
    struct descrip *dx;
{
    x->lsd = WORDLEN - 1;
    x->msd = WORDLEN;
    x->sign = 0;

    if (i == 0) {
	x->msd--;
	*DIG(x,0) = 0;}
    else if (i < 0) {
	word d = lo(i);
	if (d != 0) {
	    d = B - d;
	    i += B;}
	i = - signed_hi (i);
	x->msd--;
	*DIG(x,0) = d;
	x->sign = 1;}
	    
    while (i != 0) {
	x->msd--;
	*DIG(x,0) = lo(i);
	i = hi (i);}

    BlkLoc(*dx) = (union block *) x;}

/*
 *  string -> bignum 
 */

word bigradix (sign, r, s, result)
    char sign;				/* '-' or not */
    int r;				/* radix 2 .. 36 */
    char *s;				/* input string */
    union numeric *result;		/* output T_Integer or T_Bignum */
{   int len = ceil (strlen(s) * ln(r) / ln(B));
    BIGNUM *b = alcbignum (len);
    DIGIT *bd = DIG(b,0);
    word c;

    bzero (bd, len * sizeof (DIGIT));

   if (r < 2 || r > 36)
      return CvtFail;

    for (c = *s++; isalnum(c); c = *s++) {
	c = tonum (c);
	if (c >= r) return CvtFail;
	muli1 (bd, r, c, bd, len);}

    while (isspace (c))
	c = *s++;

    if (c != '\0')
	return CvtFail;

    if (sign == '-')
	b->sign = 1;

    return retnum (b, result);}

/*
 *  bignum -> real
 */

double bigtoreal (b)
    BIGNUM *b;
{   int i;
    double r = 0;

    for (i = b->msd; i <= b->lsd; i++) {
	r = r * B + b->digits[i];}
    
    return b->sign ? -r : r;}

/*
 *  real -> bignum
 */

realtobig (x, dp)
    double x;
    struct descrip *dp;
{   double ax = x > 0 ? x : -x;
    if (ax < 1.0) {
	MkIntT ((word) 0, dp);
	return;}
    else {
	BIGNUM *b;
	int i;
	int blen = ln(ax) / ln(B) + 0.99;

	for (i = 0; i < blen; i++)
	    ax /= B;
	if (ax >= 1.0) {
	    ax /= B;
	    blen += 1;}

	b = alcbignum (blen);
	for (i = 0; i < blen; i++) {
	    word d = (ax *= B);
	    *DIG(b,i) = d;
	    ax -= d;}
	
	b->sign = x < 0;
	return retdesc (b, dp);}}

/*
 *  bignum -> string
 */

bigtos (dn, dp, sbuf)
    struct descrip *dn, *dp;
    char *sbuf;
{   int l = DLEN(dn);
    int slen = ceil (l * ln(B) / ln(10));
    if (DBIG(dn)->sign) slen++;
    if (strreq (slen) == Error)
	return Error;
    else {
	BIGNUM *n = DBIG(dn);
	char *p, *q = alcstr ("", slen);
	DIGIT *b = tempalloc (l, DIGIT);
	bcopy (DIG(n,0), b, l * sizeof (DIGIT));
	p = q += slen;
	while (cmpi1 (b, 0, l)) {
	    *--p = '0' + divi1 (b, 10, b, l);}
	if (n->sign) *--p = '-';
	StrLen (*dp) = q - p;
	StrLoc (*dp) = p;}
    return Cvt;}

/*
 *  bignum -> file 
 */

bigprint (f, n)
    FILE *f;
    BIGNUM *n;
{   int ld = LEN(n);
    DIGIT *d = tempalloc (ld, DIGIT);
    bcopy (DIG(n,0), d, ld * sizeof (DIGIT));
    if (n->sign) putc ('-', f);
    decout (f, d, ld);}

decout (f, n, l)
    FILE *f;
    DIGIT *n;
    int l;
{   word i = divi1 (n, 10, n, l);
    if (cmpi1 (n, 0, l)) decout (f, n, l);
    putc ('0' + i, f);}

/*
 *  da -> db
 */

cpbignum (da, db)
    struct descrip *da, *db;
{   int alen = DLEN (da);
    BIGNUM *b = alcbignum (alen);
    BIGNUM *a = DBIG (da);
    bcopy (DIG(a,0), DIG(b,0), alen * sizeof (DIGIT));
    b->sign = a->sign;
    return retdesc (b, db);}

/*
 *  da + db -> dx
 */

bigadd (da, db, dx)
    struct descrip *da, *db;
    struct descrip *dx;
{   BIGNUM *x, *a, *b;
    int alen = DLEN(da);
    int blen = DLEN(db);
    word c;

    if (DBIG(da)->sign == DBIG(db)->sign) {
	if (alen > blen) {
	    x = alcbignum (alen + 1);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = add1 (DIG(a,alen-blen), DIG(b,0), DIG(x,alen-blen+1), blen);
	    *DIG(x,0) = addi1 (DIG(a,0), c, DIG(x,1), alen-blen);}
	else if (alen == blen) {
	    x = alcbignum (alen + 1);
	    a = DBIG(da);
	    b = DBIG(db);
	    *DIG(x,0) = add1 (DIG(a,0), DIG(b,0), DIG(x,1), alen);}
	else {
	    x = alcbignum (blen + 1);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = add1 (DIG(b,blen-alen), DIG(a,0), DIG(x,blen-alen+1), alen);
	    *DIG(x,0) = addi1 (DIG(b,0), c, DIG(x,1), blen-alen);}
	x->sign = a->sign;}
    else {
	if (alen > blen) {
	    x = alcbignum (alen);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = sub1 (DIG(a,alen-blen), DIG(b,0), DIG(x,alen-blen), blen);
	    subi1 (DIG(a,0), -c, DIG(x,0), alen-blen);
	    x->sign = a->sign;}
	else if (alen == blen) {
	    x = alcbignum (alen);
	    a = DBIG(da);
	    b = DBIG(db);
	    if (cmp1 (DIG(a,0), DIG(b,0), alen) > 0) {
		sub1 (DIG(a,0), DIG(b,0), DIG(x,0), alen);
		x->sign = a->sign;}
	    else {
		sub1 (DIG(b,0), DIG(a,0), DIG(x,0), alen);
		x->sign = b->sign;}}
	else {
	    x = alcbignum (blen);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = sub1 (DIG(b,blen-alen), DIG(a,0), DIG(x,blen-alen), alen);
	    subi1 (DIG(b,0), -c, DIG(x,0), blen-alen);
	    x->sign = b->sign;}}

    return retdesc (x, dx);}

/*
 *  da - db -> dx
 */ 

bigsub (da, db, dx)
    struct descrip *da, *db, *dx;
{   BIGNUM *a, *b, *x;
    int alen = DLEN(da);
    int blen = DLEN(db);
    word c;

    if (DBIG(da)->sign != DBIG(db)->sign) {
	if (alen > blen) {
	    x = alcbignum (alen + 1);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = add1 (DIG(a,alen-blen), DIG(b,0), DIG(x,alen-blen+1), blen);
	    *DIG(x,0) = addi1 (DIG(a,0), c, DIG(x,1), alen-blen);}
	else if (alen == blen) {
	    x = alcbignum (alen + 1);
	    a = DBIG(da);
	    b = DBIG(db);
	    *DIG(x,0) = add1 (DIG(a,0), DIG(b,0), DIG(x,1), alen);}
	else {
	    x = alcbignum (blen + 1);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = add1 (DIG(b,blen-alen), DIG(a,0), DIG(x,blen-alen+1), alen);
	    *DIG(x,0) = addi1 (DIG(b,0), c, DIG(x,1), blen-alen);}
	x->sign = a->sign;}
    else {
	if (alen > blen) {
	    x = alcbignum (alen);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = sub1 (DIG(a,alen-blen), DIG(b,0), DIG(x,alen-blen), blen);
	    subi1 (DIG(a,0), -c, DIG(x,0), alen-blen);
	    x->sign = a->sign;}
	else if (alen == blen) {
	    x = alcbignum (alen);
	    a = DBIG(da);
	    b = DBIG(db);
	    if (cmp1 (DIG(a,0), DIG(b,0), alen) > 0) {
		sub1 (DIG(a,0), DIG(b,0), DIG(x,0), alen);
		x->sign = a->sign;}
	    else {
		sub1 (DIG(b,0), DIG(a,0), DIG(x,0), alen);
		x->sign = 1 ^ b->sign;}}
	else {
	    x = alcbignum (blen);
	    a = DBIG(da);
	    b = DBIG(db);
	    c = sub1 (DIG(b,blen-alen), DIG(a,0), DIG(x,blen-alen), alen);
	    subi1 (DIG(b,0), -c, DIG(x,0), blen-alen);
	    x->sign = 1 ^ b->sign;}}

    return retdesc (x, dx);}

/*
 *  da * db -> dx
 */

bigmul (da, db, dx)
    struct descrip *da, *db, *dx;
{   BIGNUM *a, *b, *x;
    int alen = DLEN(da);
    int blen = DLEN(db);

    x = alcbignum (alen + blen);
    a = DBIG(da);
    b = DBIG(db);
    mul1 (DIG(a,0), DIG(b,0), DIG(x,0), alen, blen);
    x->sign = a->sign ^ b->sign;
    return retdesc (x, dx);}

/*
 *  da / db -> dx
 */
 
bigdiv (da, db, dx)
    struct descrip *da, *db, *dx;
{   BIGNUM *a, *b, *x;
    int alen = DLEN(da);
    int blen = DLEN(db);

    if (alen < blen) {
	MkIntT (0, dx);
	return;}

    x = alcbignum (alen - blen + 1);
    a = DBIG(da);
    b = DBIG(db);
    if (blen == 1) {
	divi1 (DIG(a,0), (word) *DIG(b,0), DIG(x,0), alen);}
    else {
	div1 (DIG(a,0), DIG(b,0), DIG(x,0), NULL, alen-blen, blen);}
    x->sign = a->sign ^ b->sign;
    return retdesc (x, dx);}

/*
 *  da % db -> dx
 */

bigmod (da, db, dx)
    struct descrip *da, *db, *dx;
{   BIGNUM *a, *b, *x;
    int alen = DLEN(da);
    int blen = DLEN(db);

    if (alen < blen)
	return cpbignum (da, dx);

    x = alcbignum (blen);
    a = DBIG(da);
    b = DBIG(db);
    if (blen == 1) {
	DIGIT *junk = tempalloc (alen, DIGIT);
	*DIG(x,0) = divi1 (DIG(a,0), (word) *DIG(b,0), junk, alen);}
    else {
	div1 (DIG(a,0), DIG(b,0), NULL, DIG(x,0), alen-blen, blen);}
    x->sign = a->sign;
    return retdesc (x, dx);}

/*
 *  da + i -> dx
 */

bigaddi (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   BIGNUM *a, *x; 
    if (i < 0)
	return bigsubi (da, -i, dx);
    else if (i != (DIGIT) i)
	return bigadd (da, itobigr (i), dx);
    else if (DBIG(da)->sign) {
	int alen = DLEN(da);
	x = alcbignum (alen);
	a = DBIG(da);
	subi1 (DIG(a,0), i, DIG (x,0), alen);}
    else {
	int alen = DLEN(da);
	x = alcbignum (alen + 1);
	a = DBIG(da);
	*DIG(x,0) = addi1 (DIG(a,0), i, DIG (x,1), alen);}
    x->sign = a->sign;
    return retdesc (x, dx);}

/*
 *  da - i -> dx
 */

bigsubi (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   BIGNUM *a, *x; 
    if (i < 0)
	return bigaddi (da, -i, dx);
    else if (i != (DIGIT) i)
	return bigsub (da, itobigr (i), dx);
    else if (DBIG(da)->sign) {
	int alen = DLEN(da);
	x = alcbignum (alen + 1);
	a = DBIG(da);
	*DIG(x,0) = addi1 (DIG(a,0), i, DIG (x,1), alen);}
    else {
	int alen = DLEN(da);
	x = alcbignum (alen);
	a = DBIG(da);
	subi1 (DIG(a,0), i, DIG (x,0), alen);}
    x->sign = a->sign;
    return retdesc (x, dx);}

/*
 *  -i -> dx
 */

bignegi (i, dx)
    word i;
    struct descrip *dx;
{   struct descrip *di = itobigr (i);
    DBIG(di)->sign ^= 1;
    cpbignum (di, dx);}

/*
 *  da * i -> dx
 */

bigmuli (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   BIGNUM *a, *x; 
    if (i <= -B || i >= B)
	return bigmul (da, itobigr (i), dx);
    else {
	int alen = DLEN(da);
	x = alcbignum (alen + 1);
	a = DBIG (da);
	if (i >= 0) {
	    x->sign = a->sign;}
	else {
	    x->sign = 1 ^ a->sign;
	    i = -i;}
	*DIG(x,0) = muli1 (DIG(a,0), i, 0, DIG (x,1), alen);
	return retdesc (x, dx);}}

/*
 *  da / i -> dx
 */

bigdivi (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   BIGNUM *a, *x; 
    if (i <= -B || i >= B)
	return bigdiv (da, itobigr (i), dx);
    else {
	int alen = DLEN(da);
	x = alcbignum (alen);
	a = DBIG(da);
	if (i >= 0) {
	    x->sign = a->sign;}
	else {
	    x->sign = 1 ^ a->sign;
	    i = -i;}
	divi1 (DIG(a,0), i, DIG (x,0), alen);
	return retdesc (x, dx);}}

/*
 *  da % i -> dx
 */

bigmodi (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   BIGNUM *a = DBIG(da);
    if (i <= -B || i >= B)
	return bigmod (da, itobigr (i), dx);
    else {
	int alen = LEN(a);
	DIGIT *junk = tempalloc (alen, DIGIT);
	word x = divi1 (DIG(a,0), abs(i), junk, alen);
	if (a->sign) x = -x;
	MkIntT (x, dx);}}

/*
 *  da ^ db -> dx
 */

bigpow (da, db, dx) 
    struct descrip *da, *db, *dx;
{   if (DBIG(db)->sign) {
	MkIntT ((word) 0, dx);}
    else {
	int n = DLEN(db) * NB;
	/* scan bits left to right.  skip leading 1. */
	while (--n >= 0) {
	    if ((*DIG (DBIG(db), n / NB) & (1 << (n % NB))))
		break;}
	/* then, for each zero, square the partial result;
	   for each one, square it and multiply it by a */
	*dx = *da;
	while (--n >= 0) {
	    bigmul (dx, dx, dx);
	    if ((*DIG (DBIG(db), n / NB) & (1 << (n % NB)))) {
		bigmul (dx, da, dx);}}}}

/*
 *  da ^ i -> dx
 */

bigpowi (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   if (i > 0) {
	int n = bitsizeof (i);
	/* scan bits left to right.  skip leading 1. */
	while (--n >= 0)
	    if (i & (1 << n)) break;
	/* then, for each zero, square the partial result;
	   for each one, square it and multiply it by a */
	*dx = *da;
	while (--n >= 0) {
	    bigmul (dx, dx, dx);
	    if (i & (1 << n)) {
		bigmul (dx, da, dx);}}}
    else {
	MkIntT ((word) 0, dx);}}

/*
 *  a ^ i -> dx
 */

bigpowii (a, i, dx) 
    word a, i;
    struct descrip *dx;
{   if (a == 0 || i <= 0) {		/* special cases */
	if (a == 0 && i <= 0)		/* 0 ^ negative -> error */
	    RunErr (-204, NULL);
	if (a == -1) {			/* -1 ^ [odd,even] -> [-1,+1] */
	    if (!(i & 1)) a = 1;}
	else if (a != 1) {		/* 1 ^ any -> 1 */
	    a = 0;}			/* others ^ negative -> 0 */
	MkIntT (a, dx);}
    else {
	word x, y;
	int n = bitsizeof (i);
	int isbig = 0;
	/* scan bits left to right.  skip leading 1. */
	while (--n >= 0)
	    if (i & (1 << n)) break;
	/* then, for each zero, square the partial result;
	   for each one, square it and multiply it by a */
	x = a;
	while (--n >= 0) {
	    if (isbig)
		bigmul (dx, dx, dx);
	    else if (mul3 (x, x, &y))
		x = y;
	    else {
		bigmul (itobigl (x), itobigl (x), dx);
		isbig = Type (*dx) == T_Bignum;} 
	    if (i & (1 << n)) {
		if (isbig)
		    bigmuli (dx, a, dx);
		else if (mul3 (x, a, &y))
		    x = y;
		else {
		    bigmuli (itobigl (x), a, dx);
		    isbig = Type (*dx) == T_Bignum;}}}
	if (!isbig) MkIntT (x, dx);
	return;}}

/*
 *  iand (da, db) -> dx
 */

bigand (da, db, dx) 
    struct descrip *da, *db, *dx;
{   int i;
    int alen = DLEN (da);
    int blen = DLEN (db);
    int xlen = alen > blen ? alen : blen;
    BIGNUM *x = alcbignum (xlen);
    BIGNUM *a = DBIG (da);
    BIGNUM *b = DBIG (db);
    DIGIT *ad, *bd;

    if (alen == xlen && !a->sign) {
	ad = DIG(a,0);}
    else {
	ad = tempalloc (xlen, DIGIT);
	bzero (ad, (xlen - alen) * sizeof (DIGIT));
	bcopy (DIG(a,0), &ad[xlen-alen], alen * sizeof (DIGIT));
	if (a->sign) compl1 (ad, ad, xlen);}

    if (blen == xlen && !b->sign) {
	bd = DIG(b,0);}
    else {
	bd = tempalloc (xlen, DIGIT);
	bzero (bd, (xlen - blen) * sizeof (DIGIT));
	bcopy (DIG(b,0), &bd[xlen-blen], blen * sizeof (DIGIT));
	if (b->sign) compl1 (bd, bd, xlen);}
	
    for (i = 0; i < xlen; i++) 
	*DIG(x,i) = ad[i] & bd[i];

    if (a->sign & b->sign) {
	x->sign = 1;
	compl1 (DIG(x,0), DIG(x,0), xlen);}

    retdesc (x, dx);}

/*
 *  ior (da, db) -> dx
 */

bigor (da, db, dx) 
    struct descrip *da, *db, *dx;
{   int i;
    int alen = DLEN (da);
    int blen = DLEN (db);
    int xlen = alen > blen ? alen : blen;
    BIGNUM *x = alcbignum (xlen);
    BIGNUM *a = DBIG (da);
    BIGNUM *b = DBIG (db);
    DIGIT *ad, *bd;

    if (alen == xlen && !a->sign) {
	ad = DIG(a,0);}
    else {
	ad = tempalloc (xlen, DIGIT);
	bzero (ad, (xlen - alen) * sizeof (DIGIT));
	bcopy (DIG(a,0), &ad[xlen-alen], alen * sizeof (DIGIT));
	if (a->sign) compl1 (ad, ad, xlen);}

    if (blen == xlen && !b->sign) {
	bd = DIG(b,0);}
    else {
	bd = tempalloc (xlen, DIGIT);
	bzero (bd, (xlen - blen) * sizeof (DIGIT));
	bcopy (DIG(b,0), &bd[xlen-blen], blen * sizeof (DIGIT));
	if (b->sign) compl1 (bd, bd, xlen);}
	
    for (i = 0; i < xlen; i++) 
	*DIG(x,i) = ad[i] | bd[i];

    if (a->sign | b->sign) {
	x->sign = 1;
	compl1 (DIG(x,0), DIG(x,0), xlen);}

    retdesc (x, dx);}

/*
 *  xor (da, db) -> dx
 */

bigxor (da, db, dx) 
    struct descrip *da, *db, *dx;
{   int i;
    int alen = DLEN (da);
    int blen = DLEN (db);
    int xlen = alen > blen ? alen : blen;
    BIGNUM *x = alcbignum (xlen);
    BIGNUM *a = DBIG (da);
    BIGNUM *b = DBIG (db);
    DIGIT *ad, *bd;

    if (alen == xlen && !a->sign) {
	ad = DIG(a,0);}
    else {
	ad = tempalloc (xlen, DIGIT);
	bzero (ad, (xlen - alen) * sizeof (DIGIT));
	bcopy (DIG(a,0), &ad[xlen-alen], alen * sizeof (DIGIT));
	if (a->sign) compl1 (ad, ad, xlen);}

    if (blen == xlen && !b->sign) {
	bd = DIG(b,0);}
    else {
	bd = tempalloc (xlen, DIGIT);
	bzero (bd, (xlen - blen) * sizeof (DIGIT));
	bcopy (DIG(b,0), &bd[xlen-blen], blen * sizeof (DIGIT));
	if (b->sign) compl1 (bd, bd, xlen);}
	
    for (i = 0; i < xlen; i++) 
	*DIG(x,i) = ad[i] ^ bd[i];

    if (a->sign ^ b->sign) {
	x->sign = 1;
	compl1 (DIG(x,0), DIG(x,0), xlen);}

    retdesc (x, dx);}

/*
 *  ishift (da, i) -> dx
 */

bigshifti (da, i, dx)
    struct descrip *da, *dx;
    word i;
{   int alen = DLEN (da);
    word r = i % NB;
    word q = (r >= 0 ? i : (i - (r += NB))) / NB;
    int xlen = alen + q + 1;
    if (xlen <= 0) {
	MkIntT (-DBIG(da)->sign, dx);}
    else {
	BIGNUM *x = alcbignum (xlen);
	BIGNUM *a = DBIG (da);
	DIGIT *ad;

	if (!a->sign) {
	    ad = DIG(a,0);}
	else {
	    ad = tempalloc (alen, DIGIT);
	    bcopy (DIG(a,0), ad, alen * sizeof (DIGIT));
	    compl1 (ad, ad, alen);}

	if (q >= 0) {
	    *DIG(x,0) = shifti1 (ad, r, 0, DIG(x,1), alen);
	    bzero (DIG(x,alen+1), q * sizeof (DIGIT));}
	else 
	    *DIG(x,0) = shifti1 (ad, r, ad[alen+q]>>(NB-r), DIG(x,1), alen+q);

	if (a->sign) {
	    x->sign = 1;
	    *DIG(x,0) |= B - (1 << r);
	    compl1 (DIG(x,0), DIG(x,0), xlen);}

	retdesc (x, dx);}}

/*
 *  negative if da < db
 *  zero if da == db
 *  positive if da > db
 */

word bigcmp (da, db) 
    struct descrip *da, *db;
{   BIGNUM *a = DBIG(da);
    BIGNUM *b = DBIG(db);
    int alen, blen; 

    if (a->sign != b->sign)
	return b->sign - a->sign;

    alen = LEN(a);
    blen = LEN(b);
    if (alen != blen)
	return a->sign ? blen - alen : alen - blen;

    if (a->sign)
	return cmp1 (DIG(b,0), DIG(a,0), alen);
    else
	return cmp1 (DIG(a,0), DIG(b,0), alen);}

/*
 *  negative if da < i
 *  zero if da == i
 *  positive if da > i
 */  
  
word bigcmpi (da, i)
    struct descrip *da;
    word i;
{   if (i > -B && i < B) {
	BIGNUM *a = DBIG (da);
        int alen = LEN(a);
	if (i >= 0)
	    if (a->sign) return -1;
	    else return cmpi1 (DIG(a,0), i, alen);
	else
	    if (a->sign) return -cmpi1 (DIG(a,0), -i, alen);
	    else return 1;}
    else return bigcmp (da, itobigr (i));}

/*
 *  ?da -> dx
 */  

bigrand (da, dx)
    struct descrip *da, *dx;
{   int alen = DLEN(da);
    BIGNUM *x = alcbignum (alen);
    BIGNUM *a = DBIG(da);
    DIGIT *d = tempalloc (alen+1, DIGIT);
    int i;
    double rval;

    for (i = alen; i >= 0; i--) {
	rval = RandVal;
	d[i] = rval * B;}
    
    div1 (d, DIG(a,0), NULL, DIG(x,0), 1, alen);
    addi1 (DIG(x,0), 1, DIG(x,0), alen);
    return retdesc (x, dx);}

/* These are all straight out of Knuth vol. 2, Sec. 4.3.1. */

/*
 *  (a,m+n) / (b,n) -> (q,m+1) (r,n)
 *
 *  if q or r is NULL, the quotient or remainder is discarded
 */

static void div1 (a, b, q, r, m, n)
    DIGIT *a, *b, *q, *r;
    int m, n;
{   WORD qhat, rhat;
    WORD dig, carry;
    DIGIT *u = tempalloc (m+n+1, DIGIT);
    DIGIT *v = tempalloc (n, DIGIT);
    int d;
    int i, j;

    /* D1 */
    for (d = 0; d < NB; d++) {
	if (b[0] & (1 << (NB - 1 - d))) break;}

    u[0] = shifti1 (a, d, 0, &u[1], m+n);
    shifti1 (b, d, 0, v, n);

    /* D2, D7 */
    for (j = 0; j <= m; j++) {
	/* D3 */
	if (u[j] == v[0]) {
	    qhat = B - 1;
	    rhat = v[0] + u[j+1];}
	else {
	    WORD numerator = dbl (u[j], u[j+1]);
	    qhat = numerator / (WORD) v[0];
	    rhat = numerator % (WORD) v[0];}

	while (rhat < B && qhat * v[1] > dbl (rhat, u[j+2])) {
	    qhat -= 1;
	    rhat += v[0];}
	    
    	/* D4 */
	carry = 0;
	for (i = n; i > 0; i--) {
	    dig = u[i+j] - v[i-1] * qhat + carry; 	/* -BSQ+B .. B-1 */
	    u[i+j] = lo (dig);
	    if ((UWORD) dig < B)
		carry = hi (dig);
	    else carry = hi (dig) | -B;}
	carry = (SWORD) (carry + u[j]) < 0;

	/* D5 */
	if (q) q[j] = qhat;

	/* D6 */
	if (carry) {
	    if (q) q[j] -= 1;
	    carry = 0;
	    for (i = n; i > 0; i--) {
		dig = (WORD) u[i+j] + v[i-1] + carry;
		u[i+j] = lo (dig);
		carry = hi (dig);}}}

    if (r) {
	if (d == 0)
	    shifti1 (&u[m+1], d, 0, r, n);
	else
	    r[0] = shifti1 (&u[m+1], NB - d, u[m+n]>>d, &r[1], n - 1);}}

/*
 *  (u,n) + (v,n) -> (w,n) 
 *
 *  returns carry, 0 or 1
 */

static DIGIT add1 (u, v, w, n)
    DIGIT *u, *v, *w;
    int n;
{   WORD dig, carry; 
    int i;

    carry = 0;
    for (i = n; --i >= 0; ) {
	dig = (WORD) u[i] + v[i] + carry;
	w[i] = lo (dig);
	carry = hi (dig);}
    return carry;}

/*
 *  (u,n) - (v,n) -> (w,n)
 *
 *  returns carry, 0 or -1
 */

static word sub1 (u, v, w, n)
    DIGIT *u, *v, *w;
    int n;
{   WORD dig, carry; 
    int i;

    carry = 0;
    for (i = n; --i >= 0; ) {
	dig = (WORD) u[i] - v[i] + carry;
	w[i] = lo (dig);
	carry = signed_hi (dig);}
    return carry;}

/*
 *  (u,n) * k -> (w,n)
 *
 *  k in 0 .. B-1
 *  returns carry, 0 .. B-1
 */

static DIGIT muli1 (u, k, c, w, n)
    DIGIT *u, *w;
    word k, c;
    int n;
{   WORD dig, carry;
    int i;

    carry = c;
    for (i = n; --i >= 0; ) {
	dig = (WORD) k * u[i] + carry;
	w[i] = lo (dig);
	carry = hi (dig);}
    return carry;}

/*
 *  (u,n) / k -> (w,n)
 *
 *  k in 0 .. B-1
 *  returns remainder, 0 .. B-1
 */

static DIGIT divi1 (u, k, w, n)
    DIGIT *u, *w;
    word k;
    int n;
{   WORD dig, carry;
    int i;

    carry = 0;
    for (i = 0; i < n; i++) {
	dig = dbl (carry, u[i]);
	w[i] = dig / k;
	carry = dig % k;}
    return carry;}

/*
 *  ((u,n) << k) + c -> (w,n)
 *
 *  k in 0 .. NB-1
 *  c in 0 .. B-1 
 *  returns carry, 0 .. B-1
 */

static DIGIT shifti1 (u, k, c, w, n)
    DIGIT *u, c, *w;
    word k;
    int n;
{   WORD dig;
    int i;

    if (k == 0) {
	bcopy (u, w, n * sizeof *u);
	return 0;}
    
    for (i = n; --i >= 0; ) {
	dig = ((WORD) u[i] << k) + c;
	w[i] = lo (dig);
	c = hi (dig);}
    return c;}

/*
 *  (u,n) + k -> (w,n)
 *
 *  k in 0 .. B-1
 *  returns carry, 0 or 1
 */

static DIGIT addi1 (u, k, w, n)
    DIGIT *u, *w;
    word k;
    int n;
{   WORD dig, carry;
    int i;
    
    carry = k;
    for (i = n; --i >= 0; ) {
	dig = (WORD) u[i] + carry;
	w[i] = lo (dig);
	carry = hi (dig);}
    return carry;}

/*
 *  (u,n) - k -> (w,n)
 *
 *  k in 0 .. B-1
 *  u must be greater than k
 */

static void subi1 (u, k, w, n)
    DIGIT *u, *w;
    word k;
    int n;
{   WORD dig, carry;
    int i;
    
    carry = -k;
    for (i = n; --i >= 0; ) {
	dig = (WORD) u[i] + carry;
	w[i] = lo (dig);
	carry = signed_hi (dig);}}

/*
 *  - (u,n) -> (w,n)
 *
 *  returns carry, 0 or -1
 */

static word compl1 (u, w, n)
    DIGIT *u, *w;
    int n;
{   WORD dig, carry;
    int i;
    carry = 0;
    for (i = n; --i >= 0; ) {
	dig = carry - u[i];
	w[i] = lo (dig);
	carry = signed_hi (dig);}
    return carry;}

/*
 *  (u,n) : (v,n)
 */

static word cmp1 (u, v, n)
    DIGIT *u, *v;
    int n;
{   int i;
    for (i = 0; i < n; i++)
	if (u[i] != v[i]) return u[i] - v[i];
    return 0;}

/*
 *  (u,n) : k
 */

static int cmpi1 (u, k, n)
    DIGIT *u;
    word k;
    int n;
{   int i;
    for (i = 0; i < n-1; i++)
	if (u[i]) return 1;
    return u[n-1] - k;}

/*
 *  (u,n) * (v,m) -> (w,m+n)
 */

static void mul1 (u, v, w, n, m)
    DIGIT *u, *v, *w;
    int m, n;
{   int i, j;
    WORD dig, carry;

    bzero (&w[m], n * sizeof (DIGIT));

    for (j = m; --j >= 0; ) {
	carry = 0;
	for (i = n; --i >= 0; ) {
	    dig = (WORD) u[i] * v[j] + w[i+j+1] + carry;
	    w[i+j+1] = lo (dig);
	    carry = hi (dig);}
	w[j] = carry;}}


#if 0
/* these are slow */

/* don't propagate carries in inner loop; requires WORD > DIGIT**2 */

static xmul1 (u, v, w, m, n)		/* (u,m) * (v,n) -> (w,m+n) */
    DIGIT *u, *v, *w;
    int m, n;
{   int i, j;
    SWORD dig, carry;
    SWORD *t = tempalloc (m + n - 1, SWORD);

    bzero (t, (m + n - 1) * sizeof (SWORD));

    for (i = m; --i >= 0; )
	for (j = n; --j >= 0; )
	    t[i+j] += u[i] * v[j];

    carry = 0;
    for (i = m+n; --i > 0; ) {
	dig = t[i-1] + carry;
	w[i] = lo (dig);
	carry = hi (dig);}
    w[0] = carry;}

/* halve and recurse, O(n**lg 3).  
   Doesn't propagate carries, so requires WORD > DIGIT**2 */

static xxmul1 (u, v, w, m, n)		/* (u,m) * (v,n) -> (w,m+n) */
    DIGIT *u, *v, *w;
{   int i;
    SWORD dig, carry;
    int lw = m + n;
    SWORD *wu = tempalloc (m, SWORD);
    SWORD *wv = tempalloc (n, SWORD);
    SWORD *ww = tempalloc (m + n, SWORD);

    for (i = 0; i < m; i++) wu[i] = u[m-1-i];
    for (i = 0; i < n; i++) wv[i] = v[n-1-i];

    if (m >= n) mulstep (wu, wv, ww, m, n);
    else mulstep (wv, wu, ww, n, m);

    carry = 0;
    for (i = lw; --i > 0; ) {
	dig = ww[lw-1-i] + carry;
	w[i] = lo (dig);
	carry = signed_hi (dig);}
    w[0] = carry;}

static mulstep (u, v, w, n, m)		/* (u,n) * (v,m) -> (w,m+n) */
    SWORD *u, *v, *w;
    int m, n;
{   int i;

    if (m == 1) {
	for (i = 0; i < n; i++) w[i] = v[0] * u[i];
	w[n] = 0;
	return;}
    else {
	int n1 = n >> 1;	/* high half of u */
	int n2 = n - n1;	/* low half of u */
	int m1 = m - n2;	/* high piece of v  */

	if (m1 <= 0) {
	    int lh = n1 + m;	/* high half of w */
	    SWORD *ul = &u[0];
	    SWORD *uh = &u[n2];
	    SWORD *wl = &w[0];
	    SWORD *wh = tempalloc (lh, SWORD);

	    if (m <= n1) mulstep (uh, v, wh, n1, m);
	    else mulstep (v, uh, wh, m, n1);
	    if (m <= n2) mulstep (ul, v, wl, n2, m);
	    else mulstep (v, ul, wl, m, n2);

	    for (i = 0; i < m; i++) w[i+n2] += wh[i];
	    for (     ; i < lh; i++) w[i+n2] = wh[i];
	    return;}
	else {
	    int lh = n1 + m1;	/* high half of w */
	    int ll = n2 + n2;	/* low half of w */

	    SWORD *uh = &u[n2];
	    SWORD *ul = &u[0];
	    SWORD *vh = &v[n2];
	    SWORD *vl = &v[0];
	    SWORD *wl = &w[0];
	    SWORD *wh = &wl[ll];
	    SWORD *wm = tempalloc (ll, SWORD);
	    SWORD *t1 = tempalloc (n2, SWORD);
	    SWORD *t2 = tempalloc (n2, SWORD);

	    for (i = 0; i < n1; i++) t1[i] = uh[i] + ul[i];
	    for ( ;     i < n2; i++) t1[i] = ul[i];

	    for (i = 0; i < m1; i++) t2[i] = vh[i] + vl[i];
	    for (     ; i < n2; i++) t2[i] = vl[i];

	    mulstep (uh, vh, wh, n1, m1);
	    mulstep (ul, vl, wl, n2, n2);
	    mulstep (t1, t2, wm, n2, n2);

	    for (i = 0; i < lh; i++) wm[i] -= wh[i] + wl[i];
	    for (     ; i < ll; i++) wm[i] -= wl[i];
	    for (i = 0; i < ll; i++) w[i+n2] += wm[i];
	    return;}}}
#endif
