/* code.c: produce code for the function encoded by the parse tree. */

#include <stdio.h>
#include <strings.h>
#include "fpc.h"
#include "parse.h"
#include "code.h"
#include "fp.h"

static fpexpr preoptimize ();
static void putheader ();
static void putfinish ();

extern void codeexpr ();
extern char * sprintf ();

static int varsneeded;
static int selneeded;

/* assumes that oldname ends in .fp. Returns "" if for some reason
   the file should not be opened. */
void newfname (oldname, newname)
char * oldname, * newname;
{
  int len;

  len = strlen (oldname);
  if ((oldname [len - 3] != '.') ||
      (oldname [len - 2] != 'f') ||
      (oldname [len - 1] != 'p'))
  {
    *newname = '\0';
    return;
  }
  (void) strcpy (newname, oldname);
  newname [len - 2] = 'c';	/* change .fp to .c */
  newname [len - 1] = '\0';
}

void code (fun, tree)
char * fun;
fpexpr tree;
{
  tree = preoptimize (tree);
  countvars (tree);
  putheader (fun, varsneeded, selneeded, tree);
  codeexpr (tree, "data", "res");
  putfinish (fun);
}

static void putdefine (name, val)
char * name, *val;
{
  (void) fprintf (outf, "#define %s\t%s\n", name, val);
}

static void putdefnum (name, val)
char * name;
int val;
{
  (void) fprintf (outf, "#define %s\t%d\n", name, val);
}

static void putmain ()
{
  char inproc [MAXIDLEN], outproc [MAXIDLEN];

/* implementation should be refined, for now we don't do -c */
  if (check || (makeast && rstring) || traceptr)
    (void) fprintf (outf, "#include <stdio.h>\n");
  if (makemain && makeast && rstring)
    (void) fprintf (outf, "#include <sgtty.h>\n\n");
  else
    (void) fprintf (outf, "\n");
  if (makemain)
  {
    (void) strcpy (inproc, (rstring ? "getfpstring" : "getfpdata"));
    (void) strcpy (outproc, (wstring ? "putfpstrings" : "putfpdata"));
    if (makeast)
      (void) strcpy (inproc, (rstring ? "getfpchar" : "getfpdata"));
    if (redirout)
      (void) strcpy (outproc, "putcommands");
    (void) fprintf (outf, "main (argc, argv)\nint argc;\nchar * argv [];\n{\n");
    (void) fprintf (outf, "  extern fp_data %s (), %s ();\n", inproc, mainfn);
    (void) fprintf (outf, "  extern int fpargc;\n  extern char ** fpargv;\n");
    if (check)
      if (printspace)
        (void) fprintf (outf, "  extern void printstorage ();\n");
      else
        (void) fprintf (outf, "  extern void checkstorage ();\n");
    if (makeast)
    {
      (void) fprintf (outf, "  extern struct fp_object nilobj;\n");
      (void) fprintf (outf, "  fp_data state;\n");
      (void) fprintf (outf, "  static struct fp_constant initstate = ");
      (void) fprintf (outf, "{(short) NILOBJ, (short) 2};\n");
      if (rstring)
      {
        (void) fprintf (outf, "  struct sgttyb newtty, oldtty;\n");
        (void) fprintf (outf, "  struct sgttyb * savetty;\n");
      }
    }
    (void) fprintf (outf, "  extern void %s ();\n  fp_data input, result;\n\n",
	            outproc);
    if (makeee || makedeb)
      (void) fprintf (outf,
		      "  (void) fprintf (stderr, \"entering main\\n\");\n");
    (void) fprintf (outf, "  fpargc = argc;\n  fpargv = argv;\n");
    if (makeast)	/* produce an applicative state transition system */
    {
      if (rstring)
      {
        (void) fprintf (outf, "  savetty = &oldtty;\n");
        (void) fprintf (outf, "  ioctl (0, TIOCGETP, &oldtty);\n");
        (void) fprintf (outf, "  ioctl (0, TIOCGETP, &newtty);\n");
        (void) fprintf (outf, "  newtty.sg_flags |= CBREAK;\n");
        (void) fprintf (outf, "  ioctl (0, TIOCSETP, &newtty);\n");
      }
      (void) fprintf (outf, "  state = (fp_data) & initstate;\n");
      (void) fprintf (outf, "  input = newpair ();\n");
      (void) fprintf (outf, "  input->fp_header.fp_next->fp_entry =");
      (void) fprintf (outf, " (fp_data) & nilobj;\n");
      (void) fprintf (outf, "  input->fp_entry = & nilobj;\n");
      (void) fprintf (outf, "  while (1)\n  {\n");
      (void) fprintf (outf, "    result = %s (input);\n", mainfn);
      if (check)
      {
	(void) fprintf (outf, "    if ((result->fp_type != VECTOR) ||\n");
	(void) fprintf (outf, "        (result->fp_header.fp_next == 0) ||\n");
	(void) fprintf (outf, "        (result->%s != 0))\n",
		 "fp_header.fp_next->fp_header.fp_next");
	(void) fprintf (outf,
		 "      genbottom (\"non-pair returned in AST\", result);\n");
      }
      (void) fprintf (outf,
		      "    state = result->fp_header.fp_next->fp_entry;\n");
      (void) fprintf (outf, "    %s (result->fp_entry);\n", outproc);
      (void) fprintf (outf, "    if (state->fp_type == NILOBJ)\n");
      (void) fprintf (outf, "      break;\n");
      (void) fprintf (outf, "    inc_ref (state);\n");
      (void) fprintf (outf, "    dec_ref (result);\n");
      (void) fprintf (outf, "    input = newpair ();\n");
      (void) fprintf (outf,
		      "    input->fp_header.fp_next->fp_entry = state;\n");
      (void) fprintf (outf, "    input->fp_entry = %s ();\n", inproc);
      (void) fprintf (outf, "  }\n  dec_ref (result);\n");
      if (rstring)
        (void) fprintf (outf, "  ioctl (0, TIOCSETP, &oldtty);\n");
    }
    else	/* normal, non-ast system */
    {
      if (useparms)
      {
	(void) fprintf (outf, "  if (fpargc != 1)\n");
	(void) fprintf (outf, "    input = & nilobj;\n");
	(void) fprintf (outf, "  else\n  ");
      }
      (void) fprintf (outf, "  input = %s ();\n", inproc);
      (void) fprintf (outf, "  result = %s (input);\n", mainfn);
      (void) fprintf (outf, "  %s (result);\n", outproc);
      (void) fprintf (outf, "  dec_ref (result);\n");
    }
    if (makeee || makedeb)
      (void) fprintf (outf,
		      "  (void) fprintf (stderr, \"exiting main\\n\");\n");
    if (check)
      if (printspace)
        (void) fprintf (outf, "  printstorage ();\n");
      else
        (void) fprintf (outf, "  checkstorage ();\n");
    (void) fprintf (outf, "  return (0);\n}\n\n");
  }
}

void putfileheader (in, out)
char * in;
char * out;
{
  (void) fprintf (outf, "/* %s: target file generated by fpc from source %s */\n\n",
	   out, in);
  putdefnum ("FALSEOBJ  ", FALSEOBJ);
  putdefnum ("TRUEOBJ   ", TRUEOBJ);
  putdefnum ("INTCONST  ", INTCONST);
  putdefnum ("FLOATCONST", FLOATCONST);
  putdefnum ("ATOMCONST ", ATOMCONST);
  putdefnum ("CHARCONST ", CHARCONST);
  putdefnum ("NILOBJ    ", NILOBJ);
  putdefnum ("VECTOR    ", VECTOR);
  (void) fprintf (outf, "\ntypedef struct fp_object * fp_data;\n\n");
  (void) fprintf (outf,
		  "struct fp_object\n{\n  short fp_type;\n  short fp_ref;\n");
  (void) fprintf (outf, "  union\n  {\n    long fp_int;\n    int fp_char;\n");
  (void) fprintf (outf, "    char * fp_atom;\n    float fp_float;\n");
  (void) fprintf (outf, "    fp_data fp_next;\n  } fp_header;\n");
  (void) fprintf (outf, "  fp_data fp_entry;\n};\n\n");
  (void) fprintf (outf, "struct fp_constant\n{\n  short fp_type;\n");
  (void) fprintf (outf, "  short fp_ref;\n  %s fp_value;\n", HEADERTYPE);
  (void) fprintf (outf, "  fp_data fp_entry;\n};\n\n");
  (void) fprintf (outf, "struct fp_floatc\n{\n  short fp_type;\n");
  (void) fprintf (outf, "  short fp_ref;\n  %s fp_value;\n};\n\n", HEADERFLOAT);
  (void) fprintf (outf, "struct fp_charc\n{\n  short fp_type;\n");
  (void) fprintf (outf, "  short fp_ref;\n  %s fp_value;\n};\n\n", HEADERCHAR);
  if (check)
  {
    (void) fprintf (outf, "struct stackframe\n{\n  char * st_name;\n");
    (void) fprintf (outf, "  fp_data st_data;\n");
    (void) fprintf (outf, "  struct stackframe * st_prev;\n};\n");
    (void) fprintf (outf, "extern struct stackframe * stack;\n\n");
  }
  (void) fprintf (outf, "extern fp_data newvect ();\n");
  (void) fprintf (outf, "extern fp_data newpair ();\n");
  (void) fprintf (outf, "extern fp_data newcell ();\n");
  (void) fprintf (outf, "extern fp_data newconst ();\n");
  (void) fprintf (outf, "extern void returnvect ();\n");
  (void) fprintf (outf, "extern struct fp_object nilobj;\n");
  (void) fprintf (outf, "extern struct fp_object tobj;\n");
  (void) fprintf (outf, "extern struct fp_object fobj;\n\n");
  if (makedeb || makeee || traceptr)
    (void) fprintf (outf, "extern int depthcount;\nextern int indent ();\n\n");
  if (makedeb || traceptr)
    (void) fprintf (outf, "extern void printfpdata ();\n\n");
  if (check)
    (void) fprintf (outf, "extern void genbottom ();\n\n");
  putdefine ("inc_ref(d)", "((d)->fp_ref++)");
  putdefine ("dec_ref(d)",
"if (((d)->fp_type == VECTOR) && \\\n\t\t\t\t(--((d)->fp_ref) <= 0)) returnvect (d)");
  putdefine ("abs(n)", "((n) < 0 ? - (n) : (n))");
  (void) fprintf (outf, "\n");
  putmain ();
}

void putfiletail ()
{
  (void) fprintf (outf, "\n");
}

static void traverse (tree, fn, pre)
/* traverses the tree, calling fn on each and every node */
fpexpr tree;
void ((* fn) ());
int pre;
{
  fpexpr save = tree;

  if (pre)
    (* fn) (tree);
  switch (tree->exprtype)
  {
    case COND:
      traverse (tree->fpexprv.conditional [0], (* fn), pre);
      traverse (tree->fpexprv.conditional [1], (* fn), pre);
      traverse (tree->fpexprv.conditional [2], (* fn), pre);
      break;
    case BU:
    case BUR:
      traverse (tree->fpexprv.bulr.bufun, (* fn), pre);
      traverse (tree->fpexprv.bulr.buobj, (* fn), pre);
      break;
    case WHILE:
      traverse (tree->fpexprv.whilestat [0], (* fn), pre);
      traverse (tree->fpexprv.whilestat [1], (* fn), pre);
      break;
    case COMP:
    case CONSTR:
      while (tree != 0)
      {
        traverse (tree->fpexprv.compconstr.compexpr, (* fn), pre);
	tree = tree->fpexprv.compconstr.compnext;
      }
      break;
    case AA:
    case INSERT:
    case RINSERT:
    case TREE:
    case MULTI:
      traverse (tree->fpexprv.aains, (* fn), pre);
      break;
    case LIST:
      while (tree != 0)
      {
        traverse (tree->fpexprv.listobj.listel, (* fn), pre);
	tree = tree->fpexprv.listobj.listnext;
      }
      break;
    case SEL:
    case RSEL:
    case FNCALL:
    case NIL:
    case TRUE:
    case FALSE:
    case INT:
    case FLOAT:
    case SYM:
    case CHAR:
      break;
    default:
      yyerror ("compiler error 11");
  }
  if (! pre)
   (* fn) (save);
}

static void opt (tree)
fpexpr tree;
{
  if (((tree->exprtype == INSERT) ||
       (tree->exprtype == RINSERT) ||
       (tree->exprtype == TREE)) &&
      (tree->fpexprv.aains->exprtype == FNCALL) &&
      ((strcmp (tree->fpexprv.aains->fpexprv.funcall, "plus") == 0) ||
       (strcmp (tree->fpexprv.aains->fpexprv.funcall, "times") == 0) ||
       (strcmp (tree->fpexprv.aains->fpexprv.funcall, "and") == 0) ||
       (strcmp (tree->fpexprv.aains->fpexprv.funcall, "or") == 0)))
/* means we can replace the call to insert by a call to MULTI */
    tree->exprtype = MULTI;
/* wasn't that easy, now? */
}

static fpexpr preoptimize (tree)
fpexpr tree;
{	/* as long as it doesn't change the meaning of the program,
	 * everything is fair game here */
/* the only optimization we do here is change (insert <f>), where <f>
 * is one of {plus, times, and, or} to (multi <f>)
 */
  traverse (tree, opt, 0);
  return (tree);
}

static int nodevars (tree)
fpexpr tree;
{
  char errbuf [256];

  switch (tree->exprtype)
  {
    case COND:
/* a -> b ; c : res := a; if (res) then res := b; else res := c; end */
    case FNCALL:
/* f: res := f (arg); */
    case SEL:
/* n: i1 := n; res := arg; while (--i1 > 0) res := cdr (res);
      res := car (res); */
    case RSEL:
/* n: i1 := 0; res := arg; while (res != 0) res := cdr (res); i1++;
      i1 := i1 - n; res := arg; while (--i1 != 0) res := cdr (res);
      res := car (res); */
    case NIL:
    case TRUE:
    case FALSE:
    case INT:
    case FLOAT:
    case SYM:
    case CHAR:
    case LIST:	/* called for each list element */
      return (0);

    case COMP:
/* a o b o c o d : r1 := d (arg); r2 := c (r1); r1 := b (r2); res := a (r1); */
      if ((tree->fpexprv.compconstr.compnext != 0) &&  /* should never happen */
(tree->fpexprv.compconstr.compnext->fpexprv.compconstr.compnext != 0))
        return (2);
    case CONSTR:
/* [a, b] : res := new (2); chase := res; chase->car := b (arg);
            chase = cdr (chase); chase->car := a (arg); */
    case BU:
/* bu  op v : res := v; r1 := newvect (res, arg); res := op (r1); */
    case BUR:
/* bur op v : res := v; r1 := newvect (arg, res); res := op (r1); */
    case MULTI:
/* \/f: r1 := arg; res := car (r1);
	while (r1 != 0) res := op (res, car (r1)); r1 := cdr (r1); */
      return (1);

    case RINSERT:
/* \a : res := car (arg); r1 := cdr (arg);
        while (r1 != 0) r2 := cons (res, cons (car (r1), nil));
	  res := a (r2); r1 := cdr (r1); */
    case AA:
/* aa e : if (arg == <>) then res := arg;
   else r1 := arg; res := newvect (1); r2 := res;
     while (r1 != 0) r2->el := e (car r1); r1 := cdr (r1);
       if (r1 != 0) r2->next = newvect (1); r2 = cdr (r2); */
    case WHILE:
/* while pred f : res := arg;
   while (1)
      r1 := pred (res); if (! r1) then break; arg := f (res); res := arg; */
      return (2);

    case INSERT:
/* /a : r1 := 0; r2 := arg;
	while (r2 != 0) r3 := cons (car (r2), r1); r1 := r3; r2 := cdr (r2);
        res := car (r1); r1 := cdr (r1);
        while (r1 != 0) r2 := cons (car (r1), cons (res, nil)); res := a (r2);
	  r1 := cdr (r1); */
      return (3);

    case TREE:
/* \/a: r1 := arg;
	while (cdr (r1) != 0)
	  r2 := r1; r1 := newcell (); r3 := r1;
	  while (r2 != 0)
	    if (cdr (r2) == 0) rplaca (r3, car (r2)); r2 := 0;
	    else
	      r4 := cons (car (r2), cons (cadr (r2), nil)); r2 := cddr (r2);
	      rplaca (r3, a(r4));
	      if (r2 != 0) rplacd (r3, newcell ()); r3 := cdr (r3);
	res := car (r1); */
      return (5);	/* one more needed for storage management */

    default:
      (void) sprintf (errbuf, "compiler error 12, type is %d", tree->exprtype);
      yyerror (errbuf);
      return (-1);
  }
}

static void countvar (tree)
fpexpr tree;
{
  varsneeded += nodevars (tree);
  selneeded = selneeded ||
	      (((tree->exprtype == SEL) || (tree->exprtype == RSEL)) &&
	       (tree->fpexprv.lrsel > 1));
}

static countvars (tree)
fpexpr tree;
{
  varsneeded = 0;
  selneeded = 0;
  traverse (tree, countvar, 1);
}

static int constcount;

static void declconst (tree)
fpexpr tree;
/* traverse procedure called in post-order traversal. It generates a
 * new "constant variable" for the constant and stores it in the tree.
 * It also generates a declaration for the constant itself, using
 * the "constant variables" of the elements in case of lists.
 * A constant declaration is of the form.
 * static fp_data cnn = {type, 1, val, entry}
 */
{
  static char def1 [] = "  static struct fp_constant ";
  static char def2 [] = " =\n                {(short) ";
  static char def3 [] = ", (short) 1";
  fpexpr next;

  if (tree->exprtype >= NIL)
  {
    (void) sprintf (tree->constvar, "c%d", constcount++);
/* we always use a new constant "variable" for a new constant
 * encountered. That may be updated later to allow sharing of
 * equal constants, as in equal nil/true/false and (less often)
 * numbers, strings or lists. Not a high priority item, on V.M.
 * systems */
    switch (tree->exprtype)
    {
      case FALSE:
	(void) fprintf (outf, "%s%s%s%s%s};\n", def1, tree->constvar,
		 	def2, "FALSEOBJ", def3);
	break;
      case TRUE:
	(void) fprintf (outf, "%s%s%s%s%s};\n", def1, tree->constvar,
		 	def2, "TRUEOBJ", def3);
	break;
      case NIL:
	(void) fprintf (outf, "%s%s%s%s%s};\n", def1, tree->constvar,
		 	def2, "NILOBJ", def3);
	break;
      case INT:
	(void) fprintf (outf, "%s%s%s%s%s, (%s) %d};\n", def1, tree->constvar,
		 	def2, "INTCONST", def3, HEADERTYPE,
			tree->fpexprv.intobj);
	break;
      case FLOAT:
	(void) fprintf (outf, "%s%s%s%s%s, %lf};\n",
			"  static struct fp_floatc ", tree->constvar,
		 	def2, "FLOATCONST", def3, tree->fpexprv.floatobj);
	break;
      case SYM:
	(void) fprintf (outf, "%s%s%s%s%s, (%s) \"%s\"};\n", def1,
			tree->constvar, def2, "ATOMCONST", def3,
			HEADERTYPE, tree->fpexprv.symbol);
	break;
      case CHAR:
	(void) fprintf (outf, "%s%s%s%s%s, '\\%o'};\n",
			"  static struct fp_charc ", tree->constvar,
			def2, "CHARCONST", def3, tree->fpexprv.character);
	break;
      case LIST:
	next = tree->fpexprv.listobj.listnext;
	if (next != 0)
	  declconst (next);
	(void) fprintf (outf, "%s%s%s%s%s, (%s) %c%s, (fp_data) &%s};\n", def1,
		 	tree->constvar, def2, "VECTOR", def3, HEADERTYPE,
		 	((next == 0) ? '0' : '&'),
		 	((next == 0) ? "" : next->constvar),
		 	tree->fpexprv.listobj.listel->constvar);
	break;
      default:	/* error */
        yyerror ("compiler error 13");
    }
  }	/* else it is not a constant, ignore it */
}

static char externs [MAXIDS] [MAXIDLEN];
static int extptr;

static void putoneextern (tree)
fpexpr tree;
{
  int search = 0;
  char buf [MAXIDLEN];

  if (tree->exprtype == FNCALL)
  {
    if (strcmp (tree->fpexprv.funcall, "times") == 0)
      (void) strcpy (buf, "fptimes");
    else
      (void) strcpy (buf, tree->fpexprv.funcall);
    while ((search < extptr) &&
	   (strcmp (buf, externs [search]) != 0))
      search++;
    if (search == extptr)	/* must insert new name */
      (void) strcpy (externs [extptr++], buf);
  }
}

static void putexterns (tree, fun)
fpexpr tree;
char * fun;
{
  (void) strcpy (externs [0], fun);
  extptr = 1;
  traverse (tree, putoneextern, 1);
  if (extptr > 1)
  {
    (void) fprintf (outf, "  extern fp_data");
    while (--extptr > 0)
    {
      (void) fprintf (outf, " %s ()%s", externs [extptr],
	       (extptr == 1) ? ";\n" : ",");
      if (((extptr - 1) & DCLEMASK) == DCLEMASK)
        (void) fprintf (outf, "\n\t\t");
    }
  }
}

static int freevar;

static void declvars (vars, hassel)
int vars, hassel;
{
  freevar = 0;
  if (hassel)
    (void) fprintf (outf, "  register int sel;\n");
  (void) fprintf (outf, "  fp_data");
  while (vars-- > 0)
  {
    (void) fprintf (outf, " d%d,", vars);
    if ((vars & DCLMASK) == DCLMASK)
      (void) fprintf (outf, "\n\t ");
  }
  (void) fprintf (outf, " res;\n");
  if (check)
    (void) fprintf (outf, "  struct stackframe stackentry;\n");
  (void) fprintf (outf, "\n");
}

void newvar (buf)
char * buf;
{
  (void) sprintf (buf, "d%d", freevar++);
}

static int tracingfn;

static void entertrace (fname)
char * fname;
{
  if (makeee || makedeb || tracingfn)
  {
    (void) fprintf (outf,
		    "  depthcount += 2;\n  indent (depthcount, stderr);\n");
    if (makedeb || tracingfn)
    {
      (void) fprintf (outf, "  (void) fprintf (stderr, \"entering %s, data is\\n\");\n",
	       fname);
      (void) fprintf (outf, "  printfpdata (stderr, data, depthcount);\n");
      (void) fprintf (outf, "  (void) fprintf (stderr, \"\\n\");\n");
    }
    else
      (void) fprintf (outf, "  (void) fprintf (stderr, \"entering %s\\n\");\n", fname);
  }
  if (check)		/* keep the stack */
  {
    (void) fprintf (outf, "  stackentry.st_prev = stack;\n");
    (void) fprintf (outf, "  stackentry.st_data = data;\n  inc_ref (data);\n");
    (void) fprintf (outf, "  stackentry.st_name = \"%s\";\n", fname);
    (void) fprintf (outf, "  stack = & stackentry;\n", fname);
  }
}

static void putheader (fname, vars, hassel, tree)
char * fname;
int vars, hassel;
fpexpr tree;
{
  int trace;

  for (trace = 0;
       (trace < traceptr) && (strcmp (tracefns [trace], fname) != 0);
       trace++)
    ;
  tracingfn = (trace < traceptr);	/* are we tracing this function? */
  (void) fprintf (outf, "fp_data %s (data)\nfp_data data;\n{\n", fname);
  putexterns (tree, fname);
  constcount = 0;
  traverse (tree, declconst, 0);	/* declare the static constants */
  declvars (vars, hassel);
  entertrace (fname);
}

static void putfinish (fname)
char * fname;
{
  if (makeee || makedeb || tracingfn)
  {
    (void) fprintf (outf,
		    "  indent (depthcount, stderr);\n  depthcount -= 2;\n");
    if (makedeb || tracingfn)
    {
      (void) fprintf (outf, "  (void) fprintf (stderr, \"exiting %s, result is\\n\");\n",
	       fname);
      (void) fprintf (outf, "  printfpdata (stderr, res, depthcount);\n");
      (void) fprintf (outf, "  (void) fprintf (stderr, \"\\n\");\n");
    }
    else
      (void) fprintf (outf, "  (void) fprintf (stderr, \"exiting %s\\n\");\n", fname);
  }
  if (check)		/* restore the stack */
  {
    (void) fprintf (outf, "  dec_ref (data);\n");
    (void) fprintf (outf, "  stack = stackentry.st_prev;\n");
  }
  (void) fprintf (outf, "  return (res);\n}\n\n");
  tracingfn = 0;
}
