/* parse.c: builds the parse tree for each function, then calls code (tree)
 *	    to produce the code for it.
 */

#include <stdio.h>
#include <strings.h>
#include "fpc.h"
#include "parse.h"

extern char * malloc ();
extern char * sprintf ();

void code ();
void startcomp ();
void endcomp ();

char funname [MAXIDLEN] = "no function declaration seen yet";
int inerror = 0;
static struct fpexprd basefun = { INVALID };
static fpexpr stack [STACKDEPTH] = { & basefun };
static int stackptr = 0;

/* the context holds, for level cntptr, whether a compose was
 * encountered on that level or not. */
static int context [STACKDEPTH];
static int cntptr = 0;

static fpexpr newexpr (type)
int type;
{
  register fpexpr new;

  new = (fpexpr) malloc (sizeof (struct fpexprd));
  new->exprtype = type;
  return (new);
}

void parsefnstart (fname)
char * fname;
{
  extern int verbose;	/* set in fpc.c */

#ifdef TRACE
  (void) printf ("function name is %s\n", fname);
#endif
  (void) strcpy (funname, fname);
  inerror = 0;
  stackptr = 0;
  cntptr = 0;	/* root context */
  if (verbose)
    (void) printf ("%s\n", funname);
  startcomp ();
}

void parsefnend ()
{
#ifdef TRACE
  (void) printf ("function body is finished\n");
#endif
  endcomp ();
  if (inerror)
    inerror = 0;
  else
  {
    if (stackptr != 1)
      (void) fprintf (stderr, "stackptr is %d at end\n", stackptr);
    if (cntptr != 0)
      (void) fprintf (stderr, "context pointer is %d at end\n", cntptr);
    if (stackptr != 1)
      puterror ("compiler error 1", "");
    if (cntptr != 0)
      puterror ("compiler error 5", "");
    code (funname, stack [0]);
  }
}

void parsethen ()
{ /* pop the composition being parsed off the stack and make it
   * the first (if) part of the conditional */
  fpexpr ifpart, current;

#ifdef TRACE
  (void) printf (" -> ");
#endif
  endcomp ();
  ifpart = stack [--stackptr];
  stack [stackptr++] = current = newexpr (COND);
  current->fpexprv.conditional [0] = ifpart;
  startcomp ();
}

void parseelse ()
{
/* top of stack is the then part, put it into the structure */
  fpexpr current;

#ifdef TRACE
  (void) printf (" ; ");
#endif
  endcomp ();
  current = stack [stackptr - 2];
  if (current->exprtype != COND)
    yyerror ("compiler error 2");
  current->fpexprv.conditional [1] = stack [--stackptr];
  startcomp ();
}

void parseendif ()
{
/* top of stack is the else part, put it into the structure */
  fpexpr current;

#ifdef TRACE
  (void) printf (" endif\n");
#endif
  endcomp ();
  if (stackptr < 2)
    yyerror ("compiler error 4");
  else
  {
    current = stack [stackptr - 2];
    if (current->exprtype != COND)
      yyerror ("compiler error 3");
    current->fpexprv.conditional [2] = stack [--stackptr];
  }
  startcomp ();		/* empty, only to keep the stack balanced */
}

void parsebustart (right)
int right;
{
#ifdef TRACE
  (void) printf ("starting bu%s\n", (right) ? "r" : "");
#endif
  stack [stackptr++] = newexpr (right ? BUR : BU);
  startcomp ();
}

void parsebufun ()
{
  fpexpr fun;

#ifdef TRACE
  (void) printf ("done with the bu or bur expression\n");
#endif
  endcomp ();
  fun = stack [--stackptr];
  stack [stackptr - 1]->fpexprv.bulr.bufun = fun;
}

void parsebuobj ()
{
  fpexpr obj;

#ifdef TRACE
  (void) printf ("done with the bu or bur object\n");
#endif
  obj = stack [--stackptr];
  stack [stackptr - 1]->fpexprv.bulr.buobj = obj;
}

void whilestart ()
{
#ifdef TRACE
  (void) printf ("starting the while\n");
#endif
  stack [stackptr++] = newexpr (WHILE);
  startcomp ();
}

void whilepred ()
{
  fpexpr pred;

#ifdef TRACE
  (void) printf ("while predicate done\n");
#endif
  endcomp ();
  pred = stack [--stackptr];
  stack [stackptr - 1]->fpexprv.whilestat [0] = pred;
  startcomp ();
}

void whilefun ()
{
  fpexpr loop;

#ifdef TRACE
  (void) printf ("while function done\n");
#endif
  endcomp ();
  loop = stack [--stackptr];
  stack [stackptr - 1]->fpexprv.whilestat [1] = loop;
}

void parsecomp ()
{
  fpexpr fun, next;

#ifdef TRACE
  (void) printf ("composing with next function\n");
#endif
  fun = stack [--stackptr];
  next = newexpr (COMP);
  next->fpexprv.compconstr.compexpr = fun;
  if (context [cntptr - 1])	/* node already allocated on this level */
  {	/* join the new one in front of the old list, which is now present */
#ifdef TRACE
    (void) printf ("adding to old compose list, stackptr is %d\n", stackptr);
#endif
    if (stackptr < 1)
      yyerror ("compiler error 9");
    else
    {
      next->fpexprv.compconstr.compnext = stack [stackptr - 1];
      stack [stackptr - 1] = next;
    }
  }
  else				/* create new list */
  {
#ifdef TRACE
    (void) printf ("creating new compose list, stackptr is %d\n", stackptr);
#endif
    next->fpexprv.compconstr.compnext = 0;
    stack [stackptr++] = next;
  }
  context [cntptr - 1] = 1;	/* yes, we have a compose on this level */
}

void startcomp ()
{
#ifdef TRACE
  (void) printf ("setting up possible composition\n");
#endif
  context [cntptr++] = 0;
}

void endcomp ()
{
#ifdef TRACE
  (void) printf ("closing up %s composition\n",
		 context [cntptr - 1] ? "the" : "no");
#endif
  if (cntptr <= 0)		/* endcomp does not match comp */
    yyerror ("compiler error 6");
  else if ((context [cntptr - 1]) == 0) /* composition not done */
    cntptr--;
  else if (stackptr <= 1)	/* composition done, but no composition? */
    yyerror ("compiler error 7");
  else if (stack [stackptr - 2]->exprtype != COMP)	/* same problem */
    yyerror ("compiler error 8");
  else
  {  /* we put the stack top as the last expression to be composed with */
    parsecomp ();
    cntptr--;
  }
}

void parseaa ()
{
  fpexpr exp, aa;

#ifdef TRACE
  (void) printf ("apply-to-all encountered\n");
#endif
  exp = stack [--stackptr];
  stack [stackptr++] = aa = newexpr (AA);
  aa->fpexprv.aains = exp;
}

void parseconstr ()
{
#ifdef TRACE
  (void) printf ("constructor encountered\n");
#endif
  stack [stackptr] = newexpr (CONSTR);
  stack [stackptr++]->fpexprv.compconstr.compnext = 0;
  startcomp ();
}

void constrnext ()
{	/* append new item to the end of the list */
  fpexpr fun, oldc;

#ifdef TRACE
  (void) printf ("finished item of constructor\n");
#endif
  endcomp ();
  fun = stack [--stackptr];
  oldc = stack [stackptr - 1];
  while (oldc->fpexprv.compconstr.compnext != 0)
    oldc = oldc->fpexprv.compconstr.compnext;
  oldc->fpexprv.compconstr.compexpr = fun;
  oldc->fpexprv.compconstr.compnext = newexpr (CONSTR);
  oldc->fpexprv.compconstr.compnext->fpexprv.compconstr.compnext = 0;
  startcomp ();
}

void endconstr ()
{	/* we delete the last storage box of the list, since it's unused */
  fpexpr oldc, last;

#ifdef TRACE
  (void) printf ("constructor finished\n");
#endif
  endcomp ();/* usually this one just pops the context, should always be 0 */
  oldc = stack [stackptr - 1];
  while (oldc->fpexprv.compconstr.compnext != 0)
  {
    last = oldc;
    oldc = oldc->fpexprv.compconstr.compnext;
  }
  last->fpexprv.compconstr.compnext = 0;
}

void parseinsert (type)
int type;
/* type is 0 for insert, 1 for right insert, 2 for tree insert */
{
  fpexpr ins;

#ifdef TRACE
  switch (type)
  {
    case 0: (void) printf ("insert encountered\n");
	    break;
    case 1: (void) printf ("right insert encountered\n");
	    break;
    case 2: (void) printf ("tree insert encountered\n");
	    break;
    default: (void) printf ("unknown insert found\n");
	    exit (1);
  }
#endif
  switch (type)
  {
    case 0: ins = newexpr (INSERT);
	    break;
    case 1: ins = newexpr (RINSERT);
	    break;
    default: ins = newexpr (TREE);
  }
  stack [stackptr++] = ins;
}

void endinsert ()
{
#ifdef TRACE
  (void) printf ("insert done\n");
#endif
  stackptr--;
  stack [stackptr - 1]->fpexprv.aains = stack [stackptr];
}

void parsesel (sel, right)
char * sel;
int right;
{
  fpexpr selfn;
  char errbuf [256];

#ifdef TRACE
  (void) printf ("%s selector is %s\n", (right ? "right" : "left"), sel);
#endif
  stack [stackptr++] = selfn = newexpr (right ? RSEL : SEL);
  (void) sscanf (sel, "%d", &selfn->fpexprv.lrsel);
  if (selfn->fpexprv.lrsel <= 0)
  {
    (void) sprintf (errbuf, "error: selector %d < 1", selfn->fpexprv.lrsel);
    yyerror (errbuf);
  }
}

void parsefncall (fun)
char * fun;
{
  fpexpr funblk;
  unsigned int len;

#ifdef TRACE
  (void) printf ("calling function %s\n", fun);
#endif
  stack [stackptr++] = funblk = newexpr (FNCALL);
  len = strlen (fun) + 1;
  funblk->fpexprv.funcall = malloc (len);
  (void) strcpy (funblk->fpexprv.funcall, fun);
}

void consttrue ()
{
#ifdef TRACE
  (void) printf ("constant true\n");
#endif
  stack [stackptr++] = newexpr (TRUE);
}

void constfalse ()
{
#ifdef TRACE
  (void) printf ("constant false\n");
#endif
  stack [stackptr++] = newexpr (FALSE);
}

void constnum (num)
char * num;
{
  fpexpr objblock;

#ifdef TRACE
  (void) printf ("constant number %s\n", num);
#endif
  stack [stackptr++] = objblock = newexpr (INT);
  (void) sscanf (num, "%d", &objblock->fpexprv.intobj);
}

void constsym (name)
char * name;
{
  fpexpr objblock;

#ifdef TRACE
  (void) printf ("constant symbol %s\n", name);
#endif
  stack [stackptr++] = objblock = newexpr (SYM);
  objblock->fpexprv.symbol = malloc ((unsigned) (strlen (name) + 1));
  (void) strcpy (objblock->fpexprv.symbol, name);
}

void conststr (str)
char * str;
{
  fpexpr obj, new, ch;
  char * strp = str;

#ifdef TRACE
  (void) printf ("constant string %s\n", str);
#endif
  while (*(++strp) != '\0')
    ;
  strp--;	/* strp now points to the char before the null */
  strp--;	/* strp now points to the char before the " */
  if (strp == str)	/* empty string, same as NIL */
    stack [stackptr++] = newexpr (NIL);
  else
  {
    for (obj = 0; strp != str; strp--)
/* by checking strp != str, we skip the initial " */
    {
      new = newexpr (LIST);
      new->fpexprv.listobj.listnext = obj;
      ch = newexpr (CHAR);
      ch->fpexprv.character = *strp;
      new->fpexprv.listobj.listel = ch;
      obj = new;
    }
    stack [stackptr++] = obj;
  }
}

void constchr (ch)
char * ch;
{
  fpexpr objblock;

#ifdef TRACE
  (void) printf ("constant character %s\n", ch);
#endif
  stack [stackptr++] = objblock = newexpr (CHAR);
  if (*(++ch) == '\\')
    ch++;
  objblock->fpexprv.character = *ch;
}

void constreal (num)
char * num;
{
  fpexpr objblock;

#ifdef TRACE
  (void) printf ("constant floating-point number %s\n", num);
#endif
  stack [stackptr++] = objblock = newexpr (FLOAT);
  (void) sscanf (num, "%lf", &objblock->fpexprv.floatobj);
}

void parsenil ()
{
#ifdef TRACE
  (void) printf ("constant nil\n");
#endif
  stack [stackptr++] = newexpr (NIL);
}

void liststart ()
{
  fpexpr objblock;

#ifdef TRACE
  (void) printf ("beginning of constant list\n");
#endif
  stack [stackptr++] = objblock = newexpr (LIST);
  objblock->fpexprv.listobj.listnext = 0;
}

void listnext ()
{
  fpexpr obj, oldobj, el;

#ifdef TRACE
  (void) printf ("end of element of constant list\n");
#endif
  obj = newexpr (LIST);
  el = stack [--stackptr];
  obj->fpexprv.listobj.listnext = 0;
  oldobj = stack [stackptr - 1];
  while (oldobj->fpexprv.listobj.listnext != 0)
    oldobj = oldobj->fpexprv.listobj.listnext;
  oldobj->fpexprv.listobj.listel = el;
  oldobj->fpexprv.listobj.listnext = obj;
}

void listend ()
{	/* invariant: thanks to YACC, there must have been at
	 * least one call to listnext since the call to liststart */
	/* essentially, we take the last element off the list, since
	 * that is the one and only unused one */
  fpexpr obj;

#ifdef TRACE
  (void) printf ("end of constant list\n");
#endif
  obj = stack [stackptr - 1];
  while (obj->fpexprv.listobj.listnext->fpexprv.listobj.listnext != 0)
    obj = obj->fpexprv.listobj.listnext;
  obj->fpexprv.listobj.listnext = 0;
}
