/* expr.c: produce code for the expression encoded by the parse tree. */

#include <stdio.h>
#include <strings.h>
#include "fpc.h"
#include "parse.h"
#include "code.h"
#include "fp.h"

extern void newvar ();
extern char * sprintf ();

static void codecond ();
static void codebu ();
static void codewhile ();
static void codecomp ();
static void codeaa ();
static void codeconstr ();
static void codeinsert ();
static void codesel ();
static void codefncall ();
static void codeconst ();
static void codemulti ();

void codeexpr (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
{
  int type = 0;
/* used to distinguish between slightly different functional forms that
 * use the same procedure to generate code.
 */

  switch (tree->exprtype)
  {
    case COND:
      codecond (tree, invar, outvar);
      break;
    case BUR:
      type++;
    case BU:
      codebu (tree, type, invar, outvar);
      break;
    case WHILE:
      codewhile (tree, invar, outvar);
      break;
    case COMP:
      codecomp (tree, invar, outvar);
      break;
    case AA:
      codeaa (tree, invar, outvar);
      break;
    case CONSTR:
      codeconstr (tree, invar, outvar);
      break;
    case TREE:
      type++;
    case RINSERT:
      type++;
    case INSERT:
      codeinsert (tree, type, invar, outvar);
      break;
    case MULTI:
      codemulti (tree, invar, outvar);
      break;
    case RSEL:
      type++;
    case SEL:
      codesel (tree, type, invar, outvar);
      break;
    case FNCALL:
      codefncall (tree, invar, outvar);
      break;
    default:
      if ((tree->exprtype >= NIL) && (tree->exprtype <= CHAR))
        codeconst (tree, invar, outvar);
      else
        yyerror ("compiler error 10");
  }
}

static int indlev = 1;

static void indent (plus)
int plus;
{
  if (plus > 0)
    indlev++;
  else
    indlev--;
}

static char * indentstr ()
/* returns a reference to a string with 2*indlev blanks. Notice that
 * successive calls will refer to the same string.... 'nuff said. */
{
  register char * str;
  register int count;
  static char blanks [1024] = "";

  if (indlev > 511)
    yyerror ("error: expression nesting too great");
  count = indlev;
  for (str = blanks; count > 3; *(str++) = '\t')
    count -= 4;
  count *= 2;
  for ( ; count > 0; *(str++) = ' ')
    count -= 1;
  *str = '\0';
  return (blanks);
}

static void codecond (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
/* a -> b ; c : res := a; if (res) then res := b; else res := c; end */
{
  (void) fprintf (outf, "%sinc_ref (%s);\n", indentstr (), invar);
  codeexpr (tree->fpexprv.conditional [0], invar, outvar);   /* r := a (d); */
  (void) fprintf (outf, "%sif (%s->fp_type%s)\n",	/* if (r) */
	   indentstr (), outvar, (check)? " == TRUEOBJ" : "");
  BRACE;
  codeexpr (tree->fpexprv.conditional [1], invar, outvar);   /* r := b (d); */
  UNBRACE;
  (void) fprintf (outf, "%selse", indentstr ()); /* else */
  if (check)
    (void) fprintf (outf, " if (%s->fp_type == FALSEOBJ)", outvar);
  (void) fprintf (outf, "\n");
  BRACE;
  codeexpr (tree->fpexprv.conditional [2], invar, outvar);   /* r := c (d); */
  UNBRACE;
  if (check)
    (void) fprintf (outf,
	     "%selse\n%s  genbottom (\"%s\", %s);\n",
    	     indentstr (), indentstr (), "in conditional: non-boolean pred",
	     outvar);
}

static void codebu (tree, right, invar, outvar)
fpexpr tree;
int right;
char * invar, * outvar;
/* bu  op v : res := v; r1 := newvect (res, arg); res := op (r1);
   bur op v : res := v; r1 := newvect (arg, res); res := op (r1); */
{
  char pair [MAXIDLEN];
/* later on should optimize bu/r op x for op in {=, !=, +, -, *, div, mod}
 * and for x an atomic type */

  codeconst (tree->fpexprv.bulr.buobj, "", outvar);
  newvar (pair);
  (void) fprintf (outf, "%s%s = newpair ();\n", indentstr (), pair);
  (void) fprintf (outf, "%s%s->fp_header.fp_next->fp_entry = %s;\n",
	   indentstr (), pair, (right) ? outvar : invar);
  (void) fprintf (outf, "%s%s->fp_entry = %s;\n",
	   indentstr (), pair, (right) ? invar : outvar);
  codeexpr (tree->fpexprv.bulr.bufun, pair, outvar);
}

static void codewhile (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
/* while pred f : res := arg;
   while (1)
      r1 := pred (res); if (! r1) then break; arg := f (res); res := arg; */
{
  char predicate [MAXIDLEN];
  char result [MAXIDLEN];

  newvar (predicate);
  newvar (result);
  (void) fprintf (outf, "%s%s = %s;\n%swhile (1)\n",
	          indentstr (), outvar, invar, indentstr ());
  BRACE;
  (void) fprintf (outf, "%sinc_ref (%s);\n", indentstr (), outvar);
  codeexpr (tree->fpexprv.whilestat [0], outvar, predicate);
/* notice: need not dec_ref (predicate) since the result is
   ALWAYS a boolean, so dec_ref'ing it would make no difference */
  (void) fprintf (outf, "%sif (%s %s->fp_type)\n%s  break;\n",
	   indentstr (), ((check) ? "FALSEOBJ ==" : "!"),
	   predicate, indentstr ());
  if (check)
    (void) fprintf (outf, "%selse if (%s->fp_type != TRUEOBJ)\n%s  %s%s);\n",
	     indentstr (), predicate, indentstr (),
	     "genbottom (\"predicate for while is not boolean\", ", predicate);
  codeexpr (tree->fpexprv.whilestat [1], outvar, result);
  (void) fprintf (outf, "%s%s = %s;\n", indentstr (), outvar, result);
  UNBRACE;
}

static void codecomp (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
/* a o b o c o d : r1 := d (arg); r2 := c (r1); r1 := b (r2); res := a (r1); */
/* we need to alternate use of r1 and r2 since some of the functional forms
   will generate wierd code if given the same input and output variable */
{
  char pass [2] [MAXIDLEN];
  char count = 0;

  newvar (pass [0]);
  if ((tree->fpexprv.compconstr.compnext != 0) &&  /* should never happen */
      (tree->fpexprv.compconstr.compnext->fpexprv.compconstr.compnext != 0))
/* the second expression will return false if we have (a o b) */
    newvar (pass [1]);
  while (tree != 0)
  {
    if (tree->fpexprv.compconstr.compnext != 0)
      codeexpr (tree->fpexprv.compconstr.compexpr, invar, pass [count]);
    else
      codeexpr (tree->fpexprv.compconstr.compexpr, invar, outvar);
    invar = pass [count];
    count = (count + 1) % 2;
    tree = tree->fpexprv.compconstr.compnext;
  }
}

static void codeaa (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
/* aa e : if (arg == <>) then res := arg;
   else r1 := arg; res := newcell (); r2 := res;
     while (r1 != 0) r2->el := e (car r1); r1 := cdr (r1);
       if (r1 != 0) r2->next = newcell (); r2 = cdr (r2); */
{
  char chasearg [MAXIDLEN], chaseres [MAXIDLEN], tempres [MAXIDLEN],
       tempval [MAXIDLEN];

  (void) fprintf (outf, "%sif (%s->fp_type == NILOBJ)\n%s  %s = %s;\n%selse",
	   indentstr (), invar, indentstr (), outvar, invar, indentstr ());
  if (check)
    (void) fprintf (outf, " if (%s->fp_type == VECTOR)", invar);
  newvar (chasearg);
  newvar (chaseres);
  (void) fprintf (outf, "\n");
  BRACE;
  (void) fprintf (outf, "%s%s = %s;\n%s%s = %s = newcell ();\n",
	          indentstr (), chasearg, invar,
	          indentstr (), chaseres, outvar);
  (void) fprintf (outf, "%swhile (1)\n", indentstr ());
  BRACE;
  (void) sprintf (tempres, "%s->fp_entry", chaseres);
  (void) sprintf (tempval, "%s->fp_entry", chasearg);
  (void) fprintf (outf, "%sinc_ref (%s);\n", indentstr (), tempval);
  codeexpr (tree->fpexprv.aains, tempval, tempres);
  (void) fprintf (outf, "%sif (%s = %s->fp_header.fp_next)\n",
	   indentstr (), chasearg, chasearg, indentstr ());
  (void) fprintf (outf, "%s  %s = %s->fp_header.fp_next = newcell ();\n",
	   indentstr (), chaseres, chaseres);
  (void) fprintf (outf, "%selse\n%s  break;\n", indentstr (), indentstr ());
  UNBRACE;
  UNBRACE;
  if (check)
    (void) fprintf (outf,
	     "%selse\n%s  genbottom (\"%s\", %s);\n",
	     indentstr (), indentstr (),
	     "apply-to-all called with atomic argument", invar);
  (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), invar);
}

static void codeconstr (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
/* [a, b] : res := new (2); chase := res; chase->car := b (arg);
            chase = cdr (chase); chase->car := a (arg); */
{
  int length;
  fpexpr subtree = tree;
  char chase [MAXIDLEN];
  char tempres [MAXIDLEN];

  for (length = 0; subtree != 0; length++)
    subtree = subtree->fpexprv.compconstr.compnext;
  newvar (chase);
  (void) sprintf (tempres, "%s->fp_entry", chase);
  if (length > 2)
    (void) fprintf (outf, "%s%s = %s = newvect (%d);\n", indentstr (),
		    outvar, chase, length);
  else if (length == 2)
    (void) fprintf (outf, "%s%s = %s = newpair ();\n", indentstr (),
		    outvar, chase);
  else
    (void) fprintf (outf, "%s%s = %s = newcell ();\n", indentstr (),
		    outvar, chase);
  if (length > 1)
    (void) fprintf (outf, "%s%s->fp_ref += %d;\n", indentstr (), invar,
		    length - 1);
  while (tree != 0)
  {
    codeexpr (tree->fpexprv.compconstr.compexpr, invar, tempres);
    tree = tree->fpexprv.compconstr.compnext;
    if (tree != 0)
      (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n",
	       indentstr (), chase, chase);
  }
}

static void codemulti (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
{
/* multi f: r1 := arg; res := newconst (); res->val := initval;
	    while (r1 != 0) res := op (res, car (r1)); r1 := cdr (r1); */
  char var1 [MAXIDLEN];
  int optype;	/* 0 for +, 1 for *, 2 for and, 3 for or */
  int isand;
  int isplus;
  char opchar;	/* + for +, * for * */

  newvar (var1);
  if (strcmp (tree->fpexprv.aains->fpexprv.funcall, "plus") == 0)
    optype = 0;
  else if (strcmp (tree->fpexprv.aains->fpexprv.funcall, "times") == 0)
    optype = 1;
  else if (strcmp (tree->fpexprv.aains->fpexprv.funcall, "and") == 0)
    optype = 2;
  else if (strcmp (tree->fpexprv.aains->fpexprv.funcall, "or") == 0)
    optype = 3;
  else
    yyerror ("compiler error 20");
  if (check)
  {
    (void) fprintf (outf, "%sif (%s->fp_type != VECTOR)\n",
		    indentstr (), invar);
    indent (1);
    (void) fprintf (outf,
"%sgenbottom (\"error in insert: argument not a vector\", %s);\n",
		    indentstr (), invar);
    indent (0);
  }
/* multi f: r1 := arg; */
  (void) fprintf (outf, "%s%s = %s;\n", indentstr (), var1, invar);
  if (optype > 1)
  {
    isand = (optype == 2);
/* while ((r1 != 0) && (car (r1) != true[false])) r1 := cdr (r1); */
    (void) fprintf (outf, "%swhile (%s && ", indentstr (), var1);
    if (isand)
      if (check)
        (void) fprintf (outf, "(%s->fp_entry->fp_type == TRUEOBJ))\n", var1);
      else
        (void) fprintf (outf, "%s->fp_entry->fp_type)\n", var1);
    else
      (void) fprintf (outf, "(%s->fp_entry->fp_type == FALSEOBJ))\n", var1);
    indent (1);
    (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n", indentstr (),
		    var1, var1);
    indent (0);
/* if (r1 == 0) res := default else res := other */
    (void) fprintf (outf, "%sif (%s)\n", indentstr (), var1);
    indent (1);
    if (check)
    {
      (void) fprintf (outf, "%sif (%s->fp_entry->fp_type != %sOBJ)\n",
		      indentstr (), var1, (isand ? "FALSE" : "TRUE"));
      indent (1);
      (void) fprintf (outf,
"%sgenbottom (\"error in insert %s: argument not a boolean vector\", %s);\n",
		      indentstr (), (isand ? "and" : "or"), invar);
      indent (0);
      (void) fprintf (outf, "%selse\n", indentstr ());
      indent (1);
    }
    (void) fprintf (outf, "%s%s = & %cobj;\n", indentstr (), outvar,
  	            (isand ? 'f' : 't'));
    if (check)
      indent (0);
    indent (0);
    (void) fprintf (outf, "%selse\n", indentstr ());
    indent (1);
    (void) fprintf (outf, "%s%s = & %cobj;\n", indentstr (), outvar,
  	            (isand ? 't' : 'f'));
    indent (0);
  }
  else		/* numeric */
  {
    isplus = (optype == 0);
    opchar = isplus ? '+' : '*';
/* multi f: r1 := arg; res := newconst (INT); res->val := 0|1; */
    (void) fprintf (outf, "%s%s = newconst (INTCONST);\n", indentstr (),
		    outvar);
    (void) fprintf (outf, "%sif (%s->fp_entry->fp_type == INTCONST)\n",
		    indentstr (), var1);
    BRACE;
    (void) fprintf (outf, "%s%s->fp_header.fp_int = ", indentstr (), outvar);
    (void) fprintf (outf, "%s->fp_entry->fp_header.fp_int;\n", var1);
/* while (d0 && (d0->car->type == int)) res += d0->car->val; d0 = cdr (d0); */
    (void) fprintf (outf, "%swhile ((%s = %s->fp_header.fp_next) && ",
		    indentstr (), var1, var1);
    (void) fprintf (outf, "(%s->fp_entry->fp_type == INTCONST))\n", var1);
    if (check)	/* need to check for arithmetic overflow */
    {
      BRACE;
      if (isplus)
      {
        (void) fprintf (outf, "%sif (((%s->fp_header.fp_int < 0) == ",
		        indentstr (), outvar);
        (void) fprintf (outf, "(%s->fp_entry->fp_header.fp_int < 0)) &&\n",
		        var1);
      }
      else
        (void) fprintf (outf, "%sif ((%s->fp_header.fp_int != 0) &&\n",
		        indentstr (), outvar);
      indent (1);
      indent (1);
      (void) fprintf (outf, "%s((%d %c abs (%s->fp_header.fp_int))",
		      indentstr (), MAXINT, (isplus ? '-' : '/'), outvar);
      (void) fprintf (outf, " < abs (%s->fp_entry->fp_header.fp_int)))\n",
		      var1);

      indent (0);
      (void) fprintf (outf, "%sgenbottom (\"overflow in insert %c\", %s);\n",
		      indentstr (), opchar, invar);
      indent (0);
    }
    else
      indent (1);
    (void) fprintf (outf, "%s%s->fp_header.fp_int ", indentstr (), outvar);
    (void) fprintf (outf, "%c= %s->fp_entry->fp_header.fp_int;\n",
		    opchar, var1);
    if (check)
    {
      UNBRACE;
    }
    else
      indent (0);
    UNBRACE;
    (void) fprintf (outf, "%selse\n", indentstr ());
    indent (1);
    (void) fprintf (outf, "%s%s->fp_header.fp_int = %c;\n", indentstr (),
		    outvar, (isplus ? '0' : '1'));
    indent (0);
    (void) fprintf (outf, "%sif (%s)\n", indentstr (), var1);
    BRACE;
    (void) fprintf (outf, "%s%s->fp_header.fp_float =", indentstr (), outvar);
    (void) fprintf (outf, " %s->fp_header.fp_int;\n", outvar);
    (void) fprintf (outf, "%s%s->fp_type = FLOATCONST;\n", indentstr (),
		    outvar);
    (void) fprintf (outf, "%swhile (%s)\n", indentstr (), var1);
    BRACE;
    (void) fprintf (outf, "%sif (%s->fp_entry->fp_type == FLOATCONST)\n",
		    indentstr (), var1);
    indent (1);
    (void) fprintf (outf, "%s%s->fp_header.fp_float ", indentstr (), outvar);
    (void) fprintf (outf, "%c= %s->fp_entry->fp_header.fp_float;\n",
		    opchar, var1);
    indent (0);
    if (check)
    {
      (void) fprintf (outf, "%selse if (%s->fp_entry->fp_type != INTCONST)\n",
		      indentstr (), var1);
      indent (1);
      (void) fprintf (outf,
"%sgenbottom (\"error in insert %c: argument not a numeric vector\", %s);\n",
		      indentstr (), opchar, invar);
      indent (0);
    }
    (void) fprintf (outf, "%selse\n", indentstr ());
    indent (1);
    (void) fprintf (outf, "%s%s->fp_header.fp_float ", indentstr (), outvar);
    (void) fprintf (outf, "%c= %s->fp_entry->fp_header.fp_int;\n",
		    opchar, var1);
    indent (0);
    (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n", indentstr (),
		    var1, var1);
    UNBRACE;
    UNBRACE;
  }
  (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), invar);
}

static void codeinsert (tree, type, invar, outvar)
fpexpr tree;
int type;	/* 0 for left, 1 for right, 2 for tree */
char * invar, * outvar;
/* /a : r3 := 0; r2 := arg;
	while (r2 != 0) r1 := cons (car (r2), r3); r3 := r1; r2 := cdr (r2);
        res := car (r3); r1 := cdr (r3);
        while (r1 != 0) r2 := cons (car (r1), cons (res, nil));
	  res := a (r2); r1 := cdr (r1);
   \a : res := car (arg); r1 := cdr (arg);
        while (r1 != 0) r2 := cons (res, cons (car (r1), nil));
	  res := a (r2); r1 := cdr (r1);
   \/a: r1 = arg;
        while (r1->cdr != 0)
          r2 := r1; r1 := newcell (); r3 := r1;
          while (r2 != 0)
            if (r2->cdr == 0) r3->car = r2->car; r2 = 0;
            else
              r4 = newpair (); r4->car = r2->car; r2 = r2->cdr;
              r4->cdr->car = r2->car; r2 = r2->cdr; r3->car = a (r4);
	      if (r2 != 0) r3->cdr = newcell (); r3 = r3->cdr;
        res = r1->car; */
{
  char insertname [13];
  char var1 [MAXIDLEN],
       var2 [MAXIDLEN],
       var3 [MAXIDLEN],
       var4 [MAXIDLEN],
       var5 [MAXIDLEN],		/* used for ref count in tree insert */
       argvar [MAXIDLEN],	/* this is the argument to the fn in rins */
       varcar [MAXIDLEN];

  newvar (var1);
  newvar (var2);
  switch (type)
  {
    case 0:	/* normal insert */
      (void) strcpy (insertname, "left insert");
      newvar (var3);
      (void) strcpy (argvar, var3);
      break;
    case 1:	/* right insert */
      (void) strcpy (insertname, "right insert");
      (void) strcpy (argvar, invar);
      break;
    default:	/* tree insert */
      (void) strcpy (insertname, "tree insert");
      newvar (var3);
      newvar (var4);
      newvar (var5);
      (void) sprintf (varcar, "%s->fp_entry", var3);
      break;
  }
  if (check)
  {
    (void) fprintf (outf, "%sif (%s->fp_type != VECTOR)\n",
	     	    indentstr (), invar);
    (void) fprintf (outf, "%s  genbottom (\"%s%s\", %s);\n", indentstr (),
		    "non-vector passed to ", insertname, invar);
  }
  switch (type)
  {
    case 0:	/* normal insert */
/* r3 := 0; r2 := arg; */
      (void) fprintf (outf, "%s%s = 0;\n%s%s = %s;\n", indentstr (),
	              var3, indentstr (), var2, invar);
/* while (r2 != 0) r1 := cons (car (r2), r3); r3 := r1; r2 := cdr (r2); */
/* i.e., reverse+copy arg into ra. Increment the refs of each element
   of arg, afterwards return arg, and the elements will stay. */
      (void) fprintf (outf, "%swhile (%s)\n", indentstr (), var2);
      BRACE;
      (void) fprintf (outf, "%s%s = newcell ();\n", indentstr (), var1);
      (void) fprintf (outf, "%s%s->fp_header.fp_next = %s;\n",
	       	      indentstr (), var1, var3);
      (void) fprintf (outf, "%s%s->fp_entry = %s->fp_entry;\n%s%s = %s;\n",
	       	      indentstr (), var1, var2, indentstr (), var3, var1);
      (void) fprintf (outf, "%sinc_ref (%s->fp_entry);\n", indentstr (), var3);
      (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n",
	       	      indentstr (), var2, var2);
      UNBRACE;
      (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), invar);
    case 1:	/* right insert */
/* res := car (arg/r3); r1 := cdr (arg/r3); */
      (void) fprintf (outf, "%s%s = %s->fp_entry;\n", indentstr (),
		      outvar, argvar);
      (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n", indentstr (),
		      var1, argvar);
      (void) fprintf (outf, "%sinc_ref (%s);\n", indentstr (), outvar);
/* while (r1 != 0) r2 := cons (res, cons (car (r1), nil));
		   r2 := cons (car (r1), cons (res, nil));
   res := a (r2); r1 := cdr (r1); */
      (void) fprintf (outf, "%swhile (%s)\n",
	              indentstr (), var1);
      BRACE;
      (void) fprintf (outf, "%s%s = newpair ();\n", indentstr (), var2);
      if (type == 0)
      {
	(void) fprintf (outf, "%s%s->fp_header.fp_next->fp_entry = %s;\n",
		        indentstr (), var2, outvar);
	(void) fprintf (outf, "%s%s->fp_entry = %s->fp_entry;\n",
		        indentstr (), var2, var1);
      }
      else
      {
	(void) fprintf (outf, "%s%s->fp_entry = %s;\n",
		        indentstr (), var2, outvar);
	(void) fprintf (outf,
			"%s%s->fp_header.fp_next->fp_entry = %s->fp_entry;\n",
		        indentstr (), var2, var1);
      }
      (void) fprintf (outf, "%sinc_ref (%s->fp_entry);\n", indentstr (), var1);
      codeexpr (tree->fpexprv.aains, var2, outvar);
      (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n",
	              indentstr (), var1, var1);
      UNBRACE;
      (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), argvar);
      break;
    default:	/* tree insert */
/*   \/a: r1 = arg;							*/
      (void) fprintf (outf, "%s%s = %s;\n", indentstr (), var1, invar);
/*        while (r1->cdr != 0)						*/
      (void) fprintf (outf, "%swhile (%s->fp_header.fp_next%s)\n",
		      indentstr (), var1, (check ? " != 0" : ""));
      BRACE;
/*          r2 = r1; r1 := r3 := newcell ();				*/
      (void) fprintf (outf, "%s%s = %s = %s;\n", indentstr (), var2,
		      var5, var1);
      (void) fprintf (outf, "%s%s = %s = newcell ();\n", indentstr (),
		      var1, var3);
/*          while (r2 != 0)						*/
      (void) fprintf (outf, "%swhile (%s%s)\n", indentstr (), var2,
		      (check ? " != 0" : ""));
      indent (1);
/*            if (r2->cdr == 0) r3->car := r2->car; r2 := 0;		*/
/*            else							*/
      (void) fprintf (outf, "%sif (%s->fp_header.fp_next == 0)\n",
		      indentstr (), var2);
      BRACE;
      (void) fprintf (outf, "%s%s->fp_entry = %s->fp_entry;\n",
		      indentstr (), var3, var2);
      (void) fprintf (outf, "%sinc_ref (%s->fp_entry);\n", indentstr (), var2);
      (void) fprintf (outf, "%s%s = 0;\n", indentstr (), var2);
      UNBRACE;
      (void) fprintf (outf, "%selse\n", indentstr ());
      BRACE;
/*              r4 := newpair (); r4->car := r2->car; r2 := r2->cdr;	*/
      (void) fprintf (outf, "%s%s = newpair ();\n", indentstr (), var4);
      (void) fprintf (outf, "%s%s->fp_entry = %s->fp_entry;\n",
		      indentstr (), var4, var2);
      (void) fprintf (outf, "%sinc_ref (%s->fp_entry);\n", indentstr (), var2);
      (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n",
		      indentstr (), var2, var2);
/*              r4->cdr->car := r2->car; r2 := r2->cdr; r3->car := a (r4); */
      (void) fprintf (outf,
		      "%s%s->fp_header.fp_next->fp_entry = %s->fp_entry;\n",
		      indentstr (), var4, var2);
      (void) fprintf (outf, "%sinc_ref (%s->fp_entry);\n", indentstr (), var2);
      (void) fprintf (outf, "%s%s = %s->fp_header.fp_next;\n",
		      indentstr (), var2, var2);
      codeexpr (tree->fpexprv.aains, var4, varcar);
/*	      if (r2 != 0) r3->cdr := newcell (); r3 := r3->cdr;	*/
      (void) fprintf (outf, "%sif (%s%s)\n", indentstr (), var2,
		      (check ? " != 0" : ""));
      (void) fprintf (outf,
		      "%s  %s = %s->fp_header.fp_next = newcell ();\n",
		      indentstr (), var3, var3);
/*        res := r1->car;						*/
      UNBRACE;
      indent (0);
      (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), var5);
      UNBRACE;
      (void) fprintf (outf, "%s%s = %s->fp_entry;\n",
		      indentstr (), outvar, var1);
      (void) fprintf (outf, "%sinc_ref (%s);\n", indentstr (), outvar);
      (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), var1);
      break;
  }
}

static void codesel (tree, right, invar, outvar)
fpexpr tree;
int right;
char * invar, * outvar;
/* n: i1 := n; r := d; while (--i1 != 0) r := cdr (r);
      r := car (r);
  nr: i1 := 0; r := d; while (r != 0) r := cdr (r); i1++;
      i1 := i1 - (n - 1); r := d; while (--i1 != 0) r := cdr (r);
      r := car (r); */
/* notice that selectors of 1 are special cases, since they occurr
 * very frequently and can be optimized a bit */
{
  char * ind;
  char * errmess = "argument too short for ";
  char checkstr [256];
  int selector;

  checkstr [0] = '\0';
  selector = tree->fpexprv.lrsel;
  ind = indentstr ();
  if (check)
  {
    (void) fprintf (outf, "%sif (%s->fp_type != VECTOR)\n", ind, invar);
    (void) fprintf (outf,
	     "%s  genbottom (\"selector %d%s applied to nonvector\", %s);\n",
	     ind, selector, (right) ? "r" : "", invar);
  }
  if (selector == 1)		/* first or last */
  {
    if (right)			/* last: common special case */
    {
      (void) fprintf (outf, "%s%s = %s;\n", ind, outvar, invar); /* r := d; */
      (void) fprintf (outf,			/* while (cdr (r) != 0) */
	       	      "%swhile (%s->fp_header.fp_next)\n", ind, outvar);
      (void) fprintf (outf,			/* r = cdr (r); */
	       	      "%s  %s = %s->fp_header.fp_next;\n", ind,
		      outvar, outvar);
      (void) fprintf (outf,			/* r = car (r); */
	       	      "%s%s = %s->fp_entry;\n", ind, outvar, outvar);
    }
    else			/* first: *very* common special case */
/* r := car (d); */
      (void) fprintf (outf, "%s%s = %s->fp_entry;\n", ind, outvar, invar);
  }
  else		/* selector != 1, general (i.e., non-special) case */
  {
	/* i1 := 1 or i1 := n */
    (void) fprintf (outf, "%ssel = %d;\n", ind, (right) ? 1 : selector);
    if (right)
    {
      (void) fprintf (outf, "%s%s = %s;\n", ind, outvar, invar); /* r := d; */
      (void) fprintf (outf,		/* while ((r = cdr (r)) != 0) i1++; */
	       	      "%swhile (%s = %s->fp_header.fp_next)\n%s  sel++;\n",
	       	      ind, outvar, outvar, ind);
      if (check)
        (void) fprintf (outf,
			"%sif (sel < %d)\n%s  genbottom (\"%s%dr\", %s);\n",
	                ind, selector, ind, errmess, selector, invar);
  /* i1 := i1 - (n - 1); */
      (void) fprintf (outf, "%ssel -= %d;\n", ind, selector - 1);
    }
    (void) fprintf (outf, "%s%s = %s;\n", ind, outvar, invar);	/* r := d; */
    if (check && (! right))
      (void) sprintf (checkstr,
"if (%s == 0)\n%s    genbottom (\"%ssel %d\", %s);\n%s  else\n%s    ",
	       	      outvar, ind, errmess, selector, invar, ind, ind);
	  /* while (--i1 != 0) r := cdr (r); */
    (void) fprintf (outf,
	     	    "%swhile (--sel)\n%s  %s%s = %s->fp_header.fp_next;\n",
	     	    ind, ind, checkstr, outvar, outvar);
    /*  r := car (r); */
    if (check && (! right))
      (void) fprintf (outf,
		      "%sif (%s == 0)\n%s  genbottom (\"%ssel %d\", %s);\n",
	       	      ind, outvar, ind, errmess, selector, invar);
    (void) fprintf (outf, "%s%s = %s->fp_entry;\n", ind, outvar, outvar);
  }
  (void) fprintf (outf, "%sinc_ref (%s);\n%sdec_ref (%s);\n",
	   	  ind, outvar, ind, invar);
}

static void codefncall (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
/* f: res := f (arg); */
{
  if (strcmp (tree->fpexprv.funcall, "times") == 0)
    (void) fprintf (outf, "%s%s = %s (%s);\n",
		    indentstr (), outvar, "fptimes", invar);
  else
    (void) fprintf (outf, "%s%s = %s (%s);\n",
		    indentstr (), outvar, tree->fpexprv.funcall, invar);
}

static void codeconst (tree, invar, outvar)
fpexpr tree;
char * invar, * outvar;
{
  if (*invar != '\0')
    (void) fprintf (outf, "%sdec_ref (%s);\n", indentstr (), invar);
  (void) fprintf (outf, "%s%s = (fp_data) & (%s);\n%sinc_ref (%s);\n",
	   indentstr (), outvar, tree->constvar, indentstr (), outvar);
}
