/*
 * A standard lexical analyzer
 *
 *	Copyright (c) 1986 by Andy Valencia
 */
#include "symtab.h"
#include <stdio.h>
#include <ctype.h>

static char buf[80];
static int donum();
extern YYSTYPE yylval;
extern void exit(), perror();

static FILE *cur_in = stdin;
static nextc();
char prompt;

#define MAXNEST 5		/* How deep can we get? */
static FILE *fstack[MAXNEST];	/* For nested loads */
static int fpos = 0;

    /*
     * Skip leading white space in current input stream
     */
static void
skipwhite(){
    register c;

	/*
	 * Skip leading blank space
	 */
    while( (c = nextc()) != EOF )
	if( !isspace(c) ) break;
    ungetc(c,cur_in);
}

    /*
     * Lexical analyzer for YACC
     */
yylex(){
    register char *p = buf;
    register c, c1;

	/*
	 * Skip over white space
	 */
again:
    skipwhite();
    c = nextc();

	/*
	 * Return EOF
	 */
    if( c == EOF ) return(c);

	/*
	 * An "identifier"?
	 */
    if( isalpha(c) ){
	struct symtab *q;

	    /*
	     * Assemble a "word" out of the input stream, symbol table it
	     */
	*p++ = c;
	while( isalnum(c = nextc()) ) *p++ = c;
	ungetc(c,cur_in);
	*p = '\0';
	q = lookup(buf);

	    /*
	     * yylval is always set to the symbol table entry
	     */
	yylval.YYsym = q;

	    /*
	     * For built-ins, return the token value
	     */
	if( q->sym_type == SYM_BUILTIN ) return( q->sym_val.YYint );

	    /*
	     * For user-defined (or new),
	     *	return "User Defined"--UDEF
	     */
	return( UDEF );
    }

	/*
	 * For numbers, call our number routine.
	 */
    if( isdigit(c) ) return( donum(c) );

	/*
	 * For possible unary operators, see if a digit
	 *	immediately follows.
	 */
    if( (c == '+') || (c == '-') ){
	char c2 = nextc();

	ungetc(c2,cur_in);
	if( isdigit(c2) )
	    return( donum(c) );
    }

	/*
	 * For certain C operators, need to look at following char to
	 *	assemble relationals.  Otherwise, just return the char.
	 */
    yylval.YYint = c;
    switch( c ){
    case '<':
	if( (c1 = nextc()) == '=' ) return( yylval.YYint = LE );
	ungetc( c1, cur_in );
	return(c);
    case '>':
	if( (c1 = nextc()) == '=' ) return( yylval.YYint = GE );
	ungetc( c1, cur_in );
	return(c);
    case '~':
	if( (c1 = nextc()) == '=' ) return( yylval.YYint = NE );
	ungetc( c1, cur_in );
	return(c);
    default:
	return(c);
    }
}

static int
donum(startc)
    char startc;
{
    char isdouble = 0;
    register char c, *p = buf;

    *p++ = startc;
    for(;;){
	c = nextc();
	if( isdigit(c) ){
	    *p++ = c;
	    continue;
	}
	if( c == '.' ){
	    *p++ = c;
	    isdouble = 1;
	    continue;
	}
	ungetc( c, cur_in );
	break;
    }
    *p = '\0';
    if( isdouble ){
	sscanf(buf,"%lf",&(yylval.YYdouble));
	return( FLOAT );
    } else {
	sscanf(buf,"%d",&(yylval.YYint));
	return( INT );
    }
}

    /*
     * getchar() function for lexical analyzer.  Adds a prompt if
     *	input is from keyboard, also localizes I/O redirection.
     */
static
nextc(){
    register int c;
    static saw_eof = 0;

again:
    if( cur_in == stdin ){
	if( saw_eof ) return(EOF);
	if( !stdin->_cnt )
	    putchar(prompt);
    }
    c = fgetc(cur_in);
    if( c == '#' ){
	while( (c = fgetc(cur_in)) != EOF )
	    if( c == '\n' ) goto again;
    }
	/*
	 * Pop up a level of indirection on EOF
	 */
    if( c == EOF ){
	if( cur_in != stdin ){
	    fclose(cur_in);
	    cur_in = fstack[--fpos];
	    goto again;
	} else {
	    saw_eof++;
	}
    }
    return(c);
}

    /*
     * Command processor.  The reason it's here is that we play with
     *	I/O redirection.  Shrug.
     */
void
fp_cmd(){
    char cmd[80], *p = cmd, arg[80];
    register c;
    FILE *newf;

	/*
	 * Assemble a word, the command
	 */
    skipwhite();
    if( (c = nextc()) == EOF ) return;
    *p++ = c;
    while( (c = nextc()) != EOF )
	if( isalpha(c) ) *p++ = c;
	else break;
    *p = '\0';

	/*
	 * Process the command
	 */
    if( strcmp(cmd,"load") == 0 ){	/* Load command */

	    /*
	     * Get next word, the file to load
	     */
	skipwhite();
	p = arg;
	while( (c = nextc()) != EOF )
	    if( isspace(c) ) break;
	    else *p++ = c;
	*p = '\0';

	    /*
	     * Can we push down any more?
	     */
	if( fpos == MAXNEST-1 ){
	    printf(")load'ed files nested too deep\n");
	    return;
	}

	    /*
	     * Try and open the file
	     */
	if( (newf = fopen(arg,"r")) == 0 ){
	    perror(arg);
	    return;
	}

	    /*
	     * Pushdown the current file, make this one it.
	     */
	fstack[fpos++] = cur_in;
	cur_in = newf;
	return;
    }

    if( strcmp(cmd,"quit") == 0 ){	/* Leave */
	printf("\nDone\n");
	exit( 0 );
    }
    if( strcmp(cmd,"help") == 0 ){	/* Give help */
        printf("Commands are:\n");
	printf(" quit - leave FP\n");
	printf(" help - this message\n");
	printf(" load - redirect input from a file\n");
#ifdef YYDEBUG
	printf(" yydebug - toggle parser tracing\n");
#endif
	return;
    }
#ifdef YYDEBUG
    if( strcmp(cmd,"yydebug") == 0 ){	/* Toggle parser trace */
	extern int yydebug;

	yydebug = !yydebug;
	return;
    }
#endif
    printf("Unknown command '%s'\n",cmd);
}
