/*
 * Process a format, as used by various report generators, to format a value.
 * Format characters:
 *
 * *	Digit or asterisk prefix
 * $	Digit or dollar-sign prefix
 * -	Digit or minus-sign prefix if negative
 * +	Digit or sign prefix
 * (	Digit or left-parenthesis prefix if negative
 * #	Digit or blank prefix
 * &	Digit or zero prefix
 * )	Right-parenthesis suffix if negative
 * .	Decimal point
 * ,	Comma or space prefix
 * <	Digit or space appended after format (left justification)
 *
 * This may not be the fastest possible implementation, but it's plenty fast
 * enough for my purposes.
 *
 * This routine uses only fabs(), fmod(), and floor(); it should be compatible
 * with any system that has a standard C math library.
 */

#ifdef TEST
#include <stdio.h>
#endif

extern double fabs();
extern double fmod();
extern double floor();

void
uformat(buf, val, fmt)
    char *fmt, *buf;
    double val;
{
    double decval;
    int didlead, didsign, pad, signum, overflow;
    register char *fmtp, *bufp, *decp;
    char tbuf[1024];

    signum = (val < 0.0);
    val = fabs(val);
    for (decp = fmt; *decp; decp++)
	if (*decp == '.')
	    break;
    /*
     * Make a first pass to calculate a rounding value.
     */
    decval = 0.5;
    for (fmtp = decp; *fmtp; fmtp++)
    {
	switch (*fmtp)
	{
	case '*':
	case '$':
	case '-':
	case '+':
	case '(':
	case '#':
	case '&':
	case '<':
	    decval /= 10.0;
	    break;
	}
    }
    val += decval;
    fmtp = decp;
    decval = val - floor(val);
    val = floor(val);
    pad = 0;
    didlead = 0;
    didsign = 0;
    bufp = tbuf;
#ifdef TEST
    fprintf(stderr, "fmt = %.*s, decp = %s, val = %s%.14g, decval = %.14g\n",
	    (decp - fmt), fmt, decp, (signum? "-": ""), val, decval);
#endif
    while (fmtp != fmt)
    {
	switch (*--fmtp)
	{
	case '#':
	case '<':
	    if (val < 1.0)
	    {
		if (*fmtp == '<')
		    pad++;
		else
		    *bufp++ = ' ';
		break;
	    }
	    /*FALLTHROUGH*/
	case '&':
	    *bufp++ = (int) fmod(val, 10.0) + '0';
	    val /= 10.0;
	    break;
	case '*':
	    if (val >= 1.0)
	    {
		*bufp++ = (int) fmod(val, 10.0) + '0';
		val /= 10.0;
		break;
	    }
	    *bufp++ = (didlead? ' ': '*');
	    didlead = 1;
	    break;
	case '$':
	    if (val >= 1.0)
	    {
		*bufp++ = (int) fmod(val, 10.0) + '0';
		val /= 10.0;
		break;
	    }
	    *bufp++ = (didlead? ' ': '$');
	    didlead = 1;
	    break;
	case '-':
	    if (val >= 1.0)
	    {
		*bufp++ = (int) fmod(val, 10.0) + '0';
		val /= 10.0;
		break;
	    }
	    *bufp++ = (didsign? ' ': (signum? '-': ' '));
	    didsign = 1;
	    break;
	case '+':
	    if (val >= 1.0)
	    {
		*bufp++ = (int) fmod(val, 10.0) + '0';
		val /= 10.0;
		break;
	    }
	    *bufp++ = (didsign? ' ': (signum? '-': '+'));
	    didsign = 1;
	    break;
	case '(':
	    if (val >= 1.0)
	    {
		*bufp++ = (int) fmod(val, 10.0) + '0';
		val /= 10.0;
		break;
	    }
	    *bufp++ = (didsign? ' ': (signum? '(': ' '));
	    didsign = 1;
	    break;
	case ')':
	    *bufp++ = (signum? ')': ' ');
	    break;
	case ',':
	    *bufp++ = (val < 1.0? ' ': ',');
	    break;
	default:
	    *bufp++ = *fmtp;
	}
    }
    overflow = (val >= 1.0);
    while (bufp-- != tbuf)
	*buf++ = (overflow? '*': *bufp);
    /*
     * Decimals turn out to be easy, since we can parse forward and all the
     * potential digit chars can be treated as "&".  Also, extracting digits
     * is done via (decval *= 10.0; floor(decval)) instead of slow fmod().
     */
    while (*decp)
    {
	if (overflow)
	    *buf++ = '*';
	else
	{
	    switch (*decp)
	    {
	    case '*':
	    case '$':
	    case '-':
	    case '+':
	    case '(':
	    case '#':
	    case '&':
	    case '<':
		decval *= 10.0;
		*buf++ = (int) floor(decval) + '0';
		decval -= floor(decval);
		break;
	    case ')':
		*buf++ = (signum? ')': ' ');
		break;
	    default:
		*buf++ = *decp;
		break;
	    }
	}
	decp++;
    }
    while (pad--)
	*buf++ = (overflow? '*': ' ');
    *buf = '\0';
}

#ifdef TEST

extern double atof();

main(argc, argv)
    char **argv;
{
    char buf[1024];

    if (argc != 3)
    {
	fprintf(stderr, "usage: %s format-string value\n", argv[0]);
	exit(1);
    }
    uformat(buf, atof(argv[2]), argv[1]);
    puts(buf);
    exit(0);
}

#endif
