static char S_timerc[]=
	"@(#)timer.c, 1.4 - Gregg Wonderly@a.cs.okstate.edu  -  17:10:52, 3/1/88";

/***************************************************************************
 *
 *		The routines in this file make up a generalized timer facility
 *	that maintains a list of timer entries, and sorts them according to
 *	their estimated expire times.  Thus calls to set_timer() can be made
 *	arbitrarily, and the times passed will be the elapsed time from now
 *	that the timer should go off.  set_timer() accepts as arguments the
 *	delta time from now that the timer should go off, the address of a
 *	routine to call, and a parameter to pass to that routine.  Typically
 *	this parameter qualifies the timer entry being serviced.
 *
 *		The second routine is called during servicing of the SIGALRM.  It
 *	processes all requests that expire at that time, and then set up the
 *	next alarm.
 *
 **************************************************************************/

/*  System includes.  */

#include	<stdio.h>
#include	<errno.h>
#include	<sys/types.h>
#include	<sys/ipc.h>
#include	<sys/msg.h>
#include	<signal.h>
#include	<setjmp.h>

typedef struct TIMER {
	unsigned delta_t;		/*  Time in seconds till this entry is due.  */
	union {
		long l_key;
		char *p_key;
	} u_t;
	int (*routine)();
	struct TIMER *next;
} TIMER, *TIMEPTR;

static TIMEPTR t_queue = NULL, holdq = NULL;
static int hold_timers = 0, done_timer = 0;

static int alarm_time();

did_timer()
{
	register int i = done_timer;

	done_timer = 0;

	return (i);
}

/*
 *  Process all pending timer entries.
 */

release_timer ()
{
	TIMEPTR p, q;

	hold_timers = 0;
	for (p = holdq; p != NULL;) {
		if (p->routine != NULL)
			(*(p->routine))(p->u_t.p_key);
		q = p->next;
		free (p);
		p = q;
	}
	holdq = NULL;
}

hold_timer ()
{
	hold_timers = 1;
}

/*
 *	Cancel a timer request by NULLing the routine pointer.  Quick and dirty.
 */

can_timer (rout, key)
	int (*rout)();
	char *key;
{
	TIMEPTR p;
	int left;

	/*  Get the time left till the current entry expires.  */

	left = alarm(0);

	/*  Correct the first entry to be up to date with elapsed time.  */

	if (t_queue != NULL)
		t_queue->delta_t = left;

	/*  Find the requested entry, and mark it.  */

	for (p=t_queue; p != NULL; p=p->next) {
		if (p->routine == rout && p->u_t.p_key == key) {
			p->routine = NULL;
			break;
		}
	}

	if (p == NULL) {
		alarm (left);
		return (-1);
	}

	/*
	 *  Restart the alarm on the next entry so that it doesn't go off until
	 *	necessary.
	 */

	for (p=t_queue; p != NULL; p=p->next) {

		/*  If we did not nullify the first entry then use it.  */

		if (p->routine != NULL) {

			/*
			 *	If the next available slot happens to be the second or later
			 *	entry in a group that expires at the same time, then we must
			 *	use the value of `left' as the alarm time, NOT the zero value
			 *	in the delta_t slot, which would cause the alarms to be
			 *	canceled
			 */

			if (p->delta_t == 0)
				alarm (p->delta_t = left);
			else
				alarm (p->delta_t);
			break;
		} else {

			/*  Otherwise, delete the first entry, and look at the next.  */

			t_queue = p->next;
			free (p);
			p = t_queue;
		}
	}

	return (0);
}

/*
 *	Set a new timer request to go off after the interval passed expires.
 */

set_timer (intv, rout, key)
	int intv;
	int (*rout)();
	char *key;
{
	TIMEPTR p, prevp = NULL, newp;
	unsigned t_left;

	/*  Get the remaining time, and put the alarm on hold.  */

	if (t_queue != NULL)
		t_left = alarm(0);
	else
		t_left = 0;

	/*  Get a new timer queue entry.  */

	if ((newp = (TIMEPTR) malloc (sizeof (TIMER))) == NULL)
		return (-1);

	/*  Correct the first entry to be up to date with elapsed time.  */

	if (t_queue != NULL)
		t_queue->delta_t = t_left;

	/*
	 *  Search for the insertion point.  >= makes the ordering consistant.
	 *	with the order of the calls to set_timer ().
	 */

	for (p=t_queue; p != NULL && intv >= p->delta_t; p = p->next) {
		prevp = p;
		intv -= p->delta_t;
	}

	/*  Is this the first entry, or insertion at end of list?  */

	if (p == NULL) {

		/*  If first entry, then put it in.  */

		if (prevp == NULL) {
			t_queue = newp;

		/*  If last, then just insert the entry.  */

		} else
			prevp->next = newp;

		newp->next = NULL;

	/*  If insertion at beginning or in middle.  */

	} else {

		/*  If insertion at beginning then, put in entry, and redo alarm.  */

		if (prevp == NULL) {
			newp->next = t_queue;
			t_queue = newp;

		/*  Otherwise, insertion in the middle, so don't touch alarm time.  */

		} else {
			newp->next = p;
			prevp->next = newp;
		}
		p->delta_t -= intv;
	}

	newp->delta_t = intv;
	newp->u_t.p_key = key;
	newp->routine = rout;

	/*  Reset the alarm to go off later.  */

	signal (SIGALRM, alarm_time);
	alarm (t_queue->delta_t);

	return (0);
}

/*
 *		Called when an alarm goes off, picks off the first timer entry, queues
 *	the next one, and then processes the one picked off.
 */

static int alarm_time (sig)
	unsigned sig;
{
	TIMEPTR p;

	done_timer = 1;
	signal (SIGALRM, alarm_time);

	do {
		p = t_queue;
		if ((t_queue = t_queue->next) != NULL && (t_queue->delta_t > 0)) {

			/*  Set the next alarm time.  */

			alarm (t_queue->delta_t);
		}

		/*  If holding timer entries, then place this one on the list.  */

		if (hold_timers) {
			p->next = holdq;
			holdq = p;
		} else {

			/*  Invoke the routine requested.  */

			if (p->routine != NULL)
				(*(p->routine))(p->u_t.p_key);

			/*  Free the member.  */

			free (p);
		}
	} while (t_queue != NULL && t_queue->delta_t == 0);

	/*  And back out.  */
}
