static	char	id[] = "@(#)setenv.c 2.1 89/02/22 Maarten Litmaath";

/*
 * setenv.c
 *
 * Sorted environment package.
 *
 * char *setenv(char *var, char *value);
 * Returns a pointer to the new "var=value" string if `value' has been
 * assigned to `var', (char *) NULL if the number of environment variables
 * exceeds MAX_ENV, or `var' is a NULL pointer or a malloc error occurred.
 * If `value' is a NULL pointer, the empty string is assigned to `var'.
 *
 * int unsetenv(char *var);
 * If `var' is the NULL pointer, the complete environment is unset.
 * Returns -1 if the initial number of environment variables exceeds MAX_ENV
 * or a malloc error occurred, else 0.
 *
 * int _envc;
 * The current number of environment variables.
 */

#include	"setenv.h"
#include	<stdio.h>

static	char	*envbuf[MAX_ENV] = { 0 };
static	int	initialized = 0, initenv(), envsearch();
int	_envc = 0;
extern	void	free();


char	*setenv(var, value)
char	*var, *value;
{
	extern	char	*malloc();
	char	**env, *buf;
	int	n;


	if (!initialized && initenv() == -1)
		return NULL;

	if (!var)
		return NULL;

	if (!value)
		value = "";

	n = strlen(var);

	if (!(buf = malloc(n + strlen(value) + 2)))
		return NULL;

	(void) sprintf(buf, "%s=%s", var, value);

	if (envsearch(var, n, &env) == 0) {
		free(*env);			/* unsetenv old value */
		*env = buf;			/* setenv new value */
	} else
		if (_envc == MAX_ENV)
			return NULL;
	else
		if (env == envbuf + _envc++) {
			*env++ = buf;
			*env = 0;
		}
	else {					/* *env > var */
		register char	**p, **q;


		p = envbuf + _envc;
		q = p++;
		while (q > env)
			*--p = *--q;		/* shift down */
		*env = buf;			/* insert new var */
	}

	return buf;
}


int	unsetenv(var)
char	*var;
{
	register char	**p, **q;
	char	**env;


	if (!var)
		if (!initialized) {
			environ = envbuf;
			return 0;
		} else {
			for (p = envbuf; *p; )
				free(*p++);
			*envbuf = 0;
			_envc = 0;
			return 0;
		}

	if (!initialized && initenv() == -1)
		return -1;

	if (envsearch(var, strlen(var), &env) == 1)
		return 0;

	free(*env);			/* unsetenv var */

	p = env++;
	q = env;
	while (*p++ = *q++)		/* shift up rest of environment */
		;
	--_envc;

	return 0;
}


/*
 * int initenv();
 * Copy environment to private area, sort the copy, set environ to copy,
 * initialize _envc.
 * Return -1 if the environment exceeds MAX_ENV or a malloc error occurred,
 * else 0.
 */

static	int	initenv()
{
	register char	**p = environ, **env = envbuf;
	extern	char	*malloc(), *strcpy();
	extern	void	qsort();
	static	int	error = 0;
	int	istrcmp();


	if (error == -1)
		return -1;

	if (p)
		while (*p && p < environ + MAX_ENV)
			if (!(*env = malloc(strlen(*p) + 1)))
				return error = -1;
			else
				(void) strcpy(*env++, *p++);

	if (p >= environ + MAX_ENV)
		return error = -1;

	*env = 0;
	_envc = env - envbuf;
	qsort((char *) envbuf, _envc, sizeof *envbuf, istrcmp);
	environ = envbuf;
	initialized = 1;
	return 0;
}


/*
 * int envsearch(char *var, int n, char ***pos);
 * Binarily search environment for `var', whose length is `n'.
 * If it is present, `*pos' is set to the address of `var' in the environment
 * and 0 is returned, else `*pos' is set to the address of the first variable
 * lexicographically greater than `var', or to the end of the environment,
 * and 1 is returned.
 */

static	int	envsearch(var, n, pos)
register char	*var;
register int	n;
char	***pos;
{
	register char	**env, **first = envbuf, **last = envbuf + _envc;
	register int	m;
	extern	int	strncmp();


	while (first < last) {
		env = first + ((last - first) >> 1);
		if ((m = strncmp(*env, var, n)) < 0) {
			first = env + 1;
			continue;
		}
		if (m > 0) {
			last = env;
			continue;
		}
		if ((m = (*env)[n] - '=') == 0) {
			*pos = env;
			return 0;
		}
		if (m < 0) {
			first = env + 1;
			continue;
		}
		last = env;
	}

	*pos = last;
	return 1;
}


static	int	istrcmp(p, q)			/* indirect strcmp */
char	**p, **q;
{
	register char	*s1 = *p, *s2 = *q;

	while (*s1 == *s2++)
		if (!*s1++)
			return 0;
	return *s1 - *--s2;
}


#ifdef	STANDALONE

main(argc)
int	argc;
{
	void	set(), unset(), printenv();


	printenv();

	if (argc > 1)
		unset((char *) 0);

	unset("SHELL");
	unset("PATH");
	set("SHELL", "/foo/bar/baz");
	set("FOO", "BAR");
	unset("FOOO");
	unset("FO");
	unset((char *) 0);
	set("ZORK", (char *) 0);
	set("TMP", "/tmp");
}


void	set(p, q)
char	*p, *q;
{
	void	printenv();


	printf("%s -> %s\n\n", p ? p : "(null)", (q = setenv(p, q)) ? q : "?");
	printenv();
}


void	unset(p)
char	*p;
{
	void	printenv();


	printf("%s: %d\n\n", p ? p : "(null)", unsetenv(p));
	printenv();
}


void	printenv()
{
	register char	**env;


	for (env = environ; *env; ++env)
		printf("%s\n", *env);
	printf("\n_envc=%d\n\n", _envc);
}

#endif	STANDALONE
