
/*
 * scpp.h - common declarations for the selective C preprocessor, scpp.
 *
 * Copyright (c) 1985 by
 * Tektronix, Incorporated Beaverton, Oregon 97077
 * All rights reserved.
 *
 * Permission is hereby granted for personal, non-commercial
 * reproduction and use of this program, provided that this
 * notice and all copyright notices are included in any copy.
 */

# define TRUE	1
# define FALSE	0

/*
 * sawerror - "some error was processed" If true, scpp exits non-zero.
 * Set by the error printout routines, examined when exiting.
 */
#ifdef VARS
int sawerror;
#else
extern int sawerror;
#endif

# define BSIZE	512	/*
			 * # of bytes per read -- controls how quickly
			 * istk[] is consumed.
			 */

/*
 * PENDSIZ is a tunable parameter -- it is the largest number of characters
 *  which can be waiting to be output.  This number sets a limit on:
 *  1) the longest comment;
 *  2) the largest invocation of a macro with parameters, i.e. the number
 *    of characters between the '(' and the ')'.
 *  3) the longest preprocessor control line, e.g. #define....
 * PENDSIZ also controls the input stack size, ISTK.
 *
 * Pend[] is the pending output buffer.
 *
 * Nxtout points to where within pend[] to put the next token scanned.
 * Nxtout is advanced by questr() and quec(),
 *  the primitives for putting stuff in pend[],
 * and is moved backward by dispose() and outpend(),
 *  the primitives for getting stuff out of pend[].
 *
 * Curtext points to the start of the text within pend[] of
 * the current token.  Set by gtok() and gintok().
 * For anyone who uses gtok() or gintok() to get
 *  a token, the limits of the text of the resultant
 *  token are curtext and nxtout. (be aware that the
 *  text of anything in pend[] may contain imbedded
 *  ATTN bytes.)
 */

# define PENDSIZ 8000
# define PENDHIGH 512	/* highwater mark for flushing pend[]	*/
#ifdef VARS
char pend[PENDSIZ];
char *nxtout;
char *curtext;
#else
extern char pend[];
extern char *nxtout;
char *curtext;
#endif
extern char *dispose();
#define outpend() (nxtout < &pend[PENDHIGH] ? 0 : writepend())

/*
 * filestk - the stack containing the state of the current file
 */

struct afile {
	int	af_fd;		/* the open file's file-descriptor	*/
	char	*af_name;	/* the name of the file (dynamic alloc)	*/
	int	af_line;	/* the current line in the file		*/
	int	af_raw;		/*
				 * "scanning unprocessed data rather than
				 *  pushed-back data".
				 * Used to count input lines.
				 * Also used to prevent
				 *  interpretation of "#if" expressions whose
				 *  truth or falsehood does not depend on
				 *  interpreting macros (e.g. #if '\377' > 0). 
				 */
	int	af_hide;	/*
				 * "do not output anything for this file."
				 * This file is the result of an uninterpreted
				 * "# include".
				 */
};

#define FILESIZ 11	/* max # of include files + 1 (the original file) */
#ifdef VARS
struct afile filestk[FILESIZ];
struct afile *curfile;	/* the current file.  Initially = &filestk[-1]	*/
#else
extern struct afile filestk[];
extern struct afile *curfile;
#endif

/*
 * ISTKSIZ is the size of the input/pushback stack.
 *  It contains up to one block of data for each pending file plus
 *  one pending token.
 * The input stack grows down from istk[ISTKSIZ - 1].
 *
 * Nxtin points to the next char to read from istk[].
 * Characters are popped from the stack by nxtc()
 * and are pushed back on the stack by unc() and
 * pushmac().
 */

# define ISTKSIZ (FILESIZ * BSIZE + PENDSIZ)
#ifdef VARS
char istk[ISTKSIZ];
char *nxtin;
#else
extern char istk[];
extern char *nxtin;
#endif
extern char nxtc();
extern char *pushmac();
#define unc(c) (nxtin-- < &istk[0] ? over() : (*nxtin = c))

/*
 * ATTN appears in the input stack to notify nxtc() of some condition,
 *  in the output queue to notify dispose() or outpend() of some condition,
 *  or in the value of a macro to notify gintok() of some condition.
 * ATTN means that the next byte contains a control code.
 * Input control codes are:
 *  AT_EPUSH	- end of pushed-back data.  what follows has not been
 *		 scanned before.
 *  AT_EBLK	- end of block.  read another block from the current file.
 * Output control codes are:
 *  AT_OUTOFF	- disable further output.
 *  AT_OUTON	- enable output.
 * Macro value control codes are formal parameter numbers and are not defined.
 *
 * note: to avoid breaking string operations and newline recognition,
 *  do not add an ATTN control code which has a value of '\0', '\\', or '\n'.
 */

#define ATTN		'\376'	/* this char must not appear in any file */
#define AT_EPUSH	'\001'
#define AT_EBLK		'\002'

#define AT_OUTOFF	'\006'
#define AT_OUTON	'\007'

/*
 * Ninterp - number of interpretations.  Incremented each time
 *  gintok() interprets a macro.  Since there is no
 *  overflow detection, ninterp can be used only to
 *  see if some interpretation took place -- not to
 *  count the interpretations (e.g. "oldnint != ninterp"
 *  works, but "cnt = ninterp - oldnint" may fail).
 * Used in conjunction with af_raw to prevent
 *  interpretation of  #if's which are always true
 *  or false without any macro interpretation (e.g.
 *  "#if '\377' > 0").
 */

#ifdef VARS
int ninterp;
#else
extern int ninterp;
#endif

/*
 * Falsecnt - number of currently false #if's;
 * Hidecnt  - current number of uninterpreted #include's.
 * Collectively, these variables are used to determine when
 *  to enable or disable output.
 */

#ifdef VARS
int falsecnt;
int hidecnt;
#else
extern int falsecnt;
extern int hidecnt;
#endif

/*
 * ifstk[] contains flags describing the state of all currently active #if's.
 * curif points to the currently active #if within the stack.
 * The stack grows upward, starting at ifstk[-1].
 */

#define IF_INIF		'\001'	/* "in the 'if' clause rather than 'else'" */
#define IF_TRUE		'\002'	/* "this if is currently true"		   */
#define IF_FALSE	'\004'	/* "this if is currently false"		   */
	/* uninterpreted #if statements are neither true nor false.	   */
#define IFSIZ	100		/* maximum number of nested #if's	   */
#ifdef VARS
char ifstk[IFSIZ];
char *curif;
#else
extern char ifstk[];
extern char *curif;
#endif

/*
 * expparse - "currently parsing a #if expression".
 *  Used to prevent interpretation of the macro "defined()" outside
 *  #if expressions.
 */

#ifdef VARS
int expparse;
#else
extern int expparse;
#endif

/*
 * the next set of definitions are values of parameters to pushfile().
 *  PF_NOLOOK	- the filename was given on the command line.  Don't
 *		 search any directories for it.
 *  PF_NODOT	- the include filename was enclosed in '<' and '>'.
 *		 Do not search the current directory (dot) for the it.
 *  PF_DOT	- the include filename was enclosed in double-quotes.
 *
 *  PF_HIDE	- the file is not to be interpreted (I.e. is an include file).
 *		 Do not output anything while processing this file.
 *  PF_NOHIDE	- the file is to be interpreted.
 */

# define PF_NOLOOK	(-1)
# define PF_NODOT	0
# define PF_DOT		1

# define PF_HIDE	TRUE
# define PF_NOHIDE	FALSE

/*
 * savcom - "save comments and whitespace"
 *  If false, comments and leading and trailing whitespace are removed
 *   from interpreted macro definitions.
 */

#ifdef VARS
int savcom;
#else
extern int savcom;
#endif

/*
 * catlist - the list of files to process; I.E. the filenames from
 *  the command line.  A zero pointer marks the end of the list.
 * nxtfile - points to the next element of catlist[] to be processed.
 */

# define CLSIZ		100
#ifdef VARS
char *catlist[CLSIZ];
char **nxtfile;
#else
extern char *catlist[];
extern char **nxtfile;
#endif

/*
 * dirlist - the list of directories to search for an include file.
 *  I.E. all the -I directories from the command line + /usr/include.
 *  (the search of the current directory of the file is handled separately.)
 *  A zero pointer marks the end of the list.
 */

#define DLSIZ		100
#ifdef VARS
char *dirlist[DLSIZ];
#else
extern char *dirlist[];
#endif

/*
 * The symbol table.  All macros are stored in this table.
 */

struct amacro {
	char *am_name;	/*
			 * the name of this macro (dynamically allocated).
			 * An am_name value of 0 means this slot is empty.
			 * All macros to be interpreted are allocated slots
			 * before any files are scanned.  #define and #undef
			 * do not allocate or free symbol-table slots.
			 */
	int am_npar;	/* number of parameters.  -1 == no parameters.	*/
	char *am_val;	/*
			 * the value (replacement text) of the macro.
			 * (dynamically allocated.)
			 * An am_val value of 0 means that this macro is not
			 * currently defined.
			 *
			 * am_val points to the null-terminator of the
			 * replacement text.  The replacement text is to be
			 * read backwards from (am_val - 1) until a null-
			 * terminator is found at the other end.
			 * An ATTN byte followed (well, preceeded if scanning
			 *  forward) by a one-byte integer parameter number
			 *  is replaced when expanding this macro by the
			 *  corresponding actual parameter.
			 * To avoid breaking string operations on val strings,
			 * parameter numbers begin at 1 rather than 0
			 *
			 * A visual example may help:
			 *   #define goop(name) hello there name people
			 *  results in a sym[] slot containing:
			 *
			 *  am_name:-------------|
			 *			 V
			 *			 goop\0
			 *  am_npar: 1
			 *  am_val:----------------------------|
			 *				       V
			 *	\0hello there <1><ATTN> people\0
			 */
};

#define SYMSIZ	1001
#ifdef VARS
struct amacro sym[SYMSIZ];
#else
extern struct amacro sym[];
#endif

extern struct amacro *findmac();
extern char *savtok();
extern int gintok();
extern int gtok();

/*
 * magicval - This (uninitialized) character is used to
 *  recognize special macro's (e.g. "defined()").
 * An am_val field of &magicval marks a macro
 *  as special -- it cannot be undef'ed or redefined,
 *  and macro expansion in gintok() recognizes it.
 */

#ifdef VARS
char magicval;
#else
extern char magicval;
#endif

#define MAXPARMS 40	/* max number of formal parameters to a macro	*/

/*
 * the keyword structure - one of these describes each preprocessor keyword.
 * see ctrl.c for the keyword array, key[].
 */

struct akeyword {
	char *ak_name;		/* name of this keyword (used to set ak_sym) */
	int (*ak_proc)();	/* procedure to interpret this directive     */
	struct amacro *ak_sym;	/*
				 * pointer to the symbol table slot for this
				 * keyword.  Used to recognise the keyword.
				 * All keywords in this list are effectively
				 * "-M"ed when scpp is invoked.  They are
				 * never defined.
				 *   This field is initialized at runtime.
				 */
};
extern struct akeyword *findkey();
extern char *strcpy();
