
/*
 * MENU -- a menu preprocessor for UNIX by Richard Conn 
 *
 * MENU allows the user to create menu files, containing screen displays
 * and command lines which are to be issued in response to
 * single-character commands presented by the user.  Once a user
 * issues a single-character command, MENU executes his command line
 * (as a shell procedure) and then resumes control (in most cases). 
 *
 * enhacements involving piping to a shell and changing the current *
 * working directory were done by Orest Jarosiewicz * [odj] 
 *
 * Compilation: cc -o menu menu.c 
 *
 */

#define versmaj 2		/* major version */
#define versmin 5		/* minor version */

/*
 * Version History 
 *
 * Version 1.0 - Richard Conn 
 *
 * 2.0 - Richard Conn. Rather than creating a shell procedure file and
 * then executing it via an execl, the program was redesigned to use
 * the system() library function, which I feel is a much better,
 * cleaner design. No intermediate command files are created now.
 * NOTE:  This redesign caused many problems because of what I
 * believe to be an undocumented bug with system() -- if an global
 * variables of type int are declared, calling system() results in an
 * error message about an invalid instruction exec and then a core
 * dump.  The program redesign removed all global variables. 
 *
 * 2.1 - Orest Jarosiewicz. Piping to a shell and changing the CWD
 * added. 
 *
 * 2.2 - Richard Conn. Minor code refinement, consistent style
 * reimposed, general cleanup, definition of WAIT changed to a prefix 
 *
 * 2.3 - Richard Conn. Corrected code to run under 4.3BSD UNIX on my
 * SUN3: changed kill -16 pid to kill pid and added a fork() to cause
 * a child to be run which actually sent the command to the shell
 * over the pipe (the child went to sleep via pause() after the
 * command was sent).  Ran INDENT on it. 
 *
 * 2.4 - Richard Conn. Added named menus. 
 *
 * 2.5 - Richard Conn.  Fixed bug in prcmd subshell return code
 * which was pointed out by Frank Wancho.
 *
 */

#define FALSE   0
#define TRUE    ~FALSE
#define SYS5    FALSE		/* standard definition */

#define HASH    '#'		/* menu separator */
#define GOTO    ':'		/* goto command in menu text */
#define WAIT    '!'		/* Wait command in menu text */
#define QUOTE   '\''		/* quote for user input */
#define SUBSHELL '~'		/* invoke command in subshell */
#define CHDIR   '='		/* change working directory */
#define llimit  23		/* number of lines in screen display */
#define EOB     0x7f		/* end of buffer marker */
#define ESC     0x1b		/* user abort (if allowed) */
#define BEL     '\7'
#define LF      '\n'
#define SCRSIZ  4000		/* size of buffer for screen display */
#define CMDSIZ  8000		/* size of buffer for command lines */
#define ARBSIZ  200		/* buffer size for builds --
				 * arbitrary */
#define NAMELEN 200		/* length of menu name buffer */

#include <stdio.h>
#include <ctype.h>

#ifdef SYS5
#include <sgtty.h>
#endif

/* options structure  */
#include <sys/signal.h>

int             ldisp;		/* display on/off */
int             lpage;		/* page on/off */
int             lexit;		/* exit on/off */
char            lname[NAMELEN];	/* name of menu (all caps) */

int             gdisp;		/* display on/off */
int             gpage;		/* page on/off */
int             gexit;		/* exit on/off */
char            gname[NAMELEN];	/* name of menu (all caps) */

main (argc, argv)
    int             argc;
    char           *argv[];
{
    extern char    *getenv ();	/* get environ function */
    char            file[50];	/* menu file name */
    char            inline[400];/* menu command line */
    int             cmd, ccode;	/* user command, ret code */
    int             Wait;	/* Wait flag */
    int             menunum;	/* menu number */
    char            screen[SCRSIZ];	/* screen buffer */
    char            text[CMDSIZ];	/* text buffer */
    char            mname[NAMELEN];	/* text if menu name on
					 * command line */
    char           *rover1, *rover2;
    int             pid;	/* process id */
    FILE           *Shell;
    char           *Shelle;
    char            Cmd[420];	/* Shell points to a pipe to the
				 * shell, cmd holds the commands sent
				 * to it. Shelle points to the
				 * environment var SHELL. [odj] */
    int             dummy;	/* dummy variable used by wait () */

    if (argc == 1) {		/* give user help info if no args */
	help ();
	exit (-1);
    }
    /* process options if any */
    menunum = 1;		/* set to first menu */
    *mname = '\0';		/* set no named menu to start */
    if (*argv[1] == '-') {
	if (isdigit (argv[1][1]))
	    menunum = atoi (&argv[1][1]);
	else {
	    rover1 = &argv[1][1];
	    rover2 = mname;
	    while (*rover1) {
		*rover2++ = islower (*rover1) ? toupper (*rover1) : *rover1;
		rover1++;
	    }
	    *rover2 = '\0';
	}
	if (!menunum)
	    menunum = 1;	/* set to 1 if error in arg */
    }
    /* extract menu file name */
    file[0] = '\0';		/* set no file name */
    if (argc == 2 && *argv[1] != '-')
	strcat (file, argv[1]);
    else if (argc > 2)
	strcat (file, argv[2]);
    else {
	help ();
	exit (-1);
    }

    /* open pipe to shell [odj] */
    Shelle = getenv ("SHELL");
    if (Shelle != NULL && Shelle[0] != '\0')
	strcpy (Cmd, Shelle);
    else
	strcpy (Cmd, "/bin/sh");
    Shell = popen (Cmd, "w");
    if (Shell == NULL) {
	fprintf (stderr, "%s: can't open pipe to %s\n", argv[0], Cmd);
	exit (5);
    }
    pid = getpid ();

    /* get menu from file */
    if (*mname)
	getnmenu (file, mname, screen, text);
    else
	getmenu (file, menunum, screen, text);

    /* process menu */
    while (TRUE) {		/* exit only on ESC from user (if
				 * allowed) */
	if (ldisp)
	    prdisp (screen);	/* print display */
	Wait = FALSE;		/* set no Wait */
	printf ("MENU %d.%d Command%s", versmaj, versmin,
		lexit ? " (ESC to exit)? " : "? ");	/* prompt */
	cmd = getcmd ();	/* get command from user */
	if (lexit && cmd == ESC)
	    exit (0);		/* exit */
	ccode = prcmd (cmd, text, inline, &Wait);	/* get cmnd */
	switch (ccode) {
	case 0:		/* command line */
	    /*
	     * execute command, the wait will notify the parent
	     * process when the shell (child) is done processing this
	     * line 
	     */
	    if (fork () == 0) {
		pid = getpid ();
		sprintf (Cmd, "%s;kill %d", inline, pid);
		fprintf (Shell, "%s\n", Cmd);
		fflush (Shell);
		pause ();	/* infinite wait ... child is killed
				 * by shell */
	    } else
		wait (&dummy);	/* parent waits for child to be
				 * killed */
	    if (Wait) {
		printf ("Strike any key -- ");
		setraw ();
		getchar ();
		clrraw ();
		printf ("\n");
	    }
	    break;
	case 1:		/* goto new menu */
	    menunum = atoi (inline);	/* get menu number */
	    if (!menunum)
		menunum = 1;
	    getmenu (file, menunum, screen, text);
	    break;
	case 2:		/* do a plain "system", for things
				 * like calling vi */
	    system (inline);
	    if (Wait) {
		printf ("Strike any key -- ");
		setraw ();
		getchar ();
		clrraw ();
		printf ("\n");
	    }
	    break;
	case 3:		/* change current directory; this has
				 * to happen within the shell                                                         */
	    if (chdir (inline) != 0) {
		fprintf (stderr, "Can't change to directory %s\n",
			 inline);
		break;
	    }
	    fprintf (Shell, "cd %s\n", inline);
	    fflush (Shell);
	    if (Wait) {
		printf ("Strike any key -- ");
		setraw ();
		getchar ();
		clrraw ();
		printf ("\n");
	    }
	    break;
	case 4:		/* goto new named menu */
	    getnmenu (file, inline, screen, text);
	    break;
	default:		/* invalid command */
	    putchar (BEL);
	    break;
	}
    }
}
/* print display if desired and page if desired  */
prdisp (screen, locops)
    char           *screen;
    struct opt     *locops;
{
    int             linecnt;

    linecnt = 0;		/* init line counter */
    while (*screen != EOB) {
	putchar (*screen);	/* send char */
	if (*screen++ == LF)
	    linecnt++;		/* increment line count */
    }
    if (lpage)			/* page display */
	while (linecnt++ < llimit)
	    printf ("\n");
}

/* get raw command from user  */
getcmd ()
{
    int             c;

    setraw ();			/* set no echo and immed input */
    c = getchar () & 0x7f;	/* get command and mask out MSB */
    clrraw ();			/* restore normal I/O */
    if (c > ' ')
	putchar (c);		/* echo if printing char */
    printf ("\n");		/* new line */
    return (c);
}

/* get and process command from user  */
prcmd (c, text, line, Wait)
    int             c, *Wait;
    char           *text, *line;
{
    char           *scmd (), *cline;
    int             Subshell;

    Subshell = FALSE;		/* no invocation via subshell */
    cline = scmd (c, text);	/* cline pts to command from text */
    *line = '\0';		/* init command line */
    if (!(*cline))
	return (99);		/* command not found */
    if (*cline == WAIT) {	/* WAIT prefix */
	*Wait = TRUE;		/* set flag */
	++cline;		/* pt to next char */
    }
    if (*cline == GOTO) {	/* GOTO command */
	++cline;		/* pt to first digit or letter */
	if (isdigit (*cline)) {
	    while (isdigit (*cline))
		*line++ = *cline++;	/* copy */
	    *line = '\0';	/* set end of string */
	    return (1);		/* goto return code */
	} else {
	    while (*cline > ' ') {
		*line++ = islower (*cline) ? toupper (*cline) : *cline;
		cline++;
	    }
	    *line = '\0';	/* copy and set end of line */
	    return (4);		/* goto named menu return code */
	}
    }
    if (*cline == CHDIR) {	/* check if this a directory change */
	++cline;
	docmnd (cline, line);	/* directive */
	return (3);
    }
    if (*cline == SUBSHELL) {	/* is this a subshell invokation? */
	Subshell = TRUE;	/* (good for running progs like vi) */
	++cline;
    }
    docmnd (cline, line);	/* extract and process command line */
    return (Subshell ? 0 : 2);	/* command line return code */
}

/* scan commands in command buffer for command passed  */
char           *
scmd (cmd, text)
    int             cmd;
    char           *text;
{
    while (*text != EOB) {	/* scan thru buffer */
	if (*text == cmd)
	    return (++text);	/* found command */
	while (*text++ != LF);	/* skip current line */
    }
    return ("");		/* null command if not found */
}

/* build command line  */
docmnd (cline, line)
    char           *cline, *line;
{
    char            userin[ARBSIZ];	/* buffer for build and user
					 * input */

    while (*cline != LF) {	/* advance over line */
	switch (*cline) {
	case QUOTE:
	    ++cline;		/* pt to after quote */
	    while (*cline != LF && *cline != QUOTE)
		putchar (*cline++);	/* print prompt */
	    putchar (' ');	/* print extra space */
	    gets (userin);	/* get input from user */
	    if (*cline == QUOTE)
		cline++;	/* skip ending quote */
	    break;
	default:
	    userin[0] = *cline++;	/* set up string */
	    userin[1] = '\0';
	    break;
	}
	strcat (line, userin);	/* append user input or char */
    }
}

/* extract menu from menu file */
getmenu (file, menunum, screen, text)
    int             menunum;
    char           *file, *screen, *text;
{
    FILE           *fopen (), *fd;
    int             c;

    /* open menu file */
    fd = fopen (file, "r");
    if (fd == NULL) {
	printf ("Menu %s NOT Found\n", file);
	exit (0);
    }
    /* set global options */
    gdisp = FALSE;
    gpage = FALSE;
    gexit = FALSE;
    *gname = '\0';
    if ((c = getc (fd)) == '-')
	setopts (fd, 1);
    else
	ungetc (c, fd);

    /* skip to desired menu */
    skip (fd, menunum);

    /* set local options */
    ldisp = gdisp;
    lpage = gpage;
    lexit = gexit;
    *lname = '\0';
    setopts (fd, 0);

    /* build buffers */
    build (fd, screen, SCRSIZ);	/* build screen display */
    build (fd, text, CMDSIZ);	/* build command buffer */

    /* close menu file */
    fclose (fd);
}

/* extract menu from menu file by its name */
getnmenu (file, menuname, screen, text)
    char           *menuname;
    char           *file, *screen, *text;
{
    FILE           *fopen (), *fd;
    int             c;
    char           *rover1, *rover2;
    int             looping, match;

    /* open menu file */
    fd = fopen (file, "r");
    if (fd == NULL) {
	printf ("Menu %s NOT Found\n", file);
	exit (0);
    }
    /* set global options */
    gdisp = FALSE;
    gpage = FALSE;
    gexit = FALSE;
    *gname = '\0';
    if ((c = getc (fd)) == '-')
	setopts (fd, 1);
    else
	ungetc (c, fd);

    /* get hash */
    if ((c = getc (fd)) != HASH)
	strerror ();

    /* look for desired menu */
    looping = 1;
    while (looping) {

	/* check for end of menu file */
	if ((c = getc (fd)) == HASH) {
	    printf (" Menu Named %s Not Found\n", menuname);
	    strerror ();
	} else
	    ungetc (c, fd);
	/* set local options */
	ldisp = gdisp;
	lpage = gpage;
	lexit = gexit;
	*lname = '\0';
	setopts (fd, 0);

	/* debug printout */
	/* check for menu name match */
	rover1 = lname;
	rover2 = menuname;
	match = 1;
	while (*rover1) {
	    if (*rover1++ != *rover2++) {
		match = 0;
		break;
	    }
	}
	if (match && (*rover2 == '\0'))
	    match = 1;
	else
	    match = 0;

	/* done looping if match */
	if (match)
	    looping = 0;
	else
	    skipmenu (fd);
    }

    /* build buffers */
    build (fd, screen, SCRSIZ);	/* build screen display */
    build (fd, text, CMDSIZ);	/* build command buffer */

    /* close menu file */
    fclose (fd);
}

/* set options  */
setopts (fd, opts)
    FILE           *fd;
    int             opts;
{
    int             c;
    char           *rover;
    int             disp, page, exit;
    char            name[NAMELEN];

    if (opts) {
	disp = gdisp;
	page = gpage;
	exit = gexit;
    } else {
	disp = ldisp;
	page = lpage;
	exit = lexit;
    }
    *name = '\0';

    while ((c = getc (fd)) != LF)
	switch (c) {
	case 'D':
	case 'd':
	    disp = ~disp;
	    break;
	case 'n':
	case 'N':
	    rover = name;
	    while ((c = getc (fd)) > ' ') {
		*rover++ = islower (c) ? toupper (c) : c;
	    }
	    *rover = '\0';
	    ungetc (c, fd);
	    break;
	case 'P':
	case 'p':
	    page = ~page;
	    break;
	case 'X':
	case 'x':
	    exit = ~exit;
	    break;
	case EOF:
	    premature ();
	    break;
	default:
	    break;		/* skip invalid option */
	}

    if (opts) {
	gdisp = disp;
	gpage = page;
	gexit = exit;
	strcpy (gname, name);
    } else {
	ldisp = disp;
	lpage = page;
	lexit = exit;
	strcpy (lname, name);
    }

}

/* Skip to desired menu and load buffers  */
skip (fd, menunum)
    FILE           *fd;
    int             menunum;
{
    int             mctr;

    /* skip required number of menus  */
    if (getc (fd) != HASH)
	strerror ();		/* first char must be a hash */
    for (mctr = menunum - 1; mctr; mctr--)
	skipmenu (fd);		/* skip menus */
}

/* Skip over menu  */
skipmenu (fd)
    FILE           *fd;
{
    int             c;

    /* advance over screen display  */
    while ((c = getc (fd)) != HASH)
	while (c != LF) {
	    if (c == EOF)
		strerror ();
	    c = getc (fd);
	}

    /* advance over commands  */
    while ((c = getc (fd)) != HASH)
	while (c != LF) {
	    if (c == EOF)
		strerror ();
	    c = getc (fd);
	}
}

/* Build buffers  */
build (fd, buff, size)
    FILE           *fd;
    char           *buff;
    int             size;
{
    int             c;

    while ((c = getc (fd)) != HASH) {	/* loop to end of section */
	while (c != LF) {	/* loop to end of line */
	    if (c == EOF)
		premature ();
	    *buff++ = c & 0x7f;	/* store char */
	    if (!(--size))
		overflow ();
	    c = getc (fd);	/* get next char */
	}
	*buff++ = c;		/* store LF */
	if (!(--size))
	    overflow ();
    }
    *buff = EOB;		/* set end of buffer marker */
}

/* premature EOF exit  */
premature ()
{
    printf ("Premature End of File Encountered\n");
    exit (-1);
}

/* menu structure error  */
strerror ()
{
    printf ("Menu Structure Error\n");
    exit (-1);
}

/* Buffer overflow  */
overflow ()
{
    printf ("Buffer Overflow\n");
    exit (-1);
}

/* Print Help Message  */
help ()
{
    printf ("Syntax: menu [-#] menufile\n");
    printf ("or:     menu [-name] menufile\n");
    printf ("\twhere '#' is the menu number at which to start\n");
    printf ("\tor 'name' is the menu name at which to start\n");
}

/* set byte-oriented I/O  */
setraw ()
{
#ifdef SYS5
    struct sgttyb   ttys;

    if (gtty (0, &ttys) < 0) {	/* get term characteristics */
	printf ("Can't set Byte-Oriented I/O\n");
	exit (-1);
    }
    ttys.sg_flags |= RAW;	/* set RAW */
    ttys.sg_flags &= ~ECHO;	/* set no echo */
    if (stty (0, &ttys) < 0) {	/* set term characteristics */
	printf ("Can't set Byte-Oriented I/O\n");
	exit (-1);
    }
#endif
}

/* restore normal operation for I/O  */
clrraw ()
{
#ifdef SYS5
    struct sgttyb   ttys;

    if (gtty (0, &ttys) < 0) {	/* get term characteristics */
	printf ("Can't restore normal I/O\n");
	exit (-1);
    }
    ttys.sg_flags &= ~RAW;	/* clear RAW */
    ttys.sg_flags |= ECHO;	/* enable ECHO */
    if (stty (0, &ttys) < 0) {	/* set term characteristics */
	printf ("Can't restore normal I/O\n");
	exit (-1);
    }
#endif
}
