/* CHK=0x8201 */
/*+--------------------------------------------------------------------------
	nlist.c - peruse /unix symbol table, optionally dumping kmem
	wht@n4hgf.Mt-Park.GA.US

  This is a command-level interface to nlist(3)/nlist(S) for
  exploratory hacking. 

  Compile by:     cc -O -o nlist nlist.c

  Root is usually the only user who can even read /unix to nlist
  but certainly /dev/kmem should be protected as SOP.

  Usage: nlist ksym[:length] ...

  where ksym is a kernel symbol and length is an optional parameter
  indicating the length of a block of data to dump from /dev/kmem
  for example:

$ nlist proc:0x20 sio_tty:010 v:23 ttyhog doo_dad 
     name                         type  class   value
--------------------------------  ----  -----  --------
proc                              0000   02    d007cda0
sio_tty                           0000   02    d007ba0c
v                                 0000   02    d0072214
ttyhog                            0000   02    d00741a8
doo_dad: not found

proc addr=d007cda0 length=00000020
D007CDA0 01 00 00 14 19 20 00 00 00 00 00 00 00 00 00 00 | ..... .......... |
D007CDB0 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 | ................ |

sio_tty addr=d007ba0c length=00000008
D007BA0C 00 00 00 00 00 00 00 00                         | ........         |

v addr=d0072214 length=00000017
D0072214 58 02 00 00 32 00 00 00 C8 00 00 00 40 DC 09 D0 | X...2...H...@\.P |
D0072224 C8 00 00 00 34 C4 08                            | H...4D.          |

---------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:07-17-1990-13:43-wht-adapt xlist */

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <nlist.h>

#define UNIX_KERNEL "/unix"

#define UINT8 unsigned char
#define UINT16 unsigned short
#define UINT32 unsigned long

int fd_kmem;

#define dump_putc(ch)		fputc((ch),stdout)
#define dump_puts(str)		fputs(str,stdout)

/*+-----------------------------------------------------------------------
	hex_dump#... subservient routines
------------------------------------------------------------------------*/
void hex_dump4(int4)
UINT8	int4;
{
	int4 &= 15;
	dump_putc((int4 >= 10) ? (int4 + 'A' - 10) : (int4 + '0'));
}

void hex_dump8(int8)
UINT8	int8;
{
	hex_dump4(int8 >> 4);
	hex_dump4(int8);
}

void hex_dump16(int16)
UINT16	int16;
{
	hex_dump8(int16 >> 8);
	hex_dump8(int16);
}

void hex_dump32(int32)
UINT32	int32;
{
	hex_dump16((UINT16)(int32 >> 16));
	hex_dump16((UINT16)int32);
}

/*+-----------------------------------------------------------------
	hex_dump(str,len)
------------------------------------------------------------------*/
void
hex_dump(str,len,offset)
UINT8	*str;
int len;
long offset;
{
	register int itmp;
	register int istr = 0;

	while(istr < len)
	{
		hex_dump32(offset + istr);
		for(itmp = 0; itmp < 16; ++itmp)
		{
			dump_putc(' ');
			if( istr + itmp >= len)
			{
				dump_putc(' ');
				dump_putc(' ');
				continue;
			}
			hex_dump8(str[istr + itmp]);
		}
		dump_puts(" | ");
		for(itmp = 0; itmp < 16; ++itmp)
		{
			register char dchar;
			if(istr + itmp >= len)
				dump_putc(' ');
			else
			{
				dchar = str[istr + itmp] & 0x7F;
				dump_putc(((dchar >= ' ') && (dchar < 0x7f)) ? dchar : '.' );
			}
		}
		dump_puts(" |\n");
		istr += 16;
	}   /* end of while(istr < len) */

}	/* end of hex_dump */

/*+-------------------------------------------------------------------------
	dump_kmem(seekpos,length)
--------------------------------------------------------------------------*/
void
dump_kmem(seekpos,length)
long seekpos;
int length;
{
	char *kmemdata;
	char *malloc();
	long lseek();

	if((kmemdata = malloc(length)) == NULL)
	{
		printf("cannot allocate %d bytes\n",length);
		return;
	}

	if(lseek(fd_kmem,seekpos,0) == -1L)
	{
		fprintf(stderr,"cannot seek kmem (%d) to %08lx",fd_kmem,seekpos);
		perror("");
		free(kmemdata);
		return;
	}

	if(read(fd_kmem,kmemdata,length) != length)
	{
		perror("kmem read");
		free(kmemdata);
		return;
	}

	hex_dump(kmemdata,length,seekpos);
	free(kmemdata);

}	/* end of dump_kmem */

/*+-------------------------------------------------------------------------
	open_kmem()
--------------------------------------------------------------------------*/
void
open_kmem()
{
	if((fd_kmem = open("/dev/kmem",O_RDONLY,0)) < 0)
	{
		perror("/dev/kmem");
		exit(1);
	}
}	/* end of open_kmem */

/*+-------------------------------------------------------------------------
	main(argc,argv,envp)
--------------------------------------------------------------------------*/
main(argc,argv,envp)
int argc;
char	**argv;
char	**envp;
{
	register int iargv;
	struct nlist *nlst;
	register struct nlist *nn;
	register int *count;
	char *cptr;
	char *strrchr();
	char *calloc();
	int need_kmem = 0;

	if(argc < 2)
		exit(1);

#ifdef REPEAT_ARGV
	for(iargv = 0; iargv < argc; iargv++)
		printf("%s ",argv[iargv]);
	printf("\n");
#endif

	if((nlst = (struct nlist *)calloc(argc,sizeof(struct nlist))) ==
	    (struct nlist *)0)
	{
		fprintf(stderr,"memory not available\n");
		exit(1);
	}

	if((count = (int *)calloc(argc,sizeof(int))) == (int *)0)
	{
		fprintf(stderr,"memory not available\n");
		exit(1);
	}

	nn = nlst;
	for(iargv = 1; iargv < argc; iargv++)
	{
		if(cptr = strrchr(argv[iargv],':'))
		{
			*cptr++ = 0;
			sscanf(cptr,"%i",&count[iargv - 1]);
			need_kmem = 1;
		}
		nn->n_name = argv[iargv];
		nn++;
	}

	nlist(UNIX_KERNEL,nlst);

	if(need_kmem)
		open_kmem();

	nn = nlst;
	fputs("     name                         type  class   value\n",stdout);
	fputs("--------------------------------  ----  -----  ---------\n",stdout);
	for(iargv = 1; iargv < argc; iargv++)
	{
		if(!nn->n_sclass && !nn->n_value)
		{
			printf("%s: not found\n",nn->n_name);
			nn++;
			continue;
		}

		printf("%-32.32s  %04x   %02x    %08lx\n",
		    nn->n_name,
		    nn->n_type,
		    (unsigned char)nn->n_sclass,
		    nn->n_value);

		nn++;
	}

	nn = nlst;
	for(iargv = 1; iargv < argc; iargv++)
	{
		if(!nn->n_sclass && !nn->n_value)
		{
			nn++;
			continue;
		}

		if(count[iargv - 1])
		{
			printf("\n%s addr=%08lx length=%08x\n",
				nn->n_name,nn->n_value,count[iargv - 1]);
			dump_kmem(nn->n_value,count[iargv - 1]);
		}

		nn++;
	}
	exit(0);
}	/* end of main */

/* vi: set tabstop=4 shiftwidth=4: */
/* end of nlist.c */
