#ifndef lint
static char ID[] = "@(#)nlist.c	2.1 Copyright (C) 1989 Chris Bertin 89/07/31";
#endif
/*
 * nlist -- display and monitor kernel variables. Dump kernel structures.
 *
 *	This program is free for redistribution as long as this header
 *	remains unchanged.
 *
 *				Copyleft Chris Bertin, July 1989
 *
 */
#include	<stdio.h>
#ifdef	V7
#include	<a.out.h>
#else
#include	<nlist.h>
#endif	/* V7 */
#ifdef	SYSV
#include	<string.h>
#else
#include	<strings.h>
#endif	/* SYSV */
#include	<ctype.h>
#include	<memory.h>
#include	<time.h>
#include	<sys/types.h>
#include	<sys/signal.h>
#ifdef	SYSV
#include	<sys/var.h>
#endif

#define NLISTBUG	/* see comment in fixnlist() */

#ifndef	SYSV
#define DEF_NLIST	"/vmunix"	/* default namelist */
#else
#define DEF_NLIST	"/unix"
#endif
#define DEF_KMEM	"/dev/kmem"	/* default memory device */
#define DEF_SLEEP	1		/* default pause between displays */
#define DEF_WIDTH	8		/* default print width */
#define DEF_LOOPS	-1		/* default number of loops (infinite) */
#define DEF_RDSIZE	4		/* default read size */

/* misc return values */
#define R_OK		0		/* OK */
#define R_NOPRINT	1		/* value was computed but don't print */
#define R_FLOAT		2		/* print value in float format */
#define R_SKIP		3		/* skip this entry */
#define R_ERR		-1		/* print error value */

/* misc */
#define CDEPTH		4		/* max number of push for formulas */
#define NDEC		2		/* number of decimals for float */
#define NSPACES		1		/* number of spaces between fields */

/* header special characters */
#define H_NOPRINT	'@'		/* don't print if first character */
#define H_POST		':'		/* real header start character */
#define H_SET		'='		/* set field to value */
#define H_INIT		'!'		/* initialize field to value */
#define H_SIZE		'|'		/* read size (real symbols only) */
#define H_FMT		'%'		/* field print width */
#define H_INCR		'>'		/* increment address on next loop */
#define H_NONZERO	'?'		/* quit if this address is 0 */

/* field identifiers in formulas. (For example: #1: value of field1, */
/* ^+1: previous value of next field, &.: nlist address in current field) */
#define H_CURVAL	'#'		/* current value in field number */
#define H_PREVAL	'^'		/* previous value in field number */
#define H_ADDR		'&'		/* the nlist address of the field */
#define H_CURFIELD	'.'		/* current field */

/* values for -h flag */
#define HD_NONE		0		/* don't print header */
#define HD_ONCE		1		/* print header once only */
#define HD_PAGE		2		/* print header every $LINES lines */

typedef float	val_t;
typedef	struct nlist	nl_t;

typedef	struct field {
	nl_t	*f_nl;			/* nlist entry pointer */
	char	*f_arg;			/* original argument */
	char	*f_form;		/* formula or symbol */
	char	*f_post;		/* title */
	char	*f_fmt;			/* print format (dec, oct, hex...) */
	char	*f_buf;			/* read buffer for string printf's */
	short	 f_flags;		/* features */
	short	 f_width;		/* print width */
	short	 f_rdsize;		/* read size */
	off_t	 f_offset;		/* address offset */
	off_t	 f_incr;		/* address increment */
	val_t	 f_curval;		/* current value */
	val_t	 f_preval;		/* previous value */
} fld_t;

/* f_flags bits */
#define F_TITLE		0x1		/* Title is present */
#define F_SYM		0x2		/* valid symbol */
#define F_SET		0x4		/* set value */
#define F_INIT		0x8		/* set value once only */
#define F_DONE		0x10		/* init done */
#define F_QUOTED	0x20		/* header is quoted. Don't clean it */
#define F_NONZERO	0x40		/* quit if address is 0 */
#define F_TIME		0x80		/* print the value as date/time */
#define F_READOK	0x100		/* OK to read size other that 1,2,4 */

extern	char *optarg, *getenv(), *parsearg(), *alloc(), *calloc();
extern	char *prfmt(), *spaces(), *cleanbuf();
extern	char *sys_errlist[];
extern	int errno, optind;
extern	double ceil();

/* formats */
char	decfmt[] =	"%*d";		/* decimal print mode */
char	unsfmt[] =	"%*u";		/* unsigned decimal print mode */
char	octfmt[] =	"%*o";		/* octal print mode */
char	hexfmt[] =	"%*x";		/* hex print mode */
char	flofmt[] =	"%*.*f";	/* floating point format */
char	strfmt[] =	"%*.*s";	/* string print format */
#ifndef	V7
char	extoctfmt[] =	"%#*o";		/* hex with '0' prefix */
char	exthexfmt[] =	"%#*x";		/* octal with '0x' prefix */
#endif
#define	DEF_FMT		hexfmt		/* default print format */

val_t	calc[CDEPTH];			/* calculator stack */
int	cdepth;				/* calculator stack depth */

int	tim, uniq, fmtprefix, debug, autoincr;	/* flags with default 0 */
int	strict = 1;			/* flags with default 1 */
char	*buf, *hdr;			/* line and header buffers */
char	*pname;				/* argv[0], for error messages */

char	*kmemf	= DEF_KMEM;
char	*nlistf	= DEF_NLIST;
char 	*fmt	= DEF_FMT;
int	slp	= DEF_SLEEP;
int	width	= DEF_WIDTH;
int	nloops	= DEF_LOOPS;
int	header	= HD_PAGE;

main (argc, argv)
	char		**argv;
{
	register int	mem, i;
	int		nsymbols, maxw, prtime;
	register fld_t	*baseflp, *flp;
	nl_t		*nl;
	char		*p;
	int		onfpe();

	pname = *argv;
	while ((i = getopt(argc, argv, "aD:f:h:l:m:n:ps:tux")) != -1) {
		switch (i) {
		case 'a':
			autoincr++;
			break;
		case 'D':
			debug = stoi(&optarg, "debug level", 0, 0);;
			break;
		case 'f':
			fmt = prfmt(&optarg, &width, &prtime, 0, optarg);
			break;
		case 'h':
			header = stoi(&optarg, "header value", 1, 0);
			break;
		case 'l':
			nloops = stoi(&optarg, "number of loops", 0, 0);
			break;
		case 'm':
			kmemf = optarg;
			break;
		case 'n':
			nlistf = optarg;
			break;
		case 'p':
			fmtprefix++;
			break;
		case 's':
			slp = stoi(&optarg, "sleep time", 0, 0);
			break;
		case 't':
			tim++;
			break;
		case 'u':
			uniq++;
			break;
		case 'x':
			strict = 0;
			break;
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;
	if (argc <= 0)
		usage();
	if ((mem = open (kmemf, 0)) < 0)
		perrexit (kmemf);
	baseflp = (fld_t *) alloc((int) (argc * sizeof (fld_t)));
	nl = (nl_t *) alloc((int) ((argc + 2) * sizeof (nl_t)));
	for (flp = baseflp, maxw = nsymbols = 0, i = 0; i < argc; i++, flp++) {
		p = parsearg(flp, *(argv + i));
		if (flp->f_fmt != NULL)
			maxw += flp->f_width + NSPACES;
		if (flp->f_flags & F_SYM) {
			flp->f_nl = nl + nsymbols;
			nsymbols++;
#ifdef	V7
			(void) strncpy(flp->f_nl->n_name, p, SYMLENGTH);
#else
			flp->f_nl->n_name = p;
#endif
			continue;
		}
		if (autoincr && i > 0 && flp->f_incr == 0 && *p == '\0') {
			sprintf(p = alloc(32), "%c%c%c%c-1 %d +", H_ADDR,
				H_CURFIELD, H_SET, H_ADDR, (flp - 1)->f_rdsize);
			flp->f_flags |= F_SET;
		}
		flp->f_nl = nl + (argc - i + nsymbols + 1);
		flp->f_form = p;
	}
	if (maxw == 0) {
		fprintf(stderr, "%s: nothing to print\n", pname);
		exit (1);
	}
	if (nsymbols) {
		donlist(nl, mem, nsymbols);
		for (i = 0, flp = baseflp; i < argc; i++, flp++)
			if (flp->f_offset && flp->f_nl->n_value)
				flp->f_nl->n_value += flp->f_offset;
	}
	if (header != HD_NONE) {
		hdr = alloc(maxw);
		for (p = hdr, i = 0, flp = baseflp; i < argc; i++, flp++) {
			if (flp->f_fmt == NULL)
				continue;
			sprintf(p, "%s%*.*s", spaces(NSPACES), flp->f_width,
			    flp->f_width, flp->f_post? flp->f_post: flp->f_arg);
			p += flp->f_width + NSPACES;
		}
	}
	(void) signal(SIGFPE, onfpe);
	for (;;) {
		if (process(baseflp, argc, mem, maxw, prtime)) {
			outputline();
			if (--nloops == 0)
				break;
		}
		(void) sleep((unsigned) slp);
	}
	return (0);
}

/*
 * Process the arguments once. Return 1 if 'buf' is to be printed.
 */
process(baseflp, max, mem, maxw, prtime)
	register fld_t	*baseflp;
	register int	max, mem, maxw;
{
	register int	i;
	register char	*p;
	register fld_t	*flp;
	static char	*lastbuf, *tmp;

	if (buf == NULL) {
		buf = alloc(maxw);
		lastbuf = alloc(maxw);
		tmp = alloc(BUFSIZ);
	}
	for (flp = baseflp, buf[0] = '\0', i = 0; i < max; i++, flp++) {
		tmp[0] = '\0';
		switch (analyze(baseflp, flp, max, mem)) {
		case R_NOPRINT:
			continue;
		case R_SKIP:
			sprintf(tmp, strfmt, flp->f_width, flp->f_width,
				"--------------------");
			break;
		case R_FLOAT:
			sprintf(tmp, flofmt, flp->f_width, NDEC, flp->f_curval);
			break;
		case R_OK:
			p = flp->f_fmt;
			if (p == strfmt) {
				if (prtime || (flp->f_flags & F_TIME)) {
					time_t t = (time_t) flp->f_curval;
					(void) strncpy(flp->f_buf, ctime(&t),
						flp->f_width);
				}
				sprintf(tmp, strfmt, flp->f_width, flp->f_width,
					cleanbuf(flp->f_buf, flp->f_width));
				break;
			}
#ifndef	V7
			if (fmtprefix && (long) flp->f_curval) {
				if (flp->f_fmt == hexfmt)
					p = exthexfmt;
				else if (flp->f_fmt == octfmt)
					p = extoctfmt;
			}
#endif
			sprintf(tmp, p, flp->f_width, (long) flp->f_curval);
			break;
		case R_ERR:
			if (strict) {
				outputline();
				fprintf(stderr, "%s: symbol '%s' not found\n",
					pname, flp->f_nl->n_name);
				exit (1);
			}
			sprintf(tmp, strfmt, flp->f_width, flp->f_width,
				"????????????????????");
			break;
		}
		if (strlen(tmp) > flp->f_width) {
			if (strict) {
				outputline();
				fprintf(stderr, "%s: print overflow\n", pname);
				exit (1);
			}
			sprintf(tmp, strfmt, flp->f_width, flp->f_width,
				"********************");
		}
		(void) strcat(buf, spaces(NSPACES));
		(void) strcat(buf, tmp);
	}
	for (i = 0, flp = baseflp; i < max; i++, flp++) {
		flp->f_preval = flp->f_curval;
		if (flp->f_incr && flp->f_nl->n_value)
			flp->f_nl->n_value += flp->f_incr;
	}
	if (uniq) {
		if (strcmp(buf, lastbuf) == NULL)
			return (0);
		(void) strcpy(lastbuf, buf);
	}
	return(1);
}

/*
 * Print output buffer, adding title if needed. Exit if err is set.
 */
outputline()
{
	static int	lines, nlines;
	char		*p;

	if (buf == NULL || *buf == '\0')
		return;
	if (header != HD_NONE) {
		if (nlines == 0)
			lines = nlines = (p = getenv("LINES"))? atoi(p): 24;
		if (lines == nlines) {
			printline(hdr);
			lines = 2;
		}
		lines++;
		if (header == HD_ONCE)
			header = HD_NONE;
	}
	printline(buf);
}

/*
 * Print a line, with the current time if -t flag present.
 */
printline(line)
	char			*line;
{
	time_t			t;
	register struct tm	*tm;

	if (tim) {
		(void) time(&t);
		tm = localtime(&t);
		printf("%2d:%02d:%02d", tm->tm_hour, tm->tm_min, tm->tm_sec);
	}
	printf("%s\n", line);
	fflush(stdout);
}

/*
 * Parse an argument and initialize the control structure. Return a pointer
 * to a symbol without any trailers, if the argument is a symbol name.
 */
char *
parsearg(flp, str)
	register fld_t	*flp;
	char		*str;
{
	char		*p;
	register char	*pp, *xp;
	register char	c;
	int		endsym, w, prtime;

	flp->f_rdsize = DEF_RDSIZE;
	flp->f_width = width;
	flp->f_fmt = fmt;
	(void) skipspaces(&str);
	if (*str == H_NOPRINT) {
		flp->f_fmt = NULL;
		str++;
		(void) skipspaces(&str);
	}
	if (isalpha(*str) || *str == '_')
		flp->f_flags |= F_SYM;
	flp->f_arg = str;
	(void) strcpy(pp = alloc(strlen(str) + 1), str);
	for (endsym = 0, p = pp; *p; ) {
		(void) skipspaces(&p);
		xp = p;
		switch (*p++) {
		case H_POST:
			endsym++;
			(void) skipspaces(&p);
			if ((c = *p) != '\'' && c != '"') {
				flp->f_post = p;
				break;
			}
			flp->f_post = ++p;
			flp->f_flags |= F_QUOTED;
			while (*p != c)
				if (*p++ == '\0') {
					fprintf(stderr,
					"%s: missing quote in header -- %s\n",
						pname, pp);
					exit (1);
				}
			*p++ = '\0';
			break;
		case H_FMT:
			endsym++;
			flp->f_fmt = prfmt(&p, &w, &prtime, 1, flp->f_arg);
			if (w)
				flp->f_width = w;
			if (flp->f_fmt == strfmt) {
				flp->f_buf = alloc(flp->f_width + 1);
				if (prtime)
					flp->f_flags |= F_TIME;
				else
					flp->f_flags |= F_READOK;
			}
			break;
		case H_SIZE:
			endsym++;
			flp->f_rdsize = stoi(&p, "read size", 0, 1);
			break;
		case H_INCR:
			endsym++;
			flp->f_incr = stoi(&p, "increment value", 1, 1);
			break;
		case H_SET:
			flp->f_flags |= F_SET;
			break;
		case H_INIT:
			*(p - 1) = (flp->f_flags & F_SYM)? '\0': H_SET;
			flp->f_flags |= (F_SET|F_INIT);
			flp->f_fmt = NULL;
			break;
		case '+':
		case '-':
			if (flp->f_flags & F_SYM)
				endsym++;
			--p;
			flp->f_offset = (val_t) stoi(&p, "offset", 1, 1);
			break;
		case H_NONZERO:
			endsym++;
			flp->f_flags |= F_NONZERO;
			break;
		default:
			continue;
		}
		if (! endsym)
			continue;
		*xp = '\0';
	}
	if (flp->f_buf == NULL)
		flp->f_buf = alloc(flp->f_rdsize);
	if ((flp->f_flags & F_QUOTED) == 0)
		cleanspaces(flp->f_post);
	cleanspaces(pp);
	cleanspaces(flp->f_arg);
	return (pp);
}

/*
 * Handle symbols, formulas, reads, calculator, etc...
 */
analyze(baseflp, flp, max, mem)
	register fld_t	*baseflp, *flp;
	register int	max, mem;
{
	fld_t		*nfp;
	char		*p, *form, *fmtp;
	register int	i;
	register val_t	*vp;
	int		prev, addr, seta;
	static int	skiptonext;

	if (flp == baseflp)
		skiptonext = 0;
	if (debug > 1)
		showfl(flp, flp - baseflp, "start analyze");
	if (flp->f_flags & F_DONE)
		return (R_NOPRINT);
	if (flp->f_flags & F_INIT)
		flp->f_flags |= F_DONE;
	prev = addr = seta = 0;
	fmtp = flp->f_fmt;
	p = form = flp->f_form;
	if (flp->f_flags & F_SYM) {
		skiptonext = 0;
		if (flp->f_nl->n_value == 0)
			return (R_ERR);
		flp->f_curval = (val_t) doread(flp->f_nl->n_name, kmemf, mem,
			flp->f_nl->n_value, flp->f_buf, flp->f_rdsize,
			(flp->f_flags & F_READOK));
		return ((fmtp == flofmt)? R_FLOAT: (fmtp? R_OK: R_NOPRINT));
	}
	for (i = 0; i < CDEPTH; i++)
		calc[i] = 0;
	cdepth = 0;
	vp = &flp->f_curval;
	if (flp->f_flags & F_SET) {
		nfp = baseflp + fn(baseflp, flp, &p, &prev, &seta, max, 0) - 1;
		if (seta == 0) {
			flp = nfp;
			vp = prev? &flp->f_preval: &flp->f_curval;
		}
		(void) skipspaces(&p);
		if (*p++ != H_SET)
			badf(form, --p, "invalid 'set' format");
	}
	*vp = 0;
	while (*p) {
		if (skipspaces(&p))
			continue;
		if (i = isdigit(*p))
			*vp = stoi(&p, "formula", 1, 1);
		else if (i = fn(baseflp, flp, &p, &prev, &addr, max, 1)) {
			if (addr)
				*vp =  (val_t) (baseflp + i - 1)->f_nl->n_value;
			else
				*vp = (prev? (baseflp + i - 1)->f_preval:
					     (baseflp + i - 1)->f_curval);
		}
		if (i) {
			if (push(*vp) == R_ERR)
				badf(form, p, "stack overflow");
			*vp = 0;
			continue;
		}
		switch (*p++) {
		case '+':
			calc[1] = calc[1] + calc[0];
			break;
		case '-':
			calc[1] = calc[1] - calc[0];
			break;
		case '*':
			calc[1] = calc[1] * calc[0];
			break;
		case '/':
			if (calc[0] != 0)
				calc[1] = calc[1] / calc[0];
			else if (strict)
				badf(form, --p, "zero divide");
			else
				return (R_ERR);
			break;
		default:
			badf(form, --p, "invalid operand");
		}
		*vp = 0;
		if (pop() == R_ERR)
			badf(form, --p, "stack empty");
	}
	if (*vp == 0)
		*vp = calc[0];
	if (seta)
		nfp->f_nl->n_value = (long) *vp;
	if (debug > 2)
		showfl(flp, flp - baseflp, "end analyze");
	if ((flp->f_flags & F_NONZERO) && flp->f_nl->n_value == 0)
		skiptonext++;
	if (skiptonext)
		return (fmtp? R_SKIP: R_NOPRINT);
	if (flp->f_nl->n_value)
		flp->f_curval = (val_t) doread(flp->f_nl->n_name, kmemf, mem,
			flp->f_nl->n_value, flp->f_buf, flp->f_rdsize,
			(flp->f_flags & F_READOK));
	if (fmtp == flofmt || (fmtp == decfmt &&
	    (val_t) ceil((double) flp->f_curval) != flp->f_curval))
		return (R_FLOAT);
	return (fmtp? R_OK: R_NOPRINT);
}

/*
 * Return field number. Set 'prev' and 'addr' flags, advance formula pointer.
 */
fn(baseflp, flp, p, prev, addr, max, errok)
	register fld_t	*baseflp, *flp;
	register char	**p;
	int		*prev, *addr;
{
	register int	id, plus;

	if (((*addr = (**p == H_ADDR)) == 0) &&
	    ((*prev = (**p == H_PREVAL)) == 0) && **p != H_CURVAL) {
		if (errok)
			return (0);
		badf(flp->f_form, *p, "not a field identifier");
	}
	(*p)++;
	if (**p == H_CURFIELD) {
		(*p)++;
		return ((flp - baseflp) + 1);
	}
	plus = (**p == '+');
	if ((id = stoi(p, "field number", 1, 1)) < 0 || (plus && id > 0))
		id = (int) (flp - baseflp) + 1 + id;
	if (id <= 0 || id > max) {
		char *pp;
		sprintf(pp = alloc((id / 10) + 2), "%d", id);
		badf(flp->f_form, pp, "invalid field number");
	}
	return (id);
}

/*
 * Convert a string into a positive or negative int. Handles hex and octal.
 * Advance pointer.
 */
stoi(p, what, negok, errok)
	char		*what;
	register char	**p;
{
	int		n, newval, neg, base;

	neg = 0;
	base = 10;
	(void) skipspaces(p);
	if (**p == '+' || (negok && (neg = (**p == '-'))))
		(*p)++;
	if (! strncmp(*p, "0x", 2)) {
		base = 16;
		(*p) += 2;
	}
	else if (**p == '0') {
		base = 8;
		(*p)++;
	}
	for (newval = 0; **p; (*p)++) {
		if ((n = conv(*p, base, what, errok)) < 0)
			break;
		newval = (newval * base) + n;
	}
	(void) skipspaces(p);
	return (neg? -newval: newval);
}

/*
 * Convert an ascii character into a digit.
 */
conv(p, base, what, errok)
	char		*p, *what;
{
	register char	c;

	if (isdigit(c = *p))
		return (c - '0');
	if (isupper(c))
		c = tolower(c);
	if (base == 16 && c >= 'a' && c <= 'f')
		return (10 + c - 'a');
	if (errok)
		return (-1);
	fprintf(stderr, "%s: illegal %s (%s)\n", pname, what, p);
	usage();
	/* NOTREACHED */
}

/*
 * Advance pointer to next non-space character. Return 1 pointer changed.
 */
skipspaces(p)
	register char	**p;
{
	if (! isspace(**p))
		return (0);
	while (isspace(**p))
		(*p)++;
	return (1);
}

/*
 * Remove trailing spaces from string.
 */
cleanspaces(str)
	register char	*str;
{
	if (str == NULL || *str == '\0')
		return;
	while (*str)
		str++;
	--str;
	while (isspace(*str))
		*str-- = '\0';
}

/*
 * Replace unprintable characters in a buffer by dots ('.').
 */
char *
cleanbuf(p, n)
	register char	*p;
{
	register char	*pp;

	for (pp = p; --n >= 0; pp++)
		if (! isprint(*pp))
			*pp = '.';
	return (p);
}

/*
 * Handle a format string. Return a format pointer and the width in 'width'.
 * Advance pointer.
 */
char *
prfmt(p, w, prtime, errok, str)
	char	**p;
	char	*str;
	int	*w, *prtime;
{
	*prtime = 0;
	if ((*w = stoi(p, "print width", 0, 1)) == 0)
		*w = DEF_WIDTH;
	switch (**p) {
	case 'd':
		return (decfmt);
	case 'f':
		return (flofmt);
	case 'o':
		return (octfmt);
	case 't':
		*prtime = 1;
		/* fall thru */
	case 's':
		return (strfmt);
	case 'u':
		return (unsfmt);
	case 'x':
		return (hexfmt);
	}
	if (! isalpha(**p) && errok) {
		(*p)++;
		return (DEF_FMT);
	}
	fprintf(stderr, "%s: invalid print format in '%s' (%c)\n", pname,
		str, **p);
	outputline();
	exit (1);
	/* NOTREACHED */
}

/*
 * Push argument onto calculator stack.
 */
push(n)
	register val_t	n;
{
	register int	i;

	if (cdepth++ == CDEPTH)
		return (R_ERR);
	for (i = cdepth; --i >= 0; )
		calc[i] = i? calc[i - 1]: n;
	return (R_OK);
}

/*
 * Pop one argument from calculator stack.
 */
pop()
{
	register int	i;

	if (--cdepth <= 0)
		return (R_ERR);
	for (i = 1; i <= cdepth; i++)
		if ((calc[i - 1] = calc[i]) == 0)
			break;
	return (R_OK);
}

/*
 * Do 'nlist' and validate name list. Validation is very basic. Note that
 * one extra 'nlist' entry was allocated to handle the additional symbol.
 */
donlist(nl, mem, max)
	register nl_t	*nl;
{
	char		*p;
	int		nproc, size;
#ifdef	SYSV
	struct		var v;

#ifdef	UNDERSCORESYM
	(nl + max)->n_name = "_v";
#else
	(nl + max)->n_name = "v";
#endif
	p = (char *) &v;
	size = sizeof (v);
#else	/* SYSV */
#ifdef	V7
	strncpy((nl + max)->n_name, "_nproc", SYMLENGTH);
#else
#ifdef	UNDERSCORESYM
	(nl + max)->n_name = "_nproc";
#else
	(nl + max)->n_name = "nproc";
#endif
#endif	/* V7 */
	p = (char *) &nproc;
	size = sizeof nproc;
#endif	/* SYSV */
	if (nlist (nlistf, nl) < 0) {
		fprintf(stderr, "%s: nlist(%s) failed -- ", pname, nlistf);
		perrexit("");
	}
	if ((nl + max)->n_value == 0) {
		fprintf(stderr, "%s: symbol '%s' not found in %s\n", pname,
			(nl + max)->n_name, nlistf);
		exit (1);
	}
	(void) doread((nl + max)->n_name, kmemf, mem,
		(off_t) (nl + max)->n_value, p, size, 1);
#ifdef	SYSV
	nproc = v.v_proc;
#endif
	if (nproc > 4096 || nproc < 3) {
		fprintf(stderr, "Namelist out of date\n");
		exit (1);
	}
#ifdef	NLISTBUG
	fixnlist(nl, max);
#endif
}

#ifdef	NLISTBUG
/*
 * If a symbol is more than once in the list, all but the first one will have
 * an n_value of zero. This normally doesn't matter, but 'nlist' can ask for
 * symbol+4 and symbol+8, for example, in which case the second one fails.
 * This is a bug present in all the nlist(3) I have tried.
 */
fixnlist(nl, max)
	register nl_t	*nl;
	register int	max;
{
	register nl_t	*np, *npp;

	for (np = nl; np < nl + (max - 1) ; np++)
		for (npp = np + 1; npp < nl + max; npp++)
			if (! strcmp(npp->n_name, np->n_name)) {
				npp->n_value = np->n_value;
				break;
			}
}
#endif	/* NLISTBUG */

/*
 * Generic read routine. If read size is  1, 2, 4 bytes, return that value,
 * else return 0. If okdiff is 0, don't allow read sizes != 1, 2, 4.
 */
doread(what, from, mem, offset, into, size, okdiff)
	char	*what, *from, *into;
	off_t	offset;
{
	if (debug)
		showread(what, from, offset, size, -1);
	(void) lseek(mem, offset, 0);
	if (read(mem, into, (unsigned) size) != size) {
		showread(what, from, offset, size, errno);
		if (strict) {
			outputline();
			exit (1);
		}
	}
	switch (size) {
	case sizeof (char):
		return ((int) *into);
	case sizeof (short):
		return ((int) *(short *) into);
	case sizeof (int):
		return ((int) *(int *) into);
	default:
		if (okdiff)
			return (0);
		outputline();
		fprintf(stderr, "%s: Invalid read size: %d\n", pname, size);
		exit (1);
	}
	/* NOTREACHED */
}

/*
 * Debug and error routine. Show read arguments.
 */
showread(what, from, offset, size, err)
	char	*what, *from;
	off_t	offset;
{
	fprintf(stderr, "%s: read ", pname);
	if (what)
		fprintf(stderr, "'%s', ", what);
	fprintf(stderr, "%d bytes @ offset 0x%x in %s", size, offset, from);
	if (err >= 0)
		fprintf(stderr, " -- %s", sys_errlist[err]);
	fprintf(stderr, "\n");
}

/*
 * Debug routine. Show the whole control structure.
 */
showfl(flp, n, str)
	register fld_t	*flp;
	char		*str;
{
	fprintf(stderr, "  %s: fld %d, nl %x, buf '%s', name '%s', addr %x\n",
		str, n, flp->f_nl, flp->f_buf, flp->f_nl->n_name,
		flp->f_nl->n_value);
	fprintf(stderr, "  arg '%s', form '%s', post '%s', fmt '%s'\n",
		flp->f_arg, flp->f_form, flp->f_post, flp->f_fmt);
	fprintf(stderr, "  f 0x%x, w %d, rd %d, o %d, i %d, cur %d, pre %d\n",
		flp->f_flags, flp->f_width, flp->f_rdsize, flp->f_offset,
		flp->f_incr, (long) flp->f_curval, (long) flp->f_preval);
}

/*
 * Allocate 'n' bytes on the heap.
 */
char *
alloc(n)
{
	char	*a;

	if ((a = calloc((unsigned) n, (unsigned) 1)) == 0)
		perrexit("calloc");
	return (a);
}

/*
 * Build an array of spaces, return the address.
 */
char *
spaces(n)
{
	static char	*sp;
	register char	*p;

	if (sp == NULL) {
		sp = alloc(n + 1);
		for (p = sp; --n >= 0; p++)
			*p = ' ';
	}
	return (sp);
}

badf(form, msg1, msg2)
	char	*form, *msg1, *msg2;
{
	outputline();
	fprintf(stderr, "%s: invalid formula '%s'", pname, form);
	if (msg1 && *msg1)
		fprintf(stderr, " (%s)", msg1);
	if (msg2 && *msg2)
		fprintf(stderr, " -- %s", msg2);
	fprintf(stderr, "\n");
	exit (1);
}

perrexit(str)
	char	*str;
{
	perror(str);
	exit (1);
}

/*ARGSUSED*/
onfpe(n)
	int	n;
{
	outputline();
	fprintf(stderr, "%s: Floating poing exception\n", pname);
	exit (1);
}

usage()
{
	fprintf(stderr, "Usage: %s [-aptux] [-f format] [-h header] [-l loops] [-m mem]\n\t[-n namelist] [-s sleep] [-D debug] symbol|formula...\n", pname);
	exit (1);
}
