/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *	%Z%%M% %I% %E% SMI; from UCB 7.2 12/7/87 plus MULTICAST 1.0
 */

#ifndef	_IF_ETHER_
#define	_IF_ETHER_

/*
 * The following include is for compatibility with SunOS 3.x and
 * 4.3bsd.  Newly written programs should include it separately.
 */
#include <net/if_arp.h>

/*
 * Ethernet address - 6 octets
 */
struct ether_addr {
	u_char	ether_addr_octet[6];
};

/*
 * Structure of a 10Mb/s Ethernet header.
 */
struct	ether_header {
	struct	ether_addr ether_dhost;
	struct	ether_addr ether_shost;
	u_short	ether_type;
};

#define	ETHERTYPE_PUP		0x0200		/* PUP protocol */
#define	ETHERTYPE_IP		0x0800		/* IP protocol */
#define	ETHERTYPE_ARP		0x0806		/* Addr. resolution protocol */
#define	ETHERTYPE_REVARP	0x8035		/* Reverse ARP */

/*
 * The ETHERTYPE_NTRAILER packet types starting at ETHERTYPE_TRAIL have
 * (type-ETHERTYPE_TRAIL)*512 bytes of data followed
 * by an ETHER type (as given above) and then the (variable-length) header.
 */
#define	ETHERTYPE_TRAIL		0x1000		/* Trailer packet */
#define	ETHERTYPE_NTRAILER	16

#define	ETHERMTU	1500
#define	ETHERMIN	(60-14)

#ifdef MULTICAST
#ifdef KERNEL
/*
 * Macro to map an IP multicast address to an Ethernet multicast address.
 * The high-order 25 bits of the Ethernet address are statically assigned,
 * and the low-order 23 bits are taken from the low end of the IP address.
 */
#define ETHER_MAP_IP_MULTICAST(ipaddr, enaddr)				\
	/* struct in_addr *ipaddr; */					\
	/* struct ether_addr *enaddr; */				\
{									\
	(enaddr)->ether_addr_octet[0] = 0x01;				\
	(enaddr)->ether_addr_octet[1] = 0x00;				\
	(enaddr)->ether_addr_octet[2] = 0x5e;				\
	(enaddr)->ether_addr_octet[3] = ((u_char *)ipaddr)[1] & 0x7f;	\
	(enaddr)->ether_addr_octet[4] = ((u_char *)ipaddr)[2];		\
	(enaddr)->ether_addr_octet[5] = ((u_char *)ipaddr)[3];		\
}
#endif KERNEL
#endif MULTICAST

/*
 * Ethernet Address Resolution Protocol.
 *
 * See RFC 826 for protocol description.  Structure below is adapted
 * to resolving internet addresses.  Field names used correspond to
 * RFC 826.
 */
struct	ether_arp {
	struct	arphdr ea_hdr;		/* fixed-size header */
	struct	ether_addr arp_sha;	/* sender hardware address */
	u_char	arp_spa[4];		/* sender protocol address */
	struct	ether_addr arp_tha;	/* target hardware address */
	u_char	arp_tpa[4];		/* target protocol address */
};
#define	arp_hrd	ea_hdr.ar_hrd
#define	arp_pro	ea_hdr.ar_pro
#define	arp_hln	ea_hdr.ar_hln
#define	arp_pln	ea_hdr.ar_pln
#define	arp_op	ea_hdr.ar_op

/*
 * Structure shared between the ethernet driver modules and
 * the address resolution code.  For example, each ec_softc or il_softc
 * begins with this structure.
 *
 * The structure contains a pointer to an array of multicast addresses.
 * This pointer is NULL until the first successful SIOCADDMULTI ioctl
 * is issued for the interface.
 */
#ifndef MULTICAST
#define	MCADDRMAX	64		/* multicast addr table length */
#endif  MULTICAST
struct	arpcom {
	struct	ifnet ac_if;		/* network-visible interface */
	struct	ether_addr ac_enaddr;	/* ethernet hardware address */
	struct	in_addr ac_ipaddr;	/* copy of ip address- XXX */
#ifdef MULTICAST
	struct ether_multi *ac_multiaddrs; /* list of ether multicast addrs */
#else
	struct	ether_addr *ac_mcaddr;	/* table of multicast addrs */
	u_short	ac_nmcaddr;		/* count of M/C addrs in use */
#endif MULTICAST
};

/*
 * Internet to ethernet address resolution table.
 */
struct	arptab {
	struct	in_addr at_iaddr;	/* internet address */
	union {
	    struct ether_addr atu_enaddr;	/* ethernet address */
	    long   atu_tvsec;			/* timestamp if incomplete */
	} 	at_union;
	u_char	at_timer;		/* minutes since last reference */
	u_char	at_flags;		/* flags */
	struct	mbuf *at_hold;		/* last packet until resolved/timeout */
};

# define at_enaddr at_union.atu_enaddr
# define at_tvsec at_union.atu_tvsec

/*
 * Compare two Ethernet addresses - assumes that the two given
 * pointers can be referenced as shorts.  On architectures
 * where this is not the case, use bcmp instead.  Note that like
 * bcmp, we return zero if they are the SAME.
 */
#if defined(sun2) || defined(sun3) || defined(sun3x)
/*
 * On 680x0 machines, we can do a longword compare that is NOT
 * longword aligned, as long as it is even aligned.
 */
#define ether_cmp(a,b) ( ((short *)a)[2] != ((short *)b)[2] || \
  *((long *)a) != *((long *)b) )
#endif

/*
 * On a sparc, functions are FAST
 */
#if defined(sparc)
#define ether_cmp(a,b) (sparc_ether_cmp((short *)a, (short *)b))
#endif 

#ifndef ether_cmp
#define ether_cmp(a,b) (bcmp((caddr_t)a,(caddr_t)b, 6))
#endif

/*
 * Copy Ethernet addresses from a to b - assumes that the two given
 * pointers can be referenced as shorts.  On architectures
 * where this is not the case, use bcopy instead.
 */
#if defined(sun2) || defined(sun3) || defined(sun3x)
#define ether_copy(a,b) { ((long *)b)[0]=((long *)a)[0]; \
 ((short *)b)[2]=((short *)a)[2]; }
#endif

#if defined(sparc)
#define ether_copy(a,b) { ((short *)b)[0]=((short *)a)[0]; \
 ((short *)b)[1]=((short *)a)[1]; ((short *)b)[2]=((short *)a)[2]; }
#endif

#ifndef ether_copy
#define ether_copy(a,b) (bccopy((caddr_t)a,(caddr_t)b, 6))
#endif

/*
 * Copy IP addresses from a to b - assumes that the two given
 * pointers can be referenced as shorts.  On architectures
 * where this is not the case, use bcopy instead.
 */
#if defined(sun2) || defined(sun3) || defined(sun3x)
#define ip_copy(a,b) { *((long *)b) = *((long *)a); }
#endif

#if defined(sparc)
#define ip_copy(a,b) { ((short *)b)[0]=((short *)a)[0]; \
 ((short *)b)[1]=((short *)a)[1]; }
#endif

#ifndef ip_copy
#define ip_copy(a,b) (bccopy((caddr_t)a,(caddr_t)b, 4))
#endif

#ifdef	KERNEL
struct	ether_addr etherbroadcastaddr;
#ifdef MULTICAST
struct	ether_addr ether_ipmulticast_min;
struct	ether_addr ether_ipmulticast_max;
#endif MULTICAST
struct	arptab *arptnew();
char *ether_sprintf();
#endif	KERNEL

#ifdef MULTICAST
/*
 * Ethernet multicast address structure.  There is one of these for each
 * multicast address or range of multicast addresses that we are supposed
 * to listen to on a particular interface.  They are kept in a linked list,
 * rooted in the interface's arpcom structure.  (This really has nothing to
 * do with ARP, or with the Internet address family, but this appears to be
 * the minimally-disrupting place to put it.)
 */
struct ether_multi {
	struct ether_addr   enm_addrlo;   /* low  or only address of range */
	struct ether_addr   enm_addrhi;   /* high or only address of range */
	struct arpcom      *enm_ac;	  /* back pointer to arpcom        */
	u_int               enm_refcount; /* no. claims to this addr/range */
	struct ether_multi *enm_next;	  /* ptr to next ether_multi       */
};

#ifdef KERNEL
/*
 * Structure used by macros below to remember position when stepping through
 * all of the ether_multi records.
 */
struct ether_multistep {
	struct ether_multi  *e_enm;
};

/*
 * Macro for looking up the ether_multi record for a given range of Ethernet
 * multicast addresses connected to a given arpcom structure.  If no matching
 * record is found, "enm" returns NULL.
 */
#define ETHER_LOOKUP_MULTI(addrlo, addrhi, ac, enm)			\
	/* struct ether_addr  *addrlo;    */				\
	/* struct ether_addr  *addrhi;    */				\
	/* struct arpcom      *ac;        */				\
	/* struct ether_multi *enm;       */				\
{									\
	for ((enm) = (ac)->ac_multiaddrs;				\
	     (enm) != NULL &&						\
		(ether_cmp(&(enm)->enm_addrlo, (addrlo)) != 0 ||	\
		 ether_cmp(&(enm)->enm_addrhi, (addrhi)) != 0);	\
	     (enm) = (enm)->enm_next);					\
}

/*
 * Macro to step through all of the ether_multi records, one at a time.
 * The current position is remembered in "step", which the caller must
 * provide.  ETHER_FIRST_MULTI(), below, must be called to initialize "step"
 * and get the first record.  Both macros return a NULL "enm" when there
 * are no remaining records.
 */
#define ETHER_NEXT_MULTI(step, enm)					\
	/* struct ether_multistep  step; */				\
	/* struct ether_multi     *enm;  */				\
{									\
	if (((enm) = (step).e_enm) != NULL)				\
		(step).e_enm = (enm)->enm_next;				\
}

#define ETHER_FIRST_MULTI(step, ac, enm)				\
	/* struct ether_multistep  step; */				\
	/* struct arpcom          *ac;   */				\
	/* struct ether_multi     *enm;  */				\
{									\
	(step).e_enm = (ac)->ac_multiaddrs;				\
	ETHER_NEXT_MULTI((step), (enm));				\
}
#endif KERNEL
#endif MULTICAST

#endif	_IF_ETHER_
