#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/inode.h"
#include "../h/user.h"
#include "../h/nami.h"
#include "../h/acl.h"

#define FOLLOW_LINK 1

/*
 * pack_acle() copies an acle struct into the inode.  the acle is packed
 * since the inode is rather short of space.
 */
pack_acle( ip, a, n )
struct inode *ip;
struct acle a;
int n;
{
	/* clear the type and mode fields */
	ip->i_acle[n/2] &= 017 << ((n%2) ? 4 : 0);

	/* set the type, mode, and id fields */
	ip->i_acle[n/2] |= ( a.a_type != A_GROUP ? A_USER : A_GROUP ) <<
		((n%2) ? 3 : 7);
	ip->i_acle[n/2] |= ( a.a_mode & 07 ) << ((n%2) ? 0 : 4);
	ip->i_aclid[n] = a.a_id;
}

/*
 * null_acle() puts an end-of-acl marker in the acl at entry n.
 * the end-of-acl marker looks like an acl entry for root
 * (a_type == A_USER  && a_id == 0).
 */
null_acle( ip, n )
struct inode *ip;
int n;
{
	/* clear the type and mode fields */
	ip->i_acle[n/2] &= 017 << ((n%2) ? 4 : 0);

	/* clear the id field */
	ip->i_aclid[n] = 0;
}

/*
 * getacl() returns the set of acl entries associated with the file.
 * the user must be able to access the file in order to read the acl.
 */
getacl()
{
	struct inode *ip;
	struct acle ka[MAXACL];
	struct a {
		char *fname;
		int n_acle;
		struct acle *acle_p;
	} *uap = (struct a *) u.u_ap;
	int i;

	if ( (ip = namei(uchar,LOOKUP,FOLLOW_LINK)) != NULL ) {
		/* unpack the acl entries */
		for (i = 0; i < MAXACL && ok_acle(ip,i); i++ ) {
			ka[i].a_type = acle_type( ip, i );
			ka[i].a_mode = acle_mode( ip, i );
			ka[i].a_id = acle_id( ip, i );
		}

		/* getacl doesn't need the inode any more. */
		iput( ip );

		if ( uap->n_acle < i ) {
			u.u_error = EINVAL;
		} else {
			/*
			 * copy the acl entries to the user's buffer,
			 * and return the number of entries found.
			 */
			u.u_error = copyout((caddr_t) ka, (caddr_t) uap->acle_p,
				i*sizeof(struct acle));
			if ( u.u_error == 0 ) u.u_r.r_val1 = i;
		}
	}
}

/*
 * setacl() defines the acl for an inode.  to do this, the user must
 * be the owner of the inode, or else must be the superuser.
 *
 * bugs: acl entries should be sorted by a_type , so that access() does
 * not have to read the entire acl three times.  since the number of acl
 * entries is so small, though, this would make setacl() more complex
 * (sorting the entries) without really buying that much.
 */
setacl()
{
	struct inode *ip;
	struct acle ka[MAXACL];
	int kn;
	struct a {
		char *fname;
		int n_acle;
		struct acle *acle_p;
	} *uap = (struct a *) u.u_ap;
	int i;

	/*
	 * copy the user's arguments into kernel space; kn will contain
	 * the number of acl entries, and ka will contain the entries.
	 */
	if ( (kn = uap->n_acle) > MAXACL ) {
		u.u_error = EINVAL;
	} else {
		u.u_error = copyin( (caddr_t) uap->acle_p, (caddr_t) ka,
			kn*sizeof(struct acle) );
		if ( u.u_error == 0 ) {
			if ( (ip = owner(FOLLOW_LINK)) != NULL ) {
				for ( i = 0; i < kn; i++ ) {
					pack_acle( ip, ka[i], i );
					if (! ok_acle(ip,i) ) break;
				}

				if ( kn < MAXACL ) null_acle( ip, kn );

				/* update ctime for the inode */
				ip->i_flag |= ICHG;
				iput( ip );
			}
		}
	}
}
