/* wdbread - converts raw world data bank II files into vax words

/* NOTE: like old wdbiiread, but limits consec. points to +-127 units */

/* written by Alan Paeth, University of Waterloo, January, 1984 */

/* wdbread <in >out 
   reads in hunks of twenty byte EBCDIC records, as per the FORTRAN 
   spec of WDBII, and writes them out as signed 8-bit integers (signed
   bytes), as delta values. No conversion of characters is done for other
   than the digits and few other expected sparse characters.

NOTE: the output is named map.data and map.index, both placed on the
current directory. The file >out contains a summary of processed records.
*/

/* DEFINITIONS */

#include <stdio.h>

#define PROGNAME "wdbread"
#define EBCDICFLAG 0		/* 0=ascii input, 1=ebcdic */
#define INBUFSIZE 20
#define PIPEOUT 1
#define DATANAME "map.data"
#define INDEXNAME "map.index"
#define WRITEMODE "w"
#define OPENFAIL 0 
#define LON360 1296000
#define FRAGLIMIT 20
#define maxint  2147483647
#define minint -2147483648 
#define v(arg) (inbuf[arg]-'0')

#define ABS(a) ((a)<0?(-a):(a))

FILE *mapdata, *mapindex;
char inbuf[INBUFSIZE+1], remap[256];
int rank, count;
int fragcount, totfrag, readrec, writerec, readpoint, writepoint;
int i, j, k, n, s, e, w;
short shortn, shorts, shorte, shortw;
int olat, olon, privlat, privlon, dlat, dlon, oprivlat, oprivlon;

/* THE MAIN */

main(argc,argv)
    int  argc;
    char **argv;
    {

/* open the data and index output */

    if ((mapdata = fopen(DATANAME, WRITEMODE)) == OPENFAIL)
        abort("open fail on %s", DATANAME);

    if ((mapindex = fopen(INDEXNAME, WRITEMODE)) == OPENFAIL)
        abort("open fail on %s", INDEXNAME);
    
/* set remap to map into ASCII, digit 0 for all other characters */

    if (EBCDICFLAG) 
/* EBCDIC REMAPPING STUFF */
        {
	for (i=0; i<256; i++) remap[i] = '0';		/* def. 0 */
	for (i=0; i<10; i++) remap[240+i] = 48+i;	/* digits */
	remap[103] = 'W';				/* w,W,s,S */
	remap[231] = 'W';
	remap[99] = 'S';
	remap[227] = 'S';
	}
    else
/* ASCII REMAPPING (identity xform, space -> digit 0 */
	{
	for (i=0; i<256; i++) remap[i] = i;		/* identity */
	remap[' '] = '0';				/* space -> 0 */
	}

    inbuf[INBUFSIZE] = '\0';
    readrec = 0;
    writerec = 0;
    totfrag = 0;

/* now the top level count loop */

    while( fread(&inbuf[0], 1, INBUFSIZE, stdin) == INBUFSIZE)
	{
	readrec++;
	readpoint = 0;
	writepoint = 0;
	
	for (i=0; i<INBUFSIZE; i++) inbuf[i] = remap[inbuf[i]];
	printf("/%s/\n", inbuf);
	sscanf( &inbuf[9], "%6ld", &count);
	sscanf( &inbuf[7], "%2ld", &rank);
	printf("count %d rank %d\n", count, rank);
	fflush(stdout);
	
	n = minint;
	s = maxint;
	e = minint;
	w = maxint;

/* NEW STUFF */

	fragcount = 0;
	
/* the high-speed data loop */
	for (j=0; j<count; j++)
	    {
/* optimizations */
	    register int lat, lon, rSIX, rTEN;
	    register char *base;
	    rTEN = 10;
	    rSIX = 6;

	    olat = lat;
	    olon = lon;
	    
    	    fread(&inbuf[0], 1, INBUFSIZE, stdin);
	    for (i=0; i<INBUFSIZE; i++) inbuf[i] = remap[inbuf[i]];
	    
	    readpoint++;
/*
 * optimization of the following:
 *
 *	    lat = v(0)*36000 + v(1)*3600 + v(2)*600 +
 *	      v(3)*60 + v(4)*10 + v(5);
 *	    if (inbuf[6] == 'S') lat = -lat;
 */
	    base = &inbuf[0];
	    lat = *base++;
	    lat *= rTEN;
	    lat += *base++;
	    lat *= rSIX;
	    lat += *base++;
	    lat *= rTEN;
	    lat += *base++;
	    lat *= rSIX;
	    lat += *base++;
	    lat *= rTEN;
	    lat += *base++;
	    lat -= '0'*(36000+3600+600+60+10+1); /* correct for ascii 000000 */
	    if (*base == 'S') lat = -lat;
/*
 * optimization of the following:
 *
 *	    lon = v(7)*360000 + v(8)*36000 + v(9)*3600 +
 *	      v(10)*600 + v(11)*60 + v(12)*10 + v(13);
 *  	    if (inbuf[14] == 'W') lon = -lon;
 */
	    base = &inbuf[7];
	    lon = *base++;
	    lon *= rTEN;
	    lon += *base++;
	    lon *= rTEN;
	    lon += *base++;
	    lon *= rSIX;
	    lon += *base++;
	    lon *= rTEN;
	    lon += *base++;
	    lon *= rSIX;
	    lon += *base++;
	    lon *= rTEN;
	    lon += *base++;
	    lon -= '0'*(360000+36000+3600+600+60+10+1); /* ascii 0000000 */
	    if (*base == 'W') lon = -lon;
/*
 * hemisphere check
 */
	    if (lon >= LON360) lon -= LON360;
	    if ((j != 0) && (ABS(olon-lon)>(LON360/2)))
		{
		warning("date-line crossing: %d\n", lon);
		}
/*
 * adjust bounding box
 */
	    if (lat > n) n = lat;
	    if (lat < s) s = lat;
	    if (lon > e) e = lon;
	    if (lon < w) w = lon;
	    
    	    if (j == 0)
		{
		dlat = lat;
		dlon = lon;
		fwrite(&dlat, 1, 4, mapdata);
		fwrite(&dlon, 1, 4, mapdata);
		writepoint++;
		}
	    else
		{
		dlat = lat - olat;
	        dlon = lon - olon;
	        fragcount = roundupdiv( lmax( labs(dlat), labs(dlon) ) , 127l);
		if (fragcount ==1)
		    {
		    fwrite(&dlat, 1, 1, mapdata);
		    fwrite(&dlon, 1, 1, mapdata);
		    writepoint++;
		    }
		else
		    {
		    if (fragcount == 0)
		        warning("fragcount=0 => consec. duplicate points",NULL);
		    if (fragcount > FRAGLIMIT)
		        {
			warning("excessive fragments %ld",fragcount);
			warning("occurring at segment: %d\n", j);
			if (fragcount > 2000)
			    {
			    warning("trace vars: \n");
			    warning(" lat = %d\n",  lat);
			    warning("olat = %d\n", olat);
			    warning("dlat = %d\n", dlat);
			    warning(" lon = %d\n",  lon);
			    warning("olon = %d\n", olon);
			    warning("dlon = %d\n", dlon);
			    }
			}
		    oprivlat = 0;
		    oprivlon = 0;
		    for (k = 0; k < fragcount; k++)
		        {
		        privlat = rounddiv( dlat*(k+1), fragcount) - oprivlat;
		        privlon = rounddiv( dlon*(k+1), fragcount) - oprivlon;
		        oprivlat = privlat;
		        oprivlon = privlon;

		        fwrite( &privlat, 1, 1, mapdata);
		        fwrite( &privlon, 1, 1, mapdata);

		        writepoint++;
		        }
		    printf(" frg=%ld dx=%ld dy=%ld\n", fragcount,dlat,dlon);
		    totfrag += (fragcount - 1);
		    }
	        } 
	    }
    	fwrite( &writepoint, 1, 2, mapindex);
    	fwrite( &rank, 1, 2, mapindex);
	writerec++;

/* round n,e +, round s,w - */

	shortn = round20up(n);
	shorts = round20down(s);
	shorte = round20up(e);
	shortw = round20down(w);

	fwrite( &shortn, 1, 2, mapindex);
    	fwrite( &shorts, 1, 2, mapindex);
    	fwrite( &shorte, 1, 2, mapindex);
    	fwrite( &shortw, 1, 2, mapindex);

	printf("read=%ld write=%ld n=%ld s=%ld e=%ld w=%ld\n\n", readpoint,
	   writepoint, n, s, e, w);
	fflush(stdout);
	}
    printf("\n\nTOTALS: records in=%ld records out=%ld new fragment=%ld\n",
		readrec,writerec,totfrag);
    fclose(mapdata);
    fclose(mapindex);
    exit(0);
    }

/* ROUTINES */

abort(a,b)
    char *a, *b;
    {
    fprintf(stderr,"%s error: ",PROGNAME);
    fprintf(stderr,a,b);
    fprintf(stderr,"\n");
    exit(1);
    }

warning(a,b)
    char *a, *b;
    {
    printf("\nWARNING: ");
    printf(a,b);
    printf("\n");
    fflush(stdout);
    }

roundupdiv(a,b)
    int a,b;
    {
    return (a >= 0) ? ((a - 1) / b + 1) : -roundupdiv(-a,b);
    }

rounddiv(a,b)
    int a,b;
    {
    return (a >= 0) ? (2*a + b) / (2*b) : -rounddiv(-a,b);
    }

lmax(a,b)
    int a,b;
    {
    return ((a >= b) ? a : b);
    }

labs(a)
    int a;
    {
    return ((a >= 0) ? a : -a);
    }

round20up(a)
    int a;
    {
    return ( (a >= 0) ? (a+19)/20 : (-round20down(-a)) );
    }

round20down(a)
    int a;
    {
    return ( (a >= 0) ? a/20 : (-round20up(-a)) );
    }
