/* wdbplot -  draws wdb maps on unix plot files, using converted files */

/* written by Alan Paeth, University of Waterloo, January, 1984 */

/* wdbplot [clat] [clon] [latscale]  
   draws a map centered at clat,clon degrees which spans a total of
   lonscale degrees (all values are floats).

   NOTE: this version adjusts the vertical scale by secant(lat), ie, it
   creates maps which are Mercator-like, especially at small scalings.
   Here, the scale parameter (user specified) controls lon. (width) only.

SPECIAL NOTE: y coordinates are left-handed, ie, increase as one moves
   downward from the northnmost edge of the map. Many display devices
   work this way.
   
   wN - lat  (but will revert back to the proper)   lat - wS
    
   Color (based on rank) could also use some work
*/


/* DEFINITIONS */

#define XRES	 512	/* max output width in integer device units */
#define YRES	 512	/* max output height in integer device units */


#include <stdio.h>
#include <graphics/const.h>
#include <graphics/ik_type.h>
#include <graphics/ik_const.h>

#define PROGNAME "wdbplot"
#define MASKREG 1
#define DATANAME "map.data"
#define INDEXNAME "map.index"
#define READMODE 0 
#define OPENFAIL -1
#define UNITPERDEG 3600.0

#define GRIDCOLOR  8
#define COASTCOLOR 1
#define RIVERCOLOR 2
#define BOUNDCOLOR 8


static int lastcolor = 0;
static int pengrabbed = 0;
static int lastx = -1;
static int lasty = -1;

double atof(), cos();

double xscale, yscale, clat, clon, xs, ys;
double latgridwidth, latgridcount, latbase;
double longridwidth, longridcount, lonbase;
int mapcolor[64];
int mapdata, mapindex;

short rank, count, shortn, shorts, shorte, shortw;
long n, s, e, w;
long i, totalrec;
long lat, lon;
char dlat, dlon;
long wN, wS, wE, wW;


/* ROUTINES */

double intpart(a)
double a;
    {
    return( (double) ( (int) a ) );
    }

lattopoint(latd)
double latd;
    {
    return (   (double)  (  ( ((float) wN) - latd*UNITPERDEG) * ys)   );
    }
    
lontopoint(lond)
double lond;
    {
    return (   (double)  (   ( lond*UNITPERDEG - ((float) wW) ) * xs)   );
    }
    
double pointtolat(y)
int y;
    {
    return (  ( ((float) wN) - ((float) y) / ys) / UNITPERDEG );
    }
    
double pointtolon(x)
int x;
    {
    return ( ( ((float) x) / xs + ((float) wW) )/ UNITPERDEG );
    }
    
abort(a,b)
    char *a, *b;
    {
    fprintf(stderr,"%s error: ",PROGNAME);
    fprintf(stderr,a,b);
    fprintf(stderr,"\n");
    exit(1);
    }

/* THE MAIN */

main(argc,argv)
    int argc;
    char *argv[];
    {
    if (argc != 4)
	abort("usage: [centerlat] [centerlon] [scale]", NULL);

/* open the data and index input, plotter for output */

    if ((mapdata = open(DATANAME, READMODE)) == OPENFAIL)
        abort("open fail on %s", DATANAME);

    if ((mapindex = open(INDEXNAME, READMODE)) == OPENFAIL)
        abort("open fail on %s", INDEXNAME);
    
    openplot();

/* now the initial sizing and reticle construction */

    clat =   atof(argv[1]);
    clon =   atof(argv[2]);
    xscale = atof(argv[3]);
    
    for (i =  0; i <  3; i++) mapcolor[i] = COASTCOLOR;
    for (i =  3; i < 32; i++) mapcolor[i] = RIVERCOLOR;
    for (i = 32; i < 48; i++) mapcolor[i] = BOUNDCOLOR;
    for (i = 48; i < 64; i++) mapcolor[i] = BOUNDCOLOR;
   
    yscale = xscale * cos(clat*0.01745329252);

    wN = (long)( (clat + yscale/2.0) * UNITPERDEG );
    wS = (long)( (clat - yscale/2.0) * UNITPERDEG );
    wE = (long)( (clon + xscale/2.0) * UNITPERDEG );
    wW = (long)( (clon - xscale/2.0) * UNITPERDEG );
    ys = YRES / (float)(wN - wS);	/* 512 -> 0..512 on calls to draw */
    xs = XRES / (float)(wE - wW);

    latgridcount = pointtolat(0) - pointtolat(511);
    latgridwidth = 1.0;
    while (latgridcount > 5.0)
	{
	latgridwidth *= 2.0;
	latgridcount /= 2.0;
	}
    while (latgridcount < 3.0)
	{
	latgridwidth /= 2.0;
	latgridcount *= 2.0;
	}

    longridcount = pointtolon(511) - pointtolon(0);
    longridwidth = 1.0;
    while (longridcount > 5.0)
	{
	longridwidth *= 2.0;
	longridcount /= 2.0;
	}
    while (longridcount < 3.0)
	{
	longridwidth /= 2.0;
        longridcount *= 2.0;
	}
	
    latbase = intpart( pointtolat(511)*latgridwidth ) / latgridwidth;
    lonbase = intpart( pointtolon( 0 )*longridwidth ) / longridwidth;
    
    setcolor(GRIDCOLOR);
    for (i = -1.0; i < latgridcount + 1.0; ++i)
	{
	moveto( 0.0   , (double) lattopoint(latbase + i*latgridwidth) );
	drawto( 511.0 , (double) lattopoint(latbase + i*latgridwidth) );
	}
    for (i = -1.0; i < longridcount + 1.0; ++i)
	{
	moveto( (double) lontopoint(lonbase + i*longridwidth),   0.0 );
	drawto( (double) lontopoint(lonbase + i*longridwidth), 511.0 );
	}

/* now the top level count loop */

    while( read(mapindex, &count, 2) != 0)
	{
	read(mapindex, &rank, 2);
	read(mapindex, &shortn, 2);
	read(mapindex, &shorts, 2);
	read(mapindex, &shorte, 2);
	read(mapindex, &shortw, 2);
	
	setcolor( mapcolor[rank] );
	
	n = shortn * 20;
	s = shorts * 20;
	e = shorte * 20;
	w = shortw * 20;
	
	if (s <= wN && n >= wS && w <= wE && e >= wW)
	    {
	    read(mapdata, &lat, 4);
	    read(mapdata, &lon, 4);
	    moveto( ((float)(lon - wW)) * xs, ((float)(lat - wS)) * ys);
	    for (i=0; i < (count - 1); i++)
		{
		read(mapdata, &dlat, 1);
		read(mapdata, &dlon, 1);
		lat += dlat;
		lon += dlon;
		drawto( ((float)(lon - wW)) * xs, ((float)(lat - wS)) * ys);
		}
	    }
/* skip forward from cur pos */
	else lseek(mapdata, (long)(2*count+6), 1l);
	}

    close(mapdata);
    close(mapindex);
    closeplot();
    exit(0);
    }

/*
 * INTERFACE TO UNIX PLOT UTILITIES
 */

openplot()
    {
    openpl();
    erase();
    space(0, 0, XRES, YRES);
    }
    
closeplot()
    {
    closepl();
    }
    
/* move and draw do simple clipping because grid is larger than viewport */

moveto(x, y)
    double x, y;
    {
    int ix, iy;
    ix = (int)(x+0.5);
    iy = (int)(y+0.5);
    if (ix < 0) ix = 0; else if (ix >= XRES) ix = XRES-1;
    if (iy < 0) iy = 0; else if (iy >= YRES) iy = YRES-1;
    if ((lastx != ix) || (lasty != iy)) move(ix, iy);
    lastx = ix;
    lasty = iy;
    }

drawto(x, y)
    double x, y;
    {
    int ix, iy;
/* one could filter based on the present vector "color", if desired */
    ix = (int)(x+0.5);
    iy = (int)(y+0.5);
    if (ix < 0) ix = 0; else if (ix >= XRES) ix = XRES-1;
    if (iy < 0) iy = 0; else if (iy >= YRES) iy = YRES-1;
    if ((lastx != ix) || (lasty != iy)) cont(ix, iy);
    lastx = ix;
    lasty = iy;
    }

setcolor(c)
    {
    /* globalcol = c;	/*	/* one could record present vector color */
    }
