/*
 * Copyright 1987 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of my example below.
 */

/*
 *
 *  source file:   ./filters/genlib/polyfix.c
 *
 * Joe Dellinger (SEP), June 11 1987
 *	Inserted this sample edit history entry.
 *	Please log any further modifications made to this file:
 * Joe Dellinger Feb 29, 1988
 *	Changed POLYS to 5000 so it will compile on RT's.
 *	Somebody should change the whole thing into linked lists!
 */

#include <stdio.h>
#include "../include/err.h"
#include "../include/extern.h"
#include "../include/params.h"
#define EMPTY -100
#define NOLINK -10
#define ENDOFLIST -1
#define UNCLAIMED 0
#define CLAIMED 1
#define INTERIOR 0
#define POLYS   5000	/* Memory alloted for polygon */
#define MAXPOL  500	/* Maximum number of polygons made */

/*
 * This routine breaks up the clipped polygon generated by "polygon" into
 * multiple polygons as needed, using the Thorson-Dellinger method!
 *
 *	Note 1:
 *	The output of "polygon" is a clipped polygon with different parts
 *	joined by lines around the edge of the window. (This is unsightly.)
 *	This routine seems to do a reasonably efficient job of fixing this.
 *	(It's a pretty hard problem!)
 *
 *	Note 2:
 *	This always (I hope) works for non-crossed polygons. Crossed polygons
 *	will sometimes be turned into two overlapping polygons,
 *	which can cause interior voids to be shaded. Some terminals,
 *	like the color tektronix, can be sent multiple polygons
 *	and told that they are to be shaded as one. If this is the
 *	case, the routine should always (I think) work, even for crossed
 *	polygons. To fix this problem, you would have to laboriously
 *	look for intersections of "disjoint" polygons. A new point would
 *	then be added at the intersection, and the two polygons joined
 *	into one through that point. The required routines are pretty much
 *	already in this program, but it seems that the large amount of
 *	time to check for this obscure case is not worth it. There seems
 *	to be no obvious way around this.
 *
 *		--- Joe Dellinger, May 7, 1984.
 *
 *	Looking back on this I realize that I still had a lot to learn
 *	about C coding when I wrote this; some kind soul should go back
 *	and update this to use pointers and allocated linked-lists.
 *	I don't feel like doing this myself; the thing works perfectly
 *	for us even if the code itself is a mess!
 *
 *		--- Joe Dellinger, May 3, 1987.
 */

static int      poly[POLYS][5];
/*
 * Doubly linked list that will contain the vertices of our polygons. 0-X 1-Y
 * 2-forward link 3-backward link 4-which edge? 1-bottom 2-left 4-top 8-right 
 */
static int      pols[MAXPOL];	/* Point to the start of each polygon */
static int      polsc[MAXPOL];	/* Cycle length */
static int      npols;	/* How many polygons we have */

static int      pedge[POLYS];	/* Edge points of the polygon */
static int      nedge, nedge2;	/* number of edge points */

static int      point, endlist;

int             bottom, left;


int             Allgone;/* Communicate to polygon whether we ever got
			 * anything */

polyfix (x, y, first)	/* Read in the data for polystart */
    int             x, y, *first;
{
static int      oldx, oldy;

    if (*first == 1)
    {
	*first = 0;
	point = 1;
	Allgone = 0;		/* All the points aren't gone! */
	oldx = x + 1;
	oldy = y + 1;		/* Won't match */
    }
    /*
     * Put the point x,y into the list as number point, Unless it is *
     * superfluous 
     */
    if ((x == oldx) && (y == oldy))
	return;			/* ignore repeated points */
    poly[point][0] = x;		/* X of vertex */
    poly[point][1] = y;		/* Y of vertex */
    oldx = x;
    oldy = y;
    point++;
    if (point >= POLYS)
	ERR (FATAL, name, "Not enough memory for polygon!");
}

polystart ()		/* Start working on the polygons */
{
int             i, j, k, l, ii;
int             firstpoint, flag, double_check;
int             temp1, temp2, temp;
extern int      smart_clip;

    endlist = point;		/* Last element in use */
    /* initialize array */
    for (i = 0; i <= point; i++)
    {
	poly[i][4] = INTERIOR;
	poly[i][2] = i + 1;
	poly[i][3] = i - 1;
    }
    poly[0][0] = poly[0][1] = EMPTY;
    poly[point][0] = poly[point][1] = EMPTY;

    /* fix links of free list and last point */
    poly[0][3] = NOLINK;
    poly[0][2] = point;
    poly[point][2] = ENDOFLIST;
    poly[point][3] = 0;		/* point to start of free list */

    poly[point - 1][2] = 1;	/* point to start of polygon */
    poly[1][3] = point - 1;

    /* Now we have our polygon stored as a circular double-linked list! */

    if (smart_clip)
	goto skip_it;		/* smart device can handle it from here */


    /* Identify all edge points */

    /*
     * They are identified 2 ways; each point contains the information
     * itself, and there is an array which points to all edge points. This is
     * to avoid searching through the entire polygon. 
     */

    nedge = 0;
    i = firstpoint = 1;		/* Since we are just starting, we know the
				 * polygon MUST start here */
    do
    {
	if (poly[i][4] = edge (poly[i][0], poly[i][1]))	/* NOT an == */
	{
	    /* It's on an edge */
	    nedge++;
	    pedge[nedge] = i;
	}
	i = poly[i][2];
    }
    while (firstpoint != i);

    nedge2 = nedge;

    /*
     * Look at each edge segment. See if any edge points occur inside
     * segments joining two other edge points. (Got that?) 
     */

    for (ii = 1; ii <= nedge2; ii++)
    {
	i = pedge[ii];
	if (k = (poly[i][4] & poly[poly[i][2]][4]))
	{
	    /* We have an edge link. That is, it connects to edge vertices. */
	    /* See if any other edge points lie inbetween these two. */
	    for (j = 1; j <= nedge; j++)
	    {
		if (k & poly[pedge[j]][4])
		{
		    /* This point is on the correct edge */

		    l = ((k & (8 + 2)) > 0);

		    /* l = 0 if top or bottom, 1 if left or right */

		    if (poly[i][l] > poly[poly[i][2]][l])
		    {
			if ((poly[pedge[j]][l] > poly[poly[i][2]][l])
			    && (poly[pedge[j]][l] < poly[i][l]))
			    /* Got one! */
			    /* insert it into the list */
			{
			    nedge2++;
			    insert (i, poly[pedge[j]][0],
				    poly[pedge[j]][1],
				    poly[pedge[j]][4]);
			    pedge[nedge2] = poly[i][2];
			}
		    }
		    else
		    {
			if ((poly[pedge[j]][l] < poly[poly[i][2]][l])
			    && (poly[pedge[j]][l] > poly[i][l]))
			    /* Got one! */
			    /* insert it into the list */
			{
			    nedge2++;
			    insert (i, poly[pedge[j]][0],
				    poly[pedge[j]][1],
				    poly[pedge[j]][4]);
			    pedge[nedge2] = poly[i][2];
			}
		    }
		}
		/*
		 * Whether you found one or not, continue on. If we did
		 * insert one, IT is now the next point and our interval has
		 * gotten shorter. Some sort of sorting would be more
		 * effecient, but it doesn't seem worth the extra complexity
		 * at this time of night. It would involve a lot of special
		 * cases, whereas doing it this way doesn't. 
		 */
	    }
	}
    }

    nedge = nedge2;		/* Keep track of all the new edge points we
				 * just created */

    /*
     * We have now inserted all the required points. Look for links that
     * connect the same two positions. (But different vertices... That is,
     * vertices which are not consecutive but have the same location!) 
     */
    /* Be careful that the link is not of zero length! */
    /*
     * Despite everything, it still seems to occasionally miss a spot. So as
     * an easy fix make it go through the search until it doesn't find
     * anything. In the overwhelming majority of cases, the first pass will
     * have done the job. 
     */
    double_check = 1;
    while (double_check == 1)
    {
	double_check = 0;

	for (j = 1; j <= nedge; j++)	/* Loop through all edge points */
	{
	    flag = 1;
	    while (flag == 1)
	    {
		flag = 0;
		if ((poly[pedge[j]][4] & poly[poly[pedge[j]][2]][4])
		    && ((poly[pedge[j]][0] != poly[poly[pedge[j]][2]][0])
		      || (poly[pedge[j]][1] != poly[poly[pedge[j]][2]][1])))
		{
		    /*
		     * This edge point's succesor is a point on the same
		     * edge, And the link is not of zero length 
		     */
		    for (k = j; k <= nedge; k++)	/* Loop through edge
							 * points */
		    {
			/* Look for a vertex that is at the same position */
			if ((poly[pedge[j]][0] == poly[pedge[k]][0])
			    && (poly[pedge[j]][1] == poly[pedge[k]][1]))
			{
			    /*
			     * OK, it is in the same position. See if either
			     * its succesor or predecessor is the same as the
			     * other end of the link. 
			     */
			    if ((poly[poly[pedge[j]][2]][0] ==
				 poly[poly[pedge[k]][3]][0])
				&& (poly[poly[pedge[j]][2]][1] ==
				    poly[poly[pedge[k]][3]][1]))
			    {
				/*
				 * links go in opposite directions. (The easy
				 * case) Break the links and instead link
				 * points at the same location. Delete
				 * duplicated point. 
				 */
				/*
				 * pedge[j] and pedge[k] are at the same
				 * spot, as are pedge[j]'s succesor and
				 * pedge[k]'s predecessor. 
				 */
				temp1 = poly[pedge[j]][2];
				/* remember pedge[j]'s succesor */
				poly[pedge[j]][2] = pedge[k];
				/* pedge[j]'s succesor is now pedge[k] */
				temp2 = poly[pedge[k]][3];
				/* remember pedge[k]'s predecessor */
				poly[pedge[k]][3] = pedge[j];
				/*
				 * pedge[j] and pedge[k] are now linked. Link
				 * temp1 and temp2 
				 */
				poly[temp1][3] = temp2;
				poly[temp2][2] = temp1;
				/*
				 * Done. We have just split one polygon into
				 * two! 
				 */
				/* Clean up by removing the repeated vertices */
				delete (pedge[k]);
				pedge[k] = 0;
				/*
				 * Point this to a place we know is marked as
				 * an interior point. Always fails checks to
				 * see if it is on the edge we want. 
				 */
				delete (temp2);
				flag = 1;
				double_check = 1;
			    }
			    else
			    {
				if ((j != k) &&
				    (poly[poly[pedge[j]][2]][0] ==
				     poly[poly[pedge[k]][2]][0])
				    && (poly[poly[pedge[j]][2]][1] ==
					poly[poly[pedge[k]][2]][1]))
				{
				    /*
				     * the hard case. Both links go the same
				     * direction. Do as before, but Reverse
				     * the direction of one of the two
				     * pieces. 
				     */
				    temp1 = poly[pedge[j]][2];
				    temp2 = poly[pedge[k]][2];
				    i = temp1;
				    do
				    {	/* Reverse one piece first */
					temp = poly[i][2];
					poly[i][2] = poly[i][3];
					poly[i][3] = temp;
					i = temp;
				    }
				    while ((temp2 != i) && (temp1 != i));
				    poly[temp1][2] = temp2;
				    poly[temp2][3] = temp1;
				    if (i == temp2)
				    {
					poly[pedge[j]][2] = pedge[k];
					poly[pedge[k]][3] = pedge[j];
					/*
					 * We have not created a new polygon,
					 * merely re-ordered one! 
					 */
				    }
				    else
				    {
					poly[pedge[j]][3] = pedge[k];
					poly[pedge[k]][2] = pedge[j];
					/*
					 * We have merged two polygons back
					 * into one! 
					 */
				    }
				    /* clean up */
				    delete (pedge[k]);
				    pedge[k] = 0;
				    delete (temp2);
				    flag = 1;
				    double_check = 1;
				}
			    }
			}
			if (flag == 1)
			    break;
			/*
			 * We've just altered the point we're at! So better
			 * stop this loop. 
			 */
		    }		/* End of search loop */
		}
	    }
	}
    }

    /*
     * Our polygon has been fragmented into multiple smaller polygons as
     * necessary! Output the resulting polygons 
     */

skip_it:
    scan ();

    for (i = 1; i <= npols; i++)
    {
	dev.startpoly (polsc[i]);
	j = pols[i];
	do
	{
	    dev.midpoly (poly[j][0], poly[j][1]);
	    j = poly[j][2];
	}
	while (j != pols[i]);
	dev.endpoly (i == npols);
    }
}

insert (where, x, y, z)
    int             where, x, y, z;
{
int             temp;

    /* insert the given vertex between the element where and its forward link */

    /* remove an element from the free list */
    temp = poly[0][2];		/* free element */
    if (temp == endlist)
    {
	/* Need to make our list one longer. */
	endlist++;
	if (endlist >= POLYS)
	    ERR (FATAL, name, "Ran out of memory on the polygon!");
	poly[endlist][3] = temp;
	poly[endlist][2] = ENDOFLIST;
	poly[temp][2] = endlist;
    }
    /* OK, Now you can remove it, it isn't at the end anymore. */
    poly[0][2] = poly[temp][2];
    poly[poly[temp][2]][3] = 0;

    poly[temp][0] = x;		/* Put vertex in for element */
    poly[temp][1] = y;
    poly[temp][4] = z;

    /* update links */

    poly[temp][2] = poly[where][2];
    poly[where][2] = temp;

    poly[temp][3] = poly[poly[temp][2]][3];
    poly[poly[temp][2]][3] = temp;
}

delete (where)
    int             where;
{
int             temp;

    /* Get rid of element number where; put it on the free list */
    /* This must work even if element where points to itself! */
    poly[where][0] = poly[where][1] = EMPTY;
    poly[where][4] = INTERIOR;
    temp = poly[where][3];
    poly[where][3] = 0;
    poly[temp][2] = poly[where][2];
    poly[where][2] = poly[0][2];
    poly[0][2] = where;
    poly[poly[where][2]][3] = where;
    poly[poly[temp][2]][3] = temp;
}

/*
 * See how many polygons we have, and where they start.
 * Put this information in pols.
 */
scan ()
{
int             polyon[POLYS];	/* Array to remember whom this point belongs
				 * to */
int             joe;
int             cycle;
int             firstpoint;
int             where;

    for (joe = 1; joe < endlist; joe++)
    {
	if (poly[joe][0] == EMPTY)
	{
	    polyon[joe] = EMPTY;
	}
	else
	{
	    polyon[joe] = UNCLAIMED;
	}
    }

    npols = 0;
    for (joe = 1; joe < endlist; joe++)
    {
	if (polyon[joe] == UNCLAIMED)
	{
	    /* Found the start of a polygon */
	    firstpoint = joe;
	    polyon[firstpoint] = CLAIMED;
	    cycle = 1;
	    where = poly[firstpoint][2];
	    while (firstpoint != where)
	    {
		cycle++;
		polyon[where] = CLAIMED;
		where = poly[where][2];
		if (where == NOLINK)
		    ERR (FATAL, name, "Oh, No! Polygon list damaged! (This shouldn't be able to happen.)");
	    }
	    if (cycle < 3)
	    {			/* Not really a polygon, remove it */
		where = poly[firstpoint][2];
		while (firstpoint != where)
		{
		    delete (where);
		    where = poly[firstpoint][2];
		}
		delete (firstpoint);
	    }
	    else
	    {
		/* We found another polygon */
		npols++;
		if (npols > MAXPOL)
		    ERR (FATAL, name, "Too many polygons!");
		pols[npols] = firstpoint;
		polsc[npols] = cycle;
	    }
	}
    }
}
 /*
  * Now pols is a list of indexes to the beginnings of each of the  Polygons.
  * npols gives how many there are. 
  */

int
edge (x, y)		/* Find out which edges this point is on */
    int             x, y;
{
    bottom = left = 0;
    if (x == xwmin)
	left = 2;
    if (y == ywmin)
	bottom = 1;
    if (x == xwmax)
	left = 8;
    if (y == ywmax)
	bottom = 4;
    return (bottom + left);
}
