/*
 * Copyright 1987 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of my example below.
 */

/*
 *
 *  source file:   ./filters/imaglib/imagraster.c
 *
 * Joe Dellinger (SEP), June 11 1987
 *	Inserted this sample edit history entry.
 *	Please log any further modifications made to this file:
 * Joe Dellinger Feb 16 1988
 *	Update to new raster call.
 */

#include <stdio.h>
#include "../include/extern.h"
#include "../include/enum.h"
#include "../include/err.h"
#include "imagen.h"

extern int      overlay;
extern char    *calloc ();
extern FILE    *pltout;
extern int      overlay;

/*
 * raster for the (barf) imagen-300, which can only align things on multiples
 * of 32 by 32! This routine does what is necessary to make the imagen ACT
 * as if it could place it arbitrarily. - Joe Dellinger, Nov 1986
 */

imagraster (count, out_of, xout, yout, length, orient, raster, dummy1, dummy2)
    int             count, out_of, xout, yout, length, orient, dummy1, dummy2;
    char           *raster;
{
int             ii, jj, kk, ll, base;
int             xgoof_s, ygoof_s, xgoof_e, ygoof_e, length_x, length_y;
char            byte;
static int      num_lines;
static cha
r    *array;
static int      ylength, length_p;
static int      xgoof, ygoof, xpatch;
static int      overlay_im;

    if (count == 0)
    {
/*
 * First time through. Do various things that need to be done once at the
 * beginning:
 */

	switch (overlay)
	{
	case 0:
	    overlay_im = OPAQUE;
	    break;
	case 1:
	    overlay_im = OR;
	    break;
	}

	/*
	 * Treasure the raster up into 32 byte blocks. This is actually a
	 * constant, but I'll let it be a variable so that the similarity of
	 * this code to genraster1.c is more clear. 
	 */
	num_lines = 32;

	/* End any unfinished path */
	endpath (DRAW_PATH, imag_curcolor);

	/*
	 * Imagen can only position raster every 32 pixels... 
	 *
	 * The complexity here comes about because the truncation occurs in
	 * physical, unrotated coordinates, whereas we're working in
	 * (possibly) rotated and opposite-handed ones. Ug! 
	 */

	/* Dimensions in each direction in Imagen physical coordinates  */
	if (orient == 1 || orient == 3)
	{
	    length_x = out_of;
	    length_y = length;
	}
	else
	{
	    length_x = length;
	    length_y = out_of;
	}

	/* Find extremes of padded raster area */
	if (orient == 0 || orient == 3)
	{
	    /* X direction start truncates down */
	    xgoof_s = num_lines * div_round (xout, num_lines);
	    xgoof = xout - xgoof_s;
	    /* X direction end truncates up */
	    xgoof_e = num_lines * div_round ((xout + (length_x - 1) + num_lines), num_lines);
	}
	else
	{
	    /* X direction start truncates up */
	    xgoof_s = num_lines * div_round ((xout + num_lines), num_lines);
	    xgoof = -(xout - xgoof_s) - 1;
	    /* X direction end truncates down */
	    xgoof_e = num_lines * div_round ((xout - (length_x - 1)), num_lines);
	}

	if (orient == 0 || orient == 1)
	{
	    /* Y direction start truncates up */
	    ygoof_s = num_lines * div_round ((yout + num_lines), num_lines);
	    ygoof = -(yout - ygoof_s) - 1;
	    /* Y direction end truncates down */
	    ygoof_e = num_lines * div_round ((yout - (length_y - 1)), num_lines);
	}
	else
	{
	    /* Y direction start truncates down */
	    ygoof_s = num_lines * div_round (yout, num_lines);
	    ygoof = yout - ygoof_s;
	    /* Y direction end truncates up */
	    ygoof_e = num_lines * div_round ((yout + (length_y - 1) + num_lines), num_lines);
	}

	/*
	 * Tell the imagen where to put it while we're still in physical
	 * coord's 
	 */
	putc (SET_ABS_H, pltout);
	putc (hib (ygoof_s), pltout);
	putc (lob (ygoof_s), pltout);
	putc (SET_ABS_V, pltout);
	putc (hib (xgoof_s), pltout);
	putc (lob (xgoof_s), pltout);

	if (orient == 1 || orient == 3)
	{
	    ii = xgoof;
	    xgoof = ygoof;
	    ygoof = ii;
	    ii = xgoof_s;
	    xgoof_s = ygoof_s;
	    ygoof_s = ii;
	    ii = xgoof_e;
	    xgoof_e = ygoof_e;
	    ygoof_e = ii;
	    ii = xout;
	    xout = yout;
	    yout = ii;
	}
/* 
 * xgoof and ygoof now tell us how much padding we have to insert to
 * correct for the imagen's mispositioning goofs.
 */

	/*
	 * Rotate the imagen's internal coordinate system so that the main
	 * thrust of its raster direction is in the same direction as
	 * Vplot's. 
	 */
	putc (SET_HV_SYSTEM, pltout);
	putc (0x01, pltout);

	/*
	 * If rotated coordinates, then rotate on top of this to match 
	 */
	if (orient == 1 || orient == 3)
	{
	    putc (SET_HV_SYSTEM, pltout);
	    putc (0x01, pltout);
	}

	/*
	 * If flipped coordinates, then flip on top of this to match 
	 */
	if (orient == 2 || orient == 3)
	{
	    putc (SET_HV_SYSTEM, pltout);
	    putc (0x02, pltout);
	}

	/*
	 * Allocate an array to hold 32 raster lines at a time. Use calloc so
	 * that we start out full of 0's, some of which we'll need for
	 * padding. length_p gives length rounded to the correct multiple of
	 * 32, which is needed for the raster plus the padding. xpatch tells
	 * us how many patches we'll need per line. 
	 */

	length_p = abs (xgoof_e - xgoof_s);
	xpatch = length_p / num_lines;
	array = (char *) calloc (num_lines * length_p, sizeof (char));
	if (array == NULL)
	    ERR (FATAL, name, "Can't allocate %d bytes for imagen raster buffer",
		 num_lines * length_p * sizeof (char));

	/*
	 * Displace where we start into the block the first time through to
	 * do the required padding. 
	 */
	ylength = ygoof;
    }

/*
 * Each time we're called, we get another line of the raster image.
 * Save this line in the block we're currently storing up, converting
 * to monochrome as we do so.
 */
    for (ii = 0; ii < length; ii++)
    {
	array[length_p * ylength + ii + xgoof] = (raster[ii] > 0);
    }
    ylength++;

/*
 * See if we've saved up an entire block yet. Also, if this is the last
 * call, we'd better not wait around either.
 */
    if (ylength >= num_lines || count == out_of - 1)
    {
	if (count == out_of - 1)
	{
	    /*
	     * Clear out any garbage left in the part of the patch that's
	     * padding. Only need to worry about this the last time, when we
	     * may not have filled in the entire patch. 
	     */
	    for (jj = ylength; jj < num_lines; jj++)
	    {
		for (ii = 0; ii < length; ii++)
		{
		    array[length_p * jj + ii + xgoof] = 0;
		}
	    }
	}

	/*
	 * Tell the imagen a bitmap is coming 
	 */
	putc (BITMAP, pltout);
	/* Set the overlay mode as given in Vplot */
	putc (overlay_im, pltout);
	/* How many patches needed to draw this line */
	putc (xpatch, pltout);
	/* Each Y line gets a separate call of its own */
	putc (1, pltout);

	/* Finally, here are the actual bits */
	for (ii = 0; ii < xpatch; ii++)
	{
	    /* Loop over patches */
	    for (jj = 0; jj < num_lines; jj++)
	    {
		/* Loop over 32 rows in a patch */
		for (kk = 0; kk < 4; kk++)
		{
		    /*
		     * 4 for the 4 bytes encoding 32 bits for the 32 pixels
		     * in each row of a patch 
		     */
/*
 * Here we calculate base offset into our array. The bizarre ordering
 * is due to the bizarre order in which the imagen wants the bits.
 */
		    base = ii * num_lines + jj * length_p + kk * 8;

		    byte = 0x0;
		    for (ll = 0; ll < 8; ll++)
		    {
			/* Loop over the 8 bits in each byte */

			byte = (byte << 1) | (array[base + ll] & 0x1);
		    }
		    /* Another byte rides the bus */
		    putc (byte, pltout);
		}
	    }
	}

	if (count != out_of - 1)
	{
/*
 * Now we've done a block. Reset our position in a block to 0 again, and
 * advance the imagen's current position one block-length. (Drawing raster
 * doesn't change the imagen's "current pen position".) Pointless to do this
 * stuff if we're never coming back.
 */
	    ylength = 0;
	    putc (SET_REL_V, pltout);
	    putc (hib (num_lines), pltout);
	    putc (lob (num_lines), pltout);
	}
	else
	{
/*
 * Things that have to be done before we exit the last time.
 */
	    /* Free up the array */
	    free (array);
	    /* Put the imagen's coordinate system back again */
	    if (orient == 2 || orient == 3)
	    {
		putc (SET_HV_SYSTEM, pltout);
		putc (0x02, pltout);
	    }
	    if (orient == 1 || orient == 3)
	    {
		putc (SET_HV_SYSTEM, pltout);
		putc (0x03, pltout);
	    }
	    putc (SET_HV_SYSTEM, pltout);
	    putc (0x03, pltout);
	}
    }
}

div_round (a, b)
    int             a, b;
{
    if ((a >= 0) == (b >= 0))
	return (a / b);
    else
    {
	if (b > 0)
	    return (a - (b - 1)) / b;
	else
	    return (-a - (-b - 1)) / -b;
    }
}
