/*
 * Ta2vplot - reads in a Taplot/Movie format file and outputs a vplot raster
 * file. 
 *
 * <moviefile.h  Ta2vplot xll=1. yll=1. xur=5. yur=5. nreserve=8 orient=1 invert=y
 * title=, label1=, label2=, titlefat=, titlesz= labelfat= labelsz= d1num=0.
 * o1num= d2num=0. o2num= grid=y frame=y polarity=1 coltab=y labelrot=0
 * axiscol=7 color=I backcol=red,green,blue hurry=y > vplotfile.H 
 *
 * esize:
 * 
 * If esize=1, then the input file is in "Taplot" format. Default is color=I.
 *
 * If esize=3, then the input file is assumed to contain RGB 3-byte triples.
 * The color number that corresponds to the "closest available color"
 * to that defined by the triple is used. Default is color=332.
 * Using Movie color schemes is slow. (BEWARE "color" set to the wrong
 * value by your history file!!!)
 *
 * Positioning the plot: 
 *
 * xll,yll are the coordinates of the lower left corner of the plot. xur,yur are
 * the coordinates of the upper right corner. orient=0,1,2,3 to start plot in
 * upper left, upper right, lower right, or lower left. invert=y 'flips' output
 * image to agree with movie display. 
 *
 * Annotation: 
 *
 * title,label1, and label2 are strings for axis labelling. titlefat and titlesz
 * are the fatness and size of the title. labelfat and labelsz are the fatness
 * and size of the axis labels. labelrot = 1 rotates label1 180 degrees o1num
 * and d1num are the starting value and increment for labelling the n1 axis.
 * o2num and d2num are the starting value and increment for labelling the n2
 * axis. grid=y places a grid line for each label. frame=y draws a frame around
 * the plot. axiscol determines the frame, label and title colors d1num = 0
 * turns off the labelling. d1num and d2num MUST be specified. negative fatness
 * turns off the title or labels. NOTE: labelling works properly only for
 * invert=y orient=1 at this time! 
 *
 * polarity=-1 flips the raster values black to white and white to black 
 *
 * If coltab=n, no color table definition is prepended to the vplot file. nreserve
 * is the number of colors for the plotting device that you want to be left
 * "free", ie not used in the raster color table. This prevents Ta2vplot from
 * overwriting vplot type colors with the grey scale used to display movie
 * files. 
 *
 * color= one of {I,i,F,R,W,G} just like in Movie, and also RGB, where
 * each of R,G, and B is the number of bits to allot to that primary.
 *
 * backcol=red,green,blue sets the background color. Each of {red,green,blue} are
 * between 0. and 1. 
 *
 * movish=n (if y, halves the color table like movie does) 
 *
 * The resulting vplot file can be displayed using any vplot pen filter.
 * Various device-dependent corrections are automatically made by vplot
 * to compensate for the peculiarities of each display device. Some of
 * these corrections can be controlled by the user. See "man pen" for
 * more details.
 *
 * Keywords: vplot plot movie raster hardcopy rgb color
 */
/*
 * EDIT HISTORY:
 * Author Steve Cole, March 1987
 * Joe Dellinger, Dec 1987 fixed "color value too large" bug.
 * Steve Cole, Dec 1987, changed putch call for movish to avoid problem
 *     on suns.
 * Joe Dellinger, Jan 1988, fixed some bugs involving color mapping,
 *	and added capability for RGB triples.
 * Joe Dellinger Feb 25 1988, vp_style after vp_erase
 */
#ifndef SOURCE
#define SOURCE "/usr/src/sepsrc/Ta2vplot.c"
#endif
#include <sep.main>
#include <stdio.h>
#include <strings.h>
#include <vplot.h>

#define SQUARE(A) ((A)*(A))

int             cintensityp[9] =
{
 0, 128, 255, 0, 128, 255, 0, 128, 255
};
int             cintensityn[9] =
{
 255, 128, 0, 255, 128, 0, 255, 128, 0
};
int             cflag[9] =
{
 255, 255, 0, 0, 255, 0, 0, 255, 255
};
int             cbi[9] =
{
 255, 0, 0, 0, 0, 0, 0, 0, 255
};
int             csurf[9] =
{
 0, 0, 255, 0, 0, 255, 0, 200, 255
};
int             cgsi[9] =
{
 0, 255, 255, 0, 255, 0, 0, 255, 0
};
int            *cola;

MAIN ()
{
int             n1, n2, n3, i3, i4, incr, ii, jj, kk, esize;
int             labelsz, titlesz, labelfat, titlefat;
int             labelrot, axiscol, sign;
float           d1, o1, d2, o2, d3, o3;
float           pos1, pos2;
float           xx, yy;
char            string[80];
float           d1num, d2num;
float           o1num, o2num;
int             grid, frame;
char            label1[80], label2[80], title[160];
int             offset, xpix, ypix, bit, blast, orient, invert;
float           xll, yll, xur, yur, ppi;
int             nreserve;
int             polarity, coltab;
unsigned char  *data;
char            color[10];
extern float    rd_color (), gr_color (), bl_color ();
float           backcol[3];
float           back = 0;
int             movish;
int             map[256];
int             smap[256];
float           red[256], green[256], blue[256];
float           red_lev, green_lev, blue_lev, error, error_win;
int             ired_lev, igreen_lev, iblue_lev;
int             win, redbit, greenbit, bluebit;
int             redbit2, greenbit2, bluebit2;
int             redoff, greenoff, blueoff;

    if (!fetch ("n1", "d", &n1))
	err ("n1 not given\n");
    if (!fetch ("n2", "d", &n2))
	err ("n2 not given\n");
    if (!fetch ("n3", "d", &n3))
	n3 = 1;
    if (!fetch ("esize", "d", &esize))
	err ("esize not given\n");

    if (esize != 1 && esize != 3)
	err ("esize must be 1 or 3\n");

    coltab = 1;
    fetch ("coltab", "1", &coltab);
    if (!coltab && esize == 3)
	err ("esize must be 1 if coltab=no\n");

    if (esize == 1)
	strcpy (color, "I");
    else
	strcpy (color, "332");
    fetch ("color", "s", color);

    if (!getch ("xll", "f", &xll))
	xll = 1.;
    if (!getch ("yll", "f", &yll))
	yll = 1.;
    if (!getch ("xur", "f", &xur))
	xur = 5.;
    if (!getch ("yur", "f", &yur))
	yur = 5.;
    if (!getch ("orient", "d", &orient))
	orient = 1;
    if (!getch ("nreserve", "d", &nreserve))
	nreserve = 8;
    if (!getch ("invert", "1", &invert))
	invert = 1;
    if (!getch ("movish", "1", &movish))
	movish = 0;

    putch_ ("movish", "d", &movish);

    puthead ("\tn1=-1\n");

    /* Polarity stuff */
    polarity = 1;
    fetch ("polarity", "d", &polarity);
    putch ("polarity", "d", &polarity);

    /* Axis and labelling parameters */
    strcpy (label1, "");
    strcpy (label2, "");
    strcpy (title, "");
    fetch ("label1", "s", label1);
    fetch ("label2", "s", label2);
    fetch ("title", "s", title);
    o1 = 0.;
    o2 = 0.;
    d1 = 1.;
    d2 = 1.;
    fetch ("o1", "f", &o1);
    fetch ("o2", "f", &o2);
    fetch ("d1", "f", &d1);
    fetch ("d2", "f", &d2);
    d1num = 0.;
    d2num = 0.;
    o1num = o1;
    o2num = o2;
    fetch ("d1num", "f", &d1num);
    fetch ("d2num", "f", &d2num);
    fetch ("o1num", "f", &o1num);
    fetch ("o2num", "f", &o2num);
    labelsz = 5;
    titlesz = 5;
    labelfat = 1;
    titlefat = 1;
    fetch ("labelsz", "d", &labelsz);
    fetch ("titlesz", "d", &titlesz);
    fetch ("labelfat", "d", &labelfat);
    fetch ("titlefat", "d", &titlefat);
    labelrot = 0;
    axiscol = 7;
    fetch ("labelrot", "1", &labelrot);
    fetch ("axiscol", "d", &axiscol);
    sign = -1;
    if (labelrot)
	sign = 1;

    frame = 1;
    grid = 1;
    fetch ("frame", "1", &frame);
    fetch ("grid", "1", &grid);

    if (fetch ("backcol", "f", backcol))
    {
	back = 1;
    }
    blast = 1;
    fetch ("hurry", "1", &blast);

    hclose ();

    data = (unsigned char *) alloc (n1 * n2 * esize);

    vp_erase ();
    vp_style (STANDARD);
    vp_color (axiscol);

    if (coltab)
    {

	if (color[0] >= '0' && color[0] <= '9')
	{
	    redbit = color[0] - '0';
	    greenbit = color[1] - '0';
	    bluebit = color[2] - '0';
	    if (redbit + greenbit + bluebit != 8)
		err ("You must use exactly 8 bits!\n");

	    redoff = 0;
	    greenoff = redbit;
	    blueoff = redbit + greenbit;

	    for (i3 = 0; i3 < 256; i3++)
	    {
		ii = ~(~0 << redbit);
		if (ii > 0)
		    red[i3] = (float) ((i3 >> redoff) & ii) / (float) (ii);
		else
		    red[i3] = 0.;

		ii = ~(~0 << greenbit);
		if (ii > 0)
		    green[i3] = (float) ((i3 >> greenoff) & ii) / (float) (ii);
		else
		    green[i3] = 0.;

		ii = ~(~0 << bluebit);
		if (ii > 0)
		    blue[i3] = (float) ((i3 >> blueoff) & ii) / (float) (ii);
		else
		    blue[i3] = 0.;
	    }

	    for (i3 = 256; i3 < 512; i3++)
	    {
		jj = i3 - 256;

		if (movish)
		    jj = 2 * (int) (jj / 2);
		vp_coltab (i3, red[jj], green[jj], blue[jj]);
	    }

	    for (jj = 0; jj < 256; jj++)
	    {
		ii = 0;
		greenbit2 = greenbit;
		bluebit2 = bluebit;
		redbit2 = redbit;
		kk = 0;
		while (kk < 8)
		{
		    greenbit2--;
		    if (greenbit2 >= 0)
		    {
			if (jj & (1 << (greenbit2 + greenoff)))
			    ii |= 1 << kk;
			kk++;
		    }

		    redbit2--;
		    if (redbit2 >= 0)
		    {
			if (jj & (1 << (redbit2 + redoff)))
			    ii |= 1 << kk;
			kk++;
		    }

		    bluebit2--;
		    if (bluebit2 >= 0)
		    {
			if (jj & (1 << (bluebit2 + blueoff)))
			    ii |= 1 << kk;
			kk++;
		    }
		}
		map[ii] = jj;
	    }

	    for (i3 = nreserve; i3 < 256; i3++)
	    {
		jj = i3 - nreserve;

		vp_coltab (i3, red[map[jj]],
			   green[map[jj]],
			   blue[map[jj]]);
	    }
	}
	else
	{
	    switch (*color)
	    {
	    case 'i':
		cola = cintensityn;
		break;
	    case 'F':
		cola = cflag;
		break;
	    case 'R':
		cola = cbi;
		break;
	    case 'W':
		cola = csurf;
		break;
	    case 'G':
		cola = cgsi;
		break;
	    case 'I':
	    default:
		cola = cintensityp;
		break;
	    }

	    /*
	     * set up a perfect grey scale in colors 256-511. the offset in
	     * vp_raster maps the pixels into that range. no device actually
	     * has such colors, so vplot looks at those colors the device
	     * does have (0 through some n) and picks out the one closest to
	     * the desired grey. thus by setting up black, white, 50% grey,
	     * 75% grey, 25% grey, ... in 0-255, we'll get the best grey
	     * scale for the device. 
	     *
	     * note: we leave colors 0 through nreserve-1 alone, since they are
	     * already defined for the device. 
	     *
	     */
	    for (i3 = 0; i3 < 256; i3++)
	    {
		red[i3] = rd_color ((float) i3);
		green[i3] = gr_color ((float) i3);
		blue[i3] = bl_color ((float) i3);
	    }

	    for (i3 = 256; i3 < 512; i3++)
	    {
		jj = i3 - 256;

		if (movish)
		    jj = 2 * (int) (jj / 2);
		vp_coltab (i3, red[jj], green[jj], blue[jj]);
	    }

	    /*
	     * set the lower grey scale, which will give us the best grey
	     * scale possible for the device. 
	     *
	     * we assume that the device already has colors white and black set. 
	     */

	    i3 = 0;
	    smap[i3] = 0;
	    i3++;
	    if (nreserve == 0)
	    {
		smap[i3] = 255;
		i3++;
	    }
	    for (incr = 2; incr < 512; incr *= 2)
	    {
		for (i4 = 1; i4 < incr; i4 += 2)
		{
		    jj = i4 * 256 / incr;
		    smap[i3] = jj;
		    i3++;
		    if (i3 == 256)
			break;
		}
	    }

	    for (i3 = nreserve; i3 < 256; i3++)
	    {
		jj = i3 - nreserve;

		vp_coltab (i3, red[smap[jj]],
			   green[smap[jj]],
			   blue[smap[jj]]);
	    }

	}
	if (back)
	    vp_coltab (0, backcol[0], backcol[1], backcol[2]);
    }
    /* Set the coordinate transformation */
    vp_orig (xll, yur);
    vp_uorig (o2, o1);
    vp_scale ((xur - xll) / (d2 * n2), (yll - yur) / (d1 * n1));


    /*
     * Do the plots 
     */

    for (i3 = 0; i3 < n3; i3++)
    {
	if (i3 != 0)
	{
	    vp_erase ();
	    vp_style (STANDARD);
	    vp_color (axiscol);
	}
	reed (infd, data, n1 * n2 * esize);
	if (polarity < 0)
	    for (ii = 0; ii < n1 * n2 * esize; ii++)
	    {
		data[ii] = (unsigned char) 255 - data[ii];
	    }
/*
 * If esize=3, then map the RGB triples onto the closest available
 * color.
 */
	if (esize == 3)
	{
	    if (color[0] >= '0' && color[0] <= '9')
	    {
		for (ii = 0; ii < n1 * n2; ii++)
		{
		    ired_lev = data[esize * ii];
		    igreen_lev = data[esize * ii + 1];
		    iblue_lev = data[esize * ii + 2];

		    win = 0;
		    win |= ((ired_lev >> (8 - redbit)) & ~(~0 << redbit)) << redoff;
		    win |= ((igreen_lev >> (8 - greenbit)) & ~(~0 << greenbit)) << greenoff;
		    win |= ((iblue_lev >> (8 - bluebit)) & ~(~0 << bluebit)) << blueoff;
		    data[ii] = win;
		}
	    }
	    else
	    {
		for (ii = 0; ii < n1 * n2; ii++)
		{
		    red_lev = data[esize * ii] / 255.;
		    green_lev = data[esize * ii + 1] / 255.;
		    blue_lev = data[esize * ii + 2] / 255.;
		    error_win = 8.;
		    for (jj = 0; jj < 256; jj++)
		    {
			error = 2. * SQUARE (red_lev - red[jj]) + 4. * SQUARE (green_lev - green[jj]) + SQUARE (blue_lev - blue[jj]);
			if (error < error_win)
			{
			    error_win = error;
			    win = jj;
			    if (error == 0.)
				break;
			}
		    }
		    data[ii] = win;
		}
	    }
	}

	/*
	 * Only offset the colors if we have defined a color table.
	 * Otherwise, leave them alone. 
	 */
	if (coltab)
	    offset = 256;
	else
	    offset = 0;

	xpix = n1;
	ypix = n2;
	bit = 0;
	ppi = 0;
	vp_raster (data, blast, bit, offset, xpix, ypix, xll, yll, ppi, &xur, &yur, orient, invert);

	vp_fat (1);
	if (frame)
	{
	    vp_move (xll, yll);
	    vp_draw (xll, yur);
	    vp_draw (xur, yur);
	    vp_draw (xur, yll);
	    vp_draw (xll, yll);
	}
	if (grid)
	{
	    if (d2num > 0)
	    {
		for (pos2 = o2num; pos2 <= o2 + n2 * d2; pos2 += d2num)
		{
		    vp_umove (pos2, o1);
		    vp_udraw (pos2, o1 + n1 * d1);
		}
	    }
	    if (d1num > 0)
	    {
		for (pos1 = o1num; pos1 <= o1 + n1 * d1; pos1 += d1num)
		{
		    vp_umove (o2, pos1);
		    vp_udraw (o2 + n2 * d2, pos1);
		}
	    }
	}
	if (d2num > 0)
	{
	    vp_fat (labelfat);
	    vp_tjust (TH_CENTER, TV_BASE);
	    for (pos2 = o2num; pos2 <= o2 + n2 * d2; pos2 += d2num)
	    {
		vp_umove (pos2, o1);
		vp_where (&xx, &yy);
		yy += .07;
		vp_draw (xx, yy);
		yy += .03;
		sprintf (string, "%g", pos2);
		vp_text (xx, yy, labelsz, 0, string);
	    }
	}
	if (d1num > 0)
	{
	    vp_fat (labelfat);
	    vp_tjust (TH_CENTER, TV_TOP);
	    for (pos1 = o1num; pos1 <= o1 + n1 * d1; pos1 += d1num)
	    {
		vp_umove (o2, pos1);
		vp_where (&xx, &yy);
		xx -= .07;
		vp_draw (xx, yy);
		xx -= .03;
		sprintf (string, "%g", pos1);
		vp_text (xx, yy, labelsz, -90, string);
	    }
	}
	vp_move ((xll + xur) / 2., yur);
	vp_where (&xx, &yy);
	yy += .07;
	yy += .03;
	yy += labelsz / TXPERIN;
	yy += .15;
	vp_fat (labelfat);
	vp_tjust (TH_CENTER, TV_BASE);
	vp_text (xx, yy, labelsz, 0, label2);

	yy += labelsz / TXPERIN;
	yy += .15;
	vp_fat (titlefat);
	vp_tjust (TH_CENTER, TV_BOTTOM);
	vp_text (xx, yy, titlesz, 0, title);

	vp_move (xll, (yll + yur) / 2.);
	vp_where (&xx, &yy);
	xx -= .07;
	xx -= .03;
	xx -= labelsz / TXPERIN;
	xx -= .15;
	vp_fat (labelfat);
	if (labelrot)
	    vp_tjust (TH_CENTER, TV_BOTTOM);
	else
	    vp_tjust (TH_CENTER, TV_TOP);
	vp_text (xx, yy, labelsz, sign * 90, label1);
    }
}

float
rd_color (x)
    float           x;
{
    if (x < 128)
	return (((128. - x) * (float) cola[0] / 128.) + (x * (float) cola[1] / 128.)) / 255.;
    else
	return (((255. - x) * (float) cola[1] / 127.) + ((x - 128.) * (float) cola[2] / 127.)) / 255.;
}

float
gr_color (x)
    float           x;
{
    if (x < 128)
	return (((128. - x) * (float) cola[3] / 128.) + (x * (float) cola[4] / 128.)) / 255.;
    else
	return (((255. - x) * (float) cola[4] / 127.) + ((x - 128.) * (float) cola[5] / 127.)) / 255.;
}

float
bl_color (x)
    float           x;
{
    if (x < 128)
	return (((128. - x) * (float) cola[6] / 128.) + (x * (float) cola[7] / 128.)) / 255.;
    else
	return (((255. - x) * (float) cola[7] / 127.) + ((x - 128.) * (float) cola[8] / 127.)) / 255.;
}
