#include <stdio.h>
#include <sys/types.h>
#ifdef MSC
#include <fcntl.h>
#include <stdlib.h>
#include <malloc.h>
#else
#include <sys/file.h>
#endif
#include "oldvort.h"

/*
 * oldopenimage
 *
 *	returns an image structure for image file name. Takes same arguments
 * as fopen(3), the only difference been if the file name "-" is used stdin
 * or stdout is assumed.
 */
oldimage *
oldopenimage(name, flag)
	char	*name;
	char	*flag;
{
	oldimage	*im;
	int		flags;

	if (name == (char *)0)
		return((oldimage *)NULL);

	if ((im = (oldimage *)malloc(sizeof(oldimage))) == (oldimage *)NULL)
		return((oldimage *)NULL);
	
	im->written = im->reading = 0;

#ifdef PC
	if (strcmp(flag, "r") == 0) {
		im->reading = 1;
		flags = O_RDONLY | O_BINARY;
	} else if (strcmp(flag, "w") == 0)
		flags = O_WRONLY | O_CREAT | O_TRUNC | O_BINARY;
	else {
		fprintf(stderr, "oldopenimage: bad open flag.\n");
		return((oldimage *)NULL);
	}
#else
	if (strcmp(flag, "r") == 0) {
		im->reading = 1;
		flags = O_RDONLY;
	} else if (strcmp(flag, "w") == 0)
		flags = O_WRONLY | O_CREAT | O_TRUNC;
	else {
		fprintf(stderr, "oldopenimage: bad open flag.\n");
		return((oldimage *)NULL);
	}
#endif

	if (strcmp(name, "-") == 0)
		im->fd = (im->reading) ? 0 : 1;
	else
		if ((im->fd = open(name, flags, 0664)) < 0)
			return((oldimage *)NULL);

	if (!im->reading) {
		im->ibufp = im->ibuf;		/* initialise bufp to start */
		return(im);
	}

	if (read(im->fd, &im->hdr, sizeof(pixhdr)) != sizeof(pixhdr))
		return((oldimage *)NULL);

	switch (oldimagetype(im)) {
	case OLDPIX_RGB:
	case OLDPIX_RLE:
	case OLDPIX_MIX:
	case OLDPIX_RGBA:
	case OLDPIX_RLEA:
	case OLDPIX_MIXA:
	case OLDPIX_CMAP:
	case OLDPIX_RLECMAP:
		break;
	default:
		fprintf(stderr, "oldopenimage: bad magic number in file.\n");
		return((oldimage *)NULL);
	}

	if (oldtitlelength(im) != 0) {
		oldimagetitle(im) = (char *)malloc(oldtitlelength(im));
		read(im->fd, oldimagetitle(im), oldtitlelength(im));
	} else
		oldimagetitle(im) = "";

	if (oldcolormapped(im)) {
		read(im->fd, &im->mapsize, sizeof(unsigned short));
		im->red = (unsigned char *)malloc(oldcmapsize(im));
		read(im->fd, im->red, oldcmapsize(im));
		im->green = (unsigned char *)malloc(oldcmapsize(im));
		read(im->fd, im->green, oldcmapsize(im));
		im->blue = (unsigned char *)malloc(oldcmapsize(im));
		read(im->fd, im->blue, oldcmapsize(im));
	}

	im->ebufp = im->ibufp = im->ibuf + IBUF_SIZE;

	return(im);
}

/*
 * oldimagepos
 *
 *	position the image file at pixel x, y. (at the moment this just gos to (0, 0))
 */
int
oldimagepos(im, x, y)
	oldimage	*im;
	int		x, y;
{
	int	offset;

	im->ebufp = im->ibufp = im->ibuf + IBUF_SIZE;

	if (oldimagetype(im) == OLDPIX_CMAP || oldimagetype(im) == OLDPIX_RLECMAP)
		return(lseek(im->fd, (long)(sizeof(pixhdr) + sizeof(im->mapsize) + im->hdr.tlength + 3 * oldcmapsize(im)), 0));


	return(lseek(im->fd, (long)(sizeof(pixhdr) + im->hdr.tlength), 0));
}

/*
 * oldreadbyte
 *
 *	returns a byte from the image or EOF if the end is reached
 */
int
oldreadbyte(im)
	oldimage		*im;
{
	int	count;

	if (im->ibufp == im->ebufp) {
		im->ibufp = im->ibuf;
		if ((count = read(im->fd, im->ibuf, IBUF_SIZE)) == 0)
			return(EOF);
		im->ebufp = im->ibuf + count;
	}

	return(*im->ibufp++);
}

/*
 * closeimage
 *
 *	close up an image file
 */
int
oldcloseimage(im)
	oldimage		*im;
{
	close(im->fd);
}

/*
 * oldreadrgbline
 *
 *	reads in a scanline from file in and puts it into red, green, and blue.
 */
int
oldreadrgbline(in, red, green, blue)
	oldimage		*in;
	unsigned char	*red, *green, *blue;
{
	char		alpha;
	register int	x, i;
	unsigned char	r, g, b, run = 0;

	switch (oldimagetype(in)) {
	case OLDPIX_CMAP:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			red[x] = in->red[i];
			green[x] = in->green[i];
			blue[x] = in->blue[i];
		}
		break;
	case OLDPIX_RLECMAP:
		x = 0;
		while (x < oldimagewidth(in)) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			run = i;
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			do {
				red[x] = in->red[i];
				green[x] = in->green[i];
				blue[x++] = in->blue[i];
			} while (--run != 255);
		}
		break;
	case OLDPIX_RGB:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			red[x] = i;
			green[x] = oldreadbyte(in);
			blue[x] = oldreadbyte(in);
		}
		break;
	case OLDPIX_RGBA:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			red[x] = i;
			green[x] = oldreadbyte(in);
			blue[x] = oldreadbyte(in);
			oldreadbyte(in);
		}
		break;
	case OLDPIX_RLE:
		x = 0;
		while (x < oldimagewidth(in)) {
			if (run == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;

				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);

				x++;
			} 
			if (run > 0)
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					run--;
					x++;
				}
		}
		break;
	case OLDPIX_RLEA:
		x = 0;
		while (x < oldimagewidth(in)) {
			if (run == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;
				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);
				oldreadbyte(in);
				x++;
			} 
			if (run > 0)
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					run--;
					x++;
				}
		}
		break;
	case OLDPIX_MIX:
		x = 0;
		while (x < oldimagewidth(in)) {
			if ((run & 0x7f) == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;
			} 
			if (run & 0x80) {
				while ((run & 0x7f) && x < oldimagewidth(in)) {
					red[x] = oldreadbyte(in);
					green[x] = oldreadbyte(in);
					blue[x] = oldreadbyte(in);
					run--;
					x++;
				}
			} else {
				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);
				x++;
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					run--;
					x++;
				}
			}
		}
		break;
	case OLDPIX_MIXA:
		x = 0;
		while (x < oldimagewidth(in)) {
			if (run == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;
			} 
			if (run & 0x80) {
				while ((run & 0x7f) && x < oldimagewidth(in)) {
					red[x] = oldreadbyte(in);
					green[x] = oldreadbyte(in);
					blue[x] = oldreadbyte(in);
					oldreadbyte(in);
					run--;
					x++;
				}
			} else {
				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);
				oldreadbyte(in);
				x++;
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					run--;
					x++;
				}
			}
		}
		break;
	default:
		fprintf(stderr, "oldreadrgbline: bad file format.\n");
		return(0);
	}

	return(1);
}

/*
 * oldreadrgbaline
 *
 *	reads in a scanline from file in and puts the scanline into red, green,
 * blue, and alpha.
 */
int
oldreadrgbaline(in, red, green, blue, alpha)
	oldimage		*in;
	unsigned char	*red, *green, *blue, *alpha;
{
	register int	x, i;
	unsigned char	r, g, b, a, run = 0;

	switch (oldimagetype(in)) {
	case OLDPIX_CMAP:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			red[x] = in->red[i];
			green[x] = in->green[i];
			blue[x] = in->blue[i];
			alpha[x] = 255;
		}
		break;
	case OLDPIX_RLECMAP:
		x = 0;
		while (x < oldimagewidth(in)) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			run = i;
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			do {
				red[x] = in->red[i];
				green[x] = in->green[i];
				blue[x++] = in->blue[i];
				alpha[x] = 255;
			} while (--run != 255);
		}
		break;
	case OLDPIX_RGB:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			red[x] = i;
			green[x] = oldreadbyte(in);
			blue[x] = oldreadbyte(in);
			alpha[x] = 255;
		}
		break;
	case OLDPIX_RGBA:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			red[x] = i;
			green[x] = oldreadbyte(in);
			blue[x] = oldreadbyte(in);
			alpha[x] = oldreadbyte(in);
		}
		break;
	case OLDPIX_RLE:
		x = 0;
		while (x < oldimagewidth(in)) {
			if (run == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;

				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);

				x++;
			} 
			if (run > 0)
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					alpha[x] = 255;
					run--;
					x++;
				}
		}
		break;
	case OLDPIX_RLEA:
		x = 0;
		while (x < oldimagewidth(in)) {
			if (run == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;
				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);
				a = alpha[x] = oldreadbyte(in);
				x++;
			} 
			if (run > 0)
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					alpha[x] = a;
					run--;
					x++;
				}
		}
		break;
	case OLDPIX_MIX:
		x = 0;
		while (x < oldimagewidth(in)) {
			if ((run & 0x7f) == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;
			} 
			if (run & 0x80) {
				while ((run & 0x7f) && x < oldimagewidth(in)) {
					red[x] = oldreadbyte(in);
					green[x] = oldreadbyte(in);
					blue[x] = oldreadbyte(in);
					alpha[x] = 255;
					run--;
					x++;
				}
			} else {
				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);
				x++;
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					alpha[x] = 255;
					run--;
					x++;
				}
			}
		}
		break;
	case OLDPIX_MIXA:
		x = 0;
		while (x < oldimagewidth(in)) {
			if (run == 0) {
				if ((i = oldreadbyte(in)) == EOF)
					return(0);
				run = i;
			} 
			if (run & 0x80) {
				while ((run & 0x7f) && x < oldimagewidth(in)) {
					red[x] = oldreadbyte(in);
					green[x] = oldreadbyte(in);
					blue[x] = oldreadbyte(in);
					alpha[x] = oldreadbyte(in);
					oldreadbyte(in);
					run--;
					x++;
				}
			} else {
				r = red[x] = oldreadbyte(in);
				g = green[x] = oldreadbyte(in);
				b = blue[x] = oldreadbyte(in);
				a = alpha[x] = oldreadbyte(in);
				oldreadbyte(in);
				x++;
				while (run && x < oldimagewidth(in)) {
					red[x] = r;
					green[x] = g;
					blue[x] = b;
					alpha[x] = a;
					run--;
					x++;
				}
			}
		}
		break;
	default:
		fprintf(stderr, "oldreadrgbaline: bad file format.\n");
		return(0);
	}

	return(1);
}

/*
 * oldreadmappedline
 *
 *	reads in a scanline for a colour mapped image as the
 * colourmap indexes.
 */
int
oldreadmappedline(in, line)
	oldimage	*in;
	unsigned char	*line;
{
	char		alpha;
	register int	x, i;
	unsigned char	run = 0;

	switch (oldimagetype(in)) {
	case OLDPIX_CMAP:
		for (x = 0; x < oldimagewidth(in); x++) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			line[x] = i;
		}
		break;
	case OLDPIX_RLECMAP:
		x = 0;
		while (x < oldimagewidth(in)) {
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			run = i;
			if ((i = oldreadbyte(in)) == EOF)
				return(0);
			do
				line[x++] = i;
			while (--run != 255);
		}
		break;
	default:
		fprintf(stderr, "oldreadmappedline: bad file format.\n");
		return(0);
	}

	return(1);
}
