/*
 * parser for the input language to art
 */

%{
#include <stdio.h>
#include <math.h>
#include "art.h"
#include "macro.h"

extern object	*sphereinit(),
		*boxinit(),
		*csginit(),
		*getcsgobj(),
		*getcsgexp(),
		*alginit(),
		*ringinit(),
		*torusinit(),
		*polygoninit(),
		*compinit(),
		*geometryinit(),
		*coneinit(),
		*ellipsinit(),
		*cylinit(),
		*superinit();

extern light	*lightinit();
extern tlist	*tileinit();
extern tlist	*textureinit();

extern attr	*astackp;
extern object	*oblist;
extern light	*lights;

extern vector	org, up;

extern matrix	trans;
extern float	focallength;

extern char	*title;
extern int	maxhitlevel, raysperpix;
extern long	filetype;

extern colour	backcol;

#ifndef M_PI
#define M_PI	3.14159265358979323846
#endif

%}

%union{
	object	*y_obj;
	light	*y_lht;
	vector	*y_pnt;
	details	*y_det;
	char	*y_str;
	eqn	*y_eqn;
	term	*y_trm;
	float	y_flt;
	int	y_int;
};

%token		BOX
%token		CSG
%token		CONE
%token		RING
%token		TORUS
%token		SPHERE
%token		POLYGON
%token		GEOMETRY
%token		CYLINDER
%token		ALGEBRAIC
%token		COMPOSITE
%token		ELLIPSOID
%token		SUPERQUADRIC

%token <y_flt>	FLOAT
%token <y_int>	INTEGER FILETYPE OPTION
%token <y_str>	STRING NAME

%type  <y_lht>	light lights
%type  <y_flt>	expr number
%type  <y_trm>	termlist term
%type  <y_obj>	object objects csgexpr
%type  <y_det>	body bodyitem attrib lbody litem texture_ops textitem

%token LBRACE RBRACE LP RP RADIUS RADII COLOUR CENTER VERTEX COMMA PCENT
%token MATERIAL REFI MINUS AMBIENT LIGHT INTENSITY LOCATION NAME DOLS
%token EQUATION TILE OFFFILE BASE TOP CONST COEFFS SCALE ROTATE TRANSLATE
%token TITLE REFLECTANCE DOT ON OFF LOOKAT FIELDOFVIEW TRANSPARENCY
%token RAYSPERPIXEL BACKGROUND SIZE MAXHITLEVEL OUTPUT FILETYPE ORDER
%token ABSORPTION VREF1 VREF2 NUMRAYS OBJECT TEXTURE DIRECTION ANGLE UP
%token TWENTYFIVEBIT RANGE MAP BLENDCOLOR SCALEFACTORS VORTFILE

%left  PLUS MINUS
%left  MULT DIV
%left  COMBINE
%left  POWER
%left  UMINUS

%left  EQUALS

%%

input	: hdrdata lights attrib objects
	| hdrdata lights attrib
	| hdrdata attrib objects
	| hdrdata lights objects
	| hdrdata lights 
	| hdrdata objects
	;

hdrdata : /* NULL */
	| hdrdata TITLE STRING
	  {
		title = $3;
	  }
	| hdrdata FIELDOFVIEW expr
	  {
		if ($3 == 0.0 || $3 == 360.0)
			fatal("art: idiotic angle in field of view.\n");

		focallength = -1.0 / tan(M_PI / 360.0 * $3);
	  }
	| hdrdata RAYSPERPIXEL expr
	  {
		raysperpix = $3;
	  }
	| hdrdata TWENTYFIVEBIT ON
	  {
		twentyfivebit(TRUE);
	  }
	| hdrdata TWENTYFIVEBIT OFF
	  {
		twentyfivebit(FALSE);
	  }
	| hdrdata UP LP expr COMMA expr COMMA expr RP
	  {
		up.x = $4;
		up.y = $6;
		up.z = $8;
	  }
	| hdrdata LOOKAT LP expr COMMA expr COMMA expr COMMA expr COMMA expr COMMA expr COMMA expr RP
	  {
		vector	t, u, s, v;
		matrix	m, tmp;
		double	vy, vz, sinval, cosval, d1, d2, d3;

		/*
		 * apply the twist
		 */
		sinval = sin($16 * M_PI / 180.0);
		cosval = cos($16 * M_PI / 180.0);
		mident4(m);
		m[0][0] = cosval;
		m[0][1] = -sinval;
		m[1][0] = sinval;
		m[1][1] = cosval;
		mcpy4(tmp, trans);
		mmult4(trans, tmp, m);

		/*
		 * calculate the lookat
		 */
		t.x = $4 - $10;
		t.y = $6 - $12;
		t.z = $8 - $14;

		u.x = up.x;
		u.y = up.y;
		u.z = up.z;

		normalise(t);

		normalise(u);

		vz = dprod(t, u);

		if (fabs(vz) >= 1.0)
			fatal("art: up vector and direction of view are the same.\n");

		vy = sqrt(1.0 - vz * vz);

		u.x = (u.x - vz * t.x) / vy;
		u.y = (u.y - vz * t.y) / vy;
		u.z = (u.z - vz * t.z) / vy;

		xprod(s, u, t);

		mident4(m);

		m[0][0] = s.x;
		m[0][1] = s.y;
		m[0][2] = s.z;

		m[1][0] = u.x;
		m[1][1] = u.y;
		m[1][2] = u.z;

		m[2][0] = t.x;
		m[2][1] = t.y;
		m[2][2] = t.z;

		mcpy4(tmp, trans);
		mmult4(trans, tmp, m);

		mident4(m);
		m[3][0] = $4;
		m[3][1] = $6;
		m[3][2] = $8;
		mcpy4(tmp, trans);
		mmult4(trans, tmp, m);

		astackp->mcopy = TRUE;
	  }
	| hdrdata BACKGROUND expr COMMA expr COMMA expr
	  {
		backcol.r = $3;
		backcol.g = $5;
		backcol.b = $7;
	  }
	| hdrdata MAXHITLEVEL expr
	  {
		maxhitlevel = $3;
	  }
	| hdrdata OUTPUT FILETYPE
	  {
		filetype = $3;
	  }
	;

lights	: light
	  {
		lights = $1;
		lights->nxt = (light *)NULL;
	  }
	| lights light
	  {
		$2->nxt = lights;
		lights = $2;
	  }
	;

light	: LIGHT LBRACE lbody RBRACE
	  {
		$$ = lightinit($3);
		astackp--;
	  }
	;

lbody	: /* NULL */
	  {
		$$ = (details *)NULL;
	  }
	| lbody COLOUR expr COMMA expr COMMA expr
	  {
		astackp->s.c.r = $3;
		astackp->s.c.g = $5;
		astackp->s.c.b = $7;
		$$ = $1;
	  }
	| lbody litem
	  {
		$2->nxt = $1;
		$$ = $2;
	  }
	;

litem	: LOCATION LP expr COMMA expr COMMA expr RP
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = CENTER;
		$$->u.v.x = $3;
		$$->u.v.y = $5;
		$$->u.v.z = $7;
		$$->nxt = (details *)NULL;
	  }
	| RADIUS expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = RADIUS;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| NUMRAYS expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = NUMRAYS;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| ANGLE expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = ANGLE;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| DIRECTION LP expr COMMA expr COMMA expr RP
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = DIRECTION;
		$$->u.v.x = $3;
		$$->u.v.y = $5;
		$$->u.v.z = $7;
		$$->nxt = (details *)NULL;
	  }
	;


objects	: object
	  {
		oblist = $1;
		oblist->nxt = (object *)NULL;
	  }
	| objects object
	  {
		$2->nxt = oblist;
		oblist = $2;
	  }
	;

object	: SPHERE LBRACE body RBRACE
	  {
		if (astackp->scales.x == astackp->scales.y
		    && astackp->scales.y == astackp->scales.z)
			$$ = sphereinit($3);
		else
			$$ = ellipsinit($3);

		astackp--;
	  }
	| BOX LBRACE body RBRACE
	  {
		$$ = boxinit($3);
		astackp--;
	  }
	| CSG LBRACE csgobjs csgexpr RBRACE
	  {
		$$ = $4;
		astackp--;
	  }
	| RING LBRACE body RBRACE
	  {
		$$ = ringinit($3);
		astackp--;
	  }
	| POLYGON LBRACE body RBRACE
	  {
		$$ = polygoninit($3);
		astackp--;
	  }
	| GEOMETRY LBRACE body RBRACE
	  {
		$$ = geometryinit($3);
		astackp--;
	  }
	| SUPERQUADRIC LBRACE body RBRACE
	  {
		$$ = superinit($3, FALSE);
		astackp--;
	  }
	| ALGEBRAIC LBRACE body RBRACE
	  {
		$$ = alginit($3, FALSE);
		astackp--;
	  }
	| CONE LBRACE body RBRACE
	  {
		$$ = coneinit($3);
		astackp--;
	  }
	| ELLIPSOID LBRACE body RBRACE
	  {
		$$ = ellipsinit($3);
		astackp--;
	  }
	| CYLINDER LBRACE body RBRACE
	  {
		$$ = cylinit($3);
		astackp--;
	  }
	| TORUS LBRACE body RBRACE
	  {
		$$ = torusinit($3, FALSE);
		astackp--;
	  }
	| COMPOSITE LBRACE body RBRACE
	  {
		$$ = compinit($3);
		astackp--;
	  }
	;

body	: /* NULL */
	  {
		$$ = (details *)NULL;
	  }
	| body bodyitem
	  {
		$2->nxt = $1;
		$$ = $2;
	  }
	| body attrib
	  {
		$$ = $1;
	  }
	;

bodyitem: CENTER LP expr COMMA expr COMMA expr RP
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = CENTER;
		$$->u.v.x = $3;
		$$->u.v.y = $5;
		$$->u.v.z = $7;
		$$->nxt = (details *)NULL;
	  }
	| ORDER expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = ORDER;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| RADIUS expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = RADIUS;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| RADII expr COMMA expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = RADII;
		$$->u.v.x = $2;
		$$->u.v.y = $4;
		$$->u.v.z = 0.0;
		$$->nxt = (details *)NULL;
	  }
	| RADII expr COMMA expr COMMA expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = RADII;
		$$->u.v.x = $2;
		$$->u.v.y = $4;
		$$->u.v.z = $6;
		$$->nxt = (details *)NULL;
	  }
	| VERTEX LP expr COMMA expr COMMA expr RP
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = VERTEX;
		$$->u.v.x = $3;
		$$->u.v.y = $5;
		$$->u.v.z = $7;
		$$->nxt = (details *)NULL;
	  }
	| EQUATION DOLS termlist EQUALS number DOLS
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = EQUATION;
		$$->u.t = (term *)smalloc(sizeof(term));
		$$->u.t->coef = -$5;
		$$->u.t->xp = 0;
		$$->u.t->yp = 0;
		$$->u.t->zp = 0;
		$$->u.t->nxt = $3;
		$$->nxt = (details *)NULL;
	  }
	| COEFFS expr COMMA expr COMMA expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = COEFFS;
		$$->u.v.x = $2;
		$$->u.v.y = $4;
		$$->u.v.z = $6;
		$$->nxt = (details *)NULL;
	  }
	| CONST expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = CONST;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| TOP expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = TOP;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| BASE expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = BASE;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| OFFFILE NAME
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = OFFFILE;
		$$->u.s = $2;
		$$->nxt = (details *)NULL;
	  }
	| object
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = OBJECT;
		$$->u.obj = $1;
		$$->nxt = (details *)NULL;
	  }
	;

attrib	: OPTION ON
	  {
		astackp->options |= $1;
	  }
	| OPTION OFF
	  {
		astackp->options &= ~$1;
	  }
	| VREF1 LP expr COMMA expr COMMA expr RP
	  {
		astackp->s.txtlist->txt.refset = TRUE;
		astackp->s.txtlist->txt.vref1.x = $3;
		astackp->s.txtlist->txt.vref1.y = $5;
		astackp->s.txtlist->txt.vref1.z = $7;

		normalise(astackp->s.txtlist->txt.vref1);
	  }
	| VREF2 LP expr COMMA expr COMMA expr RP
	  {
		astackp->s.txtlist->txt.vref2.x = $3;
		astackp->s.txtlist->txt.vref2.y = $5;
		astackp->s.txtlist->txt.vref2.z = $7;

		normalise(astackp->s.txtlist->txt.vref2);
	  }
	| TEXTURE NAME LBRACE {
		mident4(astackp->m);
		astackp->scales.x = astackp->scales.y = astackp->scales.z = 1.0;
	  } texture_ops RBRACE
	  {
		tlist	*tl;

		tl = textureinit($2, $5);

		astackp--;

		tl->nxt = astackp->s.txtlist;
		astackp->s.txtlist = tl;
	  }
	| TEXTURE TILE LBRACE {
		mident4(astackp->m);
		astackp->scales.x = astackp->scales.y = astackp->scales.z = 1.0;
	  } texture_ops RBRACE
	  {
		tlist	*tl;
		details	d;

		tl = textureinit("tile", $5);

		astackp--;

		tl->nxt = astackp->s.txtlist;
		astackp->s.txtlist = tl;
	  }
	| TILE NAME SIZE expr COMMA expr
	  {
		tlist	*tl;

		tl = (tlist *)smalloc(sizeof(tlist));
		tl->txt.type = TILE;
		tl->txtcol = (void (*)())NULL;
		tl->txt.refset = FALSE;
		tl->txt.scalew = 1.0;
		tl->txt.scaleh = 1.0;

		tileinit(&tl->txt, $2, $4, $6);

		tl->txt.mat[0][0] = tl->txt.mat[1][1] = tl->txt.mat[2][2] = 1.0;
		tl->txt.mat[0][1] = tl->txt.mat[0][2] = 0.0;
		tl->txt.mat[1][0] = tl->txt.mat[1][2] = 0.0;
		tl->txt.mat[2][0] = tl->txt.mat[2][1] = 0.0;

		tl->txt.scales.x = tl->txt.scales.y = tl->txt.scales.z = 1.0;
		tl->txt.trans.x = tl->txt.trans.y = tl->txt.trans.z = 0.0;

		tl->nxt = astackp->s.txtlist;
		astackp->s.txtlist = tl;
	  }
	| COLOUR expr COMMA expr COMMA expr
	  {
		astackp->s.c.r = $2;
		astackp->s.c.g = $4;
		astackp->s.c.b = $6;
	  }
	| MATERIAL expr COMMA expr COMMA expr COMMA expr
	  {
		astackp->s.ri = $2;
		astackp->s.kd = $4;
		astackp->s.ks = $6;
		astackp->s.ksexp = $8;
	  }
	| AMBIENT expr COMMA expr COMMA expr
	  {
		astackp->s.a.r = $2;
		astackp->s.a.g = $4;
		astackp->s.a.b = $6;
	  }
	| REFLECTANCE expr
	  {
		astackp->s.refl = $2;
	  }
	| TRANSPARENCY expr
	  {
		astackp->s.trans = $2;
	  }
	| ABSORPTION expr
	  {
		astackp->s.absorb = $2;
	  }
	| ROTATE LP expr COMMA NAME RP
	  {
		rotate($3, *$5);
	  }
	| TRANSLATE LP expr COMMA expr COMMA expr RP
	  {
		translate($3, $5, $7);
	  }
	| SCALE LP expr COMMA expr COMMA expr RP
	  {
		scale($3, $5, $7);
	  }
	;

texture_ops:	/* NULL */
	  {
		$$ = (details *)NULL;
	  }
	| texture_ops textitem
	  {
		$2->nxt = $1;
		$$ = $2;
	  }
	| texture_ops attrib
	  {
		$$ = $1;
	  }
	;

textitem: MAP NAME
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = MAP;
		$$->u.s = $2;
		$$->nxt = (details *)NULL;
	  }
	| RANGE expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = RANGE;
		$$->u.f = $2;
		$$->nxt = (details *)NULL;
	  }
	| BLENDCOLOR expr COMMA expr COMMA expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = BLENDCOLOR;
		$$->u.v.x = $2;
		$$->u.v.y = $4;
		$$->u.v.z = $6;
		$$->nxt = (details *)NULL;
	  }
	| SIZE expr COMMA expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = SIZE;
		$$->u.v.x = $2;
		$$->u.v.y = $4;
		$$->nxt = (details *)NULL;
	  }
	| SCALEFACTORS expr COMMA expr COMMA expr
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = SCALEFACTORS;
		$$->u.v.x = $2;
		$$->u.v.y = $4;
		$$->u.v.z = $6;
		$$->nxt = (details *)NULL;
	  }
	| VORTFILE NAME
	  {
		$$ = (details *)smalloc(sizeof(details));
		$$->type = VORTFILE;
		$$->u.s = $2;
		$$->nxt = (details *)NULL;
	  }
	;

csgobjs	: /* NULL */
	| csgobjs attrib
	| csgobjs csgobj
	;

csgobj	: SPHERE NAME LBRACE body RBRACE
	  {
		if (astackp->scales.x == astackp->scales.y
		    && astackp->scales.y == astackp->scales.z)
			savecsgobj($2, sphereinit($4));
		else
			savecsgobj($2, ellipsinit($4));
		astackp--;
	  }
	| BOX NAME LBRACE body RBRACE
	  {
		savecsgobj($2, boxinit($4));
		astackp--;
	  }
	| SUPERQUADRIC NAME LBRACE body RBRACE
	  {
		savecsgobj($2, superinit($4, TRUE));
		astackp--;
	  }
	| ALGEBRAIC NAME LBRACE body RBRACE
	  {
		savecsgobj($2, alginit($4, TRUE));
		astackp--;
	  }
	| CONE NAME LBRACE body RBRACE
	  {
		savecsgobj($2, coneinit($4));
		astackp--;
	  }
	| ELLIPSOID NAME LBRACE body RBRACE
	  {
		savecsgobj($2, ellipsinit($4));
		astackp--;
	  }
	| CYLINDER NAME LBRACE body RBRACE
	  {
		savecsgobj($2, cylinit($4));
		astackp--;
	  }
	| TORUS NAME LBRACE body RBRACE
	  {
		savecsgobj($2, torusinit($4, TRUE));
		astackp--;
	  }
	| CSG NAME LBRACE csgobjs csgexpr RBRACE
	  {
		savecsgobj($2, $5);
		astackp--;
	  }
	;


csgexpr	: NAME
	  {
		$$ = getcsgobj($1);
	  }
	| csgexpr PLUS csgexpr
	  {
		$$ = getcsgexp(UNION, $1, $3);
	  }
	| csgexpr MULT csgexpr
	  {
		$$ = getcsgexp(INTERSECT, $1, $3);
	  }
	| csgexpr MINUS csgexpr
	  {
		$$ = getcsgexp(SUBTRACT, $1, $3);
	  }
	| LP csgexpr RP
	  {
		$$ = $2;
	  }
	;

termlist: term
	  {
		$$ = $1;
	  }
	| termlist termlist %prec MULT
	  {
		term	*t, *p, *np, *prod;
		int	i;

		prod = (term *)NULL;
		for (p = $1; p != (term *)NULL; p = p->nxt) {
			for (np = $2; np != (term *)NULL; np = np->nxt) {
				t = (term *)smalloc(sizeof(term));
				*t = *np;
				t->coef *= p->coef;
				t->xp += p->xp;
				t->yp += p->yp;
				t->zp += p->zp;
				t->nxt = prod;
				prod = t;
			}
		}

		for (t = $1; t != (term *)NULL; t = np) {
			np = t->nxt;
			free(t);
		}

		for (t = $2; t != (term *)NULL; t = np) {
			np = t->nxt;
			free(t);
		}

		$$ = prod;
	  }
	| LP termlist RP 
	  {
		$$ = $2;
	  }
	| LP termlist RP POWER LBRACE INTEGER RBRACE
	  {
		term	*t, *p, *np, *prod, *nprod;
		int	i;

		prod = $2;

		for (i = 1; i != $6; i++) {		/* expand out list */
			nprod = (term *)NULL;
			for (p = $2; p != (term *)NULL; p = p->nxt) {
				for (np = prod; np != (term *)NULL; np = np->nxt) {
					t = (term *)smalloc(sizeof(term));
					*t = *np;
					t->coef *= p->coef;
					t->xp += p->xp;
					t->yp += p->yp;
					t->zp += p->zp;
					t->nxt = nprod;
					nprod = t;
				}
			}
			if (prod != $2)
				for (t = prod; t != (term *)NULL; t = np) {
					np = t->nxt;
					free(t);
				}
			prod = nprod;
		}

		for (t = $2; t != (term *)NULL; t = np) {
			np = t->nxt;
			free(t);
		}

		$$ = prod;
	  }
	| termlist PLUS termlist
	  {
		term	*p;

		for (p = $3; p->nxt != (term *)NULL; p = p->nxt)
			;
		$$ = $3;
		p->nxt = $1;
	  }
	| termlist MINUS termlist
	  {
		term	*p, *lp;

		for (p = $3; p != (term *)NULL; p = p->nxt) {
			p->coef *= -1.0;
			lp = p;
		}

		$$ = $3;
		lp->nxt = $1;
	  }
	;

term:	 NAME
	  {
		char	*p;

		$$ = (term *)smalloc(sizeof(term));
		$$->coef = 1;
		$$->xp = 0;
		$$->yp = 0;
		$$->zp = 0;
		$$->nxt = (term *)NULL;

		for (p = $1; *p != 0; p++)
			switch (*p) {
			case 'x':
				$$->xp += 1;
				break;
			case 'y':
				$$->yp += 1;
				break;
			case 'z':
				$$->zp += 1;
				break;
			default:
				fatal("art: illegal name in equation.\n");
			}
	  }
	| NAME POWER LBRACE INTEGER RBRACE
	  {
		char	*p;

		$$ = (term *)smalloc(sizeof(term));
		$$->coef = 1;
		$$->xp = 0;
		$$->yp = 0;
		$$->zp = 0;
		$$->nxt = (term *)NULL;

		for (p = $1; *p != 0; p++)
			switch (*p) {
			case 'x':
				$$->xp += 1;
				break;
			case 'y':
				$$->yp += 1;
				break;
			case 'z':
				$$->zp += 1;
				break;
			default:
				fatal("art: illegal name in equation.\n");
			}

		p--;

		switch (*p) {
		case 'x':
			$$->xp += $4 - 1;
			break;
		case 'y':
			$$->yp += $4 - 1;
			break;
		case 'z':
			$$->zp += $4 - 1;
			break;
		default:
			fatal("art: illegal name in equation.\n");
		}
	  }
	| number
	  {
		$$ = (term *)smalloc(sizeof(term));
		$$->coef = $1;
		$$->xp = 0;
		$$->yp = 0;
		$$->zp = 0;
		$$->nxt = (term *)NULL;
	  }
	;

expr	: number
	  {
		$$ = $1;
	  }
	| number PLUS number
	  {
		$$ = $1 + $3;
	  }
	| number MINUS number
	  {
		$$ = $1 - $3;
	  }
	| number MULT number
	  {
		$$ = $1 * $3;
	  }
	| number DIV number
	  {
		$$ = $1 / $3;
	  }
	| MINUS number %prec UMINUS
	  {
		$$ = -$2;
	  }
	;

number	: FLOAT
	  {
		$$ = $1;
	  }
	| INTEGER
	  {
		$$ = $1;
	  }
	;
