#include <math.h>
#include <stdio.h>
#include "art.h"
#include "macro.h"
#include "gram.h"

extern attr	*astackp;
extern hlist	*fhlist;

extern matrix	trans;

/*
 * ray details
 */
static vector	dir, org;
static float	orgu, orgv, diru, dirv;

/*
 * ray box dimensions
 */
static float	minu, maxu, minv, maxv;

/*
 * polygon object
 */
static object	*obj;

/*
 * hitlist ptr for this object
 */
static hlist	*hitlist;

/*
 * options
 */
static int	backfacing;

/*
 * addhit
 *
 *	add a hit to the hitlist
 */
static
addhit(t, indx, last)
	register float	t;
	register int	indx;
	register hlist	**last;
{
	register hlist	*lp, *tp, *hp;

	lp = (hlist *)NULL;
	for (tp = hitlist; tp != (hlist *)NULL; tp = tp->nxt) {
		if (tp->t > t)
			break;
		lp = tp;
	}
	fetch(hp);
	hp->t = t;
	hp->obj = obj;
	hp->type = indx;
	if (hitlist == (hlist *)NULL) {
		hitlist = hp;
		hitlist->nxt = (hlist *)NULL;
		*last = hitlist;
	} else if (lp == (hlist *)NULL) {
		hp->nxt = hitlist;
		hitlist = hp;
	} else {
		hp->nxt = lp->nxt;
		lp->nxt = hp;
		if (lp == *last)
			*last = hp;
	}
}

/*
 * inttris
 *
 *	generate a list of hits for the triangles from startface to endface
 * according to the environment set up in geomi or polyi. The method used
 * here is what is commonly known as the infinite half-ray approach.
 */
inttris(base, us, vs, midface, midv, startface, endface, last)
	int		base;
	register float	*us, *vs;
	polygon		*midface;
	float		midv;
	polygon		*startface, *endface;
	hlist		**last;
{
	register int	crosses;
	register vertno	*cp;
	register float	t, pu, pv, m;
	register polygon *face, *start, *end, *mid;

	/*
	 * a quick check
	 */
	if (startface->minv > maxv)
		return;

	/*
	 * do a quick search for a likely candidate
	 */
	start = startface;
	end = endface;
	while (start->maxv < minv) {
		mid = start + (end - start) / 2;
		if (start == mid)
			break;
		if (mid->maxv < minv)
			start = mid;
		else
			end = mid;
	}

	/*
	 * scan back
	 */
	if (midv < maxv) {
		end = endface;
		while (end->minv > maxv && end > start)
			end--;
	} else {
		end = midface;
		while (end->minv > maxv && end > start)
			end--;
	}

	for (face = start; face <= end; face++) {
		
		if (face->maxu < minu || face->minu > maxu)
			continue;

		if (face->maxv < minv || face->minv > maxv)
			continue;

		t = dprod(face->n, dir);

		if (t == 0.0 || (t < 0.0 && backfacing))
			continue;

		t = -(dprod(face->n, org) + face->cnst) / t;

		if (t < TOLERANCE)
			continue;

		pu = orgu + t * diru;
		pv = orgv + t * dirv;

		crosses = 0;

		cp = face->vertnos;

		if ((vs[cp[0]] - pv) * (vs[cp[1]] - pv) < 0.0) {
			if (us[cp[0]] < pu) {
				if (us[cp[1]] < pu)
					crosses++;
				else {
					m = (us[cp[1]] - us[cp[0]]) / (vs[cp[1]] - vs[cp[0]]);
					if (((us[cp[1]] - pu) - m * (vs[cp[1]] - pv)) < 0.0)
						crosses++;
				}
			} else if (us[cp[1]] < pu) {
				m = (us[cp[1]] - us[cp[0]]) / (vs[cp[1]] - vs[cp[0]]);
				if (((us[cp[1]] - pu) - m * (vs[cp[1]] - pv)) < 0.0)
					crosses++;
			}
		}

		if ((vs[cp[1]] - pv) * (vs[cp[2]] - pv) < 0.0) {
			if (us[cp[1]] < pu) {
				if (us[cp[2]] < pu)
					crosses++;
				else {
					m = (us[cp[2]] - us[cp[1]]) / (vs[cp[2]] - vs[cp[1]]);
					if (((us[cp[2]] - pu) - m * (vs[cp[2]] - pv)) < 0.0)
						crosses++;
				}
			} else if (us[cp[2]] < pu) {
				m = (us[cp[2]] - us[cp[1]]) / (vs[cp[2]] - vs[cp[1]]);
				if (((us[cp[2]] - pu) - m * (vs[cp[2]] - pv)) < 0.0)
					crosses++;
			}
		}

		if ((vs[cp[2]] - pv) * (vs[cp[0]] - pv) < 0.0) {
			if (us[cp[2]] < pu) {
				if (us[cp[0]] < pu)
					crosses++;
				else {
					m = (us[cp[0]] - us[cp[2]]) / (vs[cp[0]] - vs[cp[2]]);
					if (((us[cp[0]] - pu) - m * (vs[cp[0]] - pv)) < 0.0)
						crosses++;
				}
			} else if (us[cp[0]] < pu) {
				m = (us[cp[0]] - us[cp[2]]) / (vs[cp[0]] - vs[cp[2]]);
				if (((us[cp[0]] - pu) - m * (vs[cp[0]] - pv)) < 0.0)
					crosses++;
			}
		}

		if (crosses & 1)
			addhit(t, face - startface + base, last);
	}
}

/*
 * intpolys
 *
 *	generate a list of hits for the polygons from startface to endface
 * according to the environment set up in geomi or polyi. The method used
 * here is what is commonly known as the infinite half-ray approach.
 */
intpolys(base, us, vs, midface, midv, startface, endface, last)
	int		base;
	register float	*us, *vs;
	polygon		*midface;
	float		midv;
	polygon		*startface, *endface;
	hlist		**last;
{
	register int	crosses;
	register vertno	*cp, *np;
	register float	t, pu, pv, m;
	register polygon *face, *start, *end, *mid;

	/*
	 * a quick check
	 */
	if (startface->minv > maxv)
		return;

	/*
	 * do a quick search for a likely candidate
	 */
	start = startface;
	end = endface;
	while (start->maxv < minv) {
		mid = start + (end - start) / 2;
		if (start == mid)
			break;
		if (mid->maxv < minv)
			start = mid;
		else
			end = mid;
	}

	/*
	 * scan back
	 */
	if (midv < maxv) {
		end = endface;
		while (end->minv > maxv && end > start)
			end--;
	} else {
		end = midface;
		while (end->minv > maxv && end > start)
			end--;
	}

	for (face = start; face <= end; face++) {
		
		if (face->maxu < minu || face->minu > maxu)
			continue;

		if (face->maxv < minv || face->minv > maxv)
			continue;

		t = dprod(face->n, dir);

		if (t == 0.0 || (t < 0.0 && backfacing))
			continue;

		t = -(dprod(face->n, org) + face->cnst) / t;

		if (t < TOLERANCE)
			continue;

		pu = orgu + t * diru;
		pv = orgv + t * dirv;

		crosses = 0;

		for (cp = face->vertnos, np = &face->vertnos[face->nsides - 1];
			np >= face->vertnos; cp = np--) {

			/*
			 * If we are both above, or both below the intersection
			 * point, go onto the next one.
			 */

			if (vs[*cp] < pv) {
				if (vs[*np] < pv)
					continue;
			} else {
				if (vs[*np] >= pv)
					continue;
			}

			/*
			 * Find out if we have crossed in the negative.
			 */
			if (us[*cp] < pu) {
				if (us[*np] < pu) {	/* crossed on left */
					crosses++;
					continue;
				}
			} else if (us[*np] >= pu)		/* both to right */
				continue;

			/* 
			 * We calculate the u at the intersection of the
			 * half-ray and the edge and test it against pu. 
			 */

			m = (us[*np] - us[*cp]) / (vs[*np] - vs[*cp]);
			if (((us[*np] - pu) - m * (vs[*np] - pv)) < 0.0)
				crosses++;
		}

		if (crosses & 1)
			addhit(t, face - startface + base, last);
	}
}

/*
 * geomi
 *
 *	returns a list of hits for the ray with a polygon model defined
 * from a geometry file.
 */
hlist *
geomi(r, o, last)
	ray	*r;
	object	*o;
	hlist	**last;
{
	register int		base;
	register geometry	*geom;
	register polygon	*start;
	float			t1, t2;
	register float		*us, *vs;
	register float		minx, maxx, miny, maxy, minz, maxz;
	vector			cv;
	ray			nr;

	vadd(cv, r->org, o->bs.cent);
	t1 = dprod(cv, r->dir);

	if ((dprod(cv, cv) - o->bs.radsqu) > t1 * t1)
		return((hlist *)NULL);

	transray(o, nr, *r);

	if (!inbbox(&nr, &o->obj.geo->bb, &t1, &t2))
		return((hlist *)NULL);

	org.x = nr.org.x;
	org.y = nr.org.y;
	org.z = nr.org.z;

	dir.x = nr.dir.x;
	dir.y = nr.dir.y;
	dir.z = nr.dir.z;

	if (t2 < TOLERANCE) {
		t2 = t1;
		t1 = 0.0;
	}

	if (dir.x < 0.0) {
		maxx = org.x + t1 * dir.x;
		minx = org.x + t2 * dir.x;
	} else {
		maxx = org.x + t2 * dir.x;
		minx = org.x + t1 * dir.x;
	}

	if (dir.y < 0.0) {
		maxy = org.y + t1 * dir.y;
		miny = org.y + t2 * dir.y;
	} else {
		maxy = org.y + t2 * dir.y;
		miny = org.y + t1 * dir.y;
	}

	if (dir.z < 0.0) {
		maxz = org.z + t1 * dir.z;
		minz = org.z + t2 * dir.z;
	} else {
		maxz = org.z + t2 * dir.z;
		minz = org.z + t1 * dir.z;
	}
	
	hitlist = (hlist *)NULL;

	obj = o;

	geom = o->obj.geo;

	backfacing = geom->options & BACKFACING;

	start = geom->faces;
	base = 0;

	if (geom->npolys[XAXIS] != 0) {
		if (geom->types[XAXIS] == YZ) {
			minu = miny; maxu = maxy;
			minv = minz; maxv = maxz;
			orgu = org.y; diru = dir.y;
			orgv = org.z; dirv = dir.z;
			us = geom->ys; vs = geom->zs;
		} else {
			minu = minz; maxu = maxz;
			minv = miny; maxv = maxy;
			orgu = org.z; diru = dir.z;
			orgv = org.y; dirv = dir.y;
			us = geom->zs; vs = geom->ys;
		}
		if (geom->ntris[XAXIS] != 0)
			inttris(base, us, vs, geom->midts[XAXIS],
				geom->midtval[XAXIS], start,
				start + geom->ntris[XAXIS] - 1, last);
		intpolys(base + geom->ntris[XAXIS], us, vs, geom->midps[XAXIS],
			geom->midpval[XAXIS], start + geom->ntris[XAXIS],
			start + geom->npolys[XAXIS] - 1, last);

		start += geom->npolys[XAXIS];
		base += geom->npolys[XAXIS];
	}

	if (geom->npolys[YAXIS] != 0) {
		if (geom->types[YAXIS] == XZ) {
			minu = minx; maxu = maxx;
			minv = minz; maxv = maxz;
			orgu = org.x; diru = dir.x;
			orgv = org.z; dirv = dir.z;
			us = geom->xs; vs = geom->zs;
		} else {
			minu = minz; maxu = maxz;
			minv = minx; maxv = maxx;
			orgu = org.z; diru = dir.z;
			orgv = org.x; dirv = dir.x;
			us = geom->zs; vs = geom->xs;
		}
		if (geom->ntris[YAXIS] != 0)
			inttris(base, us, vs, geom->midts[YAXIS],
				geom->midtval[YAXIS], start,
				start + geom->ntris[YAXIS] - 1, last);
		intpolys(base + geom->ntris[YAXIS], us, vs, geom->midps[YAXIS],
			geom->midpval[YAXIS], start + geom->ntris[YAXIS],
			start + geom->npolys[YAXIS] - 1, last);

		start += geom->npolys[YAXIS];
		base += geom->npolys[YAXIS];
	}

	if (geom->npolys[ZAXIS] != 0) {
		if (geom->types[ZAXIS] == XY) {
			minu = minx; maxu = maxx;
			minv = miny; maxv = maxy;
			orgu = org.x; diru = dir.x;
			orgv = org.y; dirv = dir.y;
			us = geom->xs; vs = geom->ys;
		} else {
			minu = miny; maxu = maxy;
			minv = minx; maxv = maxx;
			orgu = org.y; diru = dir.y;
			orgv = org.x; dirv = dir.x;
			us = geom->ys; vs = geom->xs;
		}
		if (geom->ntris[ZAXIS] != 0)
			inttris(base, us, vs, geom->midts[ZAXIS],
				geom->midtval[ZAXIS], start,
				start + geom->ntris[ZAXIS] - 1, last);
		intpolys(base + geom->ntris[ZAXIS], us, vs, geom->midps[ZAXIS],
			geom->midpval[ZAXIS], start + geom->ntris[ZAXIS],
			start + geom->npolys[ZAXIS] - 1, last);
	}

	return(hitlist);
}

/*
 * polyi
 *
 *	returns a list of hits for the ray with a polygon model defined
 * from a geometry file.
 */
hlist *
polyi(r, o, last)
	ray	*r;
	object	*o;
	hlist	**last;
{
	register float		*us, *vs;
	register geometry	*geom;
	register int		crosses;
	register vertno		*cp, *np;
	register float		t, pu, pv, m;
	register polygon	 *face;

	geom = o->obj.geo;

	backfacing = geom->options & BACKFACING;

	hitlist = (hlist *)NULL;

	if (geom->npolys[XAXIS] != 0) {
		us = geom->ys; vs = geom->zs;
		orgu = r->org.y; diru = r->dir.y;
		orgv = r->org.z; dirv = r->dir.z;
	} else if (geom->npolys[YAXIS] != 0) {
		us = geom->xs; vs = geom->zs;
		orgu = r->org.x; diru = r->dir.x;
		orgv = r->org.z; dirv = r->dir.z;
	} else {
		us = geom->xs; vs = geom->ys;
		orgu = r->org.x; diru = r->dir.x;
		orgv = r->org.y; dirv = r->dir.y;
	}

	face = geom->faces;
		
	t = dprod(face->n, r->dir);

	if (t == 0.0 || (t < 0.0 && backfacing))
		return((hlist *)NULL);

	t = -(dprod(face->n, r->org) + face->cnst) / t;

	if (t < TOLERANCE)
		return((hlist *)NULL);

	pu = orgu + t * diru;
	pv = orgv + t * dirv;

	crosses = 0;

	for (cp = face->vertnos, np = &face->vertnos[face->nsides - 1];
		np >= face->vertnos; cp = np--) {

		if (vs[*cp] < pv) {
			if (vs[*np] < pv)
				continue;
		} else {
			if (vs[*np] > pv)
				continue;
		}

		if (us[*cp] < pu) {
			if (us[*np] < pu) {
				crosses++;
				continue;
			}
		} else if (us[*np] >= pu)
			continue;

		if (vs[*cp] == vs[*np])
			continue;

		m = (vs[*np] - vs[*cp]) / (us[*np] - us[*cp]);
		if ((-(vs[*np] - pv) / m + (us[*np] - pu)) < 0.0)
			crosses++;
	}

	if (crosses & 1) {
		fetch(hitlist);
		hitlist->t = t;
		hitlist->obj = o;
		hitlist->type = 0;	/* index number of polygon hit */
		hitlist->nxt = (hlist *)NULL;
		*last = hitlist;
	}

	return(hitlist);
}

/*
 * geomn
 *
 *	returns the normal to a polygon with index number indx
 */
void
geomn(n, l, o, indx)
	vector	*n, *l;
	object	*o;
	int	indx;
{
	polygon		*face;
	geometry	*geom;
	vector		v, loc;

	geom = o->obj.geo;
	face = &geom->faces[indx];

	if (geom->options & PHONGSHADING) {
		toobject(o, loc, *l);
		phong(geom, face, &loc, &v);
	} else {
		v.x = face->n.x;
		v.y = face->n.y;
		v.z = face->n.z;
	}

	if (o->mat != (mat3x3 *)NULL) {
		v3x3tmult(*n, v, (*o->mat));
	} else
		*n = v;

	normalise(*n);
}

/*
 * getaxis
 *
 *	returns the major axis of the vector v.
 */
int
getaxis(v)
	vector	v;
{
	int	axis;

	if (fabs(v.x) > fabs(v.y))
		if (fabs(v.x) > fabs(v.z))
			axis = XAXIS;
		else
			axis = ZAXIS;
	else
		if (fabs(v.y) > fabs(v.z))
			axis = YAXIS;
		else
			axis = ZAXIS;

	return(axis);
}

/*
 * reducepoly
 *
 *	reduce the polygon described by numverts vertices 
 * to its (u, v) representation returning the values in pg
 */
reducepoly(pg, type, xs, ys, zs, nsides)
	register polygon	*pg;
	int 			type;
	float			*xs, *ys, *zs;
	int			nsides;
{
	int		j;
	register float	*us, *vs;

	switch (type) {
	case XY:
		us = xs;
		vs = ys;
		break;
	case YX:
		us = ys;
		vs = xs;
		break;
	case XZ:
		us = xs;
		vs = zs;
		break;
	case ZX:
		us = zs;
		vs = xs;
		break;
	case YZ:
		us = ys;
		vs = zs;
		break;
	case ZY:
		us = zs;
		vs = ys;
		break;
	default:
		fatal("reducepoly: bad type in switch\n");
	}
		
	pg->minu = pg->maxu = us[pg->vertnos[0]];
	pg->minv = pg->maxv = vs[pg->vertnos[0]];
	for (j = 1; j != nsides; j++) {
		minmax(pg->minu, pg->maxu, us[pg->vertnos[j]]);
		minmax(pg->minv, pg->maxv, vs[pg->vertnos[j]]);
	}
}

/*
 * sortpolys
 *
 *	sort a list of polygons according to their maxv values bubbling
 * all the triangles down to the start.
 */
sortpolys(polys, npolys)
	polygon	**polys;
	int	npolys;
{
	int	i, swapped;
	polygon	*face, *next;

	swapped = TRUE;
	while (swapped) {
		swapped = FALSE;
		for (i = 0; i < npolys - 1; i++) {
			face = polys[i];
			next = polys[i + 1];
			if (face->maxv > next->maxv) {
				if (face->nsides == next->nsides) {
					polys[i + 1] = face;
					polys[i] = next;
					swapped = TRUE;
				} else if (face->nsides != 3) {
					polys[i + 1] = face;
					polys[i] = next;
					swapped = TRUE;
				}
			} else if (next->nsides == 3 && face->nsides != 3) {
				polys[i + 1] = face;
				polys[i] = next;
				swapped = TRUE;
			}
		}
	}
}

/*
 * geominit
 *
 *	reads in and sets up a polygon model from a geometry file
 */
geometry *
geominit(name)
	char	*name;
{
        float		minx, maxx, miny, maxy, minz, maxz;
	float		*xs, *ys, *zs, midminv;
	int		nverts, connections, npnts;
	int		i, j, numedges, phong, nextj;
	int		*ntris, *npolys, cvertno, base, *types;
	vertno		*vertnums, *pntno;
	FILE		*pfile;
	vector		v, v1, v2, norm, *norms;
	geometry	*gm;
	polygon		*face, **polys[DIMS], *faces;
	unsigned char	*axis;
	char		buf[BUFSIZ];

	if ((pfile = fopen(name, "r")) == NULL) {
		sprintf(buf, "art: cannot open geofile %s.\n", name);
		fatal(buf);
	}

	if (fscanf(pfile, "%d %d %d\n", &nverts, &connections, &numedges) != 3) {
		sprintf(buf, "art: can't find header info in geofile %s.\n", name);
		fatal(buf);
	}

	gm = (geometry *)smalloc(sizeof(geometry));
	gm->options = astackp->options;
	gm->xs = xs = (float *)smalloc(sizeof(float) * nverts);
	gm->ys = ys = (float *)smalloc(sizeof(float) * nverts);
	gm->zs = zs = (float *)smalloc(sizeof(float) * nverts);
	vertnums = (vertno *)smalloc(sizeof(vertno) * numedges);
	gm->faces = (polygon *)NULL;

	ntris = gm->ntris;
	npolys = gm->npolys;
	types = gm->types;

	phong = (gm->options & PHONGSHADING);

	if (phong)
		norms = gm->norms = (vector *)smalloc(sizeof(vector) * nverts);
	else
		gm->norms = (vector *)NULL;

	fscanf(pfile, "%f %f %f\n", &v1.x, &v1.y, &v1.z);

	minx = maxx = v1.x;
	miny = maxy = v1.y;
	minz = maxz = v1.z;
	xs[0] = v1.x; ys[0] = v1.y; zs[0] = v1.z;

	for (i = 1; i != nverts; i++) {
		fscanf(pfile, "%f %f %f\n", &v1.x, &v1.y, &v1.z);
		minmax(minx, maxx, v1.x);
		minmax(miny, maxy, v1.y);
		minmax(minz, maxz, v1.z);
		xs[i] = v1.x; ys[i] = v1.y; zs[i] = v1.z;
	}

	/*
	 * set up the bounding box 
	 */
	gm->bb.topfr.x = maxx; gm->bb.botbl.x = minx;
	gm->bb.topfr.y = maxy; gm->bb.botbl.y = miny;
	gm->bb.topfr.z = maxz; gm->bb.botbl.z = minz;

	/*
	 * set organisation
	 */
	types[XAXIS] = (maxy - miny > maxz - minz) ? ZY : YZ;
	types[YAXIS] = (maxx - minx > maxz - minz) ? ZX : XZ;
	types[ZAXIS] = (maxx - minx > maxy - miny) ? YX : XY;

	faces = (polygon *)smalloc(connections * sizeof(polygon));
	axis = (unsigned char *)smalloc(connections);

	if (phong)
		for (i = 0; i != nverts; i++)
			norms[i].x = norms[i].y = norms[i].z = 0.0;

	cvertno = 0;
	npolys[XAXIS] = npolys[YAXIS] = npolys[ZAXIS] = 0;
	ntris[XAXIS] = ntris[YAXIS] = ntris[ZAXIS] = 0;
	for (i = 0; i != connections; i++) {
		fscanf(pfile, "%d", &npnts);
		face = &faces[i];
		face->vertnos = pntno = &vertnums[cvertno];
		fscanf(pfile, "%hd", &pntno[0]);
		pntno[0] -= 1;
		
		for (j = 1; j != npnts; j++) {
			fscanf(pfile, "%hd", &pntno[j]);
			pntno[j] -= 1;
		}

		v1.x = xs[pntno[0]] - xs[pntno[1]];
		v1.y = ys[pntno[0]] - ys[pntno[1]];
		v1.z = zs[pntno[0]] - zs[pntno[1]];
		v2.x = xs[pntno[0]] - xs[pntno[npnts - 1]];
		v2.y = ys[pntno[0]] - ys[pntno[npnts - 1]];
		v2.z = zs[pntno[0]] - zs[pntno[npnts - 1]];

		xprod(norm, v1, v2);

		if (norm.x != 0.0 || norm.y != 0.0 || norm.z != 0.0) {
			normalise(norm);

			if (npnts > 3) {
				for (j = 1; j != npnts; j++) {
					nextj = (j == npnts - 1) ? 0 : j + 1;
					v1.x = xs[pntno[j]] - xs[pntno[nextj]];
					v1.y = ys[pntno[j]] - ys[pntno[nextj]];
					v1.z = zs[pntno[j]] - zs[pntno[nextj]];
					v2.x = xs[pntno[j]] - xs[pntno[j - 1]];
					v2.y = ys[pntno[j]] - ys[pntno[j - 1]];
					v2.z = zs[pntno[j]] - zs[pntno[j - 1]];
					xprod(v, v1, v2);
					if (v.x != 0.0 || v.y != 0.0 || v.z != 0.0) {
						normalise(v);
						vadd(norm, norm, v);
					}
				}
				smult(norm, 1.0 / npnts);
				normalise(norm);
			}
		
			if (phong) {
				for (j = 0; j != npnts; j++)
					vadd(norms[pntno[j]], norms[pntno[j]], norm);
			}

			v.x = xs[pntno[0]];
			v.y = ys[pntno[0]];
			v.z = zs[pntno[0]];
			face->cnst = -dprod(norm, v);

			cvertno += npnts;

			axis[i] = getaxis(norm);

			npolys[axis[i]]++;
			if (npnts == 3)
				ntris[axis[i]]++;

			reducepoly(face, types[axis[i]], xs, ys, zs, npnts);

			face->n = norm;

			face->nsides = npnts;
		} else {
			cvertno += npnts;

			connections--;		/* a dud polygon */
			i--;
			sprintf(buf, "polygon %d in geometry file %s not a polygon!\n", i + 2, name);
			message(buf);
		}
	}

	if (phong)
		for (i = 0; i != nverts; i++) {
			if (norms[i].x == 0.0 && norms[i].y == 0.0 && norms[i].z == 0.0) {
				sprintf(buf, "vertex %d unused in geometry file %s\n", i + 1, name);
				message(buf);
			} else 
				normalise(norms[i]);
		}

	polys[XAXIS] = (polygon **)smalloc(sizeof(polygon *) * npolys[XAXIS]);
	polys[YAXIS] = (polygon **)smalloc(sizeof(polygon *) * npolys[YAXIS]);
	polys[ZAXIS] = (polygon **)smalloc(sizeof(polygon *) * npolys[ZAXIS]);

	npolys[XAXIS] = npolys[YAXIS] = npolys[ZAXIS] = 0;

	for (i = 0; i < connections; i++)
		polys[axis[i]][npolys[axis[i]]++] = faces + i;

	gm->faces = (polygon *)smalloc(connections * sizeof(polygon));

	base = 0;
	for (i = 0; i != DIMS; i++) {
		if (npolys[i] != 0) {
			sortpolys(polys[i], npolys[i]);
			for (j = 0; j != npolys[i]; j++)
				gm->faces[j + base] = *polys[i][j];

			gm->midts[i] = &gm->faces[base + ntris[i] / 2];
			midminv = gm->midts[i]->minv;
			for (j = ntris[i] / 2; j != ntris[i]; j++)
				if (midminv > polys[i][j]->minv)
					midminv = polys[i][j]->minv;
			gm->midtval[i] = midminv;

			gm->midps[i] = &gm->faces[base + ntris[i] + (npolys[i] - ntris[i]) / 2];
			midminv = gm->midps[i]->minv;
			for (j = ntris[i] + (npolys[i] - ntris[i]) / 2; j != npolys[i]; j++)
				if (midminv > polys[i][j]->minv)
					midminv = polys[i][j]->minv;
			gm->midpval[i] = midminv;

			base += npolys[i];
		}
	}

	if (cvertno != numedges) {
		sprintf(buf, "number of edges should be %d not %d in file %s\n", cvertno, numedges, name);
		warning(buf);
	}

	free(axis);
	free(faces);

	free(polys[XAXIS]);
	free(polys[YAXIS]);
	free(polys[ZAXIS]);

	fclose(pfile);

	return(gm);
}

/*
 * geometryinit
 *
 *	sets up a geometry object, this is always a geometry file.
 */
object *
geometryinit(d)
	details	*d;
{
	object		*o;
	details		*ld;
	vector		c1, c2, cent;

	o = (object *)smalloc(sizeof(object));
	o->type = GEOMETRY;

	while (d != (details *)NULL) {
		switch (d->type) {
		case OFFFILE:
			o->intersects = geomi;
			o->normal = geomn;
			o->obj.geo = geominit(d->u.s);
			break;
		default:
			warning("art: illegal field in geometry ignored.\n");
		}
		ld = d;
		d = d->nxt;
		free(ld);
	}

	if (o->obj.geo != (geometry *)NULL) {
		c1 = o->obj.geo->bb.topfr;
		c2 = o->obj.geo->bb.botbl;
		cent.x = (c1.x + c2.x) / 2.0;
		cent.y = (c1.y + c2.y) / 2.0;
		cent.z = (c1.z + c2.z) / 2.0;

		makebsphere(&o->bs, &cent, sqr(c1.x - cent.x) + sqr(c1.y - cent.y) + sqr(c1.z - cent.z));
	} else
		fatal("art: no geometry file defined in geometry.\n");

	setattributes(o);

	return(o);
}

/*
 * polyn
 *
 *	returns the normal to a polygon.
 */
void
polyn(n, l, o, indx)
	vector	*n, *l;
	object	*o;
	int	indx;
{
	*n = o->obj.geo->faces->n;
}

/*
 * polyc
 *
 *	returns the colour of polygon o at point l
 */
void
polyc(o, txt, l, n, pcol, type)
	object  *o;
	texture *txt;
	vector  *l, *n;
	pixel   *pcol;
	int     type;
{
	int	indx, w, h;
	vector	loc, tmp;
	float	xlensq, ylensq, zlensq;

	toobject(o, tmp, *l); 

	totexture(txt, loc, tmp);

	xlensq = (loc.x - txt->vref1.x) * (loc.x - txt->vref1.x);
	ylensq = (loc.y - txt->vref1.y) * (loc.y - txt->vref1.y);
	zlensq = (loc.z - txt->vref1.z) * (loc.z - txt->vref1.z);

	switch (txt->axis) {
	case XAXIS:
		w = sqrt((double)(zlensq + xlensq)) * txt->scalew;
		h = sqrt((double)(ylensq + xlensq)) * txt->scaleh;
		break;
	case YAXIS:
		w = sqrt((double)(xlensq + ylensq)) * txt->scalew;
		h = sqrt((double)(zlensq + ylensq)) * txt->scaleh;
		break;
	case ZAXIS:
		w = sqrt((double)(xlensq + zlensq)) * txt->scalew;
		h = sqrt((double)(ylensq + zlensq)) * txt->scaleh;
		break;
	default:
		fatal("art: bad axis in polyc.\n");
	}

	indx = (w % txt->pixw + (h % txt->pixh) * txt->pixw) * 3;

	pcol->r = (unsigned char)txt->map[indx] / 255.0;
	pcol->g = (unsigned char)txt->map[indx + 1] / 255.0;
	pcol->b = (unsigned char)txt->map[indx + 2] / 255.0;
}

/*
 * polygoninit
 *
 *	initialise the function pointers and fields for a polygon
 */
object *
polygoninit(d)
	details *d;
{
	object		*o;
	geometry	*gm;
	polygon		*pg;
	int		axis, i, count;
	matrix		m;
	tlist		*tl;
	vector		v, v1, v2, ovt[MAXVERTS], vt[MAXVERTS], *vtp[MAXVERTS];
	float		maxx, maxy, maxz, minx, miny, minz;
	float		xlensq, ylensq, zlensq;
	details		*ld;

	o = (object *)smalloc(sizeof(object));
	o->type = POLYGON;
	o->normal = polyn;
	o->intersects = polyi;

	setattributes(o);

	mmult4(m, trans, astackp->m);

	gm = o->obj.geo = (geometry *)smalloc(sizeof(geometry));
	gm->faces = pg = (polygon *)smalloc(sizeof(polygon));

	count = 0;
	while (d != (details *)NULL) {
		switch (d->type) {
		case VERTEX:
					/* transform vertex into ray space */
			ovt[count] = d->u.v;
			d->u.v.x -= m[3][0];
			d->u.v.y -= m[3][1];
			d->u.v.z -= m[3][2];
			v3x3tmult(vt[count], d->u.v, m);
			vt[count].x /= astackp->scales.x;
			vt[count].y /= astackp->scales.y;
			vt[count].z /= astackp->scales.z;
			vtp[count] = vt + count;
			count++;
			break;
		default:
			warning("art: illegal field in polygon ignored.\n");
		}
		ld = d;
		d = d->nxt;
		free(ld);
	}

	gm->options = 0;
	gm->npolys[XAXIS] = gm->npolys[YAXIS] = gm->npolys[ZAXIS] = 0;
	gm->ntris[XAXIS] = gm->ntris[YAXIS] = gm->ntris[ZAXIS] = 0;
	gm->norms = (vector *)NULL;

	vsub(v1, vt[0], vt[1]);
	vsub(v2, vt[0], vt[count - 1]);

	xprod(pg->n, v1, v2);

	if (count > 3) {
		normalise(pg->n);
		for (i = 1; i != count; i++) {
			vsub(v1, vt[i], vt[(i + 1) % count]);
			vsub(v2, vt[i], vt[i - 1]);
			xprod(v, v1, v2);
			normalise(v);
			vadd(pg->n, pg->n, v);
		}

		smult(pg->n, 1.0 / count);
	}

	gm->xs = (float *)smalloc(sizeof(float) * count);
	gm->ys = (float *)smalloc(sizeof(float) * count);
	gm->zs = (float *)smalloc(sizeof(float) * count);

	pg->vertnos = (vertno *)smalloc(sizeof(vertno) * count);

	minx = maxx = ovt[0].x;
	minz = maxz = ovt[0].z;
	miny = maxy = ovt[0].y;

	gm->xs[0] = vt[0].x;
	gm->ys[0] = vt[0].y;
	gm->zs[0] = vt[0].z;

	pg->vertnos[0] = 0;

	for (i = 1; i != count; i++) {
		gm->xs[i] = vt[i].x;
		gm->ys[i] = vt[i].y;
		gm->zs[i] = vt[i].z;
		minmax(minx, maxx, ovt[i].x);
		minmax(miny, maxy, ovt[i].y);
		minmax(minz, maxz, ovt[i].z);
		pg->vertnos[i] = i;
	}

	normalise(pg->n);

	pg->cnst = -dprod(pg->n, vt[0]);

	axis = getaxis(pg->n);

	gm->npolys[axis] = 1;

	pg->nsides = count;

	for (tl = o->s.txtlist; tl != (tlist *)NULL; tl = tl->nxt) {
		if (tl->txtcol == (void (*)())NULL) {

			tl->txtcol = polyc;

			/*
			 * find major axis of polygon normal
			 */
			vsub(v1, ovt[0], ovt[1]);
			vsub(v2, ovt[0], ovt[count - 1]);

			xprod(v, v1, v2);

			tl->txt.axis = getaxis(v);

			tl->txt.vref1 = ovt[count - 1];

			/*
			 * adjust scale so relative to 0 to 1
			 */
			xlensq = (minx - maxx) * (minx - maxx);
			ylensq = (miny - maxy) * (miny - maxy);
			zlensq = (minz - maxz) * (minz - maxz);

			switch (tl->txt.axis) {
			case XAXIS:
				tl->txt.scalew /= sqrt((double)(zlensq + xlensq));
				tl->txt.scaleh /= sqrt((double)(ylensq + xlensq));
				break;
			case YAXIS:
				tl->txt.scalew /= sqrt((double)(xlensq + ylensq));
				tl->txt.scaleh /= sqrt((double)(zlensq + ylensq));
				break;
			case ZAXIS:
				tl->txt.scalew /= sqrt((double)(xlensq + zlensq));
				tl->txt.scaleh /= sqrt((double)(ylensq + zlensq));
				break;
			default:
				fatal("art: bad axis in polygoninit.\n");
			}
		}
	}

	return(o);
}
