#include <math.h>
#include <stdio.h>
#include "art.h"
#include "macro.h"
#include "gram.h"

extern attr	*astackp;
extern hlist	*fhlist;

/*
 * cyli
 *
 *	returns a list of intersection points for the ray r and cylinder o.
 *
 */
hlist *
cyli(r, o, last)
	register ray	*r;
	register object	*o;
	hlist		**last;
{
	hlist			*hitlist, *hp;
	float			ax1, ax2;
	int			entry, exit;
	ray			nr;
	vector			cv;
	register float		a, b, c, d, t, t1, t2;

	vadd(cv, r->org, o->bs.cent);
	b = dprod(cv, r->dir);

	if ((dprod(cv, cv) - o->bs.radsqu) > b * b)
		return((hlist *)NULL);

	transray(o, nr, *r);

	a = nr.dir.x * nr.dir.x + nr.dir.y * nr.dir.y;
	b = nr.org.x * nr.dir.x + nr.org.y * nr.dir.y;
	c = nr.org.x * nr.org.x + nr.org.y * nr.org.y - 1.0;

	d = b * b - a * c;

	if (d < 0.0 && nr.dir.z == 0.0)
		return((hlist *)NULL);
	
	hitlist = (hlist *)NULL;

	if (d < 0.0 || a == 0.0) {

		if (nr.org.x * nr.org.x + nr.org.y * nr.org.y > 1.0)
			return((hlist *)NULL);

		t1 = -nr.org.z / nr.dir.z;
		t2 = (1.0 - nr.org.z) / nr.dir.z;

		if (t1 < TOLERANCE && t2 < TOLERANCE)
			return((hlist *)NULL);

		if (t1 > t2) {
			t = t1;
			t1 = t2;
			t2 = t;
		}

		entry = exit = FACE;
	} else {
		entry = exit = SIDE;

		d = sqrt((double)d);

		t1 = (-b - d) / a;
		t2 = (-b + d) / a;

		ax1 = t1 * nr.dir.z + nr.org.z;
		ax2 = t2 * nr.dir.z + nr.org.z;

		if (ax1 < 0.0 && ax2 < 0.0)
			return((hlist *)NULL);

		if (ax1 > 1.0 && ax2 > 1.0)
			return((hlist *)NULL);

		if (ax1 > 1.0) {
			entry = FACE;
			t1 = (1.0 - nr.org.z) / nr.dir.z;
		} else if (ax1 < 0.0) {
			entry = FACE;
			t1 = -nr.org.z / nr.dir.z;
		} 

		if (ax2 > 1.0) {
			exit = FACE;
			t2 = (1.0 - nr.org.z) / nr.dir.z;
		} else if (ax2 < 0.0) {
			exit = FACE;
			t2 = -nr.org.z / nr.dir.z;
		}
	}

	if (t1 >= TOLERANCE) {
		fetch(hp);
		hp->type = entry;
		hitlist = hp;
		hp->t = t1;
		hp->obj = o;
		if (t2 >= TOLERANCE) {
			fetch(hp);
			hitlist->nxt = hp;
			hp->type = exit;
			hp->t = t2;
			hp->obj = o;
		} 
		hp->nxt = (hlist *)NULL;
	} else if (t2 >= TOLERANCE) {
		fetch(hp);
		hitlist = hp;
		hp->type = exit;
		hp->t = t2;
		hp->obj = o;
		hp->nxt = (hlist *)NULL;
	}

	*last = hp;

	return(hitlist);
}

/*
 * cyln
 *
 *	returns the normal to the cylinder o
 */
void
cyln(n, l, o, type)
	register vector		*n;
	register vector		*l;
	object			*o;
	int			type;
{
	vector			v;

	if (type == SIDE) {
		toobject(o, v, *l);

		v.z = 0.0;

		normalise(v);
	} else {
		v.x = 0.0;
		v.y = 0.0;
		v.z = 1.0;
	}

	if (o->mat != NULL)
		v3x3tmult(*n, v, (*o->mat))
	else
		*n = v;
}

/*
 * cylc
 *
 *	return the colour of the cylinder o at the intersection point l
 */
void
cylc(o, txt, l, n, pcol, type)
	object  *o;
	texture *txt;
	vector  *l, *n;
	pixel   *pcol;
	int     type;
{
	vector	tmp, tmp2;
	float	u;
	int	w, h, indx;

	if (type == SIDE) {
		toobject(o, tmp2, *l);

		totexture(txt, tmp, tmp2);

		if (fabs(tmp.x) > 1.0)
			tmp.x = (tmp.x < 0.0) ? -1.0 : 1.0;

		u = acos(tmp.x) / (2.0 * M_PI);

		w = ((tmp.y < 0.0) ? 1.0 - u : u) * txt->scaleh;
		h = tmp.z * txt->scalew;

		indx = (w % txt->pixw + (h % txt->pixh) * txt->pixw) * 3;

		pcol->r = (unsigned char)txt->map[indx] / 255.0;
		pcol->g = (unsigned char)txt->map[indx + 1] / 255.0;
		pcol->b = (unsigned char)txt->map[indx + 2] / 255.0;
	}
}

/*
 * cylinit
 *
 *	initialise the function pointers and fields of a cylinder object
 */
object *
cylinit(d)
	details *d;
{
	object	*o;
	int	first;
	vector	cent, c1, c2, rads;
	details	*ld;
	tlist	*tl;
	float	rad;

	o = (object *)smalloc(sizeof(object));
	o->type = CYLINDER;
	o->intersects = cyli;
	o->normal = cyln;

	first = 1;

	rads.x = rads.y = rad = 1.0;

	while (d != (details *)NULL) {
		switch (d->type) {
		case CENTER:
			if (first) {
				c1 = d->u.v;
				first = 0;
			} else
				c2 = d->u.v;
			break;
		case RADIUS:
			rads.x = rads.y = rad = d->u.f;
			break;
		case RADII:
			if (d->u.v.x > d->u.v.y)
				rad = d->u.v.x;
			else
				rad = d->u.v.y;
			rads = d->u.v;
			break;
		default:
			warning("art: illegal field in cylinder ignored.\n");
		}
		ld = d;
		d = d->nxt;
		free(ld);
	}

	cent.x = (c1.x + c2.x) / 2.0;
	cent.y = (c1.y + c2.y) / 2.0;
	cent.z = (c1.z + c2.z) / 2.0;

	makebsphere(&o->bs, &cent, sqr(c1.x - cent.x) + sqr(c1.y - cent.y) + sqr(c1.z - cent.z) + rad * rad);

	qtransform(c1, c2);

	scale(rads.x, rads.y, 1.0);

	setattributes(o);

	for (tl = o->s.txtlist; tl != (tlist *)NULL; tl = tl->nxt)
		if (tl->txtcol == (void (*)())NULL)
			tl->txtcol = cylc;

	return(o);
}
