#include <math.h>
#include <stdio.h>
#include "art.h"
#include "macro.h"
#include "gram.h"
#include "poly.h"

extern attr	*astackp;
extern hlist	*fhlist;

extern double	power();

/*
 * bounding box for all superquadrics
 */
static bbox sqbox = {
	{ 1.0, 1.0, 1.0 },
	{ -1.0, -1.0, -1.0 }
};

/*
 * superi
 *
 *	returns the first hit of the ray r on the superquadric o, unless the
 * superquadric is in a csg model in which case a list of intersection points
 * for the ray r is returned.
 */
hlist *
superi(r, o, last)
	register ray	*r;
	register object	*o;
	hlist		**last;
{
	hlist		*hitlist, *hp;
	float		t1, t2;
	ray		nr;
	vector		cv;
	superquadric	*sprq;
	register int	ord;
	register float	oldt, diff;
	register float	val, oldval;

	vadd(cv, r->org, o->bs.cent);
	val = dprod(cv, r->dir);

	if ((dprod(cv, cv) - o->bs.radsqu) > val * val)
		return((hlist *)NULL);

	transray(o, nr, *r);

	if (!inbbox(&nr, &sqbox, &t1, &t2))
		return((hlist *)NULL);
	
	hitlist = (hlist *)NULL;

	sprq = o->obj.spq;

	ord = sprq->ord;

	val = power(fabs(nr.org.x), ord) + power(fabs(nr.org.y), ord)
		+ power(fabs(nr.org.z), ord) - 1.0;
	
	if (val <= ALG_TOLERANCE)	/* we are on the surface */
		oldt = t1 + TOLERANCE;
	else
		oldt = t1 - TOLERANCE;	/* we are heading towards it */

	val = power(fabs(nr.org.x + nr.dir.x * oldt), ord)
		+ power(fabs(nr.org.y + nr.dir.y * oldt), ord)
		+ power(fabs(nr.org.z + nr.dir.z * oldt), ord)
		- 1.0;

	while (val > ALG_TOLERANCE) {
		oldval = val;
		val = power(fabs(nr.org.x + nr.dir.x * t1), ord)
			+ power(fabs(nr.org.y + nr.dir.y * t1), ord)
			+ power(fabs(nr.org.z + nr.dir.z * t1), ord)
			- 1.0;

		if (val >= oldval) {
			t1 = 0.0;
			break;
		}

		diff = (val * (t1 - oldt)) / (val - oldval);
		oldt = t1;

		t1 -= diff;
	}

	if (sprq->incsg && t2 > TOLERANCE) {
		oldt = t2 + TOLERANCE;

		val = power(fabs(nr.org.x + nr.dir.x * oldt), ord)
			+ power(fabs(nr.org.y + nr.dir.y * oldt), ord)
			+ power(fabs(nr.org.z + nr.dir.z * oldt), ord)
			- 1.0;

		while (val > ALG_TOLERANCE) {
			oldval = val;
			val = power(fabs(nr.org.x + nr.dir.x * t2), ord)
				+ power(fabs(nr.org.y + nr.dir.y * t2), ord)
				+ power(fabs(nr.org.z + nr.dir.z * t2), ord)
				- 1.0;

			if (val >= oldval) {
				t2 = 0.0;
				break;
			}

			diff = (val * (t2 - oldt)) / (val - oldval);
			oldt = t2;

			t2 -= diff;
		}

		if (t2 > TOLERANCE) {
			fetch(hitlist);
			hitlist->obj = o;
			hitlist->t = t2;
			hitlist->nxt = (hlist *)NULL;
			*last = hitlist;
		}
	}


	if (t1 > TOLERANCE) {
		fetch(hp);
		hp->obj = o;
		hp->t = t1;
		hp->nxt = hitlist;
		hitlist = hp;
		if (hp->nxt == (hlist *)NULL)
			*last = hp;
	}

	return(hitlist);
}

/*
 * supern
 *
 *	returns the normal vector to a point on a superquadric surface
 */
void
supern(n, l, o)
	register vector	*n, *l;
	register object	*o;
{
	register int	ord;
	vector		loc;

	ord = o->obj.spq->ord;

	toobject(o, loc, *l);

	loc.x = (loc.x > 0) ? power(loc.x, ord - 1) : -power(-loc.x, ord - 1);
	loc.y = (loc.y > 0) ? power(loc.y, ord - 1) : -power(-loc.y, ord - 1);
	loc.z = (loc.z > 0) ? power(loc.z, ord - 1) : -power(-loc.z, ord - 1);

	if (o->mat != (mat3x3 *)NULL) {
		v3x3tmult(*n, loc, (*o->mat));
	} else
		*n = loc;

	normalise(*n);
}

/*
 * superc
 *
 *	returns the colour of a superquadric at the position loc.
 */
void
superc(o, txt, l, n, pcol, type)
	object  *o;
	texture *txt;
	vector  *l, *n;
	pixel   *pcol;
	int     type;
{
	float   u, v;
	int     w, h, indx;
	vector	loc, tmp;

	toobject(o, tmp, *l);

	totexture(txt, loc, tmp);

	normalise(loc);

	spheremap(&loc, txt, &u, &v);

	w = u * txt->scalew;
	h = v * txt->scaleh;

	indx = (w % txt->pixw + (h % txt->pixh) * txt->pixw) * 3;

	pcol->r = (unsigned char)txt->map[indx] / 255.0;
	pcol->g = (unsigned char)txt->map[indx + 1] / 255.0;
	pcol->b = (unsigned char)txt->map[indx + 2] / 255.0;
}

/*
 * superinit
 *
 *	initialise the function pointers and fields for a superquadric object,
 * 	returning its pointer.
 */
object *
superinit(d, incsg)
	details *d;
	int	incsg;
{
	object		*o;
	superquadric	*sprq;
	int		first;
	vector		topcnr, botcnr, cent, v;
	float		xlen, ylen, zlen;
	tlist		*tl;
	details		*ld;

	o = (object *)smalloc(sizeof(object));
	o->type = SUPERQUADRIC;
	o->intersects = superi;
	o->normal = supern;

	sprq = o->obj.spq = (superquadric *)smalloc(sizeof(superquadric));

	sprq->incsg = incsg;

	first = 1;

	botcnr.x = botcnr.y = botcnr.z = -1.0;
	topcnr.x = topcnr.y = topcnr.z = 1.0;

	while (d != (details *)NULL) {
		switch (d->type) {
		case VERTEX:
			if (first) {
				botcnr.x = d->u.v.x;
				botcnr.y = d->u.v.y;
				botcnr.z = d->u.v.z;
				first = 0;
			} else {
				topcnr.x = d->u.v.x;
				topcnr.y = d->u.v.y;
				topcnr.z = d->u.v.z;
			}
			break;
		case ORDER:
			sprq->ord = d->u.f;
			break;
		default:
			warning("art: illegal field in superquadric ignored.\n");
		}
		ld = d;
		d = d->nxt;
		free(ld);
	}

	xlen = (topcnr.x - botcnr.x) / 2;
	ylen = (topcnr.y - botcnr.y) / 2;
	zlen = (topcnr.z - botcnr.z) / 2;

	cent.x = xlen + botcnr.x;
	cent.y = ylen + botcnr.y;
	cent.z = zlen + botcnr.z;

	makebsphere(&o->bs, &cent, sqr(xlen) + sqr(ylen) + sqr(zlen));

	objtranslate(cent.x, cent.y, cent.z);

	scale(xlen, ylen, zlen);

	setattributes(o);

	for (tl = o->s.txtlist; tl != (tlist *)NULL; tl = tl->nxt)
		if (tl->txtcol == (void (*)())NULL) {
			if (!tl->txt.refset) {
				tl->txt.vref1.y = 1.0;
				tl->txt.vref1.x = tl->txt.vref1.z = 0.0;
				tl->txt.vref2.x = 1.0;
				tl->txt.vref2.y = tl->txt.vref2.z = 0.0;
			}

			tl->txtcol = superc;

			v3x3tmult(v, tl->txt.vref1, astackp->m);
			tl->txt.vref1 = v;
			v3x3tmult(v, tl->txt.vref2, astackp->m);
			tl->txt.vref2 = v;
		}

	return(o);
}
