#include <math.h>
#include <stdio.h>
#include "art.h"
#include "macro.h"
#include "gram.h"

extern attr	*astackp;
extern hlist	*fhlist;

/*
 * ellipsi
 *
 *	returns a list of intersection points for the ray r and ellipsoid o.
 *
 */
hlist *
ellipsi(r, o, last)
	register ray	*r;
	register object	*o;
	hlist		**last;
{
	hlist			*hitlist, *hp;
	register float		a, b, c, d, t1, t2;
	vector			cv;
	ray			nr;

	vadd(cv, r->org, o->bs.cent);
	t1 = dprod(cv, r->dir);

	if ((dprod(cv, cv) - o->bs.radsqu) > t1 * t1)
		return((hlist *)NULL);

	transray(o, nr, *r);

	a = dprod(nr.dir, nr.dir);
	b = dprod(nr.dir, nr.org);
	c = dprod(nr.org, nr.org) - 1.0;

	d = b * b - a * c;

	if (d < 0.0)
		return((hlist *)NULL);
	
	hitlist = (hlist *)NULL;

	d = sqrt((double)d);

	t1 = (-b - d) / a;
	t2 = (-b + d) / a;

	if (t1 >= TOLERANCE) {
		fetch(hp);
		hitlist = hp;
		hp->t = t1;
		hp->obj = o;
		if (t2 >= TOLERANCE) {
			fetch(hp);
			hitlist->nxt = hp;
			hp->t = t2;
			hp->obj = o;
		} 
		hp->nxt = (hlist *)NULL;
	} else if (t2 >= TOLERANCE) {
		fetch(hp);
		hitlist = hp;
		hp->t = t2;
		hp->obj = o;
		hp->nxt = (hlist *)NULL;
	}

	*last = hp;

	return(hitlist);
}

/*
 * ellipsn
 *
 *	returns the normal to the ellipsoid o
 */
void
ellipsn(n, l, o)
	register vector		*n;
	register vector		*l;
	object			*o;
{
	vector			loc;

	toobject(o, loc, *l);

	if (o->mat != (mat3x3 *)NULL) {
		v3x3tmult(*n, loc, (*o->mat));
	} else
		*n = loc;

	normalise(*n);
}

/*
 * ellipsc
 *
 *	return the color of an ellipse o at a the intersection point l.
 *
 */
void
ellipsc(o, txt, l, n, pcol, type)
	object  *o;
	texture *txt;
	vector  *l, *n;
	pixel   *pcol;
	int     type;
{
	vector	tmp;
	float	u, v;
	int	w, h, indx;

	totexture(txt, tmp, *n); 

	spheremap(&tmp, txt, &u, &v);

	w = u * txt->scalew;
	h = v * txt->scaleh;

	indx = (w % txt->pixw + (h % txt->pixh) * txt->pixw) * 3;

	pcol->r = (unsigned char)txt->map[indx] / 255.0;
	pcol->g = (unsigned char)txt->map[indx + 1] / 255.0;
	pcol->b = (unsigned char)txt->map[indx + 2] / 255.0;
}

/*
 * ellipsinit
 *
 *	initialise the function pointers and fields of an ellipsoid
 *
 */
object *
ellipsinit(d)
	details *d;
{
	object		*o;
	details		*ld;
	float		radius;
	vector		cent, radii, v;
	tlist		*tl;

	o = (object *)smalloc(sizeof(object));
	o->type = ELLIPSOID;
	o->intersects = ellipsi;
	o->normal = ellipsn;

	cent.x = cent.y = cent.z = 0.0;
	radii.x = radii.y = radii.z = 1.0;

	while (d != (details *)NULL) {
		switch (d->type) {
		case CENTER:
			cent = d->u.v;
			break;
		case RADII:
			radii = d->u.v;
			break;
		case RADIUS:
			/*
			 * only likely if someone is scaling a sphere
			 */
			radii.x = radii.y = radii.z = d->u.f;
			break;
		default:
			warning("art: illegal field in ellipsoid ignored.\n");
		}
		ld = d;
		d = d->nxt;
		free(ld);
	}

	if (radii.x > radii.y)
		if (radii.x > radii.z)
			radius = radii.x;
		else
			radius = radii.z;
	else if (radii.y > radii.z)
			radius = radii.y;
		else
			radius = radii.z;

	makebsphere(&o->bs, &cent, radius * radius);

	objtranslate(cent.x, cent.y, cent.z);

	scale(radii.x, radii.y, radii.z);

	setattributes(o);

	for (tl = o->s.txtlist; tl != (tlist *)NULL; tl = tl->nxt)
		if (tl->txtcol == (void (*)())NULL) {
			if (!tl->txt.refset) {
				tl->txt.vref1.y = 1.0;
				tl->txt.vref1.x = tl->txt.vref1.z = 0.0;
				tl->txt.vref2.x = 1.0;
				tl->txt.vref2.y = tl->txt.vref2.z = 0.0;
			}

			tl->txtcol = ellipsc;

			v3x3tmult(v, tl->txt.vref1, astackp->m);
			tl->txt.vref1 = v;
			v3x3tmult(v, tl->txt.vref2, astackp->m);
			tl->txt.vref2 = v;
		}

	return(o);
}
