/*
 * Display a color-mapped image on a PC VGA.
 * Images are produced by ART and MEDIAN.
 * Currently, only images that are 320 x 200 & 320 x 400 are handled well.
 */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <io.h>
#include <bios.h>
#include <string.h>
#include "vort.h"
#include "pcdisp.h"


#define	MAX_ADAPTERS	2
#define	MAX_FRAMES		100
#define	MAX_IMAGE_WIDTH	2048
#define	PALETTE_SIZE	256
#define SET_TEXT()	{union REGS regs; regs.x.ax = 3; int86(0x10, &regs, &regs);}

typedef unsigned char	UCHAR;
typedef	unsigned short	USHORT;

/* adapter info */
ADAPTER	adapters[MAX_ADAPTERS];
ADAPTER	adapter;

/* for reading images */
UCHAR	*r, *g, *b, *s;

/* colour map info */
UCHAR	palette[PALETTE_SIZE][3];
int		darkest_palette = 0;
int		cmap_size = PALETTE_SIZE;

/* frame stuff */
UCHAR	*frame[MAX_FRAMES];			/* ptrs to frames loaded from image files */
int		nframes = 0;				/* number of such ptrs */
int		frame_w,					/* width & height of 1st image */
		frame_h,					/*  used for all subsequent images */
		display_by_line,			/* TRUE when image is too big for memory */
		request_abort = FALSE;		/* TRUE when user wants to cark his movie */

char	*progname;



int			main(argc, argv)
int			argc;
char		*argv[];
{
	int		ac;
	char	**av;
	int		mode = 0,
			i,
			key,
			animate = FALSE,
			hold = FALSE,
			overscan = FALSE;

	/* setup adapter structures - should ALWAYS be MAX_ADAPTERS calls here */
	setup_vga320x200(&adapters[0]);
	setup_vga320x400(&adapters[1]);

	progname = *argv;
	if (argc < 2) usage(progname);

	ac = --argc;
	av = ++argv;

	while (ac-- > 0) {
		if (**av == '-' || **av == '/') {
			switch (tolower((*av)[1])) {

				case '?' :
					usage(progname);
					break;

				case 'a' :
					animate = TRUE;
					break;

				case 'm' :
					if (((*av)[2]) == 0) {
						if (--ac >= 0) sscanf(*++av, "%d", &mode);
						else usage(progname);
						}
					else sscanf(&((*av)[2]), "%d", &mode);
					if (mode < 0 || mode >= MAX_ADAPTERS) usage(progname);
					break;

				case 'o' :
					overscan = TRUE;
					break;

				default :
					usage(progname);
					break;
				}
			}
		++av;
		}

	/* allocate buffers for maximum possible (likely) image width */
	s = (UCHAR *) malloc(MAX_IMGWIDTH);
	r = (UCHAR *) malloc(MAX_IMGWIDTH);
	g = (UCHAR *) malloc(MAX_IMGWIDTH);
	b = (UCHAR *) malloc(MAX_IMGWIDTH);

	/* set the adapter */
	adapter = adapters[mode];
	if (!animate) adapter.init();

	for (ac = argc, av = argv; ac; ac--, av++) {
		/* process all the files named */
		if (**av == '-' || **av == '/') {
			/* skip options and their parameters */
			switch ((*av)[1]) {
				case 'm' :
					if ((*av)[2] == '\0') {
						--ac;
						++av;
						}
					break;
				default:
					break;
				}
			continue;
			}

		if (animate) {
			/* load the image into memory - load colour palette for 1st image only */
			if (load_image(*av, &(frame[nframes]), nframes == 0)) nframes++;
			}
		else {
			if (load_image(*av, &(frame[0]), TRUE)) {
				/* display image from memory */
				adapter.palette(palette, cmap_size);
				if (overscan) adapter.overscan(darkest_palette);
				display_frame(frame[0]);
				free(frame[0]);
				}
			else {
				/* display image from file */
				if (display_by_line) display_image(*av, overscan);
				}
			getch();
			adapter.erase(darkest_palette);
			}
		}

	if (animate) {
		adapter.init();
		adapter.palette(palette, cmap_size);
		if (overscan) adapter.overscan(darkest_palette);
		adapter.back();
		i = 0;
		do {
			if (get_key(&key)) {
				switch (toupper(key)) {
					case 'Q' : request_abort = TRUE; break;
					case 'H' : hold = !hold; break;
					default : break;
					}
				}
			else if (!hold) {
				display_frame(frame[i]);
				adapter.swap();
				if (++i >= nframes) i = 0;
				}
			} while (!request_abort);
		}

	/* reset video mode */
	SET_TEXT();

	exit(0);
}



int			display_image(fn, overscan)
char		*fn;
int			overscan;
{
	image	*im;
	int		w, h,
			x, y;
	UCHAR	*sptr;

	if ((im = openimage(fn, "r")) == (image *) NULL) {
		SET_TEXT();
		fprintf(stderr, "%s: can't open file %s.\n", progname, fn);
		exit(1);
		}

	w = imagewidth(im);
	h = imageheight(im);

	/* remap palette */
	make_palette(im);
	adapter.palette(palette, cmap_size);
	if (overscan) adapter.overscan(darkest_palette);

	/* display image line by line */
	imagepos(im, 0, 0);
	switch (imagetype(im)) {

		case PIX_CMAP :
		case PIX_RLECMAP :
			for (y = 0; y < h; y++) {
				readmappedline(im, s);
				adapter.line(s, w, 0, y);
				}
			break;

		default :
			for (y = 0; y < h; y++) {
				readrgbline(im, r, g, b);
				sptr = s;
				for (x = 0; x < w; x++) {
					*sptr++ = (UCHAR) ((90*r[x] + 140*g[x] + 25*b[x]) >> 8);
					}
				adapter.line(s, w, 0, y);
				}
			break;
		}
	closeimage(im);
}



int			display_frame(fptr)
char		*fptr;
{
	adapter.block(fptr, frame_w, frame_h, 0, 0);
}



int			load_image(fn, frame, set_frame_params)
char		*fn;
char		**frame;
int			set_frame_params;
{
	image	*im;
	int		x, y;
	UCHAR	*sptr,
			*fptr;

	display_by_line = FALSE;
	if ((im = openimage(fn, "r")) == (image *) NULL) {
		fprintf(stderr, "%s: can't open file %s.\n", progname, fn);
		return FALSE;
		}
	fprintf(stderr, "%s\n", fn);

	if (frame_w == 0 || set_frame_params) frame_w = imagewidth(im);
	if (frame_h == 0 || set_frame_params) frame_h = imageheight(im);

	/* validate size of buffer required */
	if (((long)frame_w * (long)frame_h) > 65500L) {
		/* can't load image into memory without heaps of stuffing around, */
		/* so just display it from file line by line */
		display_by_line = TRUE;
		closeimage(im);
		return FALSE;
		}
	else display_by_line = FALSE;

	if (!(fptr = *frame = malloc(frame_w * frame_h))) {
		fprintf(stderr, "%s: can't alloc %u bytes.\n", progname, (long) (frame_w * frame_h));
		closeimage(im);
		return FALSE;
		}

	/* create palette if required */
	if (set_frame_params) make_palette(im);

	/* load image */
	imagepos(im, 0, 0);
	switch (imagetype(im)) {

		case PIX_CMAP :
		case PIX_RLECMAP :
			for (y = 0; y < frame_h; y++) {
				readmappedline(im, s);
				memcpy(fptr, s, frame_w);
				fptr += frame_w;
				}
			break;

		default :
			for (y = 0; y < frame_h; y++) {
				readrgbline(im, r, g, b);
				sptr = s;
				for (x = 0; x < frame_w; x++) {
					*fptr = (UCHAR) ((90*r[x] + 140*g[x] + 25*b[x]) >> 8);
					fptr++;
					}
				}
			break;
		}

	closeimage(im);
	return TRUE;
}


int			make_palette(im)
image		*im;
{
	int		i,
			r1, g1, b1;
	long	darkest = 999999999,
			intensity;

	/* generate palette */
	if (imagetype(im) == PIX_CMAP || imagetype(im) == PIX_RLECMAP) {
		/* set up colour palette for colour map files */
		for (i = 0; i < cmapsize(im); i++) {
			/* 6-bits only per value please */
			r1 = palette[i][0]=im->red[i] >> 2;
			g1 = palette[i][1]=im->green[i] >> 2;
			b1 = palette[i][2]=im->blue[i] >> 2;
			/* check if this colour is darker than previous darkest */
			if ((intensity = r1*r1 + g1*g1 + b1*b1) < darkest) {
				darkest = intensity;
				darkest_palette = i;
				}
			}
		cmap_size = cmapsize(im);
		if (cmap_size < PALETTE_SIZE && darkest != 0) {
			/* palette not yet full & darkest not black - add a black entry to colour map */
			palette[cmap_size][0] = 0;
			palette[cmap_size][1] = 0;
			palette[cmap_size][2] = 0;
			darkest_palette = cmap_size;
			cmap_size++;
			}
		}							  	
	else {
		/* create a B&W colour map */
		for (i = 0; i < PALETTE_SIZE; i++) {
			/* gray scale: 6-bits only per value */
			palette[i][0] = i >> 2;
			palette[i][1] = i >> 2;
			palette[i][2] = i >> 2;
			}
		darkest_palette = 0;
		cmap_size = PALETTE_SIZE;
		}
}



int			usage(prog)
char		*prog;
{
	int		i;

	fprintf(stderr,"usage: %s fname [fname...] [-m mode] [-d delay] [-o] [-?]\n\n", prog);
	fprintf(stderr,"where:\n");
	fprintf(stderr,"       fname    - filename of image to display (may be wildcard)\n");
	fprintf(stderr,"       -m mode  - video mode to use\n");
	for (i = 0; i < MAX_ADAPTERS; i++) fprintf(stderr, "                   = %2d %s\n", i, adapters[i].descr);
	fprintf(stderr,"       -d delay - delay between successive images when several specified (msec)\n");
	fprintf(stderr,"       -o       - set overscan colour to darkest colour in colour map\n");
	fprintf(stderr,"       -?       - this message\n");
	exit(1);
}


int			fail(msg)
char		*msg;
{
	fprintf(stderr, "%s", msg);
	exit(1);
}


int			get_key(key)
int			*key;
{
	*key = _bios_keybrd(_KEYBRD_READY) & 0xff;
	if (*key) _bios_keybrd(_KEYBRD_READ);
	return *key;
}



/****
 * Normalize a (LARGE/HUGE) pointer so that the offset part
 * is as small as possible. This means additions to the pointer
 * of values < (64K - 16) will work correctly.
 **/

static void *norm_pointer(p)
void *p;
{
	void *rp;
	
	FP_SEG(rp) = FP_SEG (p) + FP_OFF (p) / 16;
	FP_OFF(rp) = FP_OFF (p) % 16;
	return (rp);
}

