#include <math.h>
#include <stdio.h>
#include "art.h"
#include "macro.h"
#include "gram.h"

extern attr	*astackp;
extern hlist	*fhlist;

/*
 * interp3
 *
 *	calculates the interpolated normal for the triangle described
 * by the normals norms. In this case we have to use area (barycentric)
 * coordinates. P is the (x, y, z) point we are on.
 */
interp3(x, y, z, xs, ys, zs, norms, n)
	register float	x, y, z;
	register float	*xs, *ys, *zs;
	register vector	**norms, *n;
{
	vector		v, v1, v2;
	register float	l1, l2, l3;

	v1.x = xs[1] - xs[2];		/* area 1, 2, P */
	v1.y = ys[1] - ys[2];
	v1.z = zs[1] - zs[2];

	v2.x = xs[1] - x;
	v2.y = ys[1] - y;
	v2.z = zs[1] - z;

	xprod(v, v1, v2);
	l1 = sqrt(sqr(v.x) + sqr(v.y) + sqr(v.z));

	v1.x = xs[2] - xs[0];		/* area 0, 2, P */
	v1.y = ys[2] - ys[0];
	v1.z = zs[2] - zs[0];

	v2.x = xs[2] - x;
	v2.y = ys[2] - y;
	v2.z = zs[2] - z;

	xprod(v, v1, v2);
	l2 = sqrt(sqr(v.x) + sqr(v.y) + sqr(v.z));

	v1.x = xs[0] - xs[1];		/* area 0, 1, P */
	v1.y = ys[0] - ys[1];
	v1.z = zs[0] - zs[1];

	v2.x = xs[0] - x;
	v2.y = ys[0] - y;
	v2.z = zs[0] - z;

	xprod(v, v1, v2);
	l3 = sqrt(sqr(v.x) + sqr(v.y) + sqr(v.z));

	n->x = l1 * norms[0]->x + l2 * norms[1]->x + l3 * norms[2]->x;
	n->y = l1 * norms[0]->y + l2 * norms[1]->y + l3 * norms[2]->y;
	n->z = l1 * norms[0]->z + l2 * norms[1]->z + l3 * norms[2]->z;
}

/*
 * interp4
 *
 *	calculate a normal from the four normals provided by spliting
 * the four sided polygon pg into triangles, finding out which triangle
 * the point hit is in and using interp3 to calculate the normal at the
 * point hit.
 */
interp4(gm, pg, xs, ys, zs, nrms, l, n)
	geometry	*gm;
	polygon		*pg;
	float		*xs, *ys, *zs;
	vector		**nrms, *l, *n;
{
	vector		c, *tns[12];
	float		txs[12], tys[12], tzs[12];
	register int	i, j, nj, crosses;
	register float	u, v, *us, *vs, m;

	/*
	 * calculate the centroid
	 */
	
	c.x = (xs[0] + xs[1] + xs[2] + xs[3]) / 4;
	c.y = (ys[0] + ys[1] + ys[2] + ys[3]) / 4;
	c.z = (zs[0] + zs[1] + zs[2] + zs[3]) / 4;

	/*
	 * set the u, v values
	 */
	if (pg < &gm->faces[gm->npolys[XAXIS]]) {		/* x facing */
		u = l->y; v = l->z;
		us = tys; vs = tzs;
	} else if (pg < &gm->faces[gm->npolys[YAXIS] + gm->npolys[XAXIS]]) {	/* y facing */
		u = l->x; v = l->z;
		us = txs; vs = tzs;
	} else {				/* z facing */
		u = l->x; v = l->y;
		us = txs; vs = tys;
	}

	/*
	 * split the mother into triangles 
	 */
	txs[0] = xs[0]; tys[0] = ys[0]; tzs[0] = zs[0]; tns[0] = nrms[0];
	txs[1] = xs[1]; tys[1] = ys[1]; tzs[1] = zs[1]; tns[1] = nrms[1];

	txs[3] = xs[1]; tys[3] = ys[1]; tzs[3] = zs[1]; tns[3] = nrms[1];
	txs[4] = xs[2]; tys[4] = ys[2]; tzs[4] = zs[2]; tns[4] = nrms[2];

	txs[6] = xs[2]; tys[6] = ys[2]; tzs[6] = zs[2]; tns[6] = nrms[2];
	txs[7] = xs[3]; tys[7] = ys[3]; tzs[7] = zs[3]; tns[7] = nrms[3];

	txs[9] = xs[3]; tys[9] = ys[3]; tzs[9] = zs[3]; tns[9] = nrms[3];
	txs[10] = xs[0]; tys[10] = ys[0]; tzs[10] = zs[0]; tns[10] = nrms[0];

	txs[2] = txs[5] = txs[8] = txs[11] = c.x;
	tys[2] = tys[5] = tys[8] = tys[11] = c.y;
	tzs[2] = tzs[5] = tzs[8] = tzs[11] = c.z;
	tns[2] = tns[5] = tns[8] = tns[11] = &pg->n;

	/*
	 * find which part we have hit
	 */
	for (i = 0; i != 12; i += 3) {
		crosses = 0;
		for (j = i; j != i + 3; j++) {

			nj = (j == i + 2) ? i : j + 1;

			if (vs[j] < v) {
				if (vs[nj] < v)
					continue;
			} else {
				if (vs[nj] > v)
					continue;
			}

			if (us[j] < u) {
				if (us[nj] < u) {	/* crossed on left */
					crosses++;
					continue;
				}
			} else if (us[nj] > u)		/* both to right */
				continue;

			m = (vs[nj] - vs[j]) / (us[nj] - us[j]);
			if ((-(vs[nj] - v) / m + (us[nj] - u)) < 0.0)
				crosses++;
		}

		if (crosses & 1)
			break;
	}

	if (i == 12) {
		*n = pg->n;
	} else
		interp3(l->x, l->y, l->z, &txs[i], &tys[i], &tzs[i], &tns[i], n);
}

/*
 * phong
 *
 *	returns the interpolated normal to a polygon pg.
 */
void
phong(gm, pg, l, n)
	geometry	*gm;
	polygon		*pg;
	vector		*l, *n;
{
	vector	*norms[4];
	float	xs[4], ys[4], zs[4];
	int	i;


	if (pg->nsides == 4) {
		for (i = 0; i != 4; i++) {
			xs[i] = gm->xs[pg->vertnos[i]];
			ys[i] = gm->ys[pg->vertnos[i]];
			zs[i] = gm->zs[pg->vertnos[i]];
			norms[i] = &gm->norms[pg->vertnos[i]];
		}
		interp4(gm, pg, xs, ys, zs, norms, l, n);
	} else if (pg->nsides == 3) {
		for (i = 0; i != 3; i++) {
			xs[i] = gm->xs[pg->vertnos[i]];
			ys[i] = gm->ys[pg->vertnos[i]];
			zs[i] = gm->zs[pg->vertnos[i]];
			norms[i] = &gm->norms[pg->vertnos[i]];
		}
		interp3(l->x, l->y, l->z, xs, ys, zs, norms, n);
	} else {
		n->x = pg->n.x;
		n->y = pg->n.y;
		n->z = pg->n.z;
	}
}
