#include <math.h>
#include <stdio.h>
#include "art.h"
#include "macro.h"
#include "gram.h"

extern attr	*astackp;
extern hlist	*fhlist;

/*
 * conei
 *
 *	returns a list of intersection points for the ray r and cone o.
 *
 */
hlist *
conei(r, o, last)
	register ray	*r;
	register object	*o;
	hlist		**last;
{
	hlist		*hitlist, *hp;
	register float	a, b, c, d, t, t1, t2;
	float		tipval, ax1, ax2, x, y;
	int		face, entry, exit;
	vector		cv;
	ray		nr;

	vadd(cv, r->org, o->bs.cent);
	t = dprod(cv, r->dir);

	if ((dprod(cv, cv) - o->bs.radsqu) > t * t)
		return((hlist *)NULL);

	transray(o, nr, *r);

	a = nr.dir.x * nr.dir.x + nr.dir.y * nr.dir.y - nr.dir.z * nr.dir.z;
	b = nr.org.x * nr.dir.x + nr.org.y * nr.dir.y - nr.dir.z * nr.org.z;
	c = nr.org.x * nr.org.x + nr.org.y * nr.org.y - nr.org.z * nr.org.z;

	d = b * b - a * c;

	if (d < 0.0)
		return((hlist *)NULL);
	
	tipval = o->obj.cne->tipval;

	hitlist = (hlist *)NULL;

	d = sqrt((double)d);

	if (a == 0.0) {
		t1 = (tipval - nr.org.z) / nr.dir.z;
		t2 = (1.0 - nr.org.z) / nr.dir.z;

		if (t1 < TOLERANCE && t2 < TOLERANCE)
			return((hlist *)NULL);

		x = t2 * nr.dir.x + nr.org.x;
		y = t2 * nr.dir.y + nr.org.y;
		if (x * x + y * y > 1.0)
			return((hlist *)NULL);

                entry = exit = FACE;
	} else {
		entry = exit = SIDE;

		t1 = (-b - d) / a;
		t2 = (-b + d) / a;

		ax1 = t1 * nr.dir.z + nr.org.z;
		ax2 = t2 * nr.dir.z + nr.org.z;

		if (ax1 < tipval && ax2 < tipval)
			return((hlist *)NULL);

		if (ax1 > 1.0 && ax2 > 1.0)
			return((hlist *)NULL);

		if (ax1 > 1.0) {
			entry = FACE;
			t1 = (1.0 - nr.org.z) / nr.dir.z;
			x = t1 * nr.dir.x + nr.org.x;
			y = t1 * nr.dir.y + nr.org.y;
			if (x * x + y * y > 1.0)
				return((hlist *)NULL);
		} else if (ax1 < tipval) {
			entry = FACE;
			t1 = (tipval - nr.org.z) / nr.dir.z;
			if (tipval != 0.0) {
				x = t1 * nr.dir.x + nr.org.x;
				y = t1 * nr.dir.y + nr.org.y;
				if (x * x + y * y > tipval * tipval)
					return((hlist *)NULL);
			}
		}

		if (ax2 > 1.0) {
			exit = FACE;
			t2 = (1.0 - nr.org.z) / nr.dir.z;
			x = t2 * nr.dir.x + nr.org.x;
			y = t2 * nr.dir.y + nr.org.y;
			if (x * x + y * y > 1.0)
				return((hlist *)NULL);
		} else if (ax2 < tipval) {
			exit = FACE;
			t2 = (tipval - nr.org.z) / nr.dir.z;
			if (tipval != 0.0) {
				x = t2 * nr.dir.x + nr.org.x;
				y = t2 * nr.dir.y + nr.org.y;
				if (x * x + y * y > tipval * tipval)
					return((hlist *)NULL);
			}
		}
	}

	if (t1 > t2) {
		t = t1;
		t1 = t2;
		t2 = t;
		face = entry;
		entry = exit;
		exit = face;
	}

	if (t1 >= TOLERANCE) {
		fetch(hp);
		hp->type = entry;
		hitlist = hp;
		hp->t = t1;
		hp->obj = o;
		if (t2 >= TOLERANCE) {
			fetch(hp);
			hitlist->nxt = hp;
			hp->type = exit;
			hp->t = t2;
			hp->obj = o;
		} 
		hp->nxt = (hlist *)NULL;
	} else if (t2 >= TOLERANCE) {
		fetch(hp);
		hitlist = hp;
		hp->type = exit;
		hp->t = t2;
		hp->obj = o;
		hp->nxt = (hlist *)NULL;
	}

	*last = hp;

	return(hitlist);
}

/*
 * conen
 *
 *	returns the normal to the cone o
 */
void
conen(n, l, o, type)
	register vector		*n;
	register vector		*l;
	object			*o;
	int			type;
{
	vector			v;

	if (type == SIDE) {
		toobject(o, v, *l);

		v.z = -v.z;

		normalise(v);
	} else {
		v.x = 0.0;
		v.y = 0.0;
		v.z = 1.0;
	}

	if (o->mat != NULL)
		v3x3tmult(*n, v, (*o->mat))
	else
		*n = v;
}

/*
 * conec
 *
 *	set the colour for the cone o
 */
void
conec(o, txt, l, n, pcol, type)
	object  *o;
	texture *txt;
	vector  *l, *n;
	pixel   *pcol;
	int     type;
{
	vector  tmp, tmp2;
	float   u, v, tipval;
	int     w, h, indx;

	tipval = o->obj.cne->tipval;

	if (type == SIDE) {
		toobject(o, tmp2, *l);

		totexture(txt, tmp, tmp2);

		if (fabs(tmp.x) > 1.0)
			tmp.x = (tmp.x < 0.0) ? -1.0 : 1.0;

		v = 1.0 - (tmp.z - tipval) / (1.0 - tipval);
		if (tmp.z != 0.0) {
			u = tmp.x / tmp.z;
			if (fabs(u) > 1.0)
				u = (u < 0.0) ? -1.0 : 1.0;

			u = acos(u) / (2.0 * M_PI);
		} else
			u = 1.0;

		w = ((tmp.y < 0.0) ? 1.0 - u : u) * txt->scaleh;
		h = v * txt->scalew;

		indx = (w % txt->pixw + (h % txt->pixh) * txt->pixw) * 3;

		pcol->r = (unsigned char)txt->map[indx] / 255.0;
		pcol->g = (unsigned char)txt->map[indx + 1] / 255.0;
		pcol->b = (unsigned char)txt->map[indx + 2] / 255.0;
	}
}

/*
 * coneinit
 *
 *	initialise the function pointers and fields for a cone object,
 * returning its pointer.
 */
object *
coneinit(d)
	details *d;
{
	object		*o;
	int		first, faces;
	vector		base, tip, basescale, tipscale, tmp;
	details		*ld;
	tlist		*tl;
	float		grad, rad;
	static cone	stdcone = { 0.0 };

	o = (object *)smalloc(sizeof(object));
	o->type = CONE;
	o->intersects = conei;
	o->normal = conen;

	faces = 0;
	first = 1;

	while (d != (details *)NULL) {
		switch (d->type) {
		case VERTEX:
			tip = d->u.v;
			break;
		case CENTER:
			if (faces == 0)
				base = d->u.v;
			else 
				tip = d->u.v;
			faces++;
			break;
		case RADIUS:
			if (first) {
				basescale.x = basescale.y = d->u.f;
				first = 0;
			} else
				tipscale.x = tipscale.y = d->u.f;
			break;
		case RADII:
			 if (first) {
				basescale = d->u.v;
				first = 0;
			} else
				tipscale = d->u.v;
			break;
		default:
			warning("art: illegal field in cone ignored.\n");
		}
		ld = d;
		d = d->nxt;
		free(ld);
	}

			/* this axis is done in qtransform */
	basescale.z = tipscale.z = 1.0;

	if (faces == 2) {
		o->obj.cne = (cone *)smalloc(sizeof(cone));
		if (tipscale.x > basescale.x) {
			tmp = tip;
			tip = base;
			base = tmp;
			tmp = tipscale;
			tipscale = basescale;
			basescale = tmp;
		}
		o->obj.cne->tipval = tipscale.x / basescale.x;
		grad = (tip.x - base.x) / (basescale.x - tipscale.x);
		tip.x += grad * tipscale.x;
		grad = (tip.y - base.y) / (basescale.x - tipscale.x);
		tip.y += grad * tipscale.x;
		grad = (tip.z - base.z) / (basescale.x - tipscale.x);
		tip.z += grad * tipscale.x;
	} else
		o->obj.cne = &stdcone;

	if (basescale.x > basescale.y)
		rad = basescale.x;
	else
		rad = basescale.y;

	tmp.x = (tip.x + base.x) / 2.0;
	tmp.y = (tip.y + base.y) / 2.0;
	tmp.z = (tip.z + base.z) / 2.0;

	makebsphere(&o->bs, &tmp, sqr(tip.x - tmp.x) + sqr(tip.y - tmp.y) + sqr(tip.z - tmp.z) + rad * rad);

	qtransform(tip, base);

	scale(basescale.x, basescale.y, basescale.z);

	setattributes(o);

	for (tl = o->s.txtlist; tl != (tlist *)NULL; tl = tl->nxt)
		if (tl->txtcol == (void (*)())NULL)
			tl->txtcol = conec;

	return(o);
}
