/*
 * include file for pixel data
 */
#include "vort.h"

/*
 * types and defs for the ray tracer
 */

#define	TRUE	1
#define	FALSE	0

#define	MESLEN	100

/*
 * distance we must be from the last surface we hit before we register
 * another hit.
 */
#define	TOLERANCE	0.0001

/*
 * where we put infinity
 */
#define	HUGE_DIST	10000000.0

typedef float	matrix[4][4];
typedef float	mat3x3[3][3];

typedef struct cmplx {
	float	r;
	float	i;
} complex;

typedef struct v {
	float	x, y, z;
} vector;

typedef struct col {
	float	r, g, b;
} colour;

typedef struct pix {
	float	r, g, b, a;
} pixel;

/*
 * typedefs for texture, bumps, and tile patterns
 */
typedef struct t {
	int		type;
	char		axis;		/* major axis of normal to polygon */
	int		refset;		/* tile reference points set? */
	int		octaves;	/* range for noise function */
	mat3x3		mat;		/* rotation matrix to texture space */
	vector		trans, scales;	/* translation and scaling values */
	colour		c;
	colour		a;
	colour		blendcolour;
	float		var[3];
	vector		vref1, vref2;
	float		scalew, scaleh;
	char		*map;
	unsigned short	pixw, pixh;
} texture;

typedef struct tl {
	void		(*txtcol)();	/* texturing function */
	texture		txt;		/* texture */
	struct tl	*nxt;
} tlist;
	

/*
 * general structure for a surface
 */
typedef struct surf {
	colour	c, a;			/* surface and ambient light colour */
	float	kd, ks;			/* diffuse, specular */
	int	ksexp;			/* specular exponent */
	float	refl,			/* reflection */
		ri,			/* refractive index */
		absorb,			/* % of light absorbed in unit length */
		trans;			/* transparency */
	tlist	*txtlist;		/* texturing info */
} surface;

/*
 * attribute stack structure
 */
typedef struct at {
	matrix	m;		/* current transformation matrix */
	int	mcopy;		/* do we need to copy the matrix? */
	int	options;	/* shading options */
	float	maxscale;	/* maximum scale applied to an object */
	vector	scales,		/* scaling factors */
		trans;		/* translation factors */
	surface	s;		/* surface properties */
} attr;

/*
 * nodes for storing hit lists
 */

#define	ENTRY	1
#define	EXIT	2

typedef struct hl {
	float		t;
	int		type;
	struct o	*obj;
	struct hl	*nxt;
} hlist;

/*
 * defs for lights
 */
#define	POINT		0
#define	DIRECTIONAL	1
#define	DISTANT		2

typedef struct l {
	int		type;
	vector		org;
	vector		dir;
	colour		c;
	float		rad;
	float		cosang;
	int		rays;
	struct l	*nxt;
} light;

/*
 * the ray structure
 */
typedef struct r {
	vector	org, dir;
} ray;

/*
 * major axis flags
 */
#define	XAXIS	0
#define	YAXIS	1
#define	ZAXIS	2

#define	DIMS	3

/*
 * hit types
 */

/*
 * for boxes
 */
#define	XFACE	0
#define	YFACE	1
#define	ZFACE	2
#define	NOFACE	3

/*
 * for quadrics
 */
#define	FACE	1
#define	SIDE	2

/*
 * bounding shapes
 */
typedef struct bs {
	vector	cent;
	float	radsqu;
} bsphere;

typedef struct b {
	vector	topfr, botbl;
} bbox;

/*
 * object model definitions
 */

typedef struct s {
	float	radsqu;
} sphere;

typedef struct rg {
	vector	n;
	float	intrad, intradsqu;
} ring;

typedef struct c {
	float	tipval;
} cone;

typedef struct to {
	int	incsg;
	float	cnst;
} torus;

typedef struct sq {
	int	incsg, ord;
} superquadric;

/*
 * typedefs for polygons and polygon models
 */
#define	MAXVERTS	256		/* maximum number of vertices */

#define	BACKFACING	0x1		/* backfacing bit */
#define	PHONGSHADING	0x2		/* phong shading bit */

typedef unsigned short	vertno;

typedef struct fct {
	unsigned char	nsides;		/* number of sides */
	float		cnst;		/* const for the faces plane */
	vector		n;		/* normal */
	vertno		*vertnos;	/* vertex numbers */
	float		minu, maxu,	/* min max in u value */
			minv, maxv;	/* min max in v value */
} polygon;

/*
 * types for organisation of polygons in geometry
 */
#define	XY		0
#define	YX		1
#define	XZ		2
#define	ZX		3
#define	YZ		4
#define	ZY		5

typedef struct gty {
	bbox		bb;		/* bounding box */
	int		options;	/* shading options */
	vector		*norms;		/* normals at each vertex */
	float		*xs, *ys, *zs;	/* x, y, and z values for each vertex */
	polygon		*faces;		/* polygon array */
	int		types[DIMS];	/* organisation types in each axis */ 
	int		ntris[DIMS];	/* number of triangles in each axis */
	int		npolys[DIMS];	/* number of polygons in each axis */
	polygon		*midts[DIMS];	/* middle of triangle array */
	float		midtval[DIMS];	/* smallest minv in top half of tris */
	polygon		*midps[DIMS];	/* middle of polygon array */
	float		midpval[DIMS];	/* smallest minv in top half of polys */
} geometry;


/*
 * algebraic equation term def and node structure
 */
typedef struct trm {
	double		coef;
	int		type;
	int		xp, yp, zp;
	double		*xcoefs, *ycoefs, *zcoefs;
	struct trm	*nxt;
} term;

typedef struct pwl {
	int		pw;
	double		*coefs;
	struct pwl	*nxt;
} pwlist;

typedef struct eq {
	int	incsg;
	int     order;
	int	maxxp, maxyp, maxzp;
	pwlist	*xpws, *ypws, *zpws;
	term    *trmlist,
		*dxlist, *dylist, *dzlist;	/* terms for the partial derivatives */
} eqn;


/*
 * csg node structure and defs
 */
#define	UNION		1
#define	INTERSECT	2
#define	SUBTRACT	3

typedef struct cs {
	int		op;
	int		rightb, leftb;
	struct o	*left, *right;
} csg;

/*
 * the composite object
 */
typedef struct co {
	struct o	*oblist;
} composite;

/*
 * details to go in objects
 */
typedef struct det {
	int		type;
	union {
		vector		v;
		float		f;
		term		*t;
		char		*s;
		struct o	*obj;
	} u;
	struct det	*nxt;
} details;

/*
 * general object structure and defs
 */
typedef struct o {
	int		type;
	surface		s;
	vector		trans,
			scales;
	mat3x3		*mat;
	bsphere		bs;
	hlist		*(*intersects)();
	void		(*normal)(),
			(*shader)();
	union {
		csg		*csgt;
		eqn		*alg;
		cone		*cne;
		ring		*rng;
		torus		*trs;
		sphere		*sph;
		polygon		*ply;
		geometry	*geo;
		superquadric	*spq;
		composite	*cmp;
	} obj;
	struct o	*nxt;
} object;

#ifndef M_PI
#define M_PI    3.14159265358979323846
#endif

#ifdef MSC

extern char	*smalloc();

#define yywrap()	1
#endif
