/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * rlehdr.c - Print header from an RLE file.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Thu Jan 22 1987
 * Copyright (c) 1987, University of Utah
 */

#include <stdio.h>
#include <svfb_global.h>

/*****************************************************************
 * TAG( main )
 * 
 * Read and print in human readable form the header of an RLE file.
 *
 * Usage:
 *	rlehdr [-m] [file]
 * Inputs:
 *	-m:		Dump color map contents.
 *	-d:		Dump entire file contents.
 * 	file:		RLE file to interpret.
 * Outputs:
 * 	Prints contents of file header.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

main( argc, argv )
char **argv;
{
    char * fname = NULL;
    FILE * rlef = stdin;
    int mflag = 0, dbg_flag = 0;

    if ( scanargs( argc, argv, "% m%- d%- rlefile%s",
		   &mflag, &dbg_flag, &fname ) == 0 )
	exit( 1 );

    if ( fname != NULL && (rlef = fopen( fname, "r" )) == NULL )
    {
	fprintf( stderr, "%s: ", argv[0] );
	perror( fname );
	exit( 1 );
    }
    else if ( fname == NULL )
	fname = "stdin";

    sv_globals.svfb_fd = rlef;
    rle_get_setup_ok( &sv_globals, "rlehdr", fname );

    print_hdr( fname, &sv_globals );
    if ( mflag )
	print_map( &sv_globals );
    if ( dbg_flag )
	print_codes( &sv_globals );

    exit( 0 );			/* success */
}


/*****************************************************************
 * TAG( print_hdr )
 * 
 * Print the RLE header information given.
 *
 * Inputs:
 * 	fname:		Name of file header was read from.
 *	globals:	Header information.
 * Outputs:
 * 	Prints info on stdout.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
print_hdr( fname, globals )
char *fname;
struct sv_globals *globals;
{
    register int i;

    printf( "RLE header information for %s:\n", fname );
    printf( "Originally positioned at (%d, %d), size (%d, %d)\n",
	    globals->sv_xmin, globals->sv_ymin,
	    globals->sv_xmax - globals->sv_xmin + 1,
	    globals->sv_ymax - globals->sv_ymin + 1 );
    printf( "Saved %d color channels%s\n", globals->sv_ncolors,
	    globals->sv_alpha ? " plus Alpha channel" : "" );
    if ( globals->sv_background == 0 )
	printf( "No background information was saved\n" );
    else
    {
	if ( globals->sv_background == 1 )
	    printf( "Saved in overlay mode with original background color" );
	else
	    printf( "Screen will be cleared to background color" );
	for ( i = 0; i < globals->sv_ncolors; i++ )
	    printf( " %d", globals->sv_bg_color[i] );
	putchar( '\n' );
    }

    if ( globals->sv_ncmap > 0 )
	printf( "%d channels of color map %d entries long were saved.\n",
		globals->sv_ncmap, 1 << globals->sv_cmaplen );

    if ( globals->sv_comments != NULL && *globals->sv_comments != NULL )
    {
	printf( "Comments:\n" );
	for ( i = 0; globals->sv_comments[i] != NULL; i++ )
	    printf( "%s\n", globals->sv_comments[i] );
    }
}

/*****************************************************************
 * TAG( print_map )
 * 
 * Print the color map from a globals structure.
 * Inputs:
 * 	globals:	Sv_globals structure containing color map.
 * Outputs:
 * 	Prints color map (if present) on stdout.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
print_map( globals )
struct sv_globals *globals;
{
    register int i, j;
    int c, maplen, ncmap;
    rle_map * cmap;

    if ( globals->sv_ncmap == 0 )
	return;

    maplen = (1 << globals->sv_cmaplen);
    ncmap = globals->sv_ncmap;
    cmap = globals->sv_cmap;

    printf( "Color map contents, values are 16-bit(8-bit):\n" );
    for ( i = 0; i < maplen; i++ )
    {
	printf( "%3d:\t", i );
	for ( j = 0, c = 0; j < ncmap; j++, c += maplen )
	    printf( "%5d(%3d)%c", cmap[i+c], cmap[i+c] >> 8,
		    j == ncmap - 1 ? '\n' : '\t' );

    }
}


/*****************************************************************
 * TAG( print_codes )
 * 
 * Print the RLE opcodes in an RLE file.
 * Inputs:
 * 	globals:		Globals for RLE file (already open).
 * Outputs:
 * 	Prints file contents on stderr.
 * Assumptions:
 * 	
 * Algorithm:
 *	[None]
 */
print_codes( globals )
struct sv_globals *globals;
{
    char ** scans;

    rle_row_alloc( globals, &scans );
    rle_debug(1);
    while ( !feof( globals->svfb_fd ) && 
	    !globals->sv_private.get.is_eof )
	rle_getrow( globals, scans );
}
