/* 
 * rlebox.c - Find bounding box for an RLE image
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Wed Feb 11 1987
 * Copyright (c) 1987, University of Utah
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

/*****************************************************************
 * TAG( main )
 * 
 * Find a bounding box for an RLE image.
 *
 * Usage:
 *	rlebox [-v] [-c] [rlefile]
 * Inputs:
 *	-v:		Verbose mode - identifies numbers with text.
 *			Otherwise, just prints numbers.
 * 	-c:		Outputs corners of box in an order useful for use
 *			as arguments to the crop program:
 *			crop `rlebox -c foo.rle` foo.rle
 *	rlefile:	The input file.
 * Outputs:
 * 	Prints the bounding box for the rle file.  That is, it finds the
 *	minimum and maximum values of x and y for which there is some
 *	non-background data.
 * Assumptions:
 * 	
 * Algorithm:
 * 	Read the image file and find the smallest and largest X and Y
 *	coordinates of real image data.  Use raw interface for speed.
 */
main( argc, argv )
char **argv;
{
    char * rlefname = NULL;
    FILE * rlefile = stdin;
    int vflag = 0, cflag = 0, * nrawp, margin = 0;
    rle_op ** raw;
    register rle_op ** rawp;
    register int i;
    int xmin, xmax, ymin, ymax, y;

    if ( scanargs( argc, argv, "% v%- c%- m%-margin!d rlefile%s",
		   &vflag, &cflag, &margin, &margin,
		   &rlefname ) == 0 )
	exit( 1 );

    if ( rlefname != NULL && (rlefile = fopen( rlefname, "r" )) == NULL )
    {
	fprintf( stderr, "%s: ", argv[0] );
	perror( rlefname );
	exit( 1 );
    }
    else if ( rlefname == NULL )
	rlefname = "stdin";

    sv_globals.svfb_fd = rlefile;

    rle_get_setup( &sv_globals, argv[0], rlefname );

    rle_raw_alloc( &sv_globals, &raw, &nrawp );
    rawp = raw;

    ymax = -1;			/* smaller than it can ever be */
    ymin = 32768;		/* larger than it can ever be */
    xmax = -1;
    xmin = 32768;

    while ( ( y = rle_getraw( &sv_globals, rawp, nrawp ) ) != 32768 )
    {
	if ( y < ymin )		/* found ymin yet? */
	{
	    /* Only count it if there is really some data */
	    for ( i = -sv_globals.sv_alpha; i < sv_globals.sv_ncolors; i++ )
		if ( nrawp[i] > 0 )
		    ymin = y;
	}
	if ( y > ymax )		/* update ymax? */
	{
	    /* Only count it if there is really some data */
	    for ( i = -sv_globals.sv_alpha; i < sv_globals.sv_ncolors; i++ )
		if ( nrawp[i] > 0 )
		    ymax = y;
	}

	/* Now do xmin and xmax */
	    for ( i = -sv_globals.sv_alpha; i < sv_globals.sv_ncolors; i++ )
		if ( nrawp[i] > 0 )
		{
		    if ( rawp[i][0].xloc < xmin )
			xmin = rawp[i][0].xloc;
		    if ( xmax < rawp[i][nrawp[i]-1].xloc +
			 rawp[i][nrawp[i]-1].length - 1 )
			xmax =  rawp[i][nrawp[i]-1].xloc +
			    rawp[i][nrawp[i]-1].length - 1;
		}
    }

    /* If no data, arbitrarily use (0,0)x(0,0) */
    if ( xmax < xmin )
    {
	fprintf( stderr, "rlebox: %s is an empty image\n", rlefname );
	xmax = xmin = ymax = ymin = 0;
    }

    /* If margin, enlarge bounds.  Don't let them go negative */
    if ( margin )
    {
	xmax += margin;
	ymax += margin;
	xmin -= margin;
	if ( xmin < 0 )
	    xmin = 0;
	ymin -= margin;
	if ( ymin < 0 )
	    ymin = 0;
    }

    if ( cflag )
	printf( vflag ? "xmin=%d ymin=%d xmax=%d ymax=%d\n" : "%d %d %d %d\n",
		xmin, ymin, xmax, ymax );
    else
	printf( vflag ? "xmin=%d xmax=%d ymin=%d ymax=%d\n" : "%d %d %d %d\n",
		xmin, xmax, ymin, ymax );

    exit( 0 );
}

