/* 
 * mergechan.c - Merge channels from multiple RLE images
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Mon Nov  9 1987
 * Copyright (c) 1987, University of Utah
 *
 * Warning: this code does not intelligently deal with color maps!
 *
 */

#include <stdio.h>
#include "svfb_global.h"
#include "rle_getraw.h"

#define CHECK_MALLOC( ptr, msg ) \
  { if (! (ptr)) {fprintf( stderr, msg ); exit(-2);}}
#define MIN(i,j)   ( (i) < (j) ? (i) : (j) )

#define RLE_END 32768		/* This should be in svfb_global.h */

main( argc, argv )
int argc;
char ** argv;
{
    int nfiles;
    struct sv_globals * in_globs;
    struct sv_globals out_globals;
    int alpha_flag = 0;
    rle_op *** in_rows;		/* A Carl Sagan pointer. */
    int ** in_counts;		/* nraw */
    rle_op ** out_rows;
    int * out_count;
    int * skips;
    char **filenames;
    int minskip, y;
    register int i;

    if (! scanargs( argc, argv, "% a%- files%*s",
		    &alpha_flag, &nfiles, &filenames ))
	exit( -1 );

    if (alpha_flag) alpha_flag = 1; /* So indexing's right. */

    in_globs = (struct sv_globals *) malloc( sizeof( struct sv_globals )
					    * nfiles );
    
    in_rows = (rle_op ***) malloc( sizeof( rle_op ** ) * nfiles );
    CHECK_MALLOC( in_rows, "mergechan: malloc error\n" );
    in_counts = (int **) malloc( sizeof( int * ) * nfiles );
    CHECK_MALLOC( in_counts, "mergechan: malloc error\n" );
    skips = (int *) malloc( sizeof( int ) * nfiles );
    CHECK_MALLOC( skips, "mergechan: malloc error\n" );

    /* Open all the files, and check consistancy */

    for (i = 0; i < nfiles; i++)
    {
	if (! (in_globs[i].svfb_fd = fopen( filenames[i], "r" )))
	{
	    perror( filenames[i] );
	    exit( -2 );
	}

	rle_get_setup_ok( &(in_globs[i]), "mergechan", filenames[i] );

	/* Check that the channel's really there */
	if (((in_globs[i].sv_ncolors-1) < (i-alpha_flag)) ||
	    (! SV_BIT( in_globs[i], i-alpha_flag )))
	{
	    fprintf(stderr, "mergechan: channel %d not in file %s\n",
		    i, filenames[i] );
	    exit( -2 );
	}

	/* Check to make sure all images have the same size */
	if (i > 0)
	{
	    if (! ((in_globs[0].sv_xmin == in_globs[i].sv_xmin) &&
		   (in_globs[0].sv_xmax == in_globs[i].sv_xmax) &&
		   (in_globs[0].sv_ymin == in_globs[i].sv_ymin) &&
		   (in_globs[0].sv_ymax == in_globs[i].sv_ymax)) )
	    {
		fprintf(stderr,
		      "mergechan: image %s is not the same size as image %s\n",
			filenames[i], filenames[0] );
		exit( -2 );
	    }
	}

	if( rle_raw_alloc( &(in_globs[i]), &(in_rows[i]), &(in_counts[i]) ))
	{
	    fprintf( stderr, "mergechan: can't allocate raw storage\n" );
	    exit(-2);
	}
    }

    /* Setup output stuff */

    out_globals = in_globs[0];

    out_globals.sv_ncolors = nfiles - alpha_flag;
    out_globals.sv_alpha = alpha_flag;
    out_globals.svfb_fd = stdout;

    /* Enable all output channels. */
    for (i = -alpha_flag; i < out_globals.sv_ncolors; i++)
	SV_SET_BIT( out_globals, i );

    sv_setup( RUN_DISPATCH, &out_globals );

    out_rows = (rle_op **) malloc( sizeof( rle_op * ) * nfiles );
    CHECK_MALLOC( out_rows, "mergechan: malloc failed\n");
    out_count = (int *) malloc( sizeof( int ) * nfiles );
    CHECK_MALLOC( out_count, "mergechan: malloc failed\n");
				  

    if (alpha_flag)		/* So indexing's right (alpha == -1) */
    {
	in_globs++;
	in_rows++;
	in_counts++;

	out_rows++;
	out_count++;
	skips++;
    }

    /* Initialize counters */
    for (i = -alpha_flag; i < out_globals.sv_ncolors; i++)
    {
	skips[i] = 0;
	out_rows[i] = in_rows[i][i];
    }
    y = out_globals.sv_ymin - 1;	/* -1 'cuz we haven't read data yet */

    /*
     * Do the actual work.  Since rle_getraw may "skip" several lines
     * ahead, we need to keep track of the Y position of each channel
     * independently with skips[].  The output moves ahead by the
     * minimum of these skip values (minskip).
     */
    while (1)			/* Stops at EOF on all files */
    {
	for (i = -alpha_flag; i < out_globals.sv_ncolors; i++)
	{
	    if (! skips[i])
	    {
		skips[i] = rle_getraw( &(in_globs[i]),
				       in_rows[i], in_counts[i] );
		if (skips[i] != RLE_END)
		    skips[i] -= y;	/* Store delta to next data */
	    }

	    /* Find smallest skip distance until a channel has data again */
	    if (i == -alpha_flag)
		minskip = skips[i];
	    else
		minskip = MIN( skips[i], minskip );
	}

	if (minskip == RLE_END)
	    break;		/* Hit the end of all input files */

	if (minskip > 1)
	    sv_skiprow( &out_globals, minskip-1 );

	y += minskip;

	for (i = -alpha_flag; i < out_globals.sv_ncolors; i++)
	{
	    if (skips[i] != RLE_END) skips[i] -= minskip-1;

	    if (skips[i] == 1)	/* Has data to go out */
	    {
		out_count[i] = in_counts[i][i];
	    }
	    else
	    {
		out_count[i] = 0;
	    }
	}
	
	sv_putraw( out_rows, out_count, &out_globals );

	for (i = -alpha_flag; i < out_globals.sv_ncolors; i++)
	{
	    if (skips[i] == 1)	/* Data is written, so free the raws */
	    {
		rle_freeraw( &(in_globs[i]), &(in_rows[i]), in_counts[i] );
		skips[i] = 0;	/* Need to get more data */
	    }
	}
    }

    sv_puteof( &out_globals );
    exit( 0 );
}

