/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 *  Modified at BRL 16-May-88 by Mike Muuss to avoid Alliant STDC desire
 *  to have all "void" functions so declared.
 */
/* 
 * rle_raw_alc.c - Allocate buffers for rle_getraw/sv_putraw.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Fri Nov 14 1986
 * Copyright (c) 1986, Spencer W. Thomas
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

/*****************************************************************
 * TAG( rle_raw_alloc )
 * 
 * Allocate buffer space for use by rle_getraw and sv_putraw.
 * Inputs:
 * 	globals:	Globals structure for RLE file to be read or
 *			written.
 * Outputs:
 *	scanp:		Pointer to pointer to created opcode buffer.
 * 	nrawp:		Pointer to pointer to created length buffer.
 *			These pointers are adjusted for the alpha channel,
 *			if present.
 *	Returns 0 for success, -1 if malloc failed.
 * Assumptions:
 * 	Since buffers are built to have as many slots as there are pixels
 *	in the input scanline, it is assumed that no input scanline will
 *	have more data elements than this.
 * Algorithm:
 *	Count number of channels actually used (check bitmap).
 * 	Allocate nchan*rowlength elements, allocate a buffer
 *	to hold (ncolors+alpha) pointers.
 *	Also allocate a buffer of ncolors+alpha
 *	integers for the length buffer.
 */
rle_raw_alloc( globals, scanp, nrawp )
struct sv_globals *globals;
rle_op ***scanp;
int **nrawp;
{
    rle_op ** scanbuf, * opbuf;
    int rowlen, nchan = 0, i, ncol;
    char * malloc();

    rowlen = globals->sv_xmax - globals->sv_xmin + 1;
    if ( globals->sv_alpha && SV_BIT( *globals, SV_ALPHA ) )
	nchan++;
    for ( i = 0; i < globals->sv_ncolors; i++ )
	if ( SV_BIT( *globals, i ) )
	     nchan++;

    ncol = globals->sv_ncolors + globals->sv_alpha;

    if ( (scanbuf = (rle_op **) malloc( ncol * sizeof(rle_op *) )) == 0 )
	return -1;

    if ( (opbuf = (rle_op *)malloc( nchan * rowlen * sizeof(rle_op) )) == 0 )
    {
	free( scanbuf );
	return -1;
    }

    if ( (*nrawp = (int *)malloc( ncol * sizeof(int) )) == 0 )
    {
	free( scanbuf );
	free( opbuf );
	return -1;
    }

    if ( globals->sv_alpha )
    {
	scanbuf++;
	(*nrawp)++;
    }

    for ( i = -globals->sv_alpha; i < globals->sv_ncolors; i++ )
	if ( SV_BIT( *globals, i ) )
	{
	    scanbuf[i] = opbuf;
	    opbuf += rowlen;
	}
	else
	    scanbuf[i] = 0;
    *scanp = scanbuf;

    return 0;
}


/*****************************************************************
 * TAG( rle_raw_free )
 * 
 * Free storage allocated by rle_raw_alloc().
 * Inputs:
 * 	globals:	Globals structure as above.
 *	scanp:		Pointer to scanbuf above.
 *	nrawp:		Pointer to length buffer.
 * Outputs:
 * 	Frees storage referenced by scanp and nrawp.
 * Assumptions:
 * 	Storage was allocated by rle_raw_alloc, or by use of same
 *	algorithm, at least.
 * Algorithm:
 * 	free scanp[0], scanp, and nrawp.
 */
void
rle_raw_free( globals, scanp, nrawp )
struct sv_globals *globals;
rle_op **scanp;
int *nrawp ;
{
    int i;

    if ( globals->sv_alpha )
    {
	scanp--;
	nrawp--;
    }
    for ( i = 0; i < globals->sv_ncolors + globals->sv_alpha; i++ )
	if ( scanp[i] != 0 )
	{
	    free( (char *)scanp[i] );
	    break;
	}
    free( (char *)scanp );
    free( (char *)nrawp );
}
