/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * getfb.c - Read RLE files onto apollo screens.
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Fri Jun 27 1986
 * Copyright (c) 1986 John W. Peterson
 *
 * Some code lifted from an old getX by Spencer Thomas and showbw
 * by Jim Schimpf.
 *
 * Flags are:
 *   -l		Linear map
 *   -g gam	Gammap map, use gamma of gam (floating point number)
 *   -b		Use full 24 bit mode (DN550, DN560, DN660 only)
 *   -i		Flip white on black/black on white for grayscale display.
 *
 * WARNING: This version ignores the color map in window mode.
 */

#include <stdio.h>
#include <math.h>
#include "svfb_global.h"

#include "/sys/ins/base.ins.c"
#include "/sys/ins/gpr.ins.c"
#include "/sys/ins/error.ins.c"
#include "/sys/ins/pad.ins.c"

#define RASTERSIZE 1023
#define COLMAP_OFFSET 16

#define sysdebug if ((status.all)!=0) error_$print(status)

/* Scanline storage & RLE row pointers */
rle_pixel **rows;
int dm16[16][16];
int mod51[256], div51[256];
int gammamap[256];
float gam = 2.0;		/* Default gamma */
int inverse_flag = false;	/* If true, swap bits on B&W displays */
int gray_flag = false;		/* True if making BW image on color screen */
int borrow_mode = false;	/* DN560/660 imaging mode (borrow screen)) */
int 
    linear_flag = false,	/* Color map flags */
    gam_flag = true;
gpr_$pixel_value_t dest_pixels[RASTERSIZE];  /* avoid bludging stack */
status_$t status;

/*
 * Basic magic square for dithering
 */
int magic[4][4] =  {
 	 0, 14,  3, 13,
	11,  5,  8,  6,
	12,  2, 15,  1,
	 7,  9,  4, 10
};

int dm4[4][4] = {
		  0, 43,  9, 40,
		 34, 15, 25, 18,
		 37,  6, 46,  3,
		 21, 28, 12, 31 };

/*  S. Kitaoka Grey table */

int dmgray[8][8] = {   0 ,128 , 32 ,160 ,  8 ,136 , 40 ,168 ,
                 192 , 64 ,224 , 96 ,200 , 72 ,232 ,104 ,
                  48 ,176 , 16 ,144 , 56 ,184 , 24 ,152 ,
                 240 ,112 ,208 , 80 ,248 ,120 ,216 , 88 ,
                  12 ,140 , 44 ,172 ,  4 ,132 , 36 ,164 ,
                 204 , 76 ,236 ,108 ,196 , 68 ,228 ,100 ,
                  60 ,188 , 28 ,156 , 52 ,220 , 20 ,148 ,
                 252 ,124 ,220 , 92 ,244 ,116 ,212 ,84   } ;

main(argc, argv)
int argc;
char **argv;
{
    int i;
    short len;
    int row;
    unsigned n;
    char * fname = NULL;
    short namelen, tok_count;
    gpr_$display_config_t config;
    int bw_flag = false;
    char gstring[20];

    set_sbrk_size( 1000000 );

    if (! scanargs( argc, argv,
		   "% b%- l%- i%- w%- g%-gamma!f file%s ",
		   &borrow_mode, &linear_flag, &inverse_flag,
		   &gray_flag, &gam_flag, &gam, &fname ))
	exit( 1 );

    if ( fname )
    {
	if ((sv_globals.svfb_fd = fopen(fname, "r")) == NULL)
	{
	    fprintf(stderr, "getap: ");
	    perror(fname);
	    exit(1);
	}
    }
    else
    {
	sv_globals.svfb_fd = stdin;
    }

    if (gray_flag) fprintf(stderr, "getap: gray mode not implemented\n");

    switch( (int) rle_get_setup( &sv_globals ) )
    {
    case 0:
	break;			/* successful open */
    case -1:			/* not an RLE file */
    case -4:			/* or header is short */
	fprintf( stderr, "getap: %s is not an RLE file\n", fname );
	exit(-4);
	break;
    case -2:
	fprintf( stderr, "getap: malloc failed\n" );
	exit(-2);
	break;
    case -3:
	fprintf( stderr, "getap: input file is empty\n" );
	exit(-3);
	break;
    }

/*    SV_CLR_BIT( sv_globals, SV_ALPHA );	/* No alpha channel */

    rle_row_alloc( &sv_globals, &rows );

    if (borrow_mode)
    {
	borrow_setup();
	borrow_color_map(); 
	if (sv_globals.sv_ymax > 511)
	    sv_globals.sv_ymax = 511;
    }
    else
    {
	setup_scr(sv_globals.sv_xmax, sv_globals.sv_ymax);
	window_color_map();
	if (sv_globals.sv_ymax > RASTERSIZE)
	    sv_globals.sv_ymax = RASTERSIZE;
    }

    gpr_$inq_config( config, status );
    if ((config == gpr_$bw_800x1024) ||
	(config == gpr_$bw_1024x800) ||
	(config == gpr_$bw_1280x1024)) bw_flag = true;

    for (i = sv_globals.sv_ymin; i < sv_globals.sv_ymax; i++)
    {
	rle_getrow( &sv_globals, rows );
	if (borrow_mode)
	    put_line_borrow(i);
	else
	{
	    if (bw_flag)
		put_linebw(i);
	    else
		put_line8(i);
	}
    }
    sleep(100000);		/* Real sleaze - should wait for mouse click */
}

/*
 * Initialize the GPR package to imaging mode
 */
borrow_setup()
{
    gpr_$offset_t size;
    gpr_$bitmap_desc_t main_bitmap;      /* Master bitmap */

    size.x_size = 1024;
    size.y_size = 1024;			/* GPR will shrink to fit */

    gpr_$init(gpr_$borrow, 1, size, 7, main_bitmap, status );
    sysdebug;

    gpr_$set_imaging_format( gpr_$imaging_512x512x24, status );
    sysdebug;
    if (status.all != status_$ok)
    {
	fprintf(stderr, "getap: Wrong hardware?\n");
	exit(-1);
    }
}

/* Set up the color map (imaging mode) */
borrow_color_map()
{
    unsigned int value;
    unsigned int intgam;
    int i, map_offset;
    gpr_$color_vector_t colmap;
    
    if ( sv_globals.sv_ncmap )	/* Must read color map */
    {
	map_offset = 1 << sv_globals.sv_cmaplen;

	/* This needs more thought; handle ncmap = 1, etc. */
	if (sv_globals.sv_ncmap == 3)

	for (i=0; i < 256; i++)
	{
            /* This is bogus, we need to index sv_cmap!... */
	    
	    value = (((sv_globals.sv_cmap[i]>> 8) & 0xFF) << 16);
	    value |= (((sv_globals.sv_cmap[i+map_offset] >> 8) & 0xFF) << 8);
	    value |= ((sv_globals.sv_cmap[i+2*map_offset] >> 8) & 0xFF);
	    colmap[i] = value;
	}
    }
    else
    {	  
	if (gam_flag)
	{
	    for (i = 0; i < 256; i++)
	    {
		intgam = (int)(0.5 + 255 * pow( i / 255.0, 1.0/gam ));
		value = intgam;
		value |= (intgam << 8);
		value |= (intgam << 16);
		colmap[i] = value;
	    }
	}
	else
	{			/* Default is linear map */
	    for (i=0; i < 256; i++)
	    {
		value = (i << 16);
		value |= (i << 8); 
		value |= i;
		colmap[i] = value;
	    }
	}
    }
	    
	  
    gpr_$set_color_map( (int) 0, 256, colmap, status );
    sysdebug;
}

/*
 * Initialize the 8 bit color map.  Use gamma corrected map.
 */
window_color_map()
{
    int i, j, k, l, planes;
    unsigned int value;
    float magicfact;
    status_$t status;
    static gpr_$color_vector_t colmap;

    if (sv_globals.sv_ncmap)
	fprintf(stderr, "getap: Warning: color map in file ignored.\n");

    /* 
     * Set up the color map entries.  We don't yet know the location
     * in the map at which each will reside, so init it to 0.
     */
    for(i = 0; i < 216; i++) {
	colmap[i] = ((i%6) * 51) << 16;
	colmap[i] |= (((i/6)%6) * 51) << 8;
	colmap[i] |= (((i/36)%6) * 51);
    }

    /* Offset from DM window colors */
    gpr_$acquire_display( status );
    gpr_$set_color_map( (int) COLMAP_OFFSET, 255 - COLMAP_OFFSET,
		       colmap, status );
    gpr_$release_display( status );

    /* Compute a gamma correction map and tables */
    for ( i = 0; i < 256; i++ )
    {
	if (linear_flag)
	    gammamap[i] = i;
	else
	    gammamap[i] = (int)(0.5 + 255 * pow( i / 255.0, 1.0/gam ));
	mod51[i] = i % 51;
	div51[i] = i / 51;
    }

    /*
     * Expand 4x4 dither pattern to 16x16.  4x4 leaves obvious patterning,
     * and doesn't give us full intensity range (only 17 sublevels,
     * we want at least 51).  Note that 8x8 would be sufficient, but
     * 16x16 is easier to build.
     * 
     * magicfact is 49/16 so that we get numbers in the matrix from 0 to
     * 49: mod 51 gives numbers in 0 to 50, we want some chance that the
     * number is greater than a matrix entry, so the highest matrix
     * entry must be 49.
     */
    magicfact = 49. / 16.;
    for ( i = 0; i < 4; i++ )
	for ( j = 0; j < 4; j++ )
	    for ( k = 0; k < 4; k++ )
		for ( l = 0; l < 4; l++ )
		    dm16[4*k+i][4*l+j] = (int)(0.5 + magic[i][j]
					       * magicfact
					       + (magic[k][l] / 16.)
					       * magicfact);
}

/* All the fun stuff to set up an apollo window */
setup_scr(hsize,vsize)
int hsize,vsize;
{
    gpr_$offset_t size;
    gpr_$color_vector_t colmap;
    gpr_$attribute_desc_t hidden_desc;
    unsigned int value;
    int i;
    name_$pname_t output_file;
    short len;
    gpr_$window_t dest_box;
    pad_$window_desc_t window_shape;
    static gpr_$bitmap_desc_t main_bitmap;
    short fontid;
    static stream_$id_t out_stream;

    /* Open a separate window to display the results */

    window_shape.top = 0;
    window_shape.left = 0;
    window_shape.width = (short)hsize + 5;
    window_shape.height = (short)vsize + 5;

    pad_$create_window( "", (short) 0, pad_$transcript, (short) 1,
                        window_shape, out_stream, status );

    pad_$set_auto_close( out_stream, (short) 1, true, status );

    size.x_size = 1024;  /* GPR will shrink to fit */
    size.y_size = 1024;

    /* Initialize the graphics primitives package */

    gpr_$init(gpr_$direct,
              out_stream,               /* "unit" */
              size,                     /* size of the initial bitmap */
              7,                        /* identifier of the heightest
                                           numbered bitmap plane */
              main_bitmap,              /* resulting bitmap descriptor */
              status);

    gpr_$set_auto_refresh( true, status );  /* all yours, DM. */

    gpr_$acquire_display( status ) ;

    /* set the new bitmap descriptor to be the 'current bitmap' */
    gpr_$set_bitmap(main_bitmap,status);
    gpr_$clear( gpr_$black, status );

    gpr_$release_display( status );
}

/* Write a scanline to the frame buffer using full 24 bit mode */
put_line_borrow(y)
int y;
{
    int i;
    gpr_$window_t dest_box;
    unsigned char *redptr, *grnptr, *bluptr;


    dest_box.window_base.x_coord = 0;
    dest_box.window_base.y_coord = 511 - y;
    dest_box.window_size.x_size = 511;
    dest_box.window_size.y_size = 1;

    redptr = rows[SV_RED];
    if (sv_globals.sv_ncolors == 3)
    { 
        grnptr = rows[SV_GREEN];
        bluptr = rows[SV_BLUE];
    }
    else	/* If only one channel, get all from same one  */
    {
        grnptr = rows[0];
        bluptr = rows[0];
    }
    for (i = sv_globals.sv_xmin; i <= sv_globals.sv_xmax; i++)
    {
	dest_pixels[i] = (gpr_$pixel_value_t) ((*redptr) << 16);
	dest_pixels[i] |= (gpr_$pixel_value_t) ((*grnptr) << 8);
	dest_pixels[i] |= (gpr_$pixel_value_t) (*bluptr);
	redptr++;  grnptr++; bluptr++;
    }
    gpr_$write_pixels( dest_pixels, dest_box, status );
    sysdebug;
}

/*
 * Map a 24 bit scanline to 8 bits through the dither matrix.
 */
#define DMAP(v,x,y)	(mod51[v]>dm16[x][y] ? div51[v] + 1 : div51[v])

put_line8( y )
int y;
{
    register unsigned char *r, *g, *b;
    register int i, col, row;
    gpr_$window_t dest_box;
    int n = sv_globals.sv_xmax;

    r = rows[SV_RED];
    if (sv_globals.sv_ncolors == 3)
    { 
        g = rows[SV_GREEN];
        b = rows[SV_BLUE];
    }
    else
    {
        g = rows[0];
        b = rows[0];
    }

    for ( row = y % 16, col = 0, i = 0;
	 i < n; i++, r++, g++, b++, col = ((col + 1) & 15) )
	dest_pixels[i] = DMAP(gammamap[*r], col, row) +
	    		 DMAP(gammamap[*g], col, row) * 6 +
			 DMAP(gammamap[*b], col, row) * 36 + COLMAP_OFFSET;

    dest_box.window_base.x_coord = sv_globals.sv_xmin;
    dest_box.window_base.y_coord = sv_globals.sv_ymax - y;
    dest_box.window_size.x_size = sv_globals.sv_xmin + sv_globals.sv_xmax;
    dest_box.window_size.y_size = 1;

    gpr_$acquire_display( status ) ;
    gpr_$write_pixels( dest_pixels, dest_box, status ); 
    gpr_$release_display( status );

}

/*
 * Map a 24 bit scanline through to get dithered black and white
 */
put_linebw( y )
int y;
{
    register unsigned char *r, *g, *b;
    register int i, col, row;
    gpr_$window_t dest_box;
    int pixel;
    int n = sv_globals.sv_xmax;
    int B = 0, W = 1;

    r = rows[SV_RED];
    if (sv_globals.sv_ncolors == 3)
    { 
        g = rows[SV_GREEN];
        b = rows[SV_BLUE];
    }
    else
    {
        g = rows[0];
        b = rows[0];
    }
    if (inverse_flag)
    {
	B = 1; W = 0;		/* Swap meaning of Black and White */
    }

    for ( row = y % 8, col = 0, i = 0;
	 i < n; i++, r++, g++, b++, col = ((col + 1) & 7) )
    {
	/* Conver to BW (uses YIQ/percentage xformation) */
	pixel = (35*gammamap[*r] + 55*gammamap[*g] + 10*gammamap[*b]) / 100;
	if (pixel < 0) pixel += 256;
	dest_pixels[i] = ((pixel > dmgray[col][row]) ? W : B);
    }
    dest_box.window_base.x_coord = sv_globals.sv_xmin;
    dest_box.window_base.y_coord = sv_globals.sv_ymax - y;
    dest_box.window_size.x_size = sv_globals.sv_xmin + sv_globals.sv_xmax;
    dest_box.window_size.y_size = 1;

    gpr_$acquire_display( status ) ;
    gpr_$write_pixels( dest_pixels, dest_box, status ); 
    gpr_$release_display( status );

}

