/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notices are 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */

/* 
 * getX11.c - Put RLE images on X display.
 * 
 * Author:	Spencer W. Thomas  (x10)
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Thu Feb 20 1986
 * Copyright (c) 1986, University of Utah
 * 
 * Modified:	Andrew F. Vesper (x 11)
 *		High Performance Workstations
 *		Digital Equipment Corp
 * Date:	Fri, Aug 7, 1987
 *		Thu, Jan 14, 1988
 * Copyright (c) 1987,1988, Digital Equipment Corporation
 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <svfb_global.h>
#include <exit_status.h>

#include "circle.bitmap"
#include "circle_mask.bitmap"

#include "annotation.h"

/* Make Sys V macros map to BSD macros */
#ifndef _tolower
#define _tolower tolower
#define _toupper toupper
#endif

#define MAX_DISTANCE	50

#define PROGRAM_NAME		"getX11"

/* must be able to fit 3 * log2 (MAXIMUM_LEVELS) in a 32 bit word */

#define DEFAULT_LEVELS		256
#define MAXIMUM_LEVELS		1024

#define ANNOTATION_PAD		2

#define DESIRED_ICON_WIDTH	50
#define DESIRED_ICON_HEIGHT	50

/*
 * note: both USE_PIXMAP_FOR_ICON and USE_WINDOW_FOR_ICON can
 * be 1 at the same time.  The window manager will use the one it
 * wants.
 */

/* define this as 1  when you want the icon window refreshed by a pixmap */
#ifndef USE_PIXMAP_FOR_ICON
#  define USE_PIXMAP_FOR_ICON	1
#endif

/* define this as 1 if you want the icon to be a window */
#ifndef USE_WINDOW_FOR_ICON
#   define USE_WINDOW_FOR_ICON	0
#endif

/* number of scan lines to be drawn in incremental mode */
#define LINES_DRAWN		10

/* define this to get extra info printed */
#ifndef VERBOSE
# define VERBOSE			1
#endif

/* define this to request server backing store */
#ifndef WANT_BACKING_STORE
# define WANT_BACKING_STORE	0
#endif

#define ACTION_ANNOTATE		0
#define ACTION_INPUT		1
#define ACTION_ICONIFY		2
#define ACTION_EXIT		4

#define ACTION_DEFAULT		ACTION_ANNOTATE

/* define what to do on unshifted mouse buttons */
static int button_action[] = {
	ACTION_ANNOTATE,
	ACTION_INPUT,
	ACTION_ANNOTATE
	};

/* define what to do on shifted mouse buttons */
static int shifted_action[] = {
	ACTION_EXIT,
	ACTION_ICONIFY,
	ACTION_EXIT
	};

#define IMAGE_BORDERWIDTH	3
#define ICON_BORDERWIDTH	0

#define COUNT_OF(_array_)	( sizeof (_array_) / sizeof (_array_[0]) )

#ifndef EOS
# define EOS	('\0')
#endif

extern char	*getenv ();
extern char	*malloc ();

typedef enum {
    BOOL_FALSE = 0,
    BOOL_TRUE = 1
    } BOOLEAN;

#define IS_TRUE(_b_)	(_b_ != BOOL_FALSE)
#define IS_FALSE(_b_)	(_b_ == BOOL_FALSE)

typedef enum {
    FAILURE = 0,
    SUCCESS = 1
    } STATUS;

#define CONV_IMAGE_TO_X(_ix_,_iy_,_xx_,_xy_)	{	\
	if (IS_TRUE(left_pixel_first)) _xx_ = _ix_ - image_xmin;	\
	else _xx_ = image_xmax - (_ix_);		\
	if (IS_TRUE(top_line_first)) _xy_ = _iy_ - image_ymin;	\
	else _xy_ = image_ymax - (_iy_);		\
	}

#define CONV_X_TO_IMAGE(_xx_,_xy_,_ix_,_iy_)	{	\
	if (IS_TRUE(left_pixel_first)) _ix_ = _xx_ + image_xmin;	\
	else _ix_ = image_xmax - (_xx_);		\
	if (IS_TRUE(top_line_first)) _iy_ = _xy_ + image_ymin;	\
	else _iy_ = image_ymax - (_xy_);		\
	}

#define PUT_SCANLINES(_image_,_x_,_y_,_width_,_height_,_window_,_gc_)	\
XPutImage (dpy, _window_, _gc_, _image_, _x_, _y_, _x_, _y_, _width_, _height_)

#ifndef XLIB_ENUM
typedef int		X_VISUAL_CLASS;
#endif

typedef unsigned long	X_PIXEL;

static STATUS get_pic ();
static BOOLEAN init_color_rw ();
static BOOLEAN init_separate_color_rw ();
static BOOLEAN init_color_ro ();
static BOOLEAN init_separate_color_ro ();
static BOOLEAN init_mono_rw ();
static BOOLEAN init_mono_ro ();

/*
 * Basic magic square for dithering
 */

static int magic[4][4] =  {
 	 0, 14,  3, 13,
	11,  5,  8,  6,
	12,  2, 15,  1,
	 7,  9,  4, 10
};

static int dm4[4][4] = {
		  0, 43,  9, 40,
		 34, 15, 25, 18,
		 37,  6, 46,  3,
		 21, 28, 12, 31 };

static int dm16[16][16];

/* 
 * Color map, gamma correction map, and lookup tables 
 */

static X_PIXEL		* pixel_table;
static int		red_shift;
static int		green_shift;
static int		blue_shift;
static X_PIXEL		red_mask;
static X_PIXEL		green_mask;
static X_PIXEL		blue_mask;
static X_PIXEL		pixel_base;
static X_PIXEL		black_pixel;
static X_PIXEL		white_pixel;

static rle_pixel	** in_cmap = NULL;
static int 		modN[256];
static int		divN[256];
static BOOLEAN		binary_image = BOOL_FALSE; 
			/* if BOOL_TRUE, only two colors in image	*/
static BOOLEAN		monochrome_image = BOOL_FALSE;
static BOOLEAN		dither_image = BOOL_TRUE;

static BOOLEAN		monochrome_flag = BOOL_FALSE;
static BOOLEAN		binary_flag = BOOL_FALSE;
static BOOLEAN		dither_flag = BOOL_TRUE;
static BOOLEAN		line_at_a_time = BOOL_TRUE;

static BOOLEAN		top_line_first = BOOL_FALSE;
static BOOLEAN		left_pixel_first = BOOL_TRUE;

static BOOLEAN		repeat_annotation = BOOL_FALSE;

/*
 * Number of color map levels, (square and cube), and number of gradations
 * per level.
 */

static int		specified_levels = 0;
static int		levels = 0;
static int		levels_squared = 0;
static int		log2_levels;

/* 
 * Global variables
 */

static double		display_gamma = 2.5;	/* default gamma for display	*/
static double		image_gamma = 1.0;	/* default gamma for image 	*/
static int		iflag = 0;	/* was gamma on command line?	*/
static int		oflag = 0;	/* was origin on command line?  */

static X_VISUAL_CLASS	visual_type = (X_VISUAL_CLASS) (-1);
static Display 		* dpy = (Display *) NULL;
static Visual		* visual = NULL;
static int		visual_depth = 0;
static Window		root_window = (Window) 0;
static int		screen = 0;
static Colormap		color_map = (Colormap) 0;
static int		bitmap_pad;
static BOOLEAN		separate_colors;
static BOOLEAN		mutable_color_map;

static Window		image_window;
static int		image_width;		/* in pixels */
static int		image_height;
static int		image_width_in_bytes;
static int		image_colors;
static XSetWindowAttributes image_attributes;
static unsigned long	image_attribute_mask;
static GC		image_gc = (GC) 0;

static int		time_between_updates;

static ANNOTATION	* current_note = NULL;
static GC		note_gc = (GC) 0;

#if USE_WINDOW_FOR_ICON
static Window		icon_window;
#endif

#if USE_PIXMAP_FOR_ICON
static Pixmap		icon_pixmap;
#endif

static int		icon_width;
static int		icon_height;
static int		icon_width_in_bytes;
static XSetWindowAttributes icon_attributes;
static unsigned long	icon_attribute_mask;
#if USE_WINDOW_FOR_ICON
static GC		icon_gc = (GC) 0;
#endif
#if USE_PIXMAP_FOR_ICON
static GC		icon_pixmap_gc = (GC) 0;
#endif

static BOOLEAN		debug_flag = BOOL_FALSE;	/* set if debug mode */

static char		* image_buffer;	/* pixel values		*/
static char		* icon_buffer;

static XImage		* image_image;
static XImage		* icon_image;

static int		image_xmin;
static int		image_ymin;
static int		image_xmax;
static int		image_ymax;

static struct {
	char		*string;
	X_VISUAL_CLASS	type;
	}  visual_type_table[] = {

    { "staticgray", StaticGray},
    { "grayscale", GrayScale},
    { "staticgrey", StaticGray},
    { "greyscale", GrayScale},
    { "staticcolor", StaticColor},
    { "pseudocolor", PseudoColor},
    { "truecolor", TrueColor},
    { "directcolor", DirectColor}
    };

static struct {
	char		*string;
	BOOLEAN		top_line_first;
	BOOLEAN		left_pixel_first;
	} origin_table[] = {
    {"top-left",	BOOL_TRUE,	BOOL_TRUE},
    {"topleft",		BOOL_TRUE,	BOOL_TRUE},
    {"tl",		BOOL_TRUE,	BOOL_TRUE},
    {"top-right",	BOOL_TRUE,	BOOL_FALSE},
    {"topright",	BOOL_TRUE,	BOOL_FALSE},
    {"tr",		BOOL_TRUE,	BOOL_FALSE},
    {"bottom-left",	BOOL_FALSE,	BOOL_TRUE},
    {"bottom-left",	BOOL_FALSE,	BOOL_TRUE},
    {"bl",		BOOL_FALSE,	BOOL_TRUE},
    {"bottom-right",	BOOL_FALSE,	BOOL_FALSE},
    {"bottomright",	BOOL_FALSE,	BOOL_FALSE},
    {"br",		BOOL_FALSE,	BOOL_FALSE}
    };

static int		annotation_wanted = 0;
static FILE		* annotation_output_file = NULL;
static void		handle_annotation ();
static void		erase_note ();
static void		draw_note ();

static Font		annotation_font = NULL;
static XFontStruct	* annotation_font_info = NULL;
static char		* annotation_font_name = NULL;
static char		default_annotation_font_name[] = "variable";
static char		alternate_annotation_font_name[] = "fixed";
static int		annotation_height;
static int		annotation_y;

#ifdef _STDC_
typedef void SCANLINE_CONVERTER (
	unsigned char	*rgb[3],
	int		ncolors,
	int		width,
	int		stride,
	int		y,
	XImage		*image
	);

static SCANLINE_CONVERTER map_scanline_generic;
static SCANLINE_CONVERTER map_l_1_d_t_8;
static SCANLINE_CONVERTER map_l_1_nd_t_8;
static SCANLINE_CONVERTER map_l_23_d_nt_32;
static SCANLINE_CONVERTER map_l_23_nd_nt_32;

#else
typedef void SCANLINE_CONVERTER ();

static void map_scanline_generic ();
static void map_l_1_d_t_8 ();
static void map_l_1_nd_t_8 ();
static void map_l_23_d_nt_32 ();
static void map_l_23_nd_nt_32 ();

#endif

static SCANLINE_CONVERTER *map_scanline;

struct {
	BOOLEAN		left_to_right;
	BOOLEAN		one_color;
	BOOLEAN		dither;
	BOOLEAN		table_present;
	int		bits_per_pixel;
	SCANLINE_CONVERTER	*routine;
	} map_scanline_table[] = {

  /*   l to r    one color     dither    table pres  b/p  routine	    */

    {BOOL_TRUE,  BOOL_TRUE,  BOOL_TRUE,  BOOL_TRUE,    8, map_l_1_d_t_8},
    {BOOL_TRUE,  BOOL_TRUE,  BOOL_FALSE, BOOL_TRUE,    8, map_l_1_nd_t_8},
    {BOOL_TRUE,  BOOL_FALSE, BOOL_TRUE,  BOOL_FALSE,  32, map_l_23_d_nt_32},
    {BOOL_TRUE,  BOOL_FALSE, BOOL_FALSE, BOOL_FALSE,  32, map_l_23_nd_nt_32}
    };


/*****************************************************************
 * TAG( main )
 * 
 * Usage:
 *	getX11 [-F] [-D] [-d display] [-= window-geometry] [-l levels] [-z]
 *		[-{iI} gamma] [-g gamma] [-v visualtype] [-n] [-b] [-m] 
 *		[-a annotation-file] [-A new-annotation-file] [-o origin] 
 *		[-f font-name] [file]
 * Inputs:
 * 	-F:		Don't fork after putting image on screen.
 *	-D:		Debug mode: print input file as read.
 *	-b:		bitonal; binary black and white
 *	-m:		Black & white: reduce color images to monochrome before
 *			display.  Advantage is that smoother shading can
 *			be achieved.
 *	-d display:	Specify display name.
 *	-= window_geometry:
 *			Specify window geometry (but min size set by file).
 *	-i gamma:	Specify gamma of image. (default 1.0)
 *	-I gamma:	Specify gamma of display image was computed for.
 *	-g gamma:	Specify gamma of display. (default 2.5)
 *	-v visual_type: specify visual type to be used (integer, 0 to 5)
 *	-l levels:	number of levels to display of each color
 *	-z:		hold all output, then blast it all out at once
 *	-n:		Do not dither
 *	-a file		annotation file name
 *	-A file		output annotation file name
 *	-o origin	specifies origin: `top-left', `top-right', `bottom-left'
 *			or `bottom-right'
 *	-f font-name	font name for annotation
 *	-r time		repetitiously annotate everybody (time in seconds)
 *
 *	file:		Input Run Length Encoded file. Uses stdin if not
 *			specified.
 *
 * 	getX11 will also read picture comments from the input file to determine
 *			the image gamma.  These are
 *
 *	image_gamma=	gamma of image (equivalent to -i)
 *	display_gamma=	gamma of display image was computed for.
 *	origin=		`top-left', `top-right', `bottom-left' or `bottom-right'
 *		
 *	Any command line arguments override the values in the file.
 *		
 * Outputs:
 * 	Puts image on screen.
 * Assumptions:
 * 	Input file is in RLE format.
 * Algorithm:
 *	[None]
 */

int
main (argc, argv)

int		argc;
char 		** argv;

{
char		** infnames = NULL;
char		* infname = NULL;
char		* display_name = NULL;
char		* window_geometry = NULL;
FILE 		* infile;
int		nfile = 0;
int		forkflg = 0;
int		dflag = 0;
int		gflag = 0;
int		wflag = 0;
STATUS		status;
int		vflag = 0;
int		nflag = 0;
char		*vis_type = NULL;
int		zflag = 0;
int		bin_flag = 0;
int		mono_flag = 0;
int		a_out_flag = 0;
char		* annotation_file = NULL;
char		* annotation_output_file_name = NULL;
char		* origin;
int		fflag = 0;
int		rflag = 0;

#ifdef VMS
argc = getredirection (argc, argv);
#endif

if ( scanargs( argc, argv,
	"% F%- b%- m%- D%- l%-levels!d d%-display!s \n\
	=%-window-geometry!s Ii%-gamma!F g%-gamma!F v%-visual!s n%- \n\
	z%- a%-annotation-file!s A%-new-annotation-file!s  \n\
	o%-origin!s f%-font-file!s r%-repeat-annotation!d file%s",
	&forkflg, &bin_flag, &mono_flag, &debug_flag,
	&specified_levels, &specified_levels,
	&dflag, &display_name,
	&wflag, &window_geometry,
	&iflag, &image_gamma,
	&gflag, &display_gamma,
	&vflag, &vis_type,
	&nflag, &zflag,
	&annotation_wanted, &annotation_file,
	&a_out_flag, &annotation_output_file_name,
	&oflag, &origin,
	&fflag, &annotation_font_name,
	&rflag, &time_between_updates,
	&infname ) == 0 )
    exit(EXIT_STATUS_FAILURE);

/*
 * I would like to be able to use multiple files, but haven't
 * figured how to get scanargs to let us.  So, use this kludge for now.
 */

if ( infname != NULL ) {
    infnames = &infname;
    nfile = 1;
    }

if ( iflag == 1 )		/* -i */
	image_gamma = 1.0 / image_gamma;

if (bin_flag == 0) binary_flag = BOOL_FALSE;
else binary_flag = BOOL_TRUE;

if (mono_flag == 0) monochrome_flag = BOOL_FALSE;
else monochrome_flag = BOOL_TRUE;

if ( IS_TRUE (binary_flag) ) monochrome_flag = BOOL_TRUE;

visual_type = (-1);

if ( vflag != 0) {
    if (isdigit (vis_type[0]) ) {
	visual_type = (X_VISUAL_CLASS) atoi (vis_type);
	}
    else {
	register char * ptr;
	register int  i;
	visual_type = (X_VISUAL_CLASS) 9999;

	for (ptr = vis_type; *ptr != EOS; ptr++) {
	    if (isupper(*ptr)) *ptr = _tolower(*ptr);
	    }

	for (i = 0; i < COUNT_OF (visual_type_table); i++) {
	    if (strcmp (visual_type_table[i].string, vis_type) == 0) {
		visual_type = visual_type_table[i].type;
		break;
		}
	    }
	}

    if ((int) visual_type < 0 || (int) visual_type > 5) {
	fprintf (stderr, "Bad visual type %s, ignoring request\n", vis_type);
	visual_type = (X_VISUAL_CLASS) (-1);
	}

    }

if ( oflag != 0) {
    register char * ptr;
    register int  i;
    
    for (ptr = origin; *ptr != EOS; ptr++) {
	if (isupper(*ptr)) *ptr = _tolower(*ptr);
	}
    
    for (i = 0; i < COUNT_OF (origin_table); i++) {
	if (strcmp (origin_table[i].string, origin) == 0) {
	    top_line_first = origin_table[i].top_line_first;
	    left_pixel_first = origin_table[i].left_pixel_first;
	    break;
	    }
	}

    if (i >= COUNT_OF (origin_table) ) {
	fprintf (stderr, "Bad origin %s, ignoring request\n", origin);
	oflag = 0;
	}
	
    }

if (nflag == 0) dither_flag = BOOL_TRUE;
else dither_flag = BOOL_FALSE;

if (zflag == 0) line_at_a_time = BOOL_TRUE;
else line_at_a_time = BOOL_FALSE;

if (annotation_wanted == 0) annotation_file = NULL;

if (fflag == 0) annotation_font_name = default_annotation_font_name;

if (a_out_flag) {
#ifdef VMS
    annotation_output_file = fopen (annotation_output_file_name, "w",
	"rat=cr", "rfm=var");
#else
    annotation_output_file = fopen (annotation_output_file_name, "w");
#endif
    if (annotation_output_file == NULL) {
	fprintf (stderr, "Could not create annotation file %s\n", 
		annotation_output_file_name);
	}
    }

repeat_annotation = BOOL_FALSE;
if (rflag != 0 
	&& IS_TRUE (annotation_wanted) 
	&& annotation_file != NULL) {
    repeat_annotation = BOOL_TRUE;

    if (time_between_updates < 0 || time_between_updates > 120) {
	time_between_updates = 15;
	}

    }

/*
 * open the display
 */

if (display_name == NULL) display_name = getenv ("X11DISPLAY");

dpy = XOpenDisplay(display_name);

if (dpy == NULL) {
    fprintf(stderr, "%s: Can't open display %s\n", PROGRAM_NAME,
	    (display_name == NULL) ? "" : display_name);
    exit(EXIT_STATUS_FAILURE);
    }

/* 
 * For each file, display it.
 */

do {

    if ( --nfile >= 0 ) {


#ifdef VMS
	infile = fopen( *infnames, "rb", "ctx=bin" );
#else
	infile = fopen( *infnames, "r");
#endif

	if (infile == NULL) {
	    perror("fopen");
	    fprintf (stderr, "Could not open file %s\n", *infnames);
	    continue;
	    }

	status = get_pic( infile, *infnames, window_geometry);

	fclose( infile );

	fprintf (stderr, "Picture %s complete\n", *infnames);

	}

    else
	status = get_pic( stdin, "Standard Input", window_geometry);

    if (status == SUCCESS) {

#ifdef unix

	if ( ! forkflg ) {
    
	    if ( fork() == 0 ) {
		/* 
		* Get rid of std fds so rshd will go
		* away.
		*/
		close( 0 );
		close( 1 );
		close( 2 );
		update_pic (annotation_file);
		break;
		}
	    }
	else {
	    update_pic (annotation_file);
	    }
#else
	update_pic (annotation_file);
#endif
    
	}

    infnames++;

    } while ( --nfile >= 0 );

XCloseDisplay( dpy );

}

/* 
 * Read an image from the input file and display it.
 */

static
STATUS
get_pic( infile, file_name, window_geometry)

FILE 			* infile;
char 			* file_name;
char 			* window_geometry;

{
register int		i;
register int		image_y;
int			x;
register int		y;
unsigned char		*scan[3];
register int		icon_factor;
register int		next_y;
register int		y_base;
register int		lines_buffered;
register BOOLEAN	one_color;
register BOOLEAN	table_present;
XWMHints		wm_hints;
XIconSize		* icon_sizes;
int			icon_size_count;

/*
 * Read setup info from file. 
 */

sv_globals.svfb_fd = infile;

if (rle_get_setup(&sv_globals) < 0) {
    fprintf(stderr, "%s: Error reading setup information from %s\n", 
	    PROGRAM_NAME, (file_name != NULL) ? file_name : "stdin");
    return (FAILURE);
    }

if ( debug_flag )
	rle_debug( 1 );
else rle_debug (0);

image_xmin = sv_globals.sv_xmin;
image_xmax = sv_globals.sv_xmax;

image_ymin = sv_globals.sv_ymin;
image_ymax = sv_globals.sv_ymax;

image_width = image_xmax - image_xmin + 1;
image_height = image_ymax - image_ymin + 1;

sv_globals.sv_xmin = 0;
sv_globals.sv_xmax = image_width - 1;

/* We're only interested in R, G, & B */

SV_CLR_BIT(sv_globals, SV_ALPHA);
for (i = 3; i < sv_globals.sv_ncolors; i++)
	SV_CLR_BIT(sv_globals, i);

image_colors = sv_globals.sv_ncolors > 3 ? 3 : sv_globals.sv_ncolors;

if (image_colors == 1) monochrome_image = BOOL_TRUE;
else monochrome_image = BOOL_FALSE;

binary_image = BOOL_FALSE;

/*
 * search for the most appropriate visual and define
 * root_window, bitmap_pad, binary_image, monochrome_image,
 * image_attributes, image_attribute_mask,
 * icon_attributes, icon_attribute_mask,
 */

find_appropriate_visual (image_colors);

image_image = XCreateImage (dpy, visual, visual_depth, 
	(IS_TRUE (binary_image) ? XYBitmap : ZPixmap), 
	0, NULL, image_width, image_height, 32, 0);

if (image_image == NULL) {
    fprintf (stderr, "XCreateImage (image) failed\n");
    return (FAILURE);
    }

image_buffer = malloc (image_image->bytes_per_line * image_height);

if (image_buffer == NULL) {
    perror ("malloc image_buffer");
    return (FAILURE);
    }

image_image->data = image_buffer;

/*
 * We want the Icon to be about DESIRED_ICON_WIDTH x DESIRED_ICON_HEIGHT.  
 * First, make sure that is OK with the window manager.
 * Then figure out the icon scaling factor.
 */

icon_width = DESIRED_ICON_WIDTH;
icon_height = DESIRED_ICON_HEIGHT;

if (XGetIconSizes (dpy, root_window, &icon_sizes, &icon_size_count) 
	&& icon_size_count >= 1) {

    for (i = 0; i < icon_size_count; i++) {
	if (icon_sizes[i].min_width <= DESIRED_ICON_WIDTH
		&& icon_sizes[i].max_width >= DESIRED_ICON_WIDTH
		&& icon_sizes[i].min_height <= DESIRED_ICON_HEIGHT
		&& icon_sizes[i].max_height >= DESIRED_ICON_HEIGHT) {
	    break;
	    }
	}

    if (i >= icon_size_count) {
	icon_width = icon_sizes[0].max_width;
	icon_height = icon_sizes[0].max_height;
	}

    }

icon_factor = image_width / icon_width;
if (icon_factor < image_height / icon_height) 
	icon_factor = image_height / icon_height;
if ( icon_factor == 0 ) icon_factor = 1;
	
icon_width = 1 + (image_width / icon_factor);
icon_height = 1 + (image_height / icon_factor);
    
icon_image = XCreateImage (dpy, visual, visual_depth, 
	(IS_TRUE (binary_image) ? XYBitmap : ZPixmap), 
	0, NULL, icon_width, icon_height, 32, 0);

if (image_image == NULL) {
    fprintf (stderr, "XCreateImage (icon) failed\n");
    return (FAILURE);
    }

icon_buffer = malloc (icon_image->bytes_per_line * icon_height);

if (icon_buffer == NULL) {
    perror ("malloc icon_pixel_buffer");
    return (FAILURE);
    }

icon_image->data = icon_buffer;

/*
 * Set up for rle_getrow.  Pretend image x origin is 0. 
 */

for (i = 0; i < image_colors; i++) scan[i] = NULL;

scan[0] = (unsigned char *) malloc (image_width * image_colors);

if (scan[0] == NULL) {
    perror ("malloc scan buffer");
    return (FAILURE);
    }

for (i = 1; i < image_colors; i++)
    scan[i] = scan[i - 1] + image_width;
   
/* If no image gamma on command line, check comments in file */
   
if (iflag == 0) {

    char * v;
    if ( (v = rle_getcom( "image_gamma", &sv_globals )) != NULL ) {
	image_gamma = atof( v );
	/* Protect against bogus information */
	if ( image_gamma == 0.0 )
	    image_gamma = 1.0;
	else
	    image_gamma = 1.0 / image_gamma;
	}
    else if ( (v = rle_getcom( "display_gamma", &sv_globals )) != NULL) {
	image_gamma = atof( v );
	/* Protect */
	if ( image_gamma == 0.0 )
	    image_gamma = 1.0;
	}

    }

/* check for `origin' comment if not given on command line */

if (oflag == 0) {
    char *origin;
    if ( (origin = rle_getcom ("origin", &sv_globals) ) != NULL ) {
	register char * ptr;
	register int  i;
	
	for (ptr = origin; *ptr != EOS; ptr++) {
	    if (isupper(*ptr)) *ptr = _tolower(*ptr);
	    }
	
	for (i = 0; i < COUNT_OF (origin_table); i++) {
	    if (strcmp (origin_table[i].string, origin) == 0) {
		top_line_first = origin_table[i].top_line_first;
		left_pixel_first = origin_table[i].left_pixel_first;
		break;
		}
	    }
    
	}
    }


/*
 * Set up the color map. 
 */

/* Input map, at least 3 channels */

in_cmap = buildmap (&sv_globals, image_colors, image_gamma );

/* Get X color map */
   
init_color();
   
/*
 * Get image and icon windows of the right size 
 * Allow operator to positions the image window with the mouse.
 */

create_windows (window_geometry, file_name);

map_scanline = map_scanline_generic;

if (image_colors == 1 || IS_TRUE (monochrome_image) ) one_color = BOOL_TRUE;
else one_color = BOOL_FALSE;

if (pixel_table == NULL) table_present = BOOL_FALSE;
else table_present = BOOL_TRUE;

for (i = 0; i < COUNT_OF (map_scanline_table); i++) {
    if (map_scanline_table[i].left_to_right == left_pixel_first
	    && map_scanline_table[i].one_color == one_color
	    && map_scanline_table[i].dither == dither_image
	    && map_scanline_table[i].table_present == table_present
	    && map_scanline_table[i].bits_per_pixel 
		    == image_image->bits_per_pixel) {
	map_scanline = map_scanline_table[i].routine;
#if VERBOSE
	fprintf (stderr, "Special map_scanline routine used: ");
	if (IS_TRUE(left_pixel_first) ) fputs ("map_l_", stderr);
	else fputs ("map_r_", stderr);
	if (IS_TRUE(one_color)) fputs ("1_", stderr);
	else fputs ("23_", stderr);
	if (IS_TRUE (dither_image) ) fputs ("d_", stderr);
	else fputs ("nd_", stderr);
	if (IS_FALSE (table_present) ) fputc ('n', stderr);
	fprintf (stderr, "t_%d (index %d)\n", image_image->bits_per_pixel, i);
#endif
	break;
	}
    }

/*
 * For each scan line, dither it and display. 
 */

if (IS_TRUE (top_line_first)) next_y = 0;
else next_y = image_height - 1;

y_base = next_y;
lines_buffered = 0;

while ( (image_y = rle_getrow(&sv_globals, scan)) <= sv_globals.sv_ymax) {

    CONV_IMAGE_TO_X (image_xmin, image_y, x, y);

    if (IS_TRUE (line_at_a_time) && y != next_y) {
	if (lines_buffered > 0) {
	    if (IS_FALSE (top_line_first) ) y_base -= lines_buffered - 1;
	    PUT_SCANLINES (image_image, 0, y_base, 
		    image_width, lines_buffered, 
		    image_window, image_gc);
	    lines_buffered = 0;
	    }
	y_base = y;
	}

    if (IS_TRUE (top_line_first)) next_y = y + 1;
    else next_y = y - 1;

    if (IS_TRUE (monochrome_image) ) {

	map_rgb_to_bw ( scan, image_colors, scan[0], in_cmap, image_width );
	(*map_scanline) (scan, 1, image_width, 1, y, image_image);

	/* Subsample image to create icon */
   
	if ( y % icon_factor == 0 ) {

	    (*map_scanline) ( scan, 1, icon_width, icon_factor,
		    y / icon_factor,
		    icon_image);
	    }

	}		/* end monochrome display			*/
   
    else {		/* color display				*/

	(*map_scanline) (scan, image_colors, image_width, 1, y, image_image);
   
	/* Subsample image to create icon */
   
	if ( y % icon_factor == 0 ) {

	    (*map_scanline)( scan, image_colors, icon_width, icon_factor,
		    y / icon_factor, icon_image);
	    }

	}

    if (IS_TRUE (line_at_a_time) && ++lines_buffered >= LINES_DRAWN) {
	if (IS_FALSE (top_line_first) ) y_base -= lines_buffered - 1;
	PUT_SCANLINES (image_image, 0, y_base, 
		image_width, lines_buffered, image_window, image_gc); 
	y_base = next_y;
	lines_buffered = 0;
	}

    }			/* end color display				*/

if (IS_TRUE (line_at_a_time) && lines_buffered > 0) {
    if (IS_FALSE (top_line_first) ) y_base -= lines_buffered - 1;
    PUT_SCANLINES (image_image, 0, y_base, 
	    image_width, lines_buffered, image_window, image_gc); 
    }


if (IS_FALSE (line_at_a_time) ) {
#if VERBOSE
    fprintf (stderr, "Displaying the entire image now\n");
    fflush (stderr);
#endif
    PUT_SCANLINES (image_image, 0, 0, image_width, image_height, 
	    image_window, image_gc);
    }

wm_hints.flags = 0;

#if USE_PIXMAP_FOR_ICON

PUT_SCANLINES (icon_image, 0, 0, icon_width, icon_height,
	icon_pixmap, icon_pixmap_gc);

wm_hints.flags |= IconPixmapHint;
wm_hints.icon_pixmap = icon_pixmap;

#endif

#if USE_WINDOW_FOR_ICON

wm_hints.flags |= IconWindowHint;
wm_hints.icon_window = icon_window;

#endif

XSetWMHints (dpy, image_window, &wm_hints);

/*
 * Free temp storage 
 */
    
free(scan[0]);

return (SUCCESS);

}

/* 
 * Create a window with no help from user.
 */

static
create_windows (window_geometry, name )

char			*window_geometry;
char			*name;

{
char			default_geometry[30];
int			width;
int			height;
int			desired_height;
int			x;
int			y;
XSizeHints		size_hints;
unsigned int		mask;
Cursor			circle_cursor;
register unsigned long	gc_mask;
XGCValues		gc_values;
XEvent			event;

/* 
 * Now, make the window.
 */

annotation_font = NULL;

desired_height = image_height;

if (annotation_wanted 
	|| annotation_font_name != default_annotation_font_name) {
    annotation_font_info = XLoadQueryFont (dpy, annotation_font_name);
    if (annotation_font_info == NULL) {
	annotation_font_info = XLoadQueryFont (alternate_annotation_font_name);
	}
    if (annotation_font_info == NULL) {
	annotation_font = (Font) DefaultGC (dpy, screen);
	annotation_font_info = XQueryFont (dpy, annotation_font);
	}
    if (annotation_font_info != NULL) {
	annotation_height = 2 * ANNOTATION_PAD 
		+ annotation_font_info->ascent
		+ annotation_font_info->descent;
	desired_height += annotation_height;
	annotation_y = image_height + ANNOTATION_PAD 
		+ annotation_font_info->ascent;
	annotation_font = annotation_font_info->fid;
	}
    }

sprintf( default_geometry, "=%dx%d", image_width, desired_height );

mask = XGeometry (dpy, screen, window_geometry, default_geometry, 
	IMAGE_BORDERWIDTH, 1, 1, 0, 0,
	&x, &y, &width, &height);

if (width < image_width) width = image_width;
if (height > desired_height) height = desired_height;

if ( (mask & XValue) == 0) {
    x = (DisplayWidth(dpy,screen) - width)/2;
    }
if ( (mask & YValue) == 0) {
    y = (DisplayHeight(dpy,screen) - height)/2;
    }

#if VERBOSE
fprintf (stderr, "Xwindow at %d, %d, width %d, height %d\n", 
	x, y, width, height);
#endif

image_attribute_mask |= CWEventMask;
image_attributes.event_mask |= ButtonPressMask | ButtonReleaseMask | ExposureMask
	| VisibilityChangeMask;

image_window = XCreateWindow ( dpy, root_window, x, y, 
	width, height, 
	IMAGE_BORDERWIDTH, visual_depth, InputOutput, visual,
	image_attribute_mask, &image_attributes);

if (image_window == NULL) {
    fprintf( stderr, "%s: Window Create failed\n", PROGRAM_NAME );
    exit(EXIT_STATUS_FAILURE );
    }

XMapWindow (dpy, image_window );

XClearWindow (dpy, image_window);

do {
    XNextEvent(dpy, &event);
    }
while ( ! (event.type == Expose && event.xexpose.window == image_window) );

size_hints.flags = PMinSize;
size_hints.min_width = image_width;
size_hints.min_height = image_height;

XSetNormalHints (dpy, image_window, &size_hints);

XStoreName (dpy, image_window, name);
XSetIconName (dpy, image_window, name);

#if USE_WINDOW_FOR_ICON
icon_attribute_mask |= CWEventMask;
icon_attributes.event_mask |= ButtonReleaseMask | ExposureMask;

icon_window = XCreateWindow (dpy, root_window, 0, 0, icon_width, icon_height, 
	ICON_BORDERWIDTH, visual_depth, InputOutput, visual,
	icon_attribute_mask, &icon_attributes);

XStoreName (dpy, icon_window, icon_name);
#endif

gc_mask = 0;
gc_values.function = GXcopy;		gc_mask |= GCFunction;
gc_values.plane_mask = AllPlanes;	gc_mask |= GCPlaneMask;
gc_values.foreground = white_pixel;	gc_mask |= GCForeground;
gc_values.background = black_pixel;	gc_mask |= GCBackground;
gc_values.graphics_exposures = False;	gc_mask |= GCGraphicsExposures;
if (annotation_font != (Font) 0) {
    gc_values.font = annotation_font;   gc_mask |= GCFont;
    }

image_gc = XCreateGC (dpy, image_window, gc_mask, &gc_values);

#if USE_WINDOW_FOR_ICON
icon_gc  = XCreateGC (dpy, icon_window, gc_mask, &gc_values);
#endif

#if USE_PIXMAP_FOR_ICON

icon_pixmap = XCreatePixmap (dpy, root_window, 
	icon_width, icon_height, visual_depth);

icon_pixmap_gc = XCreateGC (dpy, icon_pixmap, gc_mask, &gc_values);

#endif

circle_cursor = XCreateFontCursor (dpy, XC_circle);
if (circle_cursor == NULL) {

    Pixmap	source;
    Pixmap	mask;
    XColor	color_1;
    XColor	color_2;

    source = XCreateBitmapFromData  (dpy, image_window, 
	    circle_bits, circle_width, circle_height);
    mask = XCreateBitmapFromData (dpy, image_window, 
	    circle_mask_bits, circle_width, circle_height);

    color_1.pixel = WhitePixel (dpy, screen);
    color_1.red   = 0xffff;
    color_1.green = 0xffff;
    color_1.blue  = 0xffff;
    color_1.flags = DoRed | DoGreen | DoBlue;

    color_1.pixel = BlackPixel (dpy, screen);
    color_2.red   = 0;
    color_2.green = 0;
    color_2.blue  = 0;
    color_2.flags = DoRed | DoGreen | DoBlue;

    circle_cursor = XCreatePixmapCursor (dpy, source, mask,
	    &color_1, &color_2, circle_x_hot, circle_y_hot);

    XFreePixmap (dpy, source);
    XFreePixmap (dpy, mask);

    }

if (circle_cursor != NULL) XDefineCursor (dpy, image_window, circle_cursor);

XFlush (dpy);

}

/*
 * Map a scanline through the dither matrix.
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	s:		Skip between pixels in original image.
 * 	y:		Y position of row (necessary for dither)
 *	line:		Pointer to output buffer for dithered color data.
 */

#define DMAP(v,x,y)	(divN[v] + (modN[v]>dm16[x][y] ? 1 : 0) )
#define NDMAP(v)	(divN[v])

static
void
map_scanline_generic (rgb, ncolors, given_width, stride, y, image)

unsigned char	*rgb[3];
int		ncolors;
int		given_width;
int		stride;
int		y;
XImage		*image;

{
register int	x;
register int	col;
register int	row;
register unsigned char 	* r;
register long	pixel;
register int	width = given_width;
register int	x_start;
register int	x_delta;

row = y % 16;
col = 0;
r = rgb[0];

if (IS_TRUE (left_pixel_first)) {
    x_start = 0;
    x_delta = 1;
    }
else {
    x_start = width - 1;
    x_delta = -1;
    }

if (ncolors == 1) {

    if (IS_TRUE(dither_image) ) {
	for (x = x_start; 
		--width >= 0; 
		r += stride, col = ((col + 1) & 15), x += x_delta) {
	    pixel = pixel_table [DMAP(*r, col, row) ];
	    if ( pixel ) pixel = 0xffffffff;
	    XPutPixel (image, x, y, pixel);
	    pixel = XGetPixel( image, x, y );
	    pixel = pixel;
	    }
	}

    else {
	for (x = x_start; 
		--width >= 0; 
		r += stride, col = ((col + 1) & 15), x += x_delta) {
	    pixel = pixel_table [NDMAP(*r)];
	    XPutPixel (image, x, y, pixel);
	    }
	}


    }			/* end monochrome display			*/

else {			/* Color display				*/
    register unsigned char *g;
    register unsigned char *b;

    g = b = rgb[1]; 
    if (ncolors >= 3) b = rgb[2];

    if (IS_TRUE (dither_image) ) {
	for (x = x_start; --width >= 0; 
		r += stride, g += stride, b += stride, 
		col = ((col + 1) & 15), x += x_delta ) {

	    if (pixel_table != NULL) {
		pixel = pixel_table [DMAP(in_cmap[0][*r], col, row) +
			DMAP(in_cmap[1][*g], col, row) * levels +
			DMAP(in_cmap[2][*b], col, row) * levels_squared];
		}
	    else {
		pixel = pixel_base
		    | ( (DMAP(in_cmap[0][*r], col, row) << red_shift  ) & red_mask )
		    | ( (DMAP(in_cmap[0][*g], col, row) << green_shift) & green_mask )
		    | ( (DMAP(in_cmap[0][*b], col, row) << blue_shift ) & blue_mask )
		    ;
		}
	    XPutPixel (image, x, y, pixel);
	    }

	}

    else {

	for (x = x_start; --width >= 0; 
		r += stride, g += stride, b += stride, x += x_delta ) {
	    if (pixel_table != NULL) {
		pixel = pixel_table [ NDMAP(in_cmap[0][*r])
		    + NDMAP(in_cmap[1][*g]) * levels
		    + NDMAP(in_cmap[2][*b]) * levels_squared ];
		}
	    else {
		pixel = pixel_base
		    | ( (NDMAP(in_cmap[0][*r]) << red_shift  ) & red_mask )
		    | ( (NDMAP(in_cmap[0][*g]) << green_shift) & green_mask )
		    | ( (NDMAP(in_cmap[0][*b]) << blue_shift ) & blue_mask )
		    ;
		}
	    XPutPixel (image, x, y, pixel);
	    }

	}
    
    }			/* Color display				*/

}

/*
 * map_l_1_d_t_8
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	s:		Skip between pixels in original image.
 * 	y:		Y position of row (necessary for dither)
 *	line:		Pointer to output buffer for dithered color data.
 */

static
void
map_l_1_d_t_8 (rgb, ncolors, given_width, stride, y, image)

unsigned char	*rgb[3];
int		ncolors;
int		given_width;
int		stride;
int		y;
XImage		*image;

{
register int	col;
register int	row;
register unsigned char 	* r;
register unsigned char	* pixel_ptr;
register int	width = given_width;

row = y % 16;
col = 0;
r = rgb[0];

pixel_ptr = ((unsigned char *) image->data) + y * image->bytes_per_line;

while (--width >= 0) {

    *pixel_ptr++ = pixel_table [DMAP(*r, col, row) ];

    r += stride;
    col = ((col + 1) & 15);

    }

}

/*
 * map_l_1_nd_t_8
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	s:		Skip between pixels in original image.
 * 	y:		Y position of row (necessary for dither)
 *	line:		Pointer to output buffer for dithered color data.
 */

static
void
map_l_1_nd_t_8 (rgb, ncolors, given_width, stride, y, image)

unsigned char	*rgb[3];
int		ncolors;
int		given_width;
int		stride;
int		y;
XImage		*image;

{
register unsigned char 	* r;
register unsigned char	* pixel_ptr;
register int	width = given_width;

r = rgb[0];

pixel_ptr = ((unsigned char *) image->data) + y * image->bytes_per_line;

while (--width >= 0) {

    *pixel_ptr++ = pixel_table [ NDMAP(*r) ];

    r += stride;

    }

}

/*
 * map_l_23_d_t_32
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	s:		Skip between pixels in original image.
 * 	y:		Y position of row (necessary for dither)
 *	line:		Pointer to output buffer for dithered color data.
 */

static
void
map_l_23_d_t_32 (rgb, ncolors, given_width, stride, y, image)

unsigned char	*rgb[3];
int		ncolors;
int		given_width;
int		stride;
int		y;
XImage		*image;

{
register int	x;
register int	col;
register int	row;
register unsigned char 	* r;
register unsigned char *g;
register unsigned char *b;
register unsigned long	* pixel_ptr;
register int	width = given_width;

row = y % 16;
col = 0;
r = rgb[0];

g = b = rgb[1]; 
if (ncolors >= 3) b = rgb[2];

pixel_ptr = (unsigned long *) ( image->data + y * image->bytes_per_line );

while (--width >= 0) {

    *pixel_ptr++ = pixel_table [DMAP(in_cmap[0][*r], col, row) +
	    DMAP(in_cmap[1][*g], col, row) * levels +
	    DMAP(in_cmap[2][*b], col, row) * levels_squared];

    r += stride;
    g += stride;
    b += stride;
    col = ((col + 1) & 15);

    }

}

/*
 * map_l_23_d_nt_32
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	s:		Skip between pixels in original image.
 * 	y:		Y position of row (necessary for dither)
 *	line:		Pointer to output buffer for dithered color data.
 */

static
void
map_l_23_d_nt_32 (rgb, ncolors, given_width, stride, y, image)

unsigned char	*rgb[3];
int		ncolors;
int		given_width;
int		stride;
int		y;
XImage		*image;

{
register int	x;
register int	col;
register int	row;
register unsigned char 	* r;
register unsigned char *g;
register unsigned char *b;
register unsigned long *pixel_ptr;
register int	width = given_width;

row = y % 16;
col = 0;
r = rgb[0];
g = rgb[1]; 
if (ncolors >= 3) b = rgb[2];
else b = rgb[1];

pixel_ptr = (unsigned long *) ( image->data + y * image->bytes_per_line );

while (--width >= 0) {

    *pixel_ptr++ = pixel_base
	    | ( (DMAP(in_cmap[0][*r], col, row) << red_shift  ) & red_mask )
	    | ( (DMAP(in_cmap[0][*g], col, row) << green_shift) & green_mask )
	    | ( (DMAP(in_cmap[0][*b], col, row) << blue_shift ) & blue_mask )
	    ;

    r += stride;
    g += stride;
    b += stride;
    col = ((col + 1) & 15);

    }

}

/*
 * map_l_23_nd_nt_32
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	s:		Skip between pixels in original image.
 * 	y:		Y position of row (necessary for dither)
 *	line:		Pointer to output buffer for dithered color data.
 */

static
void
map_l_23_nd_nt_32 (rgb, ncolors, given_width, stride, y, image)

unsigned char	*rgb[3];
int		ncolors;
int		given_width;
int		stride;
int		y;
XImage		*image;

{
register int	x;
register unsigned char 	* r;
register unsigned char *g;
register unsigned char *b;
register unsigned long *pixel_ptr;
register int	width = given_width;

r = rgb[0];
g = rgb[1]; 
if (ncolors >= 3) b = rgb[2];
else b = rgb[1];

pixel_ptr = (unsigned long *) ( image->data + y * image->bytes_per_line );

while (--width >= 0) {

    *pixel_ptr++ = pixel_base
	    | ( (NDMAP(in_cmap[0][*r]) << red_shift  ) & red_mask )
	    | ( (NDMAP(in_cmap[0][*g]) << green_shift) & green_mask )
	    | ( (NDMAP(in_cmap[0][*b]) << blue_shift ) & blue_mask )
	    ;

    r += stride;
    g += stride;
    b += stride;

    }

}

/*****************************************************************
 * TAG( map_rgb_to_bw )
 * 
 * Convert RGB to black and white through NTSC transform, but map
 * RGB through a color map first.
 * Inputs:
 * 	red_row, green_row, blue_row:	Given RGB pixel data.
 *	map:		Array[3] of pointers to pixel arrays,
 *			representing color map.
 *	rowlen:		Number of pixels in the rows.
 * Outputs:
 * 	bw_row:	    Output B&W data.  May coincide with one of the
 *		    inputs.
 * Algorithm:
 * 	BW = .35*map[0][R] + .55*map[1][G] + .10*map[2][B]
 */

static
map_rgb_to_bw( rows, ncolors, bw_row, map, rowlen )

rle_pixel 	**rows;
register int	ncolors;
register rle_pixel	*bw_row;
rle_pixel	**map;
register int	rowlen;

{
register int x;
register rle_pixel *red;
register rle_pixel *green;
register rle_pixel *blue;

if (ncolors < 1) {
    fprintf (stderr, "%s: map_rgb_to_bw given %d colors\n", 
	    PROGRAM_NAME, ncolors);
    exit (EXIT_STATUS_FAILURE);
    }

switch (ncolors) {

    case 1:
	red = rows[0];
	while (--rowlen >= 0) {
	    *bw_row++ = map[0][*red++];
	    }
	break;

    case 2:
	red = rows[0];
	green = rows[1];
	while (--rowlen >= 0) {
	    *bw_row++ = ( 35 * map[0][*red++] + 55 * map[1][*green++] ) / 100;
	    }
	break;

    default:
    case 3:
	red = rows[0];
	green = rows[1];
	blue = rows[2];
	while (--rowlen >= 0) {
	    *bw_row++ = ( 35 * map[0][*red++] + 55 * map[1][*green++] +
		10 * map[2][*blue++] ) / 100;
	    }
	break;
    }

}

/*
 * This routine builds the color map (for X window system, details may
 * differ in your system.  Note particularly the two level color
 * mapping:  X will give back 216 color map entries, but there is no
 * guarantee they are sequential.  The dithering code above wants to
 * compute a number in 0 - 215, so we must map from this to the X color
 * map entry id.
 */

/*
 * Initialize the 8 bit color map.  Use gamma corrected map.
 */

#define		IS_BINARY	1
#define		IS_NOT_BINARY	0

#define		IS_MONOCHROME	2
#define		IS_NOT_MONOCHROME	0

#define		WANT_READ_WRITE	4
#define		WANT_READ_ONLY	0

static
init_color()

{
register int		type;
register BOOLEAN	done;

done = BOOL_FALSE;

while (IS_FALSE (done) ) {

    if (IS_TRUE (binary_image) ) type = IS_BINARY;
    else type = IS_NOT_BINARY;

    if (IS_TRUE (monochrome_image) ) type |= IS_MONOCHROME;
    else type |= IS_NOT_MONOCHROME;

    if (IS_TRUE (mutable_color_map) ) type |= WANT_READ_WRITE;
    else type |= WANT_READ_ONLY;

    switch (type) {

	case IS_NOT_BINARY | IS_NOT_MONOCHROME | WANT_READ_WRITE:

	    if (IS_TRUE (separate_colors) ) {
		done = init_separate_color_rw ();
		}
	    else {
		done = init_color_rw ();
		}
	    if ( IS_FALSE (done) ) {
		fprintf( stderr,
			"%s: Sorry, not enough color map entries are available.\
\nWill make black & white image instead.\n", PROGRAM_NAME);
		monochrome_image = BOOL_TRUE;
		}
	    break;

	case IS_NOT_BINARY | IS_NOT_MONOCHROME | WANT_READ_ONLY:

	    if (IS_TRUE (separate_colors) ) {
		done = init_separate_color_ro ();
		}
	    else {
		done = init_color_ro ();
		}
    
	    if ( IS_FALSE (done) ) {
		fprintf( stderr,
			"%s: Sorry, not enough color map entries are available.\
\nWill make black & white image instead.\n", PROGRAM_NAME);
		monochrome_image = BOOL_TRUE;
		}
	    break;

	case IS_NOT_BINARY | IS_MONOCHROME | WANT_READ_WRITE:
    
	    done = init_mono_rw ();
	    if ( IS_FALSE (done) ) binary_image = BOOL_TRUE;
	    break;

	case IS_NOT_BINARY | IS_MONOCHROME | WANT_READ_ONLY:
    
	    done = init_mono_ro ();
	    if ( IS_FALSE (done) ) binary_image = BOOL_TRUE;
	    break;

	case IS_BINARY | IS_MONOCHROME | WANT_READ_ONLY:
    
	    /* All we care about, really, is the magic square */
	
	    make_square( 255.0, divN, modN, dm16 );
	    pixel_table = (X_PIXEL *) malloc (2 * sizeof (X_PIXEL));
	    if (pixel_table == NULL) {
		perror ("malloc pixel_table (2)");
		exit (EXIT_STATUS_FAILURE);
		}
	    pixel_table[0] = BlackPixel (dpy, screen);
	    pixel_table[1] = WhitePixel (dpy, screen);
	    levels = 2;
	    levels_squared = 4;
	    done = BOOL_TRUE;    
	    break;

	default:
    
	    fprintf (stderr, "Unknown type in init_colors: %d\n", type);
	    exit (EXIT_STATUS_FAILURE);

	}

    }

#ifdef VERBOSE
fprintf (stderr, "Created color map with %d entries", levels);
if (IS_FALSE (monochrome_image) ) 
    fprintf (stderr, " per color, %d total\n", levels * levels * levels);
else putc ('\n', stderr);
#endif

if (pixel_table != NULL) {
    black_pixel = pixel_table[0];
    if (IS_TRUE(monochrome_image)) {
	white_pixel = pixel_table [levels - 1];
	}
    else {
	white_pixel = pixel_table [levels * levels * levels - 1];
	}
    }
else {
    black_pixel = pixel_base;
    white_pixel = pixel_base
	    | ( ( (levels - 1) << red_shift  ) & red_mask )
	    | ( ( (levels - 1) << green_shift) & green_mask )
	    | ( ( (levels - 1) << blue_shift ) & blue_mask );
    }


}

static
BOOLEAN
init_separate_color_rw ()

{
register int		number_levels;
register int		log2_number_levels;
register int		* map;
register XColor		* color_defs;
register int		total_levels;
register int		i;

/*
 * use XAllocColorPlanes to allocate all the cells we need --
 * this makes it simple for me.
 */

pixel_table = NULL;
map = NULL;
color_defs = NULL;

for (log2_number_levels = log2_levels; 
	log2_number_levels >= 1; 
	log2_number_levels-- ) {

    number_levels = 1 << log2_number_levels;
    total_levels =  1 << (log2_number_levels * 3);

    if (map == NULL) {
	map = (int *) malloc (number_levels * sizeof (int) );
	if (map == NULL) continue;
	}

    if (color_defs == NULL) {
	color_defs = (XColor *) malloc (number_levels * sizeof (XColor) );
	if (color_defs == NULL) continue;
	}
	
    if ( XAllocColorPlanes (dpy, color_map, 1, &pixel_base, 1,
	log2_number_levels, log2_number_levels, log2_number_levels, 
	&red_mask,   &green_mask, &blue_mask) == 0) continue;

    if (log2_number_levels == 8) dither_image = BOOL_FALSE;
    else dither_image = dither_flag;

    bwdithermap ( number_levels, display_gamma, map, divN, modN, dm16 );

    red_shift   = shift_match_right (red_mask);
    green_shift = shift_match_right (green_mask);
    blue_shift  = shift_match_right (blue_mask);
   
    /* 
     * Set up the color map entries.
     */

    for (i = 0; i < number_levels; i++) {
	color_defs[i].pixel = pixel_base
		| ( (i << red_shift) & red_mask)
		| ( (i << green_shift) & green_mask)
		| ( (i << blue_shift) & blue_mask)
		;

	color_defs[i].red   = map[i] << 8;
	color_defs[i].green = map[i] << 8;
	color_defs[i].blue  = map[i] << 8;

	color_defs[i].flags = DoRed | DoGreen | DoBlue;
	}

    XStoreColors (dpy, color_map, color_defs, number_levels);

    if (levels > number_levels) {
	levels = number_levels;
	levels_squared = number_levels * number_levels;
	log2_levels = log2_number_levels;
	}

    free (map);
    free (color_defs);

    return (BOOL_TRUE);

    }

return (BOOL_FALSE);

}

static
BOOLEAN
init_color_rw ()

{
register int		number_levels;
register int		* map;
register XColor		* color_defs;
register int		total_levels;
register int		red_index;
register int		green_index;
register int		blue_index;
register int		i;

pixel_table = NULL;
map = NULL;
color_defs = NULL;

for (number_levels = levels; 
	number_levels > 1; 
	number_levels-- ) {

    total_levels =  number_levels * number_levels * number_levels;

    if (pixel_table == NULL) {
	pixel_table = (X_PIXEL *) malloc (total_levels * sizeof (X_PIXEL) );
	if (pixel_table == NULL) continue;
	}

    if (map == NULL) {
	map = (int *) malloc (number_levels * sizeof (int) );
	if (map == NULL) continue;
	}

    if (color_defs == NULL) {
	color_defs = (XColor *) malloc (total_levels * sizeof (XColor) );
	if (color_defs == NULL) continue;
	}
	
    if ( XAllocColorCells (dpy, color_map, 0, NULL, 0,
	pixel_table, total_levels) == 0) continue;

    if (number_levels == 256) dither_image = BOOL_FALSE;
    else dither_image = dither_flag;

    bwdithermap ( number_levels, display_gamma, map, divN, modN, dm16 );

    /* 
     * Set up the color map entries.
     */


    red_index = 0;
    green_index = 0;
    blue_index = 0;

    for (i = 0; i < total_levels; i++) {
	color_defs[i].pixel = pixel_table[i];
	color_defs[i].red   = map[red_index] << 8;
	color_defs[i].green = map[green_index] << 8;
	color_defs[i].blue  = map[blue_index] << 8;

	color_defs[i].flags = DoRed | DoGreen | DoBlue;

	if (++red_index >= number_levels) {
	    if (++green_index >= number_levels) {
		++blue_index;
		green_index = 0;
		}
	    red_index = 0;
	    }

	}

    XStoreColors (dpy, color_map, color_defs, total_levels);

    if (levels > number_levels) {
	levels = number_levels;
	levels_squared = number_levels * number_levels;
	for (log2_levels = 1, i = 2; i < levels; i <<= 1, log2_levels++) {
	    /* do nothing */
	    }
	}

    free (map);
    free (color_defs);

    return (BOOL_TRUE);

    }

return (BOOL_FALSE);

}

static
BOOLEAN
init_separate_color_ro ()

{
register int		number_levels;
register int		log2_number_levels;
register int		* map;
register int		total_levels;
register int		i;

pixel_table = NULL;
map = NULL;

for (log2_number_levels = log2_levels; 
	log2_number_levels >= 1; 
	log2_number_levels-- ) {

    number_levels = 1 << log2_number_levels;
    total_levels = 1 << (3 * log2_number_levels);

    if (map == NULL) {
	map = (int *) malloc (number_levels * sizeof (int) );
	if (map == NULL) continue;
	}
	
    if (number_levels == 256) dither_image = BOOL_FALSE;
    else dither_image = dither_flag;

    bwdithermap ( number_levels, display_gamma, map, divN, modN, dm16 );

    red_mask   = visual->red_mask;
    green_mask = visual->green_mask;
    blue_mask  = visual->blue_mask;

    red_shift   = shift_match_left (red_mask, log2_number_levels);
    green_shift = shift_match_left (green_mask, log2_number_levels);
    blue_shift  = shift_match_left (blue_mask, log2_number_levels);
   
    if (levels > number_levels) {
	levels = number_levels;
	levels_squared = number_levels * number_levels;
	log2_levels = log2_number_levels;
	}

    free (map);

    return (BOOL_TRUE);

    }

return (BOOL_FALSE);

}

static
BOOLEAN
init_color_ro ()

{
register int		number_levels;
register int		total_levels;
register XColor		color_def;
register int		red_index;
register int		green_index;
register int		blue_index;
register int		* map;
register int		i;

pixel_table = NULL;
map = NULL;

for (number_levels = levels; 
	number_levels >= 2; 
	number_levels = 
	    (number_levels > 16) ? number_levels / 2 : number_levels - 2
	) {

    total_levels = number_levels * number_levels * number_levels;

    if (pixel_table == NULL) {
	pixel_table = (X_PIXEL *) malloc (total_levels * sizeof (X_PIXEL) );
	if (pixel_table == NULL) continue;
	}

    if (map == NULL) {
	map = (int *) malloc (number_levels * sizeof (int) );
	if (map == NULL) continue;
	}

    if (number_levels == 256) dither_image = BOOL_FALSE;
    else dither_image = dither_flag;

    bwdithermap ( number_levels, display_gamma, map, divN, modN, dm16 );

    /* Get a color map entry for each color.  Assume enough exist! */

    red_index = green_index = blue_index = 0;

    for ( i = 0; i < total_levels; i++ ) {
	color_def.red   = map[red_index] << 8;
	color_def.green = map[green_index] << 8;
	color_def.blue  = map[blue_index] << 8;

	if ( XAllocColor (dpy, color_map, &color_def ) == 0 ) {
	    break;
	    }

	pixel_table[i] = color_def.pixel;
   
	if (++red_index >= number_levels) {
	    if (++green_index >= number_levels) {
		++blue_index;
		green_index = 0;
		}
	    red_index = 0;
	    }

	}
   
    /* Check if the colors are available */
   
    if ( i < total_levels) {

	/* Free the colors already obtained */
	XFreeColors (dpy, color_map, pixel_table, i, 0 );
	continue;		/* adjust level & repeat		*/
	}

    levels = number_levels;
    levels_squared = number_levels * number_levels;

    for (log2_levels = 1, i = 2; i < levels; i <<= 1, log2_levels++) {
	/* do nothing */
	}


    free (map);

    return (BOOL_TRUE);

    }

return (BOOL_FALSE);

}

static
BOOLEAN
init_mono_rw ()

{
register int		number_levels;
register int		log2_number_levels;
register int		red_shift;
register int		green_shift;
register int		blue_shift;
register int		* map;
register XColor		* color_defs;
register int		i;

pixel_table = NULL;
map = NULL;
color_defs = NULL;

for (number_levels = levels; 
	number_levels >= 2; 
	number_levels = 
		( (number_levels > 16) ? number_levels / 2 : number_levels - 1)
	) {

    if (pixel_table == NULL) {
	pixel_table = (X_PIXEL *) malloc (number_levels * sizeof (X_PIXEL) );
	if (pixel_table == NULL) continue;
	}

    if (map == NULL) {
	map = (int *) malloc (number_levels * sizeof (int) );
	if (map == NULL) continue;
	}

    if (color_defs == NULL) {
	color_defs = (XColor *) malloc (number_levels * sizeof (XColor) );
	if (color_defs == NULL) continue;
	}


    if (XAllocColorCells (dpy, color_map, 0, NULL, 0, 
	    pixel_table, number_levels) == 0) continue;

    if (number_levels == 256) dither_image = BOOL_FALSE;
    else dither_image = dither_flag;

    bwdithermap ( number_levels, display_gamma, map, divN, modN, dm16 );

    /* 
     * Set up the color map entries
     */

    for ( i = 0; i < number_levels; i++ ) {
	color_defs[i].red = map[i] << 8;
	color_defs[i].green = map[i] << 8;
	color_defs[i].blue = map[i] << 8;
	color_defs[i].pixel = pixel_table[i];
	color_defs[i].flags = DoRed | DoGreen | DoBlue;
	}

    XStoreColors (dpy, color_map, color_defs, number_levels);

    levels = number_levels;
    levels_squared = number_levels * number_levels;

    for (log2_levels = 1, i = 2; i < levels; i <<= 1, log2_levels++) {
	/* do nothing */
	}

    free (map);
    free (color_defs);

    return (BOOL_TRUE);

    }

}

static
BOOLEAN
init_mono_ro ()

{
register int		number_levels;
register XColor		color_def;
register int		* map;
register int		i;

pixel_table = NULL;
map = NULL;

for (number_levels = levels; 
	number_levels >= 2; 
	number_levels = 
	    (number_levels > 16) ? number_levels / 2 : number_levels - 2
	) {

    if (pixel_table == NULL) {
	pixel_table = (X_PIXEL *) malloc (number_levels * sizeof (X_PIXEL) );
	if (pixel_table == NULL) continue;
	}

    if (map == NULL) {
	map = (int *) malloc (number_levels * sizeof (int) );
	if (map == NULL) continue;
	}

    if (number_levels == 256) dither_image = BOOL_FALSE;
    else dither_image = dither_flag;

    bwdithermap ( number_levels, display_gamma, map, divN, modN, dm16 );

    /* Get a color map entry for each color.  Assume enough exist! */

    for ( i = 0; i < number_levels; i++ ) {
	color_def.red   = map[i] << 8;
	color_def.green = map[i] << 8;
	color_def.blue  = map[i] << 8;

	if ( XAllocColor (dpy, color_map, &color_def ) == 0 ) {
	    break;
	    }

	pixel_table[i] = color_def.pixel;
	}
   
    /* Check if the colors are available */
   
    if ( i < number_levels) {

	/* Free the colors already obtained */

	XFreeColors (dpy, color_map, pixel_table, i, 0 );
	continue;		/* adjust level & repeat		*/
	}

    levels = number_levels;
    levels_squared = number_levels * number_levels;

    for (log2_levels = 1, i = 2; i < levels; i <<= 1, log2_levels++) {
	/* do nothing */
	}


    free (map);

    return (BOOL_TRUE);

    }

return (BOOL_FALSE);

}

static
int
shift_match_left (mask, high_bit_index)

X_PIXEL		mask;
int			high_bit_index;

{
register int		shift;
register X_PIXEL	high_bit;

if (mask == 0) return (0);

high_bit = 0x80000000;

for (shift = (32 - high_bit_index); (high_bit & mask) == 0; shift--) {
    high_bit >>= 1;
    }

return (shift);

}

static
int
shift_match_right (mask)

X_PIXEL		mask;

{
register int		shift;
register X_PIXEL	low_bit;

if (mask == 0) return (0);

low_bit = 1;

for (shift = 0; (low_bit & mask) == 0; shift++) {
    low_bit <<= 1;
    }

return (shift);

}

#define BINARY_TABLE_INDEX	0
#define MONOCHROME_TABLE_INDEX	1
#define COLOR_TABLE_INDEX	2

static X_VISUAL_CLASS desired_class_table [][6] = {
    { StaticGray, GrayScale, PseudoColor, DirectColor, TrueColor, StaticColor},
    { GrayScale, PseudoColor, StaticGray, DirectColor, TrueColor, StaticColor},
    { DirectColor, PseudoColor, TrueColor, StaticColor, GrayScale, StaticGray}
    };

static
find_appropriate_visual (ncolors)

int		ncolors;

{
static XVisualInfo	* visual_info = NULL;
static int		num_visuals = 0;
register XVisualInfo	* vi;
int			desired_class;
int			desired_depth;
int			desired_depth_delta;
register int		i;
register XVisualInfo	* found_vi;
XVisualInfo		* XGetVisualInfo ();
X_VISUAL_CLASS		vt;

if (visual_info == NULL) {

    visual_info = XGetVisualInfo (dpy, VisualNoMask, NULL, &num_visuals);
    if (visual_info == NULL || num_visuals == 0) {
	fprintf (stderr, "XGetVisualInfo failed\n");
	exit (EXIT_STATUS_FAILURE);
	}	    
    }

levels = specified_levels;
if (levels == 0) levels = DEFAULT_LEVELS;	/* default starting point */
if (levels > MAXIMUM_LEVELS) levels = MAXIMUM_LEVELS;
levels_squared = levels * levels;

desired_depth = 1;

for (i = 2; i < levels; i <<= 1) {
    desired_depth++;
    }

if ( IS_TRUE (binary_flag) ) binary_image = BOOL_TRUE;
if ( IS_TRUE (monochrome_flag) ) monochrome_image = BOOL_TRUE;

if ( IS_TRUE (binary_image) ) {
    desired_class = BINARY_TABLE_INDEX;
    desired_depth = 1;
    desired_depth_delta = 1;
    }
else if ( IS_TRUE (monochrome_image) ) {
    desired_class = MONOCHROME_TABLE_INDEX;
    desired_depth_delta = 1;
    }
else {
    desired_class = COLOR_TABLE_INDEX;
    desired_depth *= 3;			/* needed if separate colors	*/
    desired_depth_delta = 3;
    }

#if VERBOSE
{
 register char *type;

switch (visual_type) {
    case DirectColor:	type = "DirectColor";		break;
    case PseudoColor:	type = "PseudoColor";		break;
    case TrueColor:	type = "TrueColor";		break;
    case StaticColor:	type = "StaticColor";		break;
    case GrayScale:	type = "GrayScale";		break;
    case StaticGray:	type = "StaticGray";		break;
    default: 		type = "any";		break;
    }

fprintf (stderr, "Searching for %s visual with desired depth >= %d\n", 
	type, desired_depth);
}
#endif

found_vi = NULL;

/*
 * find visual such that:
 *
 *	1. desired_depth is as large as possible (up to true desired depth)
 *	2. visual depth is the smallest of those supported >= desired_depth
 *	3. visual class is the `most desired' for the image type
 */

if ( (int) visual_type < 0) {
    for ( ; desired_depth >= 0; desired_depth -= desired_depth_delta) {

	for (i = 0; i < 6; i++) {  /* search for class and depth	*/

	    vt = desired_class_table[desired_class][i];

	    for (vi = visual_info; vi < visual_info + num_visuals; vi++) {
		if (vi->class == vt && vi->depth >= desired_depth) {
		    if (found_vi == NULL || found_vi->depth > vi->depth) {
			found_vi = vi;
			}
		    if (vi->depth == desired_depth) break;
		    }

		}	/* end for (vi = visual_info ...		*/

	    }		/* end for (i = 0; i < 6; ...			*/

	if (found_vi != NULL) break;

	}		/* end for ( ; desired_depth >= 1; ... )	*/

    }

else {

    for ( ; desired_depth >= 0; desired_depth -= desired_depth_delta) {

	for (vi = visual_info; vi < visual_info + num_visuals; vi++) {
	    if (vi->class == visual_type && vi->depth >= desired_depth) {
		if (found_vi == NULL || found_vi->depth > vi->depth) {
		    found_vi = vi;
		    }
		if (vi->depth == desired_depth) break;

		}

	    }		/* end for (vi = visual_info ...		*/

	if (found_vi != NULL) break;
	}		/* end for ( ; desired_depth >= 1; ... )	*/

    }


if (found_vi == NULL) {
    fprintf (stderr, "Could not find appropriate visual type\n");
    exit (EXIT_STATUS_FAILURE);
    }

screen = found_vi->screen;

root_window = RootWindow (dpy, screen);

visual_depth = found_vi->depth;
visual = found_vi->visual;

color_map = XCreateColormap (dpy, root_window, visual, AllocNone );
if (color_map == NULL) {
    fprintf (stderr, "Could not create color map for visual\n");
    exit (EXIT_STATUS_FAILURE);
    }

XInstallColormap (dpy, color_map);

if (found_vi->depth == 1) {

    binary_image = BOOL_TRUE;
    monochrome_image = BOOL_TRUE;
    separate_colors = BOOL_FALSE;
    mutable_color_map = BOOL_FALSE;

    levels = 2;
    levels_squared = 4;		/* not used, but ...			*/
    log2_levels = 1;
    bitmap_pad = BitmapPad (dpy);

    }

else {

    binary_image = BOOL_FALSE;
    if (found_vi->class == GrayScale || found_vi->class == StaticGray) {
	monochrome_image = BOOL_TRUE;
	desired_depth_delta = 1;
	}

    if (found_vi->class == DirectColor || found_vi->class == PseudoColor
	    || found_vi->class == GrayScale) {
	mutable_color_map = BOOL_TRUE;
	}
    else mutable_color_map = BOOL_FALSE;

    if (found_vi->class == DirectColor || found_vi->class == TrueColor) {
	separate_colors = BOOL_TRUE;

	i = 1 << (found_vi->depth / desired_depth_delta);
    
	if ( levels > i) {
	    levels = i;
	    levels_squared = i * i;
	    }
	}

    else {
	separate_colors = BOOL_FALSE;

	if (desired_depth_delta == 1) {
	    i = 1 << found_vi->depth;
	    }
	else {
	    for (i = 1; i * i * i < (1 << found_vi->depth); i++) {}
	    i--;
	    }
    
	if ( levels > i) {
	    levels = i;
	    levels_squared = i * i;
	    }
	}

    for (log2_levels = 1, i = 2;
	    i < levels;
	    i <<= 1, log2_levels++) {
	/* do nothing */
	}

    bitmap_pad = BitmapPad (dpy);

    }

image_attribute_mask = CWBackPixel | CWBorderPixel | CWColormap;
#if WANT_BACKING_STORE
image_attribute_mask |= CWBackingStore;	/* attempt to minimize host redrawing */
image_attributes.backing_store = Always;
#endif

image_attributes.background_pixel = BlackPixel (dpy, screen);
image_attributes.border_pixel = WhitePixel (dpy, screen);
image_attributes.colormap = color_map;

#if USE_WINDOW_FOR_ICON
icon_attribute_mask = CWBackPixel | CWBorderPixel | CWColormap;
# if WANT_BACKING_STORE
icon_attribute_mask |= CWBackingStore;	/* attempt to minimize host redrawing */
icon_attributes.backing_store = Always;
# endif

icon_attributes.background_pixel = BlackPixel (dpy, screen);
icon_attributes.border_pixel = WhitePixel (dpy, screen);
icon_attributes.colormap = color_map;
#endif

#if VERBOSE
{ register char *type;

switch (visual->class) {
    case DirectColor:	type = "DirectColor";		break;
    case PseudoColor:	type = "PseudoColor";		break;
    case TrueColor:	type = "TrueColor";		break;
    case StaticColor:	type = "StaticColor";		break;
    case GrayScale:	type = "GrayScale";		break;
    case StaticGray:	type = "StaticGray";		break;
    default: 		type = "Unknown";		break;
    }

fprintf (stderr, "Visual type %s, depth %d, screen %d\n", 
	type, visual_depth, screen);

fprintf (stderr, "levels: %d, log(2) levels: %d\n", levels, log2_levels);

}
#endif
}

/* 
 * Track events & redraw image when necessary.
 */

static
update_pic (annotation_file)

char		* annotation_file;

{
int             i;
int             npix;
int             pixscan;
int             lastscan;
int             gotpix;
long            bufsize;
long		pixsize;
XEvent		event;
register int	offset;
ANNOTATION	* closest;
ANNOTATION	dummy_note;
char		line [80];
int		length;
int		image_x;
int		image_y;
int		action;
BOOLEAN		do_annotation;
int		timer;
int		x;
int		y;
int		width;
int		height;

dummy_note.next = NULL;
dummy_note.string = line;

if (annotation_wanted && annotation_file != NULL) do_annotation = BOOL_TRUE;
else do_annotation = BOOL_FALSE;

note_gc = XCreateGC (dpy, image_window, 0, NULL);
XCopyGC (dpy, image_gc, ~0, note_gc);
XSetFunction (dpy, note_gc, GXinvert);
XSetLineAttributes (dpy, note_gc, 2, LineSolid, CapNotLast, JoinBevel);

if (IS_TRUE (do_annotation)) {
    setup_annotation (annotation_file, 
	    (annotation_output_file == NULL) ? NULL 
			: (char *) handle_annotation);

    }


/*
 * Basic event loop:  handle expose events on window & icon, and exit on
 * (shifted) mouse button event. 
 */

timer = 0;

while (1) {

    if (IS_TRUE (repeat_annotation)) {

	while (--timer >= 0) {
	    if (XPending (dpy) > 0) break;
	    sleep (1);
	    }

	if (timer <= 0) {
	    timer = time_between_updates;
	    if (current_note != NULL) erase_note (current_note);
	    current_note = find_next_annotation (current_note);
	    if (current_note != NULL) draw_note (current_note);
	    continue;
	    }

	}

    XNextEvent(dpy, &event);

    switch (event.type) {

      case ButtonPress:
	continue;

      case ButtonRelease:

	i = event.xbutton.button - Button1;
	if (i < 0 || i > COUNT_OF(button_action) ) i = ACTION_DEFAULT;
	else if ( event.xbutton.state & ShiftMask ) action = shifted_action[i];
	else action = button_action[i];

#if USE_WINDOW_FOR_ICON
	if (event.xbutton.window == icon_window 
		&&  (
		    action == ACTION_ANNOTATE
		    || action == ACTION_INPUT
		    )
		) {
	    action = ACTION_ICONIFY;
	    }
#endif

	switch (action) {

	  case ACTION_EXIT:
	    break; 			/* exit */

	  case ACTION_ICONIFY:
#if USE_WINDOW_FOR_ICON
	    if (event.xbutton.window == icon_window) {
		XUnmapWindow (dpy, icon_window);
		XMapWindow (dpy, image_window);
		}
	    else {
		XUnmapWindow (dpy, image_window);
		XMapWindow (dpy, icon_window);
		}
#endif
	    continue;

	  case ACTION_ANNOTATE:
	  case ACTION_INPUT:
	  default:

	    CONV_X_TO_IMAGE (event.xbutton.x, event.xbutton.y, 
		    image_x, image_y);
    
	    if (IS_FALSE (do_annotation) ) {
		sprintf (line, "[%d,%d]", image_x, image_y);
		}

	    else {

		closest = find_closest_annotation (image_x, image_y, 
			MAX_DISTANCE);

		if (closest != NULL) {
		    if (current_note != NULL) erase_note (current_note);
		    draw_note (closest);
		    current_note = closest;
		    timer = time_between_updates;
	
		    }
	    
		else {
		    sprintf (line, "[%d,%d] nothing close enough", 
			    image_x, image_y);
		    if (current_note != NULL) erase_note (current_note);
		    if (current_note != &dummy_note) {
			dummy_note.next = current_note;
			}
		    current_note = &dummy_note;
		    dummy_note.x = image_x;
		    dummy_note.y = image_y;
		    draw_note (current_note);
		    timer = time_between_updates;
		    }
		}
	    

	    if (action == ACTION_INPUT 
		    && annotation_output_file != NULL) {
    
		handle_annotation (image_x, image_y, NULL);
		XFlush (dpy);
			
		fprintf (stderr, "[%d,%d]: ", image_x, image_y);
		fgets (line, 80, stdin);
		length = strlen (line);
		if (line[length - 1] == '\n') {
		    line[--length] = '\0';
		    }
		if (length != 0) {
		    add_to_annotation (image_x, image_y, line);
		    fprintf (annotation_output_file, "[%d,%d]%s\n", 
			    image_x, image_y, line);
		    }
		}
    
	    continue;
    
	  }		/* end of switch (action)			*/
    
	break;		/* exit						*/


      case Expose:

	/*
	 * For icon exposure, just redraw whole thing - it's quick and
	 * much easier. 
	 * Since we are redrawing everything, don't bother to do
	 * anything if there are more exposure events to follow.
	 */

#if USE_WINDOW_FOR_ICON
	if (event.xexpose.window == icon_window) {
	    if (event.xexpose.count == 0) {
# if USE_PIXMAP_FOR_ICON
		XCopyArea (dpy, icon_pixmap, icon_window, icon_gc,
			0, 0, icon_width, icon_height, 0, 0);
# else
		PUT_SCANLINES (icon_image, 0, 0, icon_width, icon_height, 
			icon_window, icon_gc);
# endif
		}

	    continue;
	    }
#endif
	
	/*
	 * If window has been resized (bigger), don't bother redrawing
	 * the area outside the image. 
	 */

	x = event.xexpose.x;
	y = event.xexpose.y;
	width = event.xexpose.width;
	height = event.xexpose.height;
    
	if (x < 0) {
	    width += x;
	    x = 0;
	    }

	if (y < 0) {
	    height += y;
	    y = 0;
	    }

	if (y + height >= image_height)
	    height = image_height - y;

	/*
	 * round beginning pixel to beginning of word
	 */

	if ( IS_TRUE (binary_image) ) {
	    offset = x % bitmap_pad;
	    x -= offset;
	    width += offset;
	    }
    

	if (x + width >= image_width)
	    width = image_width - x;

	if (current_note != NULL) erase_note (current_note);
    
	PUT_SCANLINES (image_image, x, y, width, height,
		image_window, image_gc);
    
	if (current_note != NULL) draw_note (current_note);
    
	continue;

      case VisibilityNotify:

	if (event.xvisibility.window != image_window) continue;

	/* if fully obscured, wait here for partial or full visibility */

	while (event.xvisibility.window == image_window
		&& event.xvisibility.state == VisibilityFullyObscured) {
	    XNextEvent (dpy, &event);
	    }

	continue;

      default: 
	fprintf(stderr, "%s: Event type %x?\n", PROGRAM_NAME, event.type);
	break;

      }			/* end switch (event.type)			*/

    break;

    }			/* end while loop 				*/

if (annotation_output_file != NULL) fclose (annotation_output_file);

XUnmapWindow (dpy, image_window);
#if USE_WINDOW_FOR_ICON
XUnmapWindow (dpy, icon_window);
#endif

/*
 * make everything go away by exiting
 */

}

static void
erase_note (note)

ANNOTATION	    *note;

{
register int	    x;
register int	    y;

if (note == NULL) return;

XClearArea (dpy, image_window, 0, image_height,
			image_width, annotation_height, False);

CONV_IMAGE_TO_X (note->x, note->y, x, y);

XDrawArc (dpy, image_window, note_gc, x - MAX_DISTANCE/2, y - MAX_DISTANCE/2,
	MAX_DISTANCE, MAX_DISTANCE, 0, 360 * 64);

}

static void
draw_note (note)

ANNOTATION	    *note;

{
char		    line[160];
register int	    length;
register int	    width;
register int	    x;
register int	    y;
		
if (note == NULL) return;
	    
length = strlen (note->string);
width = XTextWidth (annotation_font_info, note->string, length);
x = (image_width - width) / 2;

XDrawString (dpy, image_window, image_gc, x, annotation_y, 
	note->string, length);

CONV_IMAGE_TO_X (note->x, note->y, x, y);

XDrawArc (dpy, image_window, note_gc, x - MAX_DISTANCE/2, y - MAX_DISTANCE/2,
	MAX_DISTANCE, MAX_DISTANCE, 0, 360 * 64);

}
    

static
void
handle_annotation (image_x, image_y, string)

int		image_x;
int		image_y;
char		* string;

{

if (string != NULL && string[0] != '\0') {
    fprintf (annotation_output_file, "[%d,%d]%s\n", image_x, image_y, string);
    }

}

