/* 
 * get4d.c - Put RLE images on the Iris/4D display under the window manager.
 * 
 * Author:	Russell D. Fish (Based on getmex.c .)
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Thu May 26 20:49:11 1988
 * Copyright (c) 1988, University of Utah
 * 
 */

#include <stdio.h>
#include <math.h>
#include "gl.h"
#include "device.h"
#include "svfb_global.h"
#define MAX(i,j)   ( (i) > (j) ? (i) : (j) )
#define MIN(i,j)   ( (i) < (j) ? (i) : (j) )

/* Global variables. */
long window_number;			/* Window number from MAX. */
int x_size, y_size;			/* Size of image. */
int dbg = 0;				/* Set if debug mode. */
int forkflg = 0;			/* Set if not to run in background. */
int bwflag = 0;				/* Set for greyscale output. */
int gt;					/* Whether we're on a GT. */

long *rect_image;			/* GT Image data buffer pointer. */
char *rgb_image;			/* Non-GT Image data buffer pointer. */

/*****************************************************************
 * TAG( main )
 * 
 * Usage:
 *	get4d [-f] [-D] [-w] [file]
 * Inputs:
 * 	-f:	Don't fork after putting image on screen.
 *	-D:	Debug mode: print input file as read.
 *	-w:	Black & white: reduce color images to B&W before display.
 *		Advantage is that smoother shading can be achieved.
 *	file:	Input Run Length Encoded file.  Uses stdin if not
 *		specified.
 * Outputs:
 * 	Puts image in a window on the screen.
 * Assumptions:
 * 	Input file is in RLE format.
 */

main(argc, argv)
char **argv;
{
    char *infname = NULL;
    FILE * infile = stdin;
    char vbuff[50];

    /* Handle arguments. */
    if ( scanargs( argc, argv, "% f%- w%- D%- file%s",
	&forkflg, &bwflag, &dbg, &infname ) == 0 )
	    exit( 1 );
    if ( infname != NULL )
	if ( (infile = fopen( infname, "r" )) == NULL )
	{
		perror( infname );
		exit( 1 );
	}

    /* See if we're on a GT. */
    gversion( vbuff );
    gt = strncmp( vbuff+4, "GT", 2 ) == 0;

    get_pic( infile, infname );	/* Read image and make a window for it. */
    update_pic();		/* Keep drawing the window. */
}

/* 
 * Read an image from the input file and display it.
 */
get_pic( infile, infname )
FILE * infile;
char * infname;
{
    register int i, y;
    int ncolors;
    unsigned char *scan[3];

    /*
     * Read setup info from file. 
     */
    sv_globals.svfb_fd = infile;
    if ( rle_get_setup( &sv_globals ) < 0 )
    {
	fprintf(stderr, "getmex: Error reading setup information from %s\n",
		infname ? infname : "stdin");
	exit(1);
    }

    if ( dbg )
	rle_debug( 1 );

    /* We`re only interested in R, G, & B */
    SV_CLR_BIT(sv_globals, SV_ALPHA);
    for (i = 3; i < sv_globals.sv_ncolors; i++)
	SV_CLR_BIT(sv_globals, i);
    ncolors = sv_globals.sv_ncolors > 3 ? 3 : sv_globals.sv_ncolors;
    if ( ncolors == 1 ) bwflag = TRUE;

    /*
     * Compute image size and allocate storage for colormapped image. 
     */
    x_size = (sv_globals.sv_xmax - sv_globals.sv_xmin + 1);
    y_size = (sv_globals.sv_ymax - sv_globals.sv_ymin + 1);
    if ( gt )
	rect_image = (long *) malloc(x_size * y_size * sizeof( long ));
    else
	rgb_image = (char *) malloc(x_size * y_size * 3 * sizeof( char ));

    /*
     * Set up for rle_getrow.  Pretend image x origin is 0. 
     */
    for (i = 0; i < 3; i++)
	scan[i] = (unsigned char *) malloc(x_size);
    sv_globals.sv_xmax -= sv_globals.sv_xmin;
    sv_globals.sv_xmin = 0;

    /* For each scan line, pack RGBs into the image memory. */
    while ((y = rle_getrow(&sv_globals, scan)) <= sv_globals.sv_ymax)
    {
	if ( bwflag && ncolors > 1 )
	{
	    rgb_to_bw( scan[0], scan[1], scan[ncolors - 1], scan[0], x_size );
	    /* Note: pack_scanline only uses channel 0 for B&W */
	}

	if ( gt )
	    pack_rect_scanline( scan, x_size,
		&rect_image[(y - sv_globals.sv_ymin) * x_size] );
	else
	    pack_rgb_scanline( scan, x_size,
		&rgb_image[(y - sv_globals.sv_ymin) * x_size * 3] );
    }

    /*
     * Free temp storage 
     */
    for (i = 0; i < 3; i++)
	free(scan[i]);

    /*
     * Get a window of the right size (user positions it with the mouse). 
     */
    if ( forkflg ) foreground();	/* Don`t fork. */
    maxsize( x_size, y_size );
    window_number = winopen( "getmex" );
    if ( infname ) wintitle( infname );

    RGBmode();
    gconfig();

    qdevice( REDRAW );
    qdevice( LEFTMOUSE );	      /* Pan the image under mouse control. */
    qdevice( SETUPKEY );		/* Reset panning. */
    unqdevice( INPUTCHANGE );		/* We don`t pay attention to these. */

    /* There was a redraw event sent when the window was created,
     * but we weren`t listening for them yet.
     */
    qenter( REDRAW, window_number );
}

/* 
 * Track events & redraw image when necessary.
 */
update_pic()
{
    short data;
    long event;
    int window_x_size, window_y_size, window_x_origin, window_y_origin;
    int x_min, x_max, y_min, y_max, x_start, y_start, x_end, y_end, x_len;
    int x_origin, y_origin, new_x_center, new_y_center;
    int x_center, y_center, saved_x_center, saved_y_center;

    register int y;

    /* Looking at the center, at first. */
    x_center = saved_x_center = x_size / 2;
    y_center = saved_y_center = y_size / 2;

    /* Redraw the window when necessary. */
    while ( TRUE )
    {
        event = qread( &data );
# ifdef DEBUG
	printf( "event %d, data %d\n", event, data );
#endif	
	switch ( event )
	{
	    case REDRAW:
		winset( window_number );
		reshapeviewport();
		RGBcolor( 128, 128, 128 );
		clear();

		/* Lower left corner of screen, in image coordinates.
		 * (Keep the center of the image in the center of the window.)
		 */
		getsize( &window_x_size, &window_y_size );
		x_min = x_center - window_x_size/2;
		x_max = x_min + (window_x_size-1);
		y_min = y_center - window_y_size/2;
		y_max = y_min + (window_y_size-1);

		/* Coordinate bounds have half a pixel added all around. */
		ortho2( x_min - .5, x_max + .5, y_min - .5, y_max + .5 );

		/* Draw just the part of the image in the window. */
		x_start = MAX( x_min, 0 );
		y_start = MAX( y_min, 0 );
		x_end = MIN( x_max, x_size-1 );
		y_end = MIN( y_max, y_size-1 );
		x_len = x_end - x_start + 1;

		/* Dump the scanlines.  Check once in a while for another
		 * redraw event queued up, and quit early if one is seen.
		 */
		if ( gt )
		{
		    register long *y_ptr;
		    y_ptr = rect_image + y_start*x_size + x_start;
		    for ( y = y_start;
			  y <= y_end && (y%16 != 0 || qtest() != REDRAW);
			  y++, y_ptr += x_size )
		    {
			lrectwrite( x_start, y, x_end, y, y_ptr );
		    }
		}
		else
		{
		    register char *y_ptr;
		    y_ptr = rgb_image + y_start*x_size*3 + x_start;
		    for ( y = y_start;
			  y <= y_end && (y%16 != 0 || qtest() != REDRAW);
			  y++, y_ptr += x_size * 3 )
		    {
			cmov2i( x_start, y );
			writeRGB( x_len,
				  y_ptr, y_ptr + x_size, y_ptr + x_size * 2 );
		    }
		}
		break;

	    /* Setup key - Reset viewing to look at the center of the image.
	     * Shift-Setup - Restores a saved view center.
	     * Control-Setup - Saves the current view center for Shift-Setup.
	     */
	    case SETUPKEY:
		if ( data == 1 )	/* Ignore button up events. */
		{
		    if ( getbutton(RIGHTSHIFTKEY) || getbutton(LEFTSHIFTKEY) )
		    {
			x_center = saved_x_center;	/* Restore. */
			y_center = saved_y_center;
			qenter( REDRAW, window_number );
		    }
		    else if ( getbutton(CTRLKEY) )
		    {
			saved_x_center = x_center;	/* Save. */
			saved_y_center = y_center;
		    }
		    else
		    {
			x_center = x_size / 2;		/* Reset. */
			y_center = y_size / 2;
			qenter( REDRAW, window_number );
		    }

		}
		break;

	    /* Pan a point picked with the left mouse button to the center
	     * of attention.  Beep if cursor is not on the image.
	     */
	    case LEFTMOUSE:
		if ( data == 1 ) 	/* Ignore button up events. */
		{
		    getorigin( &x_origin, &y_origin );
		    new_x_center = getvaluator( MOUSEX ) - x_origin + x_min;
		    new_y_center = getvaluator( MOUSEY ) - y_origin + y_min;
		    if ( new_x_center >= x_start &&
			 new_x_center <= x_end &&
			 new_y_center >= y_start &&
			 new_y_center <= y_end )
		    {
			x_center = new_x_center;
			y_center = new_y_center;
			qenter( REDRAW, window_number );
		    }
		    else
			ringbell();
		}
		break;
	}
    }
}

/*
 * Pack a scanline into a vector of RGB longs.
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	line:		Pointer to output buffer for packed color data.
 */
pack_rect_scanline( rgb, n, line )
unsigned char *rgb[3];
int n;
long *line;
{
    register int i;
    register long *dest;

    if ( !bwflag )			/* Color display. */
    {
	register unsigned char *r, *g, *b;

	for ( i = 0, r = rgb[0], g = rgb[1], b = rgb[2], dest = line;
	      i < n; i++, r++, g++, b++ )
	    /* Combine 3 8-bit colors into a long. */
	    *dest++ = *r + (*g<<8) + (*b<<16);
    }
    else				/* Gray scale display. */
    {
	register unsigned char *bw;

	for ( i = 0, bw = rgb[0]; i < n; i++, bw++ )
	    *dest++ = *bw + (*bw<<8) + (*bw<<16);
    }
}

/*
 * Pack a scanline into an RGB trio of vectors of bytes.
 * 
 * Inputs:
 * 	rgb:		Pointers to buffers containing the red, green,
 *			and blue color rows.
 *	n:		Length of row.
 *	line:		Pointer to output buffer for packed color data.
 */
pack_rgb_scanline( rgb, n, lines )
unsigned char *rgb[3];
int n;
char *lines;
{
    int chnl;
    register int i;
    register unsigned char *src, *dest;

    for ( chnl = 0, dest = lines; chnl < 3; chnl++ )
    {
	src = rgb[ bwflag ? 0 : chnl ];	/* Use just channel 0 for greyscale. */
	for ( i = 0; i < n; i++ )
	    *dest++ = *src++;
    }	    
}
