/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/*
 * rastorle  --
 *   convert Sun Rasterfile to Utah RLE format 
 *   reads Sun rasterfile or stdin and writes stdout.
 *
 * Author:         Berry Kercheval (berry@mordor.s1.gov)
 *                 Lawrence Livermore National Laboratory
 *                 Livermore, CA.
 * Date:           9 March 1987
 * History:  
 *   27 March 1987:  bbk: make it understand depth one (B&W) rasterfiles.
 *
 * Usage is:
 *   rastorle [ infile.ras ] | rleflip -v > out.rle
 *
 *   (The flip is necessary because Sun numbers the rows differently.)
 *
 * This will be difficult to compile on a non-sun, as it uses the pixrect
 * library include files.
 */

#include <stdio.h>
#include <pixrect/pixrect_hs.h>
#include <sysexits.h>
#include "svfb_global.h"

#define	USAGE	"Usage: rastorle [ rasterfile ]"
#define MAXLINE        1280	/* max width of a line    */
#define MAXVAL          255	/* max value for pixel */
#define CMAPSIZE	256	/* Max size of color map  */
/* #define DEBUG                /* debug output turned on */

unsigned char *outrows[4];	/* array of rows for RLE output */
unsigned char redline[MAXLINE], /* rle red row values */
              grnline[MAXLINE],
	      bluline[MAXLINE], 
	      alfline[MAXLINE];	/* alpha channel values */

unsigned char red[CMAPSIZE],	/* red colormap entries */
            green[CMAPSIZE],	/* Green ditto */
	    blue[CMAPSIZE];	/* see a pattern? */

main(argc, argv)
int argc;
char *argv[];
{
    FILE             *rasfile;	/* where input comes from */
    int               i;	/* useful index */
    int               p;	/* pixel value read from a file -- 
				 * used to index rasterfile color map. */
    int               h;	/* index for looping along a row */
    struct rasterfile rashdr;	/* standard header of a Sun rasterfile*/
    
    switch(argc)
    {
      case 1:			/* read stdin */
	rasfile = stdin;
	break;
      case 2:			/* read a file */
	rasfile = fopen(argv[1], "r");
	if(rasfile == (FILE *)NULL)
	{
	    perror(argv[1]);
	    exit(EX_NOINPUT);
	}
	break;
      default:			/* no other options allowed */
	fprintf(stderr, USAGE);
	exit (EX_USAGE);
    }
    
    /* first read the rasterfile header */
    if(fread(&rashdr, sizeof(rashdr), 1, rasfile) != 1)
    {
	fprintf(stderr, "Can't read rasterfile header.\n");
	exit(EX_DATAERR);
    }
    /* it has to start with the magic number... */
    if (rashdr.ras_magic != RAS_MAGIC)
    {
	fprintf(stderr, "Error: \"%s\" is not a rasterfile.\n", 
		rasfile==stdin?"stdin":argv[1]);
	exit(EX_DATAERR);
    }
#ifdef DEBUG
    fprintf (stderr, "rasterfile width     =  %d\n", rashdr.ras_width);
    fprintf (stderr, "rasterfile height    =  %d\n", rashdr.ras_height);
    fprintf (stderr, "rasterfile depth     =  %d\n", rashdr.ras_depth);
    fprintf (stderr, "rasterfile length    =  %d\n", rashdr.ras_length);
    fprintf (stderr, "rasterfile type      =  %d\n", rashdr.ras_type);
    fprintf (stderr, "rasterfile maplength =  %d\n", rashdr.ras_maplength);
#endif

    /* read in color map */
    switch(rashdr.ras_maptype)
    {
      case RMT_NONE:
#ifdef DEBUG
	fprintf (stderr, "No color map\n");
#endif
	break;
      case RMT_RAW:
#ifdef DEBUG
	fprintf (stderr, "Raw color map\n");
#endif
	break;
      case RMT_EQUAL_RGB:
#ifdef DEBUG
	fprintf (stderr, "RGB color map\n");
#endif
	/* read red */
	for (i = 0; i < rashdr.ras_maplength/3; i++)
	  red[i] = getc(rasfile);
	/* read green */
	for (i = 0; i < rashdr.ras_maplength/3; i++)
	  green[i] = getc(rasfile);
	/* read blue */
	for (i = 0; i < rashdr.ras_maplength/3; i++)
	  blue[i] = getc(rasfile);
	break;
      default:
	fprintf (stderr, "Unknown color map type (%d)\n", rashdr.ras_maptype);
	exit (EX_DATAERR);
    }
    /* we need to work on this... */
    switch(rashdr.ras_depth)
    {
      default:
	fprintf(stderr,
	  "Sorry, I can't deal with a rasterfile depth other than 1 or 8\n");
	break;

      case 1:			/* black & white */
	/* next few lines stolen from painttorle.c */
	SV_SET_BIT(sv_globals, SV_RED);
	SV_SET_BIT(sv_globals, SV_GREEN);
	SV_SET_BIT(sv_globals, SV_BLUE);
	SV_SET_BIT(sv_globals, SV_ALPHA);
	sv_globals.svfb_fd = stdout;
	sv_globals.sv_xmax = rashdr.ras_width;
	sv_globals.sv_ymax = rashdr.ras_height;
	sv_globals.sv_alpha = 1;
	
	outrows[0] = alfline;
	/* all three can have the same value since it's B&W */
	outrows[1] = redline;
	outrows[2] = redline;
	outrows[3] = redline;
	
	sv_setup( RUN_DISPATCH, &sv_globals );
	for (i=0; i<sv_globals.sv_ymax; i++) 
	{
	    for(h = 0; h < sv_globals.sv_xmax; h++)
	    {
		p = getbit(rasfile, 0);
		redline[h] = p?0:MAXVAL;
		alfline[h] = p?MAXVAL:0;
	    }
	    /* write a line to the rle file */
	    sv_putrow(&outrows[1], sv_globals.sv_ymax, &sv_globals);
	    getbit(NULL, 1);
	}
	break; /* end case 1: */

      case 8:
	/* next few lines stolen from painttorle.c */
	SV_SET_BIT(sv_globals, SV_RED);
	SV_SET_BIT(sv_globals, SV_GREEN);
	SV_SET_BIT(sv_globals, SV_BLUE);
	SV_SET_BIT(sv_globals, SV_ALPHA);
	sv_globals.svfb_fd = stdout;
	sv_globals.sv_xmax = rashdr.ras_width;
	sv_globals.sv_ymax = rashdr.ras_height;
	sv_globals.sv_alpha = 1;
	
	outrows[0] = alfline;
	outrows[1] = redline;
	outrows[2] = grnline;
	outrows[3] = bluline;
	
	sv_setup( RUN_DISPATCH, &sv_globals );
	/* now just loop through the rows */
	
	for (i=0; i<sv_globals.sv_ymax; i++)
	{
	    /* read a line from the rasterfile */
	    for (h=0; h < sv_globals.sv_xmax; h++)
	    {
		p = getc(rasfile);
		redline[h] = red[p];
		grnline[h] = green[p];
		bluline[h] = blue[p];
		/* fake up an alpha channel */
		if (redline[h] || grnline[h] || bluline[h])
		  alfline[h] = 255;
		else
		  alfline[h] = 0;
	    }
	    /* write a line to the rle file */
	    sv_putrow(&outrows[1], sv_globals.sv_ymax, &sv_globals);
	}
	break; /* end case 8: */
    } /* end switch */
}

/*
 * get a bit from the file fp.  Read a byte and return its bits
 * one at a time.  (actually returns zero or non-zero).
 */

getbit(fp, force)
FILE *fp;
int force;			/* true if we should read a byte anyway */
{
    static int c;		/* the char we are picking apart */
    static unsigned int mask
      = 0x0;			/* mask to get the next bit -- init to huge
				 * so that we get a byte the first time */
    int val;			/* value to be returned */
    if (force)			/* read a new byte next time */
    {
	mask = 0x0;
	return 0;
    }
    if(mask == 0)		/* time to get the next byte */
    {
	c = getc(fp);
	mask = 0x80;		/* reset the mask */
    }
    val = c & mask;		/* true if this bit on */
    mask >>= 1;			/* shift mask over one bit */
    return val;			/* the bit we saved goes back to caller */
}
