/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * to8.c - Convert color images to 8 bit dithered.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Fri Feb 28 1986
 * Copyright (c) 1986, University of Utah
 * 
 */
#ifndef lint
static char rcs_ident[] = "$Header:$";
#endif

#include <stdio.h>
#include <math.h>
#include "svfb_global.h"

short map[3][256];			/* for sv_setup */
int colmap[216][3];
rle_pixel ** in_cmap;		/* color map in input file */

int dm16[16][16];


int modN[256], divN[256];

double disp_gam = 2.5;
double img_gam = 1.0;

/*****************************************************************
 * TAG( main )
 * 
 * Usage: to8 [-{iI} gamma] [-g gamma] [-o outfile] [infile]
 * 
 * Inputs:
 *	-i gamma:	Specify gamma of image. (default 1.0)
 *	-I gamma:	Specify gamma of display image was computed for.
 * 	to8 will also read picture comments from the input file to determine
 *			the image gamma.  These are
 *	image_gamma=	gamma of image (equivalent to -i)
 *	display_gamma=	gamma of display image was computed for.
 *			Command line arguments override values in the file.
 *		
 *	-g gamma:	Specify gamma of display. (default 2.5)
 * 	infile:		Input (color) RLE file.  Stdin used if not
 *			specified.
 * Outputs:
 * 	outfile:	Output dithered RLE file.  Stdout used
 *			if not specified.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

main( argc, argv )
char **argv;
{
    char * infname = NULL, * outfname = NULL;
    char comment[80];		/* for gamma comment */
    FILE * infile = stdin, * outfile = stdout;
    int oflag = 0, y, nrow, nscan, i, iflag = 0, gflag = 0;
    long file_index;
    struct sv_globals out_glob;
    unsigned char ** scan, *outscan[2];
    unsigned char * buffer;

    if ( scanargs( argc, argv,
		   "% Ii%-gamma!F g%-gamma!f o%-outfile!s infile%s",
		   &iflag, &img_gam, &gflag, &disp_gam,
		   &oflag, &outfname, &infname ) == 0 )
	exit( 1 );

    if ( infname != NULL && (infile = fopen( infname, "r" )) == NULL )
    {
	perror( infname );
	exit( 1 );
    }
    sv_globals.svfb_fd = infile;

    rle_get_setup_ok( &sv_globals, "to8", infname );

    if ( sv_globals.sv_ncolors == 1 )
    {
	fprintf( stderr, "%s is already black & white\n",
		    infname ? infname : "stdin" );
	exit( 0 );
    }
    if ( sv_globals.sv_ncolors < 3 )
    {
	fprintf( stderr, "%s is not RGB",
		    infname ? infname : "stdin" );
	exit( 0 );
    }

    /* If no image gamma on command line, check comments in file */
    if ( ! iflag )
    {
	char * v;
	if ( (v = rle_getcom( "image_gamma", &sv_globals )) != NULL )
	{
	    img_gam = atof( v );
	    /* Protect against bogus information */
	    if ( img_gam == 0.0 )
		img_gam = 1.0;
	    else
		img_gam = 1.0 / img_gam;
	}
	else if ( (v = rle_getcom( "display_gamma", &sv_globals )) != NULL )
	{
	    img_gam = atof( v );
	    /* Protect */
	    if ( img_gam == 0.0 )
		img_gam = 1.0;
	}
    }

    if ( outfname != NULL && (outfile = fopen( outfname, "w" )) == NULL )
    {
	perror( outfname );
	exit( 1 );
    }

    out_glob = sv_globals;

    sv_globals.sv_xmax -= sv_globals.sv_xmin;
    sv_globals.sv_xmin = 0;
    nrow = sv_globals.sv_xmax;
    nscan = (sv_globals.sv_ymax - sv_globals.sv_ymin + 1);
    buffer = (unsigned char *)malloc( nrow );
    scan = (unsigned char **) malloc( (sv_globals.sv_ncolors +
				       sv_globals.sv_alpha) *
				      sizeof( unsigned char * ) );

    for ( i = 0;
	  i < sv_globals.sv_ncolors + sv_globals.sv_alpha;
	  i++ )
	scan[i] = (unsigned char *)malloc( nrow );

    if ( sv_globals.sv_alpha )
    {
	scan++;
	outscan[0] = scan[-1];
    }
    outscan[1] = buffer;

    /* Use input color map, too */
    in_cmap = buildmap( &sv_globals, 3, img_gam );

    init_color();
    out_glob.sv_ncolors = 1;
    out_glob.sv_ncmap = 3;
    out_glob.sv_cmaplen = 8;		/* 256 entries */
    out_glob.sv_cmap = (rle_map *)map;

    /* Record gamma color map was computed for */
    sprintf( comment, "display_gamma=%g", disp_gam );
    rle_putcom( comment, &out_glob );

    /*
     * Give it a background color of black, since the real background
     * will be dithered anyway.
     */
    if ( sv_globals.sv_background != NULL )
    {
	out_glob.sv_bg_color = (int *)malloc( sizeof( int ) );
	out_glob.sv_bg_color[0] = 0;
    }

    out_glob.svfb_fd = outfile;

    sv_setup( RUN_DISPATCH, &out_glob );

    while ( (y = rle_getrow( &sv_globals, scan )) <=
	    sv_globals.sv_ymax )
    {
	map_scanline( scan, nrow, y, buffer );
	sv_putrow( &outscan[1], nrow, &out_glob );
    }

    sv_puteof( &out_glob );
    exit( 0 );
}


/*
 * Initialize the 8 bit color map.  Choice of
 * "alpha-1" (perceptual flavor) or linear maps
 */
init_color()
{
    int     i;

    dithermap( 6, disp_gam, colmap, divN, modN, dm16 );

    for (i = 0; i < 216; i++)
    {
	map[0][i] = colmap[i][0] << 8;
	map[1][i] = colmap[i][1] << 8;
	map[2][i] = colmap[i][2] << 8;
    }

}

/*
 * Map a scanline to 8 bits through the dither matrix.
 */
#define DMAP(v,x,y)	(modN[v]>dm16[x][y] ? divN[v] + 1 : divN[v])

map_scanline( rgb, n, y, line )
unsigned char *rgb[3], *line;
{
	register unsigned char *r, *g, *b;
	register int i, col, row;
	for ( row = y % 16, col = 0, i = 0, r = rgb[0], g = rgb[1], b = rgb[2];
		  i < n; i++, r++, g++, b++, col = ((col + 1) & 15) )
		line[i] = DMAP(in_cmap[0][*r], col, row) +
			  DMAP(in_cmap[1][*g], col, row) * 6 +
			  DMAP(in_cmap[2][*b], col, row) * 36;
}

