/* 
 * rlezoom.c - Zoom an RLE file.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Thu Feb 26 1987
 * Copyright (c) 1987, University of Utah
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

/*****************************************************************
 * TAG( main )
 * 
 * Zoom an RLE file by an integral factor.
 *
 * Usage:
 * 	rlezoom factor [y-factor] [rlefile]
 * Inputs:
 * 	factor:		Factor to zoom by.  Must be a positive integer.
 * 	y-factor:	If provided, Y axis will be zoomed by this
 * 			factor, X axis by 'factor'.
 * 	rlefile:	Input RLE file.  Default is stdin.
 * Outputs:
 * 	Writes magnified RLE file to standard output.
 * Assumptions:
 * 	...
 * Algorithm:
 *      Read input file in raw mode.  For each line, expand it by the
 *      X expansion factor.  If the factor is > 3, convert pixel data
 *      into runs.  Write each line a number of times equal to the Y
 *      factor. 
 */
main( argc, argv )
char **argv;
{
    struct sv_globals in_glob, out_glob;
    char * rlename = NULL;
    FILE * rlefile = stdin;
    int xfact, yfact = 0;
    rle_op ** in_raw, ** out_raw;
    int * in_nraw, * out_nraw;
    int y, ynext, i;

    if ( scanargs( argc, argv, "% factor!d y-factor%d rlefile%s",
		   &xfact, &yfact, &rlename ) == 0 )
	exit( 1 );

    /* Error check */
    if ( xfact < 1 )
    {
	fprintf( stderr, "%s: Zoom factor must be >= 1\n", argv[0] );
	exit( 1 );
    }
    if ( yfact < 0 )
    {
	fprintf( stderr, "%s: Y zoom factor must be >= 1\n", argv[0] );
	exit( 1 );
    }

    /* If yfact is 0, it wasn't specified, set it to xfact */
    if ( yfact == 0 )
	yfact = xfact;

    /* Open input rle file and read header */
    if ( rlename != NULL && (rlefile = fopen( rlename, "r" )) == NULL )
    {
	fprintf( stderr, "%s: ", argv[0] );
	perror( rlename );
	exit( 1 );
    }
    else if ( rlename == NULL )
	rlename = "stdin";

    in_glob.svfb_fd = rlefile;

    rle_get_setup_ok( &in_glob, argv[0], rlename );
    
    /* Figure out output file size and parameters */
    out_glob = in_glob;
    out_glob.svfb_fd = stdout;
    /* more error checks */
    if ( (float)(out_glob.sv_xmax + 1) * (float)xfact > 32767 )
    {
	fprintf( stderr,
		 "%s: X zoom factor (%d) makes image too large (%g)\n",
		 argv[0], xfact,
		 (float)(out_glob.sv_xmax + 1) * (float)xfact );
	exit( 1 );
    }
    if ( (float)(out_glob.sv_ymax + 1) * (float)yfact > 32767 )
    {
	fprintf( stderr,
		 "%s: Y zoom factor (%d) makes image too large (%g)\n",
		 argv[0], yfact,
		 (float)(out_glob.sv_ymax + 1) * (float)yfact );
	exit( 1 );
    }
    out_glob.sv_xmin *= xfact;
    out_glob.sv_xmax = xfact * out_glob.sv_xmax + xfact - 1;
    out_glob.sv_ymin *= yfact;
    out_glob.sv_ymax = yfact * out_glob.sv_ymax + yfact - 1;
    sv_setup( RUN_DISPATCH, &out_glob );

    /* Create raw arrays for input and output files */
    rle_raw_alloc( &in_glob, &in_raw, &in_nraw );
    rle_raw_alloc( &out_glob, &out_raw, &out_nraw );

    y = in_glob.sv_ymin;
    while ( (ynext = rle_getraw( &in_glob, in_raw, in_nraw )) != 32768 )
    {
	if ( ynext - y > 1 )
	    sv_skiprow( &out_glob, yfact * (ynext - y) );
	expand_raw( &in_glob, in_raw, in_nraw, xfact, out_raw, out_nraw );
	for ( i = 0; i < yfact; i++ )
	    sv_putraw( out_raw, out_nraw, &out_glob );
	rle_freeraw( &in_glob, in_raw, in_nraw );
	rle_freeraw( &out_glob, out_raw, out_nraw );
	y = ynext;
    }
    sv_puteof( &out_glob );
    exit( 0 );
}

/*****************************************************************
 * TAG( expand_raw )
 * 
 * Zoom the input scanline by an integer factor.
 * Inputs:
 * 	glob:		Used for ncolors and alpha channel indication.
 *	in_raw:		Input raw opcodes.
 * 	in_nraw:	Input raw opcode counts.
 *	xfact:		Magnification factor.
 * Outputs:
 * 	out_raw:	Output raw opcodes.
 *	out_nraw:	Output counts.
 * Assumptions:
 * 	
 * Algorithm:
 * 	Replicate pixels by xfact.  If xfact > 3, turn pixel data into
 * 	runs.
 */
expand_raw( glob, in_raw, in_nraw, xfact, out_raw, out_nraw )
struct sv_globals * glob;
rle_op **in_raw;
int *in_nraw;
rle_op **out_raw;
int *out_nraw;
{
    register rle_op * inp, * outp;
    register rle_pixel * inc, * outc;
    int chan, i, j, k;

    for ( chan = -glob->sv_alpha; chan < glob->sv_ncolors; chan++ )
    {
	for ( inp = in_raw[chan], outp = out_raw[chan], i = in_nraw[chan];
	      i > 0; i--, inp++ )
	{
	    *outp = *inp;		/* copy, then modify */
	    outp->xloc *= xfact;
	    if ( inp->opcode == RRunDataOp )
	    {
		/* Just change length */
		outp->length *= xfact;
		outp++;
	    }
	    else		/* must be byte data */
		if ( xfact < 4 )
		{
		    /* Be cheap, replicate pixel data for small zooms */
		    outp->length *= xfact;
		    outp->u.pixels =
			(rle_pixel *)malloc( outp->length * sizeof(rle_pixel));
		    for ( inc = inp->u.pixels, outc = outp->u.pixels,
			  j = inp->length;
			  j > 0; j--, inc++ )
			for ( k = 0; k < xfact; k++ )
			    *outc++ = *inc;
		    outp++;
		}
		else
		{
		    /*
		     * Change pixels to runs, coalesce adjacent
		     * identical pixels
		     */
		    for ( inc = inp->u.pixels, j = 0;
			  j < inp->length; j++, inc++ )
			if ( j > 0 && outp[-1].u.run_val == *inc )
			    outp[-1].length += xfact;
			else
			{
			    outp->opcode = RRunDataOp;
			    outp->xloc = (inp->xloc + j) * xfact;
			    outp->length = xfact;
			    outp->u.run_val = *inc;
			    outp++;
			}
		}
	}
	/* Remember how many in this row */
	out_nraw[chan] = outp - out_raw[chan];
    }
}
