/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * rleswap.c - Swap the channels in an RLE file around
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Thu Jan 22 1987
 * Copyright (c) 1987, University of Utah
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

/*****************************************************************
 * TAG( main )
 * 
 * Usage:
 *	rleswap [-p channel-pairs,...] [-i input-channels,...]
 *		[-o output-channels,...] [-d delete-channels,...] [rlefile]
 * Inputs:
 *	-p, -i, -o, -d:	Only one of these may be specified, they indicate
 *			different ways of indicating the channel swapping
 *			from input to output.
 *
 * 	channel-pairs:	A comma separated list of pairs of channel numbers
 *			The first channel of each pair indicates a channel
 *			in the input file that will be mapped to the
 *			the channel in the output file indicated by the
 *			second number in the pair.  No output channel
 *			number may appear more than once.  Any input channel
 *			not mentioned will not appear in the output file.
 *			Any output channel not mentioned will not receive
 *			image data.
 *
 *	input-channels:	A comma separated list of numbers indicating the
 *			input channel that maps to each output channel
 *			in sequence.  I.e., the first number indicates
 *			the input channel mapping to output channel 0.
 *			The alpha channel will be passed through unchanged
 *			if present.  Any input channels not mentioned
 *			in the list will not appear in the output.
 *
 *	output-channels:A comma separated list of numbers indicating the
 *			output channel to which each input channel, in
 *			sequence, will map.  I.e., the first number gives
 *			the output channel to which the first input channel
 *			will map.  No number may be repeated in this list.
 *			The alpha channel will be passed through unchanged
 *			if present.  Any output channel not mentioned in
 *			the list will not receive image data.  If there
 *			are fewer numbers in the list than there are input
 *			channels, the excess input channels will be ignored.
 *			If there are more numbers than input channels, it
 *			is an error.
 *
 *	delete-channels:A comma separated list of input channels that should
 *			not appear in the output.  All other channels will
 *			be passed through unchanged.
 *
 *	rlefile:	Optional input file name specification.
 *
 * Outputs:
 * 	An RLE file similar to the input file, but with the color channels
 *	remapped as indicated, will be written to the output.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
main( argc, argv )
char **argv;
{
    int nspec, * specs;
    int iflag = 0, oflag = 0, dflag = 0, pflag = 0, verbose = 0;
    char * fname = NULL;
    FILE * rlef = stdin;
    static char * argfmt =
	"% v%- i%-input-channels!,d o%-output-channels!,d \n\
\td%-delete-channels!,d p%-channel-pairs!,d rlefile%s";
    register int i;
    int j, errs = 0;
    int * outchan, noutput, outalpha = -2;
    struct sv_globals outglob;
    rle_op ** scanraw, ** outraw;
    int * nraw, * outnraw, y, nskip;


    if ( scanargs( argc, argv, argfmt, &verbose,
		   &iflag, &nspec, &specs,
		   &oflag, &nspec, &specs,
		   &dflag, &nspec, &specs,
		   &pflag, &nspec, &specs,
		   &fname ) == 0 )
	exit( 1 );

    /* Do some sanity checks */
    if ( iflag + oflag + dflag + pflag != 1 )
    {
	fprintf( stderr,
		 "%s: You must specify exactly one of -d, -i, -o, or -p\n",
		 argv[0] );
	/*
	 * Generate usage message.
	 * This is safe because no args will actually be parsed.
	 */
	scan_usage( argv, argfmt );
	exit( 1 );
    }

    if ( pflag && (nspec % 2) != 0 )
    {
	fprintf( stderr, "%s: You must specify pairs of channels with -p\n",
		 argv[0] );
	exit( 1 );
    }
    /* More later, after we have the RLE header in hand */

    /* Open input */
    if ( fname != NULL && (rlef = fopen( fname, "r" )) == NULL )
    {
	fprintf( stderr, "%s: ", argv[0] );
	perror( fname );
	exit( 1 );
    }
    else if ( fname == NULL )
	fname = "stdin";

    sv_globals.svfb_fd = rlef;

    /* Read in header */
    rle_get_setup_ok( &sv_globals, argv[0], fname );

    /* Set up mapping and sanity check simultaneously */
    if ( iflag )		/* where does output come from? */
    {
	outchan = specs;
	noutput = nspec;
	/* Sanity check */
	for ( i = 0; i < noutput; i++ )
	    if ( outchan[i] >= sv_globals.sv_ncolors ||
		 outchan[i] < -1 ||
		 outchan[i] < 0 && !sv_globals.sv_alpha )
	    {
		fprintf( stderr, "%s: No input channel %d\n",
			 argv[0], outchan[i] );
		errs++;
	    }

	if ( sv_globals.sv_alpha )
	    outalpha = -1;	/* pass alpha channel through */
    }

    if ( oflag )		/* where does input go */
    {
	if ( nspec > sv_globals.sv_ncolors )
	{
	    fprintf( stderr,
	     "%s: Input file has %d channels, can't swap %d channels\n",
		     argv[0], sv_globals.sv_ncolors, nspec );
	    errs++;
	}
	else
	{
	    /* Find highest output channel */
	    noutput = -1;	/* assume none */
	    for ( i = 0; i < nspec; i++ )
		if ( specs[i] > noutput )
		    noutput = specs[i];
	    noutput++;

	    /* Allocate space for output pointers */
	    outchan = (int *)malloc( noutput * sizeof( int ) );
	    if ( outchan == NULL )
	    {
		fprintf( stderr, "%s: Malloc failed\n", argv[0] );
		exit( 1 );
	    }

	    /* Initialize to empty state */
	    for ( i = 0; i < noutput; i++ )
		outchan[i] = -2;

	    /* Fill it in */
	    for ( i = 0; i < nspec; i++ )
	    {
		if ( specs[i] < -1 )
		{
		    fprintf( stderr, "%s: No channel %d in output\n",
			     argv[0], specs[i] );
		    errs++;
		}
		else if ( specs[i] < 0 )
		{
		    if ( outalpha > -2 )
		    {
			fprintf( stderr,
	 "%s: Output alpha channel can't come from both inputs %d and %d.\n",
				 argv[0], outalpha, i );
			errs++;
		    }
		    else
			outalpha = i;
		}
		else
		{
		    if ( outchan[specs[i]] > -2 )
		    {
			fprintf( stderr,
	 "%s: Output channel %d can't come from both inputs %d and %d.\n",
				 argv[0], specs[i], outchan[specs[i]], i );
			errs++;
		    }
		    else
			outchan[specs[i]] = i;
		}
	    }
	    if ( outalpha < -1 && sv_globals.sv_alpha )
		outalpha = -1;	/* map alpha channel through */
	}
    }

    if ( dflag )		/* Delete some input channels */
    {
	/* First, set up identity mapping from input to output */
	noutput = sv_globals.sv_ncolors;

	/* Allocate space for output pointers */
	outchan = (int *)malloc( noutput * sizeof( int ) );
	if ( outchan == NULL )
	{
	    fprintf( stderr, "%s: Malloc failed\n", argv[0] );
	    exit( 1 );
	}

	for ( i = 0; i < noutput; i++ )
	    outchan[i] = i;
	if ( sv_globals.sv_alpha )
	    outalpha = -1;

	/* Now, delete indicated channels from mapping */
	for ( i = 0; i < nspec; i++ )
	    if ( specs[i] < -sv_globals.sv_alpha ||
		 specs[i] >= sv_globals.sv_ncolors )
		fprintf( stderr, "%s: Warning: No channel %d in input\n",
			 argv[0], specs[i] );
	    else if ( specs[i] == -1 && outalpha == -2 ||
		      outchan[specs[i]] == -2 )
		fprintf( stderr, "%s: Warning: Deleted channel %d twice\n",
			 argv[0], specs[i] );
	    else if ( specs[i] >= 0 )
		outchan[specs[i]] = -2;	/* null it out */
	    else
		outalpha = -2;

	/* Figure out how many output channels we really have */
	while ( outchan[noutput - 1] == -2 )
	    noutput--;
    }

    if ( pflag )		/* Do pairs of mappings */
    {
	/* Find highest output channel */
	noutput = -1;		/* assume none */
	for ( i = 1; i < nspec; i += 2 )
	    if ( specs[i] > noutput )
		noutput = specs[i];
	noutput++;

	/* Allocate space for output pointers */
	outchan = (int *)malloc( noutput * sizeof( int ) );
	if ( outchan == NULL )
	{
	    fprintf( stderr, "%s: Malloc failed\n", argv[0] );
	    exit( 1 );
	}

	/* Initialize to empty state */
	for ( i = 0; i < noutput; i++ )
	    outchan[i] = -2;

	/* Fill it in */
	for ( i = 0; i < nspec; i += 2 )
	{
	    if ( specs[i] < -sv_globals.sv_alpha ||
		 specs[i] >= sv_globals.sv_ncolors )
	    {
		fprintf( stderr, "%s: No channel %d in input\n",
			 argv[0], specs[i] );
		errs++;
	    }
	    if ( specs[i+1] < -1 )
	    {
		fprintf( stderr, "%s: No channel %d in output\n",
			 argv[0], specs[i+1] );
		errs++;
	    }
	    else if ( specs[i+1] < 0 )
	    {
		if ( outalpha > -2 )
		{
		    fprintf( stderr,
     	"%s: Output alpha channel can't come from both inputs %d and %d.\n",
			     argv[0], outalpha, specs[i] );
		    errs++;
		}
		else
		    outalpha = specs[i];
	    }
	    else
	    {
		if ( outchan[specs[i+1]] > -2 )
		{
		    fprintf( stderr,
	     "%s: Output channel %d can't come from both inputs %d and %d.\n",
			     argv[0], specs[i+1],
			     outchan[specs[i+1]], specs[i] );
		    errs++;
		}
		else
		    outchan[specs[i+1]] = specs[i];
	    }
	}
    }

    /* If errors, go away */
    if ( errs )
	exit( 1 );

    /* Be verbose if requested */
    if ( verbose )
    {
	fprintf( stderr, "Channel mapping from input to output:\n" );
	if ( outalpha != -2 )
	    if ( outalpha < 0 )
		fprintf( stderr, "alpha\t-> alpha\n" );
	    else
		fprintf( stderr, "%d\t-> alpha\n", outalpha );
	else
	    fprintf( stderr, "No output alpha channel\n" );
	for ( i = 0; i < noutput; i++ )
	    if ( outchan[i] == -2 )
		fprintf( stderr, "nothing\t-> %d\n", i );
	    else if ( outchan[i] == -1 )
		fprintf( stderr, "alpha\t-> %d\n", i );
	    else
		fprintf( stderr, "%d\t-> %d\n", outchan[i], i );
    }

    /* Set up output globals now */
    outglob = sv_globals;	/* same as input, basically */
    /* Except for a few changes */
    outglob.svfb_fd = stdout;
    outglob.sv_ncolors = noutput;
    if ( outalpha != -2 )
    {
	outglob.sv_alpha = 1;
	SV_SET_BIT( outglob, SV_ALPHA );
    }
    else
    {
	outglob.sv_alpha = 0;
	SV_CLR_BIT( outglob, SV_ALPHA );
    }
    for ( i = 0; i < noutput; i++ )
	if ( outchan[i] != -2 )
	    SV_SET_BIT( outglob, i );
	else
	    SV_CLR_BIT( outglob, i );

    /* Do background color */
    if ( outglob.sv_background )
	outglob.sv_bg_color =
	    (int *)malloc( outglob.sv_ncolors * sizeof(int) );

    for ( i = 0; i < noutput; i++ )
	if ( outchan[i] < 0 )
	    outglob.sv_bg_color[i] = 0;	/* got to be something */
	else
	    outglob.sv_bg_color[i] = sv_globals.sv_bg_color[outchan[i]];

    /* And color map!? */
    if ( sv_globals.sv_ncmap > 1 )	/* only one channel color map stays */
    {
	int cmaplen = 1 << outglob.sv_cmaplen;
	int cmapshift = 16 - outglob.sv_cmaplen;

	outglob.sv_cmap = (rle_map *)malloc( noutput * cmaplen *
					     sizeof(rle_map) );
	outglob.sv_ncmap = noutput;

	/* If input channel is in color map, copy it, else use identity? */
	for ( i = 0; i < noutput; i++ )
	    if ( outchan[i] >= 0 && outchan[i] < sv_globals.sv_ncmap )
	    {
		register rle_map * imap, * omap;

		imap = &sv_globals.sv_cmap[outchan[i] * cmaplen];
		omap = &outglob.sv_cmap[i * cmaplen];

		for ( j = 0; j < cmaplen; j++ )
		    *omap++ = *imap++;
	    }
	    else
	    {
		register rle_map * omap;

		omap = &outglob.sv_cmap[i * cmaplen];
		for ( j = 0; j < cmaplen; j++ )
		    *omap++ = j << cmapshift;
	    }
    }

    /* Write output header */
    sv_setup( RUN_DISPATCH, &outglob );

    /* Allocate raw buffers for input */
    rle_raw_alloc( &sv_globals, &scanraw, &nraw );

    /* Allocate buffer pointers for output */
    rle_raw_alloc( &outglob, &outraw, &outnraw );
    for ( i = -outglob.sv_alpha; i < outglob.sv_ncolors; i++ )
	if ( outraw[i] )
	{
	    free( (char *)outraw[i] );	/* don't need these */
	    break;
	};

    /* Finally, do the work */
    y = sv_globals.sv_ymin - 1;
    while ( (nskip = rle_getraw( &sv_globals, scanraw, nraw )) != 32768 )
    {
	nskip -= y;		/* difference from previous line */
	y += nskip;
	if ( nskip > 1 )
	    sv_skiprow( &outglob, nskip - 1 );

	if ( outalpha != -2 )
	{
	    outraw[-1] = scanraw[outalpha];
	    outnraw[-1] = nraw[outalpha];
	}

	for ( i = 0; i < noutput; i++ )
	    if ( outchan[i] != -2 )
	    {
		outraw[i] = scanraw[outchan[i]];
		outnraw[i] = nraw[outchan[i]];
	    }
	    else
		outnraw[i] = 0;

	sv_putraw( outraw, outnraw, &outglob );

	rle_freeraw( &sv_globals, scanraw, nraw );
    }

    sv_puteof( &outglob );	/* all done! */
    exit( 0 );
}
