/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * flip.c - Invert, reflect or 90-degree rotate an rle image.
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Mon Jun 23 1986
 * Copyright (c) 1986, University of Utah
 * 
 * Usage is:
 *   flip  {-v | -h | -l | -r} < infile > outfile
 *
 * Where the flags mean:
 *	-v : Vertical flip (top to bottom)
 *      -h : Horizontal flip (right to left)
 *      -r : Rotate image 90 degrees right (clockwise)
 *  	-l : Rotate image 90 degrees left (counter-clockwise)
 */


#include <stdio.h>
#include "svfb_global.h"

#define and &&
#define or ||

#define VERT_FLAG  0x01		/* Command line flags */
#define HORIZ_FLAG 0x02		/* must match appearance in scanargs */
#define LEFT_FLAG  0x04
#define RIGHT_FLAG 0x08

main(argc, argv)
int  argc;
char *argv[];
{
    int flags = 0;
    char *infilename = NULL;
    int xlen, ylen, i, j, chan;
    int xlinewidth;
    struct sv_globals inglobals, outglobals;
    rle_pixel *rastptr, *rasterbase;
    rle_pixel **temp_line;
    rle_pixel **rows;
    char *malloc();
    int nchan;			/* Number of channels actually used. */

    if (scanargs(argc, argv, "% rlhv!- infile%s", &flags, &infilename) == 0)
    {
	exit(-1);
    }

    if (infilename)
    {
	inglobals.svfb_fd = fopen(infilename, "r");
	if (! inglobals.svfb_fd)
	{
	    fprintf( stderr, "rleflip: can't open %s\n", infilename);
	    exit(-1);
	}
    }
    else
	inglobals.svfb_fd = stdin;

    rle_get_setup_ok( &inglobals, "rleflip", infilename );
    
    outglobals = inglobals;
    outglobals.svfb_fd = stdout;

    nchan = outglobals.sv_alpha + outglobals.sv_ncolors;

    /* Make row pointers for all seasons. */
    rows = (rle_pixel **) malloc( nchan * sizeof( rle_pixel * ) );
    if (! rows)
	fprintf(stderr, "rlepixel: malloc failed\n");

    xlen = inglobals.sv_xmax - inglobals.sv_xmin + 1;
    ylen = inglobals.sv_ymax - inglobals.sv_ymin + 1;

    /* getrow and putrow assume the scanline starts at pixel 0 */
    xlinewidth = inglobals.sv_xmax + 1;
    
    /* Note:
     * When you read in a row of pixels with rle_getrow, it places blank
     * pixels between 0 and sv_xmin of your buffer.  However, when you
     * use sv_putrow to write them out, the buffer must be pointing at
     * where the data actually starts (i.e., at [sv_xmin] of the getrow
     * buffer.  */

    /* WARNING: Heavy-duty pointer munging ahead */

    rasterbase = (rle_pixel *) malloc( xlinewidth * ylen * nchan );
    if (rasterbase == NULL)
    {
	fprintf(stderr, "flip: No memory for raster\n");
	exit(-1);
    }
    rastptr = rasterbase;

    /****************************************************
     * Read in all of the pixels
     ****************************************************/

    for (i = inglobals.sv_ymin; i <= inglobals.sv_ymax; i++)
    {
        for (chan=0; chan < nchan; chan++)
        {
            rows[chan] = rastptr;
	    /* Increment pointer by xlinewidth */
	    rastptr = &(rastptr[xlinewidth]);
	}
	rle_getrow( &inglobals, &rows[outglobals.sv_alpha] );
    }

    /****************************************************
     * Invert along vertical axis
     ****************************************************/

    if (flags == VERT_FLAG)
    {
	sv_setup( RUN_DISPATCH, &outglobals );

	/* Find last row in raster */
	rastptr = &(rasterbase[xlinewidth * (ylen - 1) * nchan]);

	for (i = outglobals.sv_ymin; i <= outglobals.sv_ymax; i++)
	{
	    for (chan=0; chan < nchan; chan++)
	    {
		rows[chan] = &(rastptr[outglobals.sv_xmin]);
		/* Increment pointer by xlinewidth */
		rastptr = &(rastptr[xlinewidth]);
	    }
	    sv_putrow( &rows[outglobals.sv_alpha], xlen, &outglobals );
	    rastptr = &(rastptr[ - 2 * nchan * xlinewidth ]);
	}
    }
    else

    /****************************************************
     * Reflect across horizontal axis
     ****************************************************/

    if (flags == HORIZ_FLAG)
    {
	register rle_pixel *inpxl, *outpxl;

	sv_setup( RUN_DISPATCH, &outglobals );

	if (rle_row_alloc( &outglobals, &temp_line ))
	    fprintf(stderr, "rleflip: Ran out of malloc space\n");
	if (outglobals.sv_alpha)
	    temp_line--;	/* Use zero based (vs. -1 based) addressing */

	/* Temp row used to swap pixel order */
	for (chan = 0; chan < nchan; chan++)
	    rows[chan] = &(temp_line[chan][outglobals.sv_xmin]);

	for (i = 0; i < ylen; i++)
	{
	    rastptr = &(rasterbase[i * xlinewidth * nchan]);

	    for (chan = 0; chan < nchan; chan++)
	    {
		inpxl = &(rastptr[chan * xlinewidth + xlinewidth - xlen]);
		outpxl = &(temp_line[chan][xlinewidth-1]);

		for (j = 0; j < xlen; j++)
		    *outpxl-- = *inpxl++;
	    }
	    sv_putrow( &rows[outglobals.sv_alpha], xlen, &outglobals );
	}
    }
    else

    /****************************************************
     * Rotation
     ****************************************************/
    if ((flags == RIGHT_FLAG) or (flags == LEFT_FLAG))
    {
	int linebytes, chan_offset;
	int lineoff;
	int oxlen, oylen;
	register rle_pixel *outptr;

	/* Must first adjust size of output image... */
	outglobals.sv_xmax = inglobals.sv_xmin + ylen - 1;
	outglobals.sv_ymax = inglobals.sv_ymin + xlen - 1;

	oxlen = outglobals.sv_xmax - outglobals.sv_xmin + 1;
	oylen = outglobals.sv_ymax - outglobals.sv_ymin + 1;

	sv_setup( RUN_DISPATCH, &outglobals );

	if (rle_row_alloc( &outglobals, &temp_line ))
	    fprintf(stderr, "rleflip: Ran out of malloc space\n");
	if (outglobals.sv_alpha)
	    temp_line--;	/* Use zero based (vs. -1 based) addressing */

	/* Temp row used to swap pixel order */
	for (chan = 0; chan < nchan; chan++)
	    rows[chan] = temp_line[chan];

	linebytes = nchan * xlinewidth;	/* Bytes in entire input scanline */

	if (flags == LEFT_FLAG)
	{
	/****************************************************
	 * Rotate left
	 ****************************************************/
    	for (i = 0; i < oylen; i++)
	    {
		lineoff = xlinewidth - xlen + i;
		for (chan = 0; chan < nchan; chan++)
		{
		    /* Bytes upto input chan */
		    chan_offset = lineoff + xlinewidth * chan;  
		    outptr = temp_line[chan];
		    for (j = oxlen - 1; j >= 0; j--)
			*outptr++ =
			  rasterbase[j * linebytes + chan_offset];
		}
		sv_putrow( &rows[outglobals.sv_alpha], oxlen, &outglobals );
	    }
	}
	else
	{
	/****************************************************
	 * Rotate right
	 ****************************************************/

	    for (i = 0; i < oylen; i++)
	    {
		for (chan = 0; chan < nchan; chan++)
		{
		    /* Bytes upto input chan */
		    chan_offset = xlinewidth * chan + (xlinewidth - 1 - i);
		    outptr = temp_line[chan];
		    for (j = 0; j < oxlen; j++)
		    {
			*outptr++ = rasterbase[j * linebytes + chan_offset];
		    }
		}
		sv_putrow( &rows[outglobals.sv_alpha], oxlen, &outglobals );
	    }
	}
    }

    sv_puteof( &outglobals );
    exit( 0 );
}

