/* 
 * unexp.c - take an exponential RLE file and make an integer file.
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Wed Aug 21 17:02:45 1985
 *
 *		Based on code by Spencer Thomas
 */

#include <stdio.h>
#include <math.h>
#include "svfb_global.h"

#define MAX(i,j)   ( (i) > (j) ? (i) : (j) )

#define SV_EXPONENT (in_globals.sv_ncolors-1) /* Assume last channel is exp */

rle_pixel **in_rows;
rle_pixel **out_rows;

int truncations = 0;
struct sv_globals in_globals, out_globals;
rle_pixel r_in, g_in, b_in;

main(argc,argv)
char *argv[];
int argc;
{
    register int x, chan;
    int y;
    int scan_flag = 0;
    int max_flag = 0;
    int verbose_flag = 0;
    int print_flag = 0;
    int oflag = 0;
    float maxval, tmp, expnt, tmp_max;
    char *infilename = NULL, *outfilename = NULL;

    if (scanargs(argc, argv, "% v%- p%- s%- m%-maxval!f infile!s outfile%s",
                 &verbose_flag, &print_flag, &scan_flag, &max_flag, &maxval, 
		 &infilename, &outfilename ) == 0)
    {
	exit(-1);
    }

    if ((in_globals.svfb_fd = fopen(infilename, "r")) == 0)
    {
	fprintf(stderr, "unexp ");
	perror(infilename);
	exit(-4);
    }

    rle_get_setup_ok( &in_globals, "unexp", infilename );
    if (! rle_getcom( "exponential_data", &in_globals ))
	fprintf(stderr, "unexp: warning - no exponential_data comment\n");

    if (in_globals.sv_ncolors < 2)
    {
	fprintf(stderr, 
	        "unexp: File does not contain exponent channel.\n");
	exit(-4);
    }

    if (rle_row_alloc( &in_globals, &in_rows ))
    {
	fprintf(stderr, "unexp: no heap space\n");
	exit(-2);
    }

    /* If maximum value isn't given, slosh through file to find it */
    if (! max_flag)
    {
	maxval = 0.0;
	for (y = in_globals.sv_ymin; y <= in_globals.sv_ymax; y++)
	{
	    if ((verbose_flag) && (y % 100) == 0)
		fprintf(stderr, "Scanning row %d...\n", y);
	    
	    rle_getrow( &in_globals, in_rows );

	    for (x = in_globals.sv_xmin; x <= in_globals.sv_xmax; x++)
	    {
		expnt = ldexp( 1/256.0, in_rows[SV_EXPONENT][x] - 127);
		tmp = -2000.0;
		for (chan = 0; chan < SV_EXPONENT; chan++)
		    tmp = MAX( tmp, (float)in_rows[chan][x] );

		tmp *= expnt;
		maxval = MAX( maxval, tmp );
	    }
	}
	if (scan_flag || print_flag || verbose_flag)
	    fprintf(stderr, "Maximum value: %1.8g\n", maxval);

	if (scan_flag) exit(0);

	rewind( in_globals.svfb_fd );
    
	rle_get_setup( &in_globals );
    }

    /* Open output file */

    out_globals = in_globals;
    rle_delcom( "exponential_data", &out_globals );
    out_globals.sv_ncolors = in_globals.sv_ncolors - 1;

    if ((out_globals.svfb_fd = 
	 ((outfilename) ? fopen(outfilename, "w") : stdout)) == 0)
    {
	fprintf(stderr, "unexp (output) - ");
	perror(outfilename);
	exit(3);
    }

    sv_setup( RUN_DISPATCH, &out_globals );

    out_globals.sv_xmin = 0;
    out_globals.sv_xmax -= in_globals.sv_xmin;

    if (rle_row_alloc( &out_globals, &out_rows ))
    {
	fprintf( stderr, "unexp: out of heap space\n" );
	exit( -2 );
    }

    /*
     * Convert byte/exponent form into straight RLE.
     * Alpha is passed straight through.
     */

    if ( maxval == 0.0 )
        maxval = 1.0;
    else
	maxval /= 255;			/* Pre-scale to 0-255 */

    for (y = in_globals.sv_ymin; y <= in_globals.sv_ymax; y++)
    {
	if (((y % 100) == 0) && verbose_flag)
	    fprintf(stderr, "Processing row %d...\n", y);
	
	rle_getrow( &in_globals, in_rows );

	for (x = in_globals.sv_xmin; x <= in_globals.sv_xmax; x++)
	{
	    /* Only bother with pixels with coverage */
	    if (in_rows[SV_ALPHA][x]) 
	    {
		expnt = ldexp( 1/256.0, in_rows[SV_EXPONENT][x] - 127 );
		tmp_max = -1000;
		for( chan = 0; chan < SV_EXPONENT; chan++ )
		    tmp_max = MAX( tmp_max, in_rows[chan][x] );

		tmp = expnt / maxval * tmp_max;
		if (tmp > 255)
		{
		    tmp = 255 / tmp;
		    truncations++;
		}
		else
		    tmp = 1.0;

		tmp *= expnt / maxval;
		for( chan = 0; chan < SV_EXPONENT; chan++ )
		    out_rows[chan][x-in_globals.sv_xmin] = (rle_pixel) (int)
					(in_rows[chan][x] * tmp);
		out_rows[SV_ALPHA][x-in_globals.sv_xmin] = 
		    in_rows[SV_ALPHA][x];
	    }
	    else
		for( chan = SV_ALPHA; chan < SV_EXPONENT; chan++ )
		    out_rows[chan][x-in_globals.sv_xmin] = 0;
	    	
	}
	sv_putrow( out_rows, (in_globals.sv_xmax - in_globals.sv_xmin) + 1,
		   &out_globals );
    }
    sv_puteof( &out_globals );

    if (truncations)
	fprintf(stderr,"unexp: %d bytes truncated (sorry...)\n", truncations);
    exit( 0 );
}
