/* 
 * rlesplit.c - Split concatenated RLE files into separate files.
 * 
 * Author:	Spencer W. Thomas
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Mon May  4 1987
 * Copyright (c) 1987, University of Utah
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>

/*****************************************************************
 * TAG( main )
 * 
 * Usage:
 *     rlesplit [-n number [digits]] [-p prefix] [rlefile]
 * Inputs:
 *	-n number:	If specified, output file numbering will start
 *			with this value (see below).  Otherwise, numbering
 *			starts at 1.
 *
 * 	digits:		The number of digits to be used in the numeric
 * 			portion of the output file names.  Defaults to 3.
 * 			All numbers will be leading zero filled.
 * 
 * 	-p prefix:	If specified, output files will be named
 *			"prefix.#.rle".  If not specified, and
 *			an rlefile is specified, then output files
 *			will be "rlefileroot.#.rle", where
 *			"rlefileroot" is rlefile with any ".rle" suffix
 *			stripped off.  If no arguments are specified,
 *			output files will be "#.rle".  In any case, "#"
 *			represents a sequentially increasing number.
 *
 *	rlefile:	If specified, input will be read from here,
 *			otherwise, input will be read from stdin.
 * Outputs:
 * 	Writes each rle image in the input stream to an output file
 *	whose name is generated as above.
 * Assumptions:
 * 	Each RLE image in the input stream must be terminated with
 *	an EOF opcode.
 * Algorithm:
 *	[None]
 */

main( argc, argv )
char **argv;
{
    register char * cp, * slashp;
    int num = 1, pflag = 0, digits = 3;
    int rle_err, ynext, y;
    char * infname = NULL, * prefix = "", * format = "%s%0*d.rle";
    char filebuf[BUFSIZ];
    FILE * infile = stdin, * outfile;
    struct sv_globals in_glob, out_glob;
    rle_op ** scan;
    int * nraw;

    if ( scanargs( argc, argv, "% n%-number!ddigits%d p%-prefix!s rlefile%s",
		   &num, &num, &digits, &pflag, &prefix, &infname ) == 0 )
	exit( 1 );

    /* Open input file */
    if ( infname != NULL && (infile = fopen( infname, "r" )) == NULL )
    {
	fprintf( stderr, "rlesplit: Can't open %s: ", infname );
	perror( "" );
	exit( 1 );
    }
    in_glob.svfb_fd = infile;

    if ( pflag || infname )
    {
	format = "%s.%0*d.rle";
	if ( !pflag )
	{
	    /* Strip ".rle" suffix from input file name */
	    /* Avoid strrchr, rindex problem */
	    for ( cp = infname; *cp; cp++ )
		;		/* find end of name */
	    /* Look for last slash */
	    for ( slashp = cp - 1; *slashp != '/' && slashp > infname; )
		slashp--;
	    if ( *slashp == '/' )
		slashp++;
	    /* Look for last dot */
	    while ( *--cp != '.' && cp > infname )
		;		/* find last . */
	    if ( strcmp( cp, ".rle" ) != 0 )
		cp = infname + strlen( infname );
	    /* Make null full string buffer */
	    prefix = (char *)calloc( cp - slashp + 1, 1 );
	    /* Copy everything but suffix */
	    strncpy( prefix, infname, cp - slashp );
	}
    }

    if ( infname == NULL )
	infname == "standard input";

    while ( (rle_err = rle_get_setup( &in_glob )) == RLE_SUCCESS )
    {
	/* Copy input to output file */
	sprintf( filebuf, format, prefix, digits, num++ );
	if ( (outfile = fopen( filebuf, "w" )) == NULL )
	{
	    fprintf( stderr, "rlesplit: Can't create %s: ", filebuf );
	    perror( "" );
	    exit( 1 );
	}
	fprintf( stderr, "rlesplit: Creating %s\n", filebuf );
	out_glob = in_glob;
	out_glob.svfb_fd = outfile;
	sv_setup( RUN_DISPATCH, &out_glob );

	rle_raw_alloc( &in_glob, &scan, &nraw );


	for ( y = in_glob.sv_ymin;
	      (ynext = rle_getraw( &in_glob, scan, nraw )) != 32768;
	      y = ynext )
	{
	    if ( ynext - y > 1 )
		sv_skiprow( &out_glob, ynext - y );
	    sv_putraw( scan, nraw, &out_glob );
	    rle_freeraw( &in_glob, scan, nraw );
	}
	sv_puteof( &out_glob );
	fclose( out_glob.svfb_fd );

	rle_raw_free( &in_glob, scan, nraw );
    }

    if ( rle_err != RLE_EOF && rle_err != RLE_EMPTY )
	rle_get_error( rle_err, argv[0], infname );

    exit( 0 );
}
