/* 
 * rlescale.c - Generate a gray-scale RLE file.
 * 
 * Author:	Spencer W. Thomas
 * 		Electrical Engineering & Computer Science Dept.
 * 		University of Michigan
 * Date:	Mon Jun 13 1988
 * Copyright (c) 1988, University of Michigan
 *
 * Usage:
 * 	rlescale [-c] [-n nsteps] [xsize] [ysize]
 *
 * Generates a xsize x ysize gray-scale RLE file (by default 480x512).
 * Has color squares along the bottom, with nsteps (default 16) gray
 * log stepped intensity gray rectangles above that.  If -c is
 * specified, does separate white, red, green, and blue scales. 
 */

#include <stdio.h>
#include <svfb_global.h>
#include <rle_getraw.h>
#include <math.h>

main( argc, argv )
char ** argv;
{
    int xsize = 512, ysize = 480, nsteps = 16, cflag = 0;
    int x, y, i, * nscan;
    rle_op ** scans;
    rle_pixel r, g, b;
    char buf[80];

    if ( scanargs( argc, argv, "% c%- n%-nsteps!d xsize%d ysize%d",
		   &cflag, &nsteps, &nsteps, &xsize, &ysize ) == 0 )
	exit( 1 );

    /* Sanity check parameters -- Runs must be at least 3 pixels long */
    if ( xsize < 3 * nsteps || xsize < 24 )
    {
	fprintf( stderr, "Image isn't wide enough for %d steps\n", nsteps );
	exit( 1 );
    }

    sv_globals.sv_ncolors = 3;
    sv_globals.sv_alpha = 0;	/* No coverage mask */
    sv_globals.sv_xmax = xsize - 1;
    sv_globals.sv_ymax = ysize - 1;
    sv_globals.svfb_fd = stdout;

    sprintf( buf, "IMAGE_TYPE=%s scale image with %d log steps",
	     cflag ? "Color" : "Gray", nsteps );  
    rle_putcom( buf, &sv_globals );

    /* Allocate storage for the output rows */
    if ( rle_raw_alloc( &sv_globals, &scans, &nscan ) )
    {
	fprintf( stderr, "rlescale: malloc failed\n" );
	exit( -2 );
    }

    /* Create the header in the output file */
    sv_setup( RUN_DISPATCH, &sv_globals );

    /* Create the scanline for the color squares */
    for ( i = 0; i < 8; i++ )
    {
	scans[0][i].u.run_val = (i & 1) ? 255 : 0;
	scans[0][i].opcode = RRunDataOp;
	scans[0][i].xloc = i * xsize / 8;
	scans[0][i].length = (i + 1) * xsize / 8 - scans[0][i].xloc;
	scans[1][i] = scans[0][i];
	scans[1][i].u.run_val = (i & 2) ? 255 : 0;
	scans[2][i] = scans[0][i];
	scans[2][i].u.run_val = (i & 4) ? 255 : 0;
    }
    nscan[0] = 8;
    nscan[1] = 8;
    nscan[2] = 8;

    /* Write the color squares */
    for ( y = 0; y < ysize / 8; y++ )
	sv_putraw( scans, nscan, &sv_globals );

    /* Create the data for the scale */
    for ( i = 0; i < nsteps; i++ )
    {
	scans[0][i].u.run_val = (int)(255.0 / pow(2.0, i*(8.0/nsteps)) + 0.5);
	scans[0][i].opcode = RRunDataOp;
	scans[0][i].xloc = i * xsize / nsteps;
	scans[0][i].length = (i + 1) * xsize / nsteps - scans[0][i].xloc;
	scans[1][i] = scans[0][i];
	scans[2][i] = scans[0][i];
    }
    nscan[0] = nsteps;
    nscan[1] = nsteps;
    nscan[2] = nsteps;

    /* Draw the scale */
    if ( !cflag )
	for ( ; y < ysize; y++ )
	    sv_putraw( scans, nscan, &sv_globals );
    else
    {
	/* blue scale */
	nscan[0] = nscan[1] = 0;
	for ( ; y < ysize * 11./32. ; y++ )
	    sv_putraw( scans, nscan, &sv_globals );
	/* green scale */
	nscan[1] = nsteps;
	nscan[2] = 0;
	for ( ; y < ysize * 18./32.; y++ )
	    sv_putraw( scans, nscan, &sv_globals );
	/* red scale */
	nscan[0] = nsteps;
	nscan[1] = 0;
	for ( ; y < ysize * 25./32.; y++ )
	    sv_putraw( scans, nscan, &sv_globals );
	/* white scale */
	nscan[1] = nscan[2] = nsteps;
	for ( ; y < ysize; y++ )
	    sv_putraw( scans, nscan, &sv_globals );
    }

    rle_raw_free( &sv_globals, nscan, scans );

    sv_puteof( &sv_globals );
}
