/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is 
 * preserved on all copies.
 * 
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the 
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/* 
 * fant.c - Perform spacial transforms on images. (should be "remap")
 * 
 * Author:	John W. Peterson
 * 		Computer Science Dept.
 * 		University of Utah
 * Date:	Wed Jun 25 1986
 * Copyright (c) 1986 John W. Peterson
 * 
 */

/* 
 * This program performs spatial transforms on images.  For full
 * details, consult the paper:
 *      Fant, Karl M. "A Nonaliasing, Real-Time, Spatial Transform
 *                     Technique", IEEE CG&A, January 1986, p. 71
 *
 * Editorial note:  This is not a particularly elegant example of toolkit
 * programming.
 */

#include <stdio.h>
#include <math.h>
#include "svfb_global.h"

#define MAXCHAN 4
#define PI 3.141592

#define and &&
#define or ||				/* C readability */
#define not !

#define H_PASS 0
#define V_PASS 1

/* Conversion macros */

#define FRAC(x) ((x) - ((int) (x)))
#define ROUND(x) ((int)((x) + 0.5))
#define DEG(x) ((x) * PI / 180.0)

#ifdef DEBUG
#define INDEX(ptr,x,y) (*arr_index(ptr,x,y))
#define FINDEX(ptr,x,y) ((float)(*arr_index(ptr,x,y)))
#else
#define INDEX(ptr,x,y) ((ptr)[array_width*(y)+(x)])
#define FINDEX(ptr,x,y) ((float)((ptr)[array_width*(y)+(x)]))
#endif

#define MIN(x,y) ((x) < (y) ? (x) : (y))
#define MAX(x,y) ((x) > (y) ? (x) : (y))

typedef struct point
{
    float x,y;
} point;

/*
 * Each channel is stored in it's own raster (an array_lines * array_width  
 * array of pixels).  This allows each channel to be transformed separately.
 * We need two copies of each set of channels so we can flip
 * between them for the two passes of the algorithm.
 */
rle_pixel * in_rast[MAXCHAN];	/* Assumes channels RGBA */
rle_pixel * out_rast[MAXCHAN];

int const_ind;			/* Constant index */
int cur_chan;			/* Which channel we're munging */
int pass;			/* Which pass we're on (h or v) */
struct sv_globals inglobals, outglobals;
int nchan;
int array_width;		/* Width of getrow line (0..sv_xmax) */
int outlinewidth, array_lines;
rle_pixel *rows[MAXCHAN];	/* Pointers for getrow/putrow */
int verboseflag;		/* Be chatty (Fant can be slow) */
int originflag;			/* Center picture on given orig instead of center */
int X_origin, Y_origin;
int vpassonlyflag;		/* If true, we only need the vertical pass */

#ifdef DEBUG
rle_pixel * arr_index(ptr,x,y)
rle_pixel * ptr;
{
    if ( x >= array_width )
    {
	fprintf( stderr, "X=%d >= array_width=%d\n", x, array_width );
	abort();
    }
    if ( x < 0 )
    {
	fprintf( stderr, "X=%d < 0\n", x );
	abort();
    }
    if ( y >= array_lines )
    {
	fprintf( stderr, "Y=%d >= array_lines=%d\n", y, array_lines );
	abort();
    }
    if ( y < 0 )
    {
	fprintf( stderr, "Y=%d < 0\n", y );
	abort();
    }
    return &ptr[array_width*(y)+(x)];
}
#endif /* DEBUG */

main(argc,argv)
int argc;
char *argv[];
{
    rle_pixel * malloc();
    char *infilename = NULL;
    float xscale, yscale, angle;
    int scaleflag, angleflag;
    int i;
    point p[5];			/* "5" so we can use 1-4 indices Fant does. */

    /* Need to get around an HP C compiler bug. */
#ifdef hpux
    int tmp_int1, tmp_int2;
#endif hpux

    xscale = 1.0; yscale = 1.0; scaleflag = 0; angleflag = 0;
    angle = 0.0;
    verboseflag = 0;
    originflag = 0;
    vpassonlyflag = 0;

    if (scanargs( argc, argv, 
"% s%-xscale!fyscale!f a%-angle!f v%- o%-xoff!dyoff!d infile%s ",
                 &scaleflag, &xscale, &yscale, &angleflag, &angle,
		 &verboseflag, &originflag, &X_origin, &Y_origin,
		 &infilename ) == 0)
	exit(-1);

    if (infilename)
    {
	inglobals.svfb_fd = fopen(infilename, "r");
	if (! inglobals.svfb_fd)
	{
	    fprintf( stderr, "fant: can't open %s\n", infilename);
	    exit(-1);
	}
    }
    else
	inglobals.svfb_fd = stdin;

    if (fabs(angle) > 45.0)
    	fprintf(stderr,
    	  "fant: Warning: angle larger than 45 degrees, image will blur.\n");

    if ((xscale == 1.0) and (angle == 0.0))
    {
	vpassonlyflag = 1;
	if (verboseflag)
	    fprintf(stderr, "fant: Only performing vertical pass\n");
    }
	
    rle_get_setup_ok( &inglobals, "fant", infilename );

    nchan = inglobals.sv_ncolors + inglobals.sv_alpha;

    outglobals = inglobals;
    outglobals.svfb_fd = stdout;

    /*
     * To define the output rectangle, we start with a set of points
     * defined by the original image, and then rotate and scale them
     * as desired.
     */
    p[1].x = inglobals.sv_xmin;		p[1].y = inglobals.sv_ymin;
    p[2].x = inglobals.sv_xmax; 	p[2].y = inglobals.sv_ymin;
    p[3].x = inglobals.sv_xmax; 	p[3].y = inglobals.sv_ymax;
    p[4].x = inglobals.sv_xmin;		p[4].y = inglobals.sv_ymax;

    xform_points(p, xscale, yscale, angle );

    outglobals = inglobals;
    outglobals.svfb_fd = stdout;

    /* Make sure the output image is large enough */

#ifdef hpux
    tmp_int1 = p[1].x;
    tmp_int2 = p[4].x;
    outglobals.sv_xmin = MAX( 0, MIN( tmp_int1, tmp_int2 ) );
    tmp_int1 = p[1].y;
    tmp_int2 = p[2].y;
    outglobals.sv_ymin = MAX( 0, MIN( tmp_int1, tmp_int2 ) );

    tmp_int1 = ROUND( p[2].x );
    tmp_int2 = ROUND( p[3].x );
    outglobals.sv_xmax = MAX( tmp_int1, tmp_int2 );
    tmp_int1 = ROUND( p[3].y );
    tmp_int2 = ROUND( p[4].y );
    outglobals.sv_ymax = MAX( tmp_int1, tmp_int2 );
#else
    outglobals.sv_xmin = MAX( 0, MIN( (int) p[1].x, (int) p[4].x ) );
    outglobals.sv_ymin = MAX( 0, MIN( (int) p[1].y, (int) p[2].y ) );

    outglobals.sv_xmax = MAX( ROUND( p[2].x ), ROUND( p[3].x ) );
    outglobals.sv_ymax = MAX( ROUND( p[3].y ), ROUND( p[4].y ) );
#endif hpux

    /*
     * Need to grab the largest dimensions so the buffers will hold the
     * picture.  The arrays for storing the pictures extend from 0 to
     * outglobals.sv_xmax in width and from outglobals.sv_ymin to ymax in
     * height.  The reason X goes from 0 to xmax is because rle_getrow
     * returns the entire row when the image is read in.
     */
    array_width = MAX( outglobals.sv_xmax, inglobals.sv_xmax ) + 1;
    array_lines = MAX( outglobals.sv_ymax - outglobals.sv_ymin,
                inglobals.sv_ymax - inglobals.sv_ymin ) + 1;
    outlinewidth = outglobals.sv_xmax - outglobals.sv_xmin + 1;

    /*
     * Since the array begins at sv_ymin, the four output corner points must be
     * translated to this coordinate system.
     */
    for (i = 1; i <= 4; i++)
	p[i].y -= MIN( inglobals.sv_ymin, outglobals.sv_ymin );

    /* Oink. */
    for (i = 0; i < nchan; i++)
    {
	in_rast[i] = malloc( array_width * array_lines );
	out_rast[i] = malloc( array_width * array_lines );
	if ((in_rast[i] == NULL) or (out_rast[i] == NULL))
	{
	    fprintf(stderr, "fant: Not enough memory for raster\n");
	    exit(-1);
	}
    }

    getraster(in_rast);

    /* Transform each channel */
    cur_chan = 0;
    for (cur_chan = 0; cur_chan < nchan; cur_chan++) 
	xform_image(p);

    putraster(out_rast);

    sv_puteof( &outglobals );
    exit( 0 );
}

/* 
 * This transforms one channel (defined by cur_chan) of the image.
 * The result image is based on the points p, using linear
 * interpolation per scanline.
 */
xform_image(p)
point *p;
{
    float real_outpos, sizefac, delta;
    rle_pixel * tmprast;
    int ystart, yend;
    int xinlen = inglobals.sv_xmax - inglobals.sv_xmin + 1;
    int yinlen = inglobals.sv_ymax - inglobals.sv_ymin + 1;

    /* Need to get around an HP C compiler bug. */
#ifdef hpux
    int tmp_int1, tmp_int2;
#endif hpux

    if (verboseflag)
	fprintf(stderr, "transforming channel %d...\n",
		cur_chan - inglobals.sv_alpha);

    /* Vertical pass */
    pass = V_PASS;
    clear_raster( out_rast );

    real_outpos = p[1].y;

    sizefac = (p[4].y-p[1].y) / (yinlen-1);
    delta = (p[2].y - p[1].y) / xinlen;

    for ( const_ind = inglobals.sv_xmin;
          const_ind <= inglobals.sv_xmax; const_ind++ )
    {
	interp_row( sizefac, 0, real_outpos,
	            inglobals.sv_ymax - inglobals.sv_ymin + 1,
		    array_lines - 1 );
	real_outpos += delta;
    }

    if (! vpassonlyflag )
    {
	/* Flip buffers */
	tmprast = in_rast[cur_chan];
	in_rast[cur_chan] = out_rast[cur_chan];
	out_rast[cur_chan] = tmprast;

	/* Horizontal pass */
	pass = H_PASS;
	clear_raster( out_rast );

	real_outpos = ( ((p[2].y - p[4].y) * (p[1].x - p[4].x))
		       / (p[1].y - p[4].y) ) + p[4].x;
	sizefac = (p[2].x - real_outpos) / (xinlen-1);
	delta = (p[4].x - real_outpos)
            / ((float) ((int) p[4].y) - ((int) p[2].y) + 1.0 );

	/* If we're moving backwards, start at p1 (needs more thought...) */
	if (delta < 0)
	    real_outpos = p[1].x;

#ifdef hpux
    	tmp_int1 = p[2].y;
    	tmp_int2 = p[1].y;
	ystart = MIN(tmp_int1, tmp_int2);
    	tmp_int1 = p[4].y;
    	tmp_int2 = p[3].y;
	yend = MAX(tmp_int1, tmp_int2);
#else
	ystart = MIN((int) p[2].y, (int) p[1].y);
	yend = MAX((int) p[4].y, (int) p[3].y);
#endif hpux

	if (ystart < 0)		/* Ensure start isn't negative */
	{
	    real_outpos += delta * abs(ystart);
	    ystart = 0;
	}

	for ( const_ind = ystart; const_ind < yend; const_ind++ )
	{
	    interp_row( sizefac, inglobals.sv_xmin, real_outpos,
		       inglobals.sv_xmax,
		       outglobals.sv_xmax );
	    real_outpos += delta;
	}
    }
}

/* 
 * Transform the points p according to xscale, yscale and angle.
 * Rotation is done first, this allows the separate scaling factors to
 * be used to adjust aspect ratios.  Note the image quality of the
 * resulting transform degrades sharply if the angle is > 45 degrees.
 */
xform_points(p, xscale, yscale, angle)
point *p;
float xscale, yscale, angle;
{
    float s, c, xoff, yoff;		
    float tmp;
    int i;

    /* Sleazy - should build real matrix */

    c = cos(DEG(angle));
    s = sin(DEG(angle));
    if (!originflag)
    {
	xoff =  ((float) (inglobals.sv_xmax - inglobals.sv_xmin) / 2.0
		 + inglobals.sv_xmin);
	yoff =  ((float) (inglobals.sv_ymax - inglobals.sv_ymin) / 2.0
		 + inglobals.sv_ymin);
    }
    else
    {
	xoff = X_origin;
	yoff = Y_origin;
    }
    if (verboseflag)
	fprintf(stderr, "Output rectangle:\n");

    for ( i = 1; i <= 4; i++ )
    {
	p[i].x -= xoff;		/* translate to origin */
	p[i].y -= yoff;

	tmp = p[i].x * c + p[i].y * s; /* Rotate... */
	p[i].y = -p[i].x * s + p[i].y * c;
	p[i].x = tmp;

	p[i].x *= xscale;	/* Scale */
	p[i].y *= yscale;

	p[i].x += ( xoff );	/* translate back from origin */
	p[i].y += ( yoff );

	if (verboseflag)
	    fprintf(stderr, "  %4.1f\t%4.1f\n", p[i].x, p[i].y);
    }
}

/* 
 * Interpolate a row or column pixels.  This is a straightforward
 * (floating point) implementation of the paper in the algorithm.
 * Sizefac is the amount the row is scaled by, ras_strt is the
 * position to start in the input raster, and real_outpos is the place
 * (in floating point coordinates) to output pixels.  The algorithm
 * loops until it's read pixels up to inmax or output up to outmax.
 */
interp_row (sizefac, ras_strt, real_outpos, inmax, outmax)
float sizefac;
int ras_strt;
float real_outpos;
int inmax, outmax;
{
    float inoff, inseg, outseg, accum, insfac, pv;
    int inptr, outptr;
    float getpxl();

    insfac = 1.0 / sizefac;

    if (real_outpos > 0.0)
    {
	inseg = 1.0;
	outseg = insfac * (1.0 - FRAC(real_outpos));
	outptr = (int) real_outpos;
	inptr = ras_strt;
    }
    else				/* Must clip */
    {
	inoff = -real_outpos * insfac;
	inseg = 1.0 - FRAC( inoff );
	outseg = insfac;
	inptr = ras_strt + ((int) inoff);
	outptr = 0;
    }
    accum = 0.0;

    while (( inptr < inmax) and (outptr < outmax))
    {
	pv = getpxl( inptr, inseg );

	if ( outseg > inseg )
	{
	    accum += pv * inseg;
	    outseg -= inseg;
	    inseg = 1.0;
	    inptr++;
	}
	else
	{
	    accum += pv * outseg;
	    inseg -= outseg;
	    outseg = insfac;
	    putpxl( outptr, accum * sizefac );
	    outptr++;
	    accum = 0.0;
	}
    }

    /* If we ran out of input, output partially completed pixel */
    if (outptr <= outmax)
	putpxl( outptr, accum * sizefac );
}

/*
 * Grab inseg's worth of pixels from the current input raster.  If inseg
 * is less than one, we perform linear interpolation (Fant's "expansion
 * smoothing") between adjacent pixels.
 *
 * Pixel indexing is determined by two globals; cur_chan, which determines
 * which raster we read from, and pass, which tells if we're indexing rows
 * or columns.
 */
float
getpxl(index, inseg)
int index;
float inseg;
{
    float result;

    if (pass == V_PASS)
    {
	if (index < inglobals.sv_ymax)
	    result = inseg * FINDEX(in_rast[cur_chan], const_ind, index)
	             + (1.0-inseg)
		     * FINDEX(in_rast[cur_chan], const_ind, index + 1);
	else
	    result = inseg * FINDEX(in_rast[cur_chan], const_ind, index);
    }
    else
    {
	if (index < inglobals.sv_xmax)
	    result = inseg * FINDEX(in_rast[cur_chan], index, const_ind)
	             + (1.0-inseg)
		     * FINDEX(in_rast[cur_chan], index + 1, const_ind);
	else
	    result = inseg * FINDEX(in_rast[cur_chan], index, const_ind);
    }
    return ( result );
}

/*
 * Write a pixel into the current output channel.
 */
putpxl(index, pxlval)
int index;
float pxlval;
{
    pxlval = ROUND(pxlval);
    if (pxlval > 255)
    {
	fprintf(stderr, "pixel overflow: %f at %d %d\n",
	       pxlval, ((pass==V_PASS) ? const_ind : index),
	       ((pass==V_PASS) ? index : const_ind));
	pxlval = 255;
    }
    if (pass == V_PASS)
	INDEX(out_rast[cur_chan], const_ind, index) = (int) pxlval;
    else
	INDEX(out_rast[cur_chan], index, const_ind) = (int) pxlval;
}

/*
 * Read all channels of the picture in, placing each channel directly
 * into a separate array.
 */
getraster(ras_ptrs)
rle_pixel *ras_ptrs[];
{
    int i, chan;
    rle_pixel *ptrs[MAXCHAN];

    for ( chan = 0; chan < nchan; chan++ )
	ptrs[chan] = ras_ptrs[chan];

    for (i = inglobals.sv_ymin; i <= inglobals.sv_ymax; i++)
    {
	for ( chan = 0; chan < nchan; chan++ )
	    rows[chan] = ptrs[chan];
	rle_getrow( &inglobals, &(rows[inglobals.sv_alpha]) );
	

	/* Bump pointers */
	for ( chan = 0; chan < nchan; chan++ )
	    ptrs[chan] = &((ptrs[chan])[array_width]);	
    }
}

/* Write out the rasters */
putraster(ras_ptrs)
rle_pixel *ras_ptrs[];
{
    int i, chan;
    rle_pixel *ptrs[MAXCHAN];

    sv_setup( RUN_DISPATCH, &outglobals );

    /* 
     * If the output image is smaller than the input, we must offset
     * into the pixel array by the difference between the two.
     */
    if (inglobals.sv_ymin < outglobals.sv_ymin)
	for ( chan = 0; chan < nchan; chan++ )
	    ptrs[chan] = &((ras_ptrs[chan])[array_width
	                      * (outglobals.sv_ymin - inglobals.sv_ymin)]);
    else 
	for ( chan = 0; chan < nchan; chan++ )
	    ptrs[chan] = ras_ptrs[chan];

    for (i = outglobals.sv_ymin; i <= outglobals.sv_ymax; i++)
    {
	for ( chan = 0; chan < nchan; chan++ )
	    rows[chan] = &((ptrs[chan])[outglobals.sv_xmin]);
	sv_putrow( &(rows[outglobals.sv_alpha]), outlinewidth, &outglobals );

	/* Bump pointers */
	for ( chan = 0; chan < nchan; chan++ )
	    ptrs[chan] = &((ptrs[chan])[array_width]);	
    }
}

/* Clear the raster's cur_chan channel */
clear_raster(ras_ptr)
rle_pixel *ras_ptr[];
{
    bzero( ras_ptr[cur_chan], array_width * array_lines );
}

/*
 * Dump out a raster (used for debugging).
 */
dumpraster(ras_ptrs)
rle_pixel *ras_ptrs[];
{
    int j, i, chan;
    for (i = inglobals.sv_ymin; i <= inglobals.sv_ymax; i++)
    {
	for (j=inglobals.sv_xmin; j <= inglobals.sv_xmax; j++)
	{
	    for ( chan = 0; chan < nchan; chan++ )
		fprintf(stderr, "%2x", (ras_ptrs[chan])[i * array_width + j ]);
	    fprintf(stderr," ");
	}
	fprintf(stderr,"\n");
    }
}

